/*
 * Decompiled with CFR 0.152.
 */
package io.trino.plugin.jdbc;

import com.google.common.base.Ticker;
import com.google.common.cache.CacheStats;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.ImmutableMap;
import com.google.common.collect.MoreCollectors;
import com.google.common.util.concurrent.Futures;
import io.airlift.concurrent.Threads;
import io.airlift.testing.TestingTicker;
import io.airlift.units.Duration;
import io.trino.plugin.base.session.SessionPropertiesProvider;
import io.trino.plugin.jdbc.BaseJdbcConfig;
import io.trino.plugin.jdbc.CachingJdbcClient;
import io.trino.plugin.jdbc.ExtraCredentialsBasedIdentityCacheMapping;
import io.trino.plugin.jdbc.ForwardingJdbcClient;
import io.trino.plugin.jdbc.IdentityCacheMapping;
import io.trino.plugin.jdbc.JdbcClient;
import io.trino.plugin.jdbc.JdbcColumnHandle;
import io.trino.plugin.jdbc.JdbcProcedureHandle;
import io.trino.plugin.jdbc.JdbcQueryRelationHandle;
import io.trino.plugin.jdbc.JdbcRelationHandle;
import io.trino.plugin.jdbc.JdbcTableHandle;
import io.trino.plugin.jdbc.PreparedQuery;
import io.trino.plugin.jdbc.SingletonIdentityCacheMapping;
import io.trino.plugin.jdbc.TestingDatabase;
import io.trino.plugin.jdbc.credential.ExtraCredentialConfig;
import io.trino.spi.connector.ColumnMetadata;
import io.trino.spi.connector.ConnectorSession;
import io.trino.spi.connector.ConnectorTableMetadata;
import io.trino.spi.connector.SchemaTableName;
import io.trino.spi.connector.TableNotFoundException;
import io.trino.spi.predicate.TupleDomain;
import io.trino.spi.security.ConnectorIdentity;
import io.trino.spi.session.PropertyMetadata;
import io.trino.spi.statistics.Estimate;
import io.trino.spi.statistics.TableStatistics;
import io.trino.spi.testing.InterfaceTestUtils;
import io.trino.spi.type.IntegerType;
import io.trino.spi.type.Type;
import io.trino.testing.TestingConnectorSession;
import io.trino.testing.TestingNames;
import java.sql.Connection;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Statement;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.OptionalLong;
import java.util.Set;
import java.util.concurrent.Callable;
import java.util.concurrent.CyclicBarrier;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.function.Function;
import java.util.stream.Stream;
import org.assertj.core.api.AbstractCollectionAssert;
import org.assertj.core.api.AbstractLongAssert;
import org.assertj.core.api.Assertions;
import org.assertj.core.api.ListAssert;
import org.junit.jupiter.api.AfterEach;
import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;
import org.junit.jupiter.api.TestInstance;
import org.junit.jupiter.api.Timeout;

@TestInstance(value=TestInstance.Lifecycle.PER_METHOD)
public class TestCachingJdbcClient {
    private static final Duration FOREVER = new Duration(1.0, TimeUnit.DAYS);
    private static final ImmutableList<PropertyMetadata<?>> PROPERTY_METADATA = ImmutableList.of((Object)PropertyMetadata.stringProperty((String)"session_name", (String)"Session name", null, (boolean)false));
    private static final Set<SessionPropertiesProvider> SESSION_PROPERTIES_PROVIDERS = Set.of(() -> PROPERTY_METADATA);
    private static final ConnectorSession SESSION = TestingConnectorSession.builder().setPropertyMetadata(PROPERTY_METADATA).build();
    private static final TableStatistics NON_EMPTY_STATS = TableStatistics.builder().setRowCount(Estimate.zero()).build();
    private TestingDatabase database;
    private JdbcClient jdbcClient;
    private String schema;
    private ExecutorService executor;

    @BeforeEach
    public void setUp() throws Exception {
        this.database = new TestingDatabase();
        this.jdbcClient = this.database.getJdbcClient();
        this.schema = (String)this.jdbcClient.getSchemaNames(SESSION).iterator().next();
        this.executor = Executors.newCachedThreadPool(Threads.daemonThreadsNamed((String)"TestCachingJdbcClient-%s"));
    }

    @AfterEach
    public void tearDown() throws Exception {
        this.executor.shutdownNow();
        this.executor = null;
        this.database.close();
        this.database = null;
    }

    @Test
    public void testSchemaNamesCached() {
        CachingJdbcClient cachingJdbcClient = this.createCachingJdbcClient();
        String phantomSchema = "phantom_schema";
        this.jdbcClient.createSchema(SESSION, phantomSchema);
        TestCachingJdbcClient.assertSchemaNamesCache(cachingJdbcClient).misses(1L).loads(1L).afterRunning(() -> Assertions.assertThat((Collection)cachingJdbcClient.getSchemaNames(SESSION)).contains((Object[])new String[]{phantomSchema}));
        this.jdbcClient.dropSchema(SESSION, phantomSchema, false);
        Assertions.assertThat((Collection)this.jdbcClient.getSchemaNames(SESSION)).doesNotContain((Object[])new String[]{phantomSchema});
        TestCachingJdbcClient.assertSchemaNamesCache(cachingJdbcClient).hits(1L).afterRunning(() -> Assertions.assertThat((Collection)cachingJdbcClient.getSchemaNames(SESSION)).contains((Object[])new String[]{phantomSchema}));
    }

    @Test
    public void testTableNamesCached() {
        CachingJdbcClient cachingJdbcClient = this.createCachingJdbcClient();
        SchemaTableName phantomTable = new SchemaTableName(this.schema, "phantom_table");
        this.createTable(phantomTable);
        TestCachingJdbcClient.assertTableNamesCache(cachingJdbcClient).misses(1L).loads(1L).afterRunning(() -> Assertions.assertThat((List)cachingJdbcClient.getTableNames(SESSION, Optional.of(this.schema))).contains((Object[])new SchemaTableName[]{phantomTable}));
        this.dropTable(phantomTable);
        Assertions.assertThat((List)this.jdbcClient.getTableNames(SESSION, Optional.of(this.schema))).doesNotContain((Object[])new SchemaTableName[]{phantomTable});
        TestCachingJdbcClient.assertTableNamesCache(cachingJdbcClient).hits(1L).afterRunning(() -> Assertions.assertThat((List)cachingJdbcClient.getTableNames(SESSION, Optional.of(this.schema))).contains((Object[])new SchemaTableName[]{phantomTable}));
    }

    @Test
    public void testTableHandleCached() {
        CachingJdbcClient cachingJdbcClient = this.createCachingJdbcClient();
        SchemaTableName phantomTable = new SchemaTableName(this.schema, "phantom_table");
        this.createTable(phantomTable);
        Optional cachedTable = cachingJdbcClient.getTableHandle(SESSION, phantomTable);
        this.dropTable(phantomTable);
        Assertions.assertThat((Optional)this.jdbcClient.getTableHandle(SESSION, phantomTable)).isEmpty();
        Assertions.assertThat((Optional)cachingJdbcClient.getTableHandle(SESSION, phantomTable)).isEqualTo((Object)cachedTable);
    }

    @Test
    public void testTableHandleOfQueryCached() throws Exception {
        CachingJdbcClient cachingJdbcClient = this.cachingClientBuilder().delegate(this.jdbcClientWithTableStats()).build();
        SchemaTableName phantomTable = new SchemaTableName(this.schema, "phantom_table");
        this.createTable(phantomTable);
        PreparedQuery query = new PreparedQuery(String.format("SELECT * FROM %s.phantom_table", this.schema), (List)ImmutableList.of());
        JdbcTableHandle cachedTable = TestCachingJdbcClient.assertTableHandleByQueryCache(cachingJdbcClient).misses(1L).loads(1L).calling(() -> cachingJdbcClient.getTableHandle(SESSION, query));
        TestCachingJdbcClient.assertCacheStats(cachingJdbcClient).afterRunning(() -> cachingJdbcClient.getColumns(SESSION, cachedTable));
        TestCachingJdbcClient.assertStatisticsCacheStats(cachingJdbcClient).misses(1L).loads(1L).afterRunning(() -> cachingJdbcClient.getTableStatistics(SESSION, cachedTable));
        this.dropTable(phantomTable);
        Assertions.assertThatThrownBy(() -> this.jdbcClient.getTableHandle(SESSION, query)).hasMessageContaining("Failed to get table handle for prepared query");
        TestCachingJdbcClient.assertTableHandleByQueryCache(cachingJdbcClient).hits(1L).afterRunning(() -> {
            Assertions.assertThat((Object)cachingJdbcClient.getTableHandle(SESSION, query)).isEqualTo((Object)cachedTable);
            Assertions.assertThat((List)cachingJdbcClient.getColumns(SESSION, cachedTable)).hasSize(0);
        });
        TestCachingJdbcClient.assertCacheStats(cachingJdbcClient).afterRunning(() -> Assertions.assertThat((List)cachingJdbcClient.getColumns(SESSION, cachedTable)).hasSize(0));
        TestCachingJdbcClient.assertStatisticsCacheStats(cachingJdbcClient).hits(1L).afterRunning(() -> cachingJdbcClient.getTableStatistics(SESSION, cachedTable));
        cachingJdbcClient.createTable(SESSION, new ConnectorTableMetadata(phantomTable, Collections.emptyList()));
        TestCachingJdbcClient.assertTableHandleByQueryCache(cachingJdbcClient).misses(1L).loads(1L).afterRunning(() -> {
            Assertions.assertThat((Object)cachingJdbcClient.getTableHandle(SESSION, query)).isEqualTo((Object)cachedTable);
            Assertions.assertThat((List)cachingJdbcClient.getColumns(SESSION, cachedTable)).hasSize(0);
        });
        TestCachingJdbcClient.assertCacheStats(cachingJdbcClient).afterRunning(() -> Assertions.assertThat((List)cachingJdbcClient.getColumns(SESSION, cachedTable)).hasSize(0));
        TestCachingJdbcClient.assertStatisticsCacheStats(cachingJdbcClient).misses(1L).loads(1L).afterRunning(() -> cachingJdbcClient.getTableStatistics(SESSION, cachedTable));
        cachingJdbcClient.onDataChanged(phantomTable);
        TestCachingJdbcClient.assertTableHandleByQueryCache(cachingJdbcClient).hits(1L).afterRunning(() -> {
            Assertions.assertThat((Object)cachingJdbcClient.getTableHandle(SESSION, query)).isEqualTo((Object)cachedTable);
            Assertions.assertThat((List)cachingJdbcClient.getColumns(SESSION, cachedTable)).hasSize(0);
        });
        TestCachingJdbcClient.assertCacheStats(cachingJdbcClient).afterRunning(() -> Assertions.assertThat((List)cachingJdbcClient.getColumns(SESSION, cachedTable)).hasSize(0));
        TestCachingJdbcClient.assertStatisticsCacheStats(cachingJdbcClient).misses(1L).loads(1L).afterRunning(() -> cachingJdbcClient.getTableStatistics(SESSION, cachedTable));
        this.dropTable(phantomTable);
    }

    @Test
    public void testProcedureHandleCached() throws Exception {
        CachingJdbcClient cachingJdbcClient = this.createCachingJdbcClient();
        SchemaTableName phantomTable = new SchemaTableName(this.schema, "phantom_table");
        this.createTable(phantomTable);
        this.createProcedure("test_procedure");
        JdbcProcedureHandle.ProcedureQuery query = new JdbcProcedureHandle.ProcedureQuery("CALL %s.test_procedure ('%s')".formatted(this.schema, phantomTable));
        JdbcProcedureHandle cachedProcedure = TestCachingJdbcClient.assertProcedureHandleByQueryCache(cachingJdbcClient).misses(1L).loads(1L).calling(() -> cachingJdbcClient.getProcedureHandle(SESSION, query));
        Assertions.assertThat((List)((List)cachedProcedure.getColumns().orElseThrow())).hasSize(0);
        this.dropProcedure("test_procedure");
        Assertions.assertThatThrownBy(() -> this.jdbcClient.getProcedureHandle(SESSION, query)).hasMessageContaining("Failed to get table handle for procedure query");
        TestCachingJdbcClient.assertProcedureHandleByQueryCache(cachingJdbcClient).hits(1L).afterRunning(() -> Assertions.assertThat((Object)cachingJdbcClient.getProcedureHandle(SESSION, query)).isEqualTo((Object)cachedProcedure));
        this.dropTable(phantomTable);
    }

    @Test
    public void testTableHandleInvalidatedOnColumnsModifications() {
        CachingJdbcClient cachingJdbcClient = this.createCachingJdbcClient();
        JdbcTableHandle table = this.createTable(new SchemaTableName(this.schema, "a_table"));
        JdbcColumnHandle existingColumn = this.addColumn(table, "a_column");
        this.assertTableHandlesByNameCacheIsInvalidated(cachingJdbcClient, table);
        JdbcColumnHandle newColumn = this.addColumn((JdbcClient)cachingJdbcClient, table, "new_column");
        this.assertTableHandlesByNameCacheIsInvalidated(cachingJdbcClient, table);
        cachingJdbcClient.setColumnComment(SESSION, table, newColumn, Optional.empty());
        this.assertTableHandlesByNameCacheIsInvalidated(cachingJdbcClient, table);
        cachingJdbcClient.renameColumn(SESSION, table, newColumn, "new_column_name");
        this.assertTableHandlesByNameCacheIsInvalidated(cachingJdbcClient, table);
        cachingJdbcClient.dropColumn(SESSION, table, existingColumn);
        this.assertTableHandlesByNameCacheIsInvalidated(cachingJdbcClient, table);
        this.dropTable(table);
    }

    private void assertTableHandlesByNameCacheIsInvalidated(CachingJdbcClient cachingJdbcClient, JdbcTableHandle table) {
        SchemaTableName tableName = table.asPlainTable().getSchemaTableName();
        TestCachingJdbcClient.assertTableHandleByNameCache(cachingJdbcClient).misses(1L).loads(1L).afterRunning(() -> Assertions.assertThat((Object)((JdbcTableHandle)cachingJdbcClient.getTableHandle(SESSION, tableName).orElseThrow())).isEqualTo((Object)table));
        TestCachingJdbcClient.assertTableHandleByNameCache(cachingJdbcClient).hits(1L).afterRunning(() -> Assertions.assertThat((Object)((JdbcTableHandle)cachingJdbcClient.getTableHandle(SESSION, tableName).orElseThrow())).isEqualTo((Object)table));
    }

    @Test
    public void testEmptyTableHandleIsCachedWhenCacheMissingIsTrue() {
        CachingJdbcClient cachingJdbcClient = this.cachingClientBuilder().config(TestCachingJdbcClient.enableCache().setCacheMissing(true)).build();
        SchemaTableName phantomTable = new SchemaTableName(this.schema, "phantom_table");
        Assertions.assertThat((Optional)cachingJdbcClient.getTableHandle(SESSION, phantomTable)).isEmpty();
        this.createTable(phantomTable);
        Assertions.assertThat((Optional)cachingJdbcClient.getTableHandle(SESSION, phantomTable)).isEmpty();
        this.dropTable(phantomTable);
    }

    @Test
    public void testEmptyTableHandleNotCachedWhenCacheMissingIsFalse() {
        CachingJdbcClient cachingJdbcClient = this.cachingClientBuilder().config(TestCachingJdbcClient.enableCache().setCacheMissing(false)).build();
        SchemaTableName phantomTable = new SchemaTableName(this.schema, "phantom_table");
        Assertions.assertThat((Optional)cachingJdbcClient.getTableHandle(SESSION, phantomTable)).isEmpty();
        this.createTable(phantomTable);
        Assertions.assertThat((Optional)cachingJdbcClient.getTableHandle(SESSION, phantomTable)).isPresent();
        this.dropTable(phantomTable);
    }

    private JdbcTableHandle createTable(SchemaTableName phantomTable) {
        this.jdbcClient.createTable(SESSION, new ConnectorTableMetadata(phantomTable, Collections.emptyList()));
        return (JdbcTableHandle)this.jdbcClient.getTableHandle(SESSION, phantomTable).orElseThrow();
    }

    private void createProcedure(String procedureName) throws SQLException {
        try (Statement statement = this.database.getConnection().createStatement();){
            statement.execute("CREATE ALIAS %s.%s FOR \"io.trino.plugin.jdbc.TestCachingJdbcClient.generateData\"".formatted(this.schema, procedureName));
        }
    }

    private void dropProcedure(String procedureName) throws SQLException {
        try (Statement statement = this.database.getConnection().createStatement();){
            statement.execute("DROP ALIAS %s.%s".formatted(this.schema, procedureName));
        }
    }

    public static ResultSet generateData(Connection connection, String table) throws SQLException {
        return connection.createStatement().executeQuery("SELECT * FROM " + table);
    }

    private void dropTable(JdbcTableHandle tableHandle) {
        this.jdbcClient.dropTable(SESSION, tableHandle);
    }

    private void dropTable(SchemaTableName phantomTable) {
        JdbcTableHandle tableHandle = (JdbcTableHandle)this.jdbcClient.getTableHandle(SESSION, phantomTable).orElseThrow();
        this.jdbcClient.dropTable(SESSION, tableHandle);
    }

    @Test
    public void testColumnsCached() {
        CachingJdbcClient cachingJdbcClient = this.createCachingJdbcClient();
        JdbcTableHandle table = this.getAnyTable(this.schema);
        JdbcColumnHandle phantomColumn = this.addColumn(table);
        TestCachingJdbcClient.assertColumnCacheStats(cachingJdbcClient).loads(1L).misses(1L).afterRunning(() -> Assertions.assertThat((List)cachingJdbcClient.getColumns(SESSION, table)).contains((Object[])new JdbcColumnHandle[]{phantomColumn}));
        this.jdbcClient.dropColumn(SESSION, table, phantomColumn);
        Assertions.assertThat((List)this.jdbcClient.getColumns(SESSION, table)).doesNotContain((Object[])new JdbcColumnHandle[]{phantomColumn});
        TestCachingJdbcClient.assertColumnCacheStats(cachingJdbcClient).hits(1L).afterRunning(() -> Assertions.assertThat((List)cachingJdbcClient.getColumns(SESSION, table)).contains((Object[])new JdbcColumnHandle[]{phantomColumn}));
    }

    @Test
    public void testColumnsCachedPerSession() {
        CachingJdbcClient cachingJdbcClient = this.createCachingJdbcClient();
        ConnectorSession firstSession = TestCachingJdbcClient.createSession("first");
        ConnectorSession secondSession = TestCachingJdbcClient.createSession("second");
        JdbcTableHandle table = this.getAnyTable(this.schema);
        JdbcColumnHandle phantomColumn = this.addColumn(table);
        TestCachingJdbcClient.assertColumnCacheStats(cachingJdbcClient).loads(1L).misses(1L).afterRunning(() -> Assertions.assertThat((List)cachingJdbcClient.getColumns(firstSession, table)).contains((Object[])new JdbcColumnHandle[]{phantomColumn}));
        TestCachingJdbcClient.assertColumnCacheStats(cachingJdbcClient).loads(1L).misses(1L).afterRunning(() -> Assertions.assertThat((List)cachingJdbcClient.getColumns(secondSession, table)).contains((Object[])new JdbcColumnHandle[]{phantomColumn}));
        TestCachingJdbcClient.assertColumnCacheStats(cachingJdbcClient).hits(1L).afterRunning(() -> Assertions.assertThat((List)cachingJdbcClient.getColumns(secondSession, table)).contains((Object[])new JdbcColumnHandle[]{phantomColumn}));
        cachingJdbcClient.dropColumn(firstSession, table, phantomColumn);
        Assertions.assertThat((List)this.jdbcClient.getColumns(firstSession, table)).doesNotContain((Object[])new JdbcColumnHandle[]{phantomColumn});
        TestCachingJdbcClient.assertColumnCacheStats(cachingJdbcClient).loads(2L).misses(2L).afterRunning(() -> {
            Assertions.assertThat((List)cachingJdbcClient.getColumns(firstSession, table)).doesNotContain((Object[])new JdbcColumnHandle[]{phantomColumn});
            Assertions.assertThat((List)cachingJdbcClient.getColumns(secondSession, table)).doesNotContain((Object[])new JdbcColumnHandle[]{phantomColumn});
        });
        TestCachingJdbcClient.assertColumnCacheStats(cachingJdbcClient).hits(2L).afterRunning(() -> {
            Assertions.assertThat((List)cachingJdbcClient.getColumns(firstSession, table)).doesNotContain((Object[])new JdbcColumnHandle[]{phantomColumn});
            Assertions.assertThat((List)cachingJdbcClient.getColumns(secondSession, table)).doesNotContain((Object[])new JdbcColumnHandle[]{phantomColumn});
        });
    }

    @Test
    public void testColumnsCacheInvalidationOnTableDrop() {
        CachingJdbcClient cachingJdbcClient = this.createCachingJdbcClient();
        ConnectorSession firstSession = TestCachingJdbcClient.createSession("first");
        ConnectorSession secondSession = TestCachingJdbcClient.createSession("second");
        JdbcTableHandle firstTable = this.createTable(new SchemaTableName(this.schema, "first_table"));
        JdbcTableHandle secondTable = this.createTable(new SchemaTableName(this.schema, "second_table"));
        JdbcColumnHandle firstColumn = this.addColumn(firstTable, "first_column");
        JdbcColumnHandle secondColumn = this.addColumn(secondTable, "second_column");
        TestCachingJdbcClient.assertColumnCacheStats(cachingJdbcClient).loads(4L).misses(4L).afterRunning(() -> {
            Assertions.assertThat((List)cachingJdbcClient.getColumns(firstSession, firstTable)).contains((Object[])new JdbcColumnHandle[]{firstColumn});
            Assertions.assertThat((List)cachingJdbcClient.getColumns(firstSession, secondTable)).contains((Object[])new JdbcColumnHandle[]{secondColumn});
            Assertions.assertThat((List)cachingJdbcClient.getColumns(secondSession, firstTable)).contains((Object[])new JdbcColumnHandle[]{firstColumn});
            Assertions.assertThat((List)cachingJdbcClient.getColumns(secondSession, secondTable)).contains((Object[])new JdbcColumnHandle[]{secondColumn});
        });
        TestCachingJdbcClient.assertColumnCacheStats(cachingJdbcClient).hits(2L).afterRunning(() -> {
            Assertions.assertThat((List)cachingJdbcClient.getColumns(firstSession, firstTable)).contains((Object[])new JdbcColumnHandle[]{firstColumn});
            Assertions.assertThat((List)cachingJdbcClient.getColumns(secondSession, secondTable)).contains((Object[])new JdbcColumnHandle[]{secondColumn});
        });
        cachingJdbcClient.renameColumn(firstSession, firstTable, firstColumn, "another_column");
        TestCachingJdbcClient.assertColumnCacheStats(cachingJdbcClient).loads(1L).misses(1L).afterRunning(() -> ((ListAssert)Assertions.assertThat((List)cachingJdbcClient.getColumns(secondSession, firstTable)).doesNotContain((Object[])new JdbcColumnHandle[]{firstColumn})).containsAll((Iterable)this.jdbcClient.getColumns(SESSION, firstTable)));
        cachingJdbcClient.dropTable(secondSession, firstTable);
        TestCachingJdbcClient.assertColumnCacheStats(cachingJdbcClient).loads(2L).misses(2L).afterRunning(() -> {
            Assertions.assertThatThrownBy(() -> cachingJdbcClient.getColumns(firstSession, firstTable)).isInstanceOf(TableNotFoundException.class);
            Assertions.assertThatThrownBy(() -> cachingJdbcClient.getColumns(secondSession, firstTable)).isInstanceOf(TableNotFoundException.class);
        });
        TestCachingJdbcClient.assertColumnCacheStats(cachingJdbcClient).hits(2L).afterRunning(() -> {
            Assertions.assertThat((List)cachingJdbcClient.getColumns(firstSession, secondTable)).contains((Object[])new JdbcColumnHandle[]{secondColumn});
            Assertions.assertThat((List)cachingJdbcClient.getColumns(secondSession, secondTable)).contains((Object[])new JdbcColumnHandle[]{secondColumn});
        });
        cachingJdbcClient.dropTable(secondSession, secondTable);
    }

    @Test
    public void testColumnsNotCachedWhenCacheDisabled() {
        CachingJdbcClient cachingJdbcClient = this.cachingClientBuilder().config(new BaseJdbcConfig().setMetadataCacheTtl(Duration.ZERO)).build();
        ConnectorSession firstSession = TestCachingJdbcClient.createSession("first");
        ConnectorSession secondSession = TestCachingJdbcClient.createSession("second");
        JdbcTableHandle firstTable = this.createTable(new SchemaTableName(this.schema, "first_table"));
        JdbcTableHandle secondTable = this.createTable(new SchemaTableName(this.schema, "second_table"));
        JdbcColumnHandle firstColumn = this.addColumn(firstTable, "first_column");
        JdbcColumnHandle secondColumn = this.addColumn(secondTable, "second_column");
        TestCachingJdbcClient.assertColumnCacheStats(cachingJdbcClient).loads(4L).misses(4L).afterRunning(() -> {
            Assertions.assertThat((List)cachingJdbcClient.getColumns(firstSession, firstTable)).containsExactly((Object[])new JdbcColumnHandle[]{firstColumn});
            Assertions.assertThat((List)cachingJdbcClient.getColumns(secondSession, firstTable)).containsExactly((Object[])new JdbcColumnHandle[]{firstColumn});
            Assertions.assertThat((List)cachingJdbcClient.getColumns(firstSession, secondTable)).containsExactly((Object[])new JdbcColumnHandle[]{secondColumn});
            Assertions.assertThat((List)cachingJdbcClient.getColumns(secondSession, secondTable)).containsExactly((Object[])new JdbcColumnHandle[]{secondColumn});
        });
        TestCachingJdbcClient.assertColumnCacheStats(cachingJdbcClient).loads(4L).misses(4L).afterRunning(() -> {
            Assertions.assertThat((List)cachingJdbcClient.getColumns(firstSession, firstTable)).containsExactly((Object[])new JdbcColumnHandle[]{firstColumn});
            Assertions.assertThat((List)cachingJdbcClient.getColumns(secondSession, firstTable)).containsExactly((Object[])new JdbcColumnHandle[]{firstColumn});
            Assertions.assertThat((List)cachingJdbcClient.getColumns(firstSession, secondTable)).containsExactly((Object[])new JdbcColumnHandle[]{secondColumn});
            Assertions.assertThat((List)cachingJdbcClient.getColumns(secondSession, secondTable)).containsExactly((Object[])new JdbcColumnHandle[]{secondColumn});
        });
        this.jdbcClient.dropTable(SESSION, firstTable);
        this.jdbcClient.dropTable(SESSION, secondTable);
        TestCachingJdbcClient.assertColumnCacheStats(cachingJdbcClient).loads(2L).misses(2L).afterRunning(() -> {
            Assertions.assertThatThrownBy(() -> cachingJdbcClient.getColumns(firstSession, firstTable)).isInstanceOf(TableNotFoundException.class);
            Assertions.assertThatThrownBy(() -> cachingJdbcClient.getColumns(firstSession, secondTable)).isInstanceOf(TableNotFoundException.class);
        });
    }

    @Test
    public void testGetTableStatistics() {
        CachingJdbcClient cachingJdbcClient = this.cachingClientBuilder().delegate(this.jdbcClientWithTableStats()).build();
        ConnectorSession session = TestCachingJdbcClient.createSession("first");
        JdbcTableHandle first = this.createTable(new SchemaTableName(this.schema, "first"));
        JdbcTableHandle second = this.createTable(new SchemaTableName(this.schema, "second"));
        TestCachingJdbcClient.assertStatisticsCacheStats(cachingJdbcClient).loads(1L).misses(1L).afterRunning(() -> Assertions.assertThat((Object)cachingJdbcClient.getTableStatistics(session, first)).isEqualTo((Object)NON_EMPTY_STATS));
        TestCachingJdbcClient.assertStatisticsCacheStats(cachingJdbcClient).hits(1L).afterRunning(() -> Assertions.assertThat((Object)cachingJdbcClient.getTableStatistics(session, first)).isEqualTo((Object)NON_EMPTY_STATS));
        TestCachingJdbcClient.assertStatisticsCacheStats(cachingJdbcClient).loads(1L).misses(1L).afterRunning(() -> Assertions.assertThat((Object)cachingJdbcClient.getTableStatistics(session, second)).isEqualTo((Object)NON_EMPTY_STATS));
        TestCachingJdbcClient.assertStatisticsCacheStats(cachingJdbcClient).hits(1L).afterRunning(() -> Assertions.assertThat((Object)cachingJdbcClient.getTableStatistics(session, first)).isEqualTo((Object)NON_EMPTY_STATS));
        cachingJdbcClient.dropTable(SESSION, first);
        JdbcTableHandle secondFirst = this.createTable(new SchemaTableName(this.schema, "first"));
        TestCachingJdbcClient.assertStatisticsCacheStats(cachingJdbcClient).loads(1L).misses(1L).afterRunning(() -> Assertions.assertThat((Object)cachingJdbcClient.getTableStatistics(session, secondFirst)).isEqualTo((Object)NON_EMPTY_STATS));
        TestCachingJdbcClient.assertStatisticsCacheStats(cachingJdbcClient).hits(1L).afterRunning(() -> Assertions.assertThat((Object)cachingJdbcClient.getTableStatistics(session, secondFirst)).isEqualTo((Object)NON_EMPTY_STATS));
        this.jdbcClient.dropTable(SESSION, first);
        this.jdbcClient.dropTable(SESSION, second);
    }

    @Test
    public void testCacheGetTableStatisticsWithQueryRelationHandle() {
        CachingJdbcClient cachingJdbcClient = this.cachingClientBuilder().delegate(this.jdbcClientWithTableStats()).build();
        ConnectorSession session = TestCachingJdbcClient.createSession("some test session name");
        JdbcTableHandle first = this.createTable(new SchemaTableName(this.schema, "first"));
        JdbcTableHandle second = this.createTable(new SchemaTableName(this.schema, "second"));
        JdbcTableHandle queryOnFirst = new JdbcTableHandle((JdbcRelationHandle)new JdbcQueryRelationHandle(new PreparedQuery("SELECT * FROM first", List.of())), TupleDomain.all(), (List)ImmutableList.of(), Optional.empty(), OptionalLong.empty(), Optional.empty(), Optional.of(Set.of(new SchemaTableName(this.schema, "first"))), 0, Optional.empty(), (List)ImmutableList.of());
        TestCachingJdbcClient.assertStatisticsCacheStats(cachingJdbcClient).loads(1L).misses(1L).afterRunning(() -> Assertions.assertThat((Object)cachingJdbcClient.getTableStatistics(session, queryOnFirst)).isEqualTo((Object)NON_EMPTY_STATS));
        TestCachingJdbcClient.assertStatisticsCacheStats(cachingJdbcClient).hits(1L).afterRunning(() -> Assertions.assertThat((Object)cachingJdbcClient.getTableStatistics(session, queryOnFirst)).isEqualTo((Object)NON_EMPTY_STATS));
        cachingJdbcClient.dropTable(SESSION, second);
        TestCachingJdbcClient.assertStatisticsCacheStats(cachingJdbcClient).hits(1L).afterRunning(() -> Assertions.assertThat((Object)cachingJdbcClient.getTableStatistics(session, queryOnFirst)).isEqualTo((Object)NON_EMPTY_STATS));
        cachingJdbcClient.dropTable(SESSION, first);
        TestCachingJdbcClient.assertStatisticsCacheStats(cachingJdbcClient).loads(1L).misses(1L).afterRunning(() -> Assertions.assertThat((Object)cachingJdbcClient.getTableStatistics(session, queryOnFirst)).isEqualTo((Object)NON_EMPTY_STATS));
    }

    @Test
    public void testTruncateTable() {
        CachingJdbcClient cachingJdbcClient = this.cachingClientBuilder().delegate(this.jdbcClientWithTableStats()).build();
        ConnectorSession session = TestCachingJdbcClient.createSession("table");
        JdbcTableHandle table = this.createTable(new SchemaTableName(this.schema, "table"));
        TestCachingJdbcClient.assertStatisticsCacheStats(cachingJdbcClient).loads(1L).misses(1L).afterRunning(() -> Assertions.assertThat((Object)cachingJdbcClient.getTableStatistics(session, table)).isEqualTo((Object)NON_EMPTY_STATS));
        TestCachingJdbcClient.assertStatisticsCacheStats(cachingJdbcClient).hits(1L).afterRunning(() -> Assertions.assertThat((Object)cachingJdbcClient.getTableStatistics(session, table)).isEqualTo((Object)NON_EMPTY_STATS));
        cachingJdbcClient.truncateTable(SESSION, table);
        TestCachingJdbcClient.assertStatisticsCacheStats(cachingJdbcClient).loads(1L).misses(1L).afterRunning(() -> Assertions.assertThat((Object)cachingJdbcClient.getTableStatistics(session, table)).isEqualTo((Object)NON_EMPTY_STATS));
        TestCachingJdbcClient.assertStatisticsCacheStats(cachingJdbcClient).hits(1L).afterRunning(() -> Assertions.assertThat((Object)cachingJdbcClient.getTableStatistics(session, table)).isEqualTo((Object)NON_EMPTY_STATS));
        this.jdbcClient.dropTable(SESSION, table);
    }

    @Test
    public void testCacheEmptyStatistics() {
        CachingJdbcClient cachingJdbcClient = this.cachingClientBuilder().config(TestCachingJdbcClient.enableCache().setCacheMissing(true)).build();
        ConnectorSession session = TestCachingJdbcClient.createSession("table");
        JdbcTableHandle table = this.createTable(new SchemaTableName(this.schema, "table"));
        TestCachingJdbcClient.assertStatisticsCacheStats(cachingJdbcClient).loads(1L).misses(1L).afterRunning(() -> Assertions.assertThat((Object)cachingJdbcClient.getTableStatistics(session, table)).isEqualTo((Object)TableStatistics.empty()));
        TestCachingJdbcClient.assertStatisticsCacheStats(cachingJdbcClient).hits(1L).afterRunning(() -> Assertions.assertThat((Object)cachingJdbcClient.getTableStatistics(session, table)).isEqualTo((Object)TableStatistics.empty()));
        this.jdbcClient.dropTable(SESSION, table);
    }

    @Test
    public void testGetTableStatisticsDoNotCacheEmptyWhenCachingMissingIsDisabled() {
        CachingJdbcClient cachingJdbcClient = this.cachingClientBuilder().config(TestCachingJdbcClient.enableCache().setCacheMissing(false)).build();
        ConnectorSession session = TestCachingJdbcClient.createSession("table");
        JdbcTableHandle table = this.createTable(new SchemaTableName(this.schema, "table"));
        TestCachingJdbcClient.assertStatisticsCacheStats(cachingJdbcClient).loads(1L).misses(1L).afterRunning(() -> Assertions.assertThat((Object)cachingJdbcClient.getTableStatistics(session, table)).isEqualTo((Object)TableStatistics.empty()));
        TestCachingJdbcClient.assertStatisticsCacheStats(cachingJdbcClient).loads(1L).hits(1L).misses(1L).afterRunning(() -> Assertions.assertThat((Object)cachingJdbcClient.getTableStatistics(session, table)).isEqualTo((Object)TableStatistics.empty()));
        this.jdbcClient.dropTable(SESSION, table);
    }

    @Test
    public void testDifferentIdentityKeys() {
        CachingJdbcClient cachingJdbcClient = this.cachingClientBuilder().identityCacheMapping((IdentityCacheMapping)new ExtraCredentialsBasedIdentityCacheMapping(new ExtraCredentialConfig().setUserCredentialName("user").setPasswordCredentialName("password"))).build();
        ConnectorSession alice = TestCachingJdbcClient.createUserSession("alice");
        ConnectorSession bob = TestCachingJdbcClient.createUserSession("bob");
        JdbcTableHandle table = this.createTable(new SchemaTableName(this.schema, "table"));
        TestCachingJdbcClient.assertTableNamesCache(cachingJdbcClient).loads(2L).misses(2L).afterRunning(() -> {
            Assertions.assertThat((List)cachingJdbcClient.getTableNames(alice, Optional.empty())).contains((Object[])new SchemaTableName[]{table.getRequiredNamedRelation().getSchemaTableName()});
            Assertions.assertThat((List)cachingJdbcClient.getTableNames(bob, Optional.empty())).contains((Object[])new SchemaTableName[]{table.getRequiredNamedRelation().getSchemaTableName()});
        });
        TestCachingJdbcClient.assertTableNamesCache(cachingJdbcClient).hits(2L).afterRunning(() -> {
            Assertions.assertThat((List)cachingJdbcClient.getTableNames(alice, Optional.empty())).contains((Object[])new SchemaTableName[]{table.getRequiredNamedRelation().getSchemaTableName()});
            Assertions.assertThat((List)cachingJdbcClient.getTableNames(bob, Optional.empty())).contains((Object[])new SchemaTableName[]{table.getRequiredNamedRelation().getSchemaTableName()});
        });
        this.jdbcClient.dropTable(SESSION, table);
    }

    @Test
    public void testFlushCache() {
        CachingJdbcClient cachingJdbcClient = this.cachingClientBuilder().delegate(this.jdbcClientWithTableStats()).build();
        ConnectorSession session = TestCachingJdbcClient.createSession("asession");
        JdbcTableHandle first = this.createTable(new SchemaTableName(this.schema, "atable"));
        TestCachingJdbcClient.assertStatisticsCacheStats(cachingJdbcClient).loads(1L).misses(1L).afterRunning(() -> Assertions.assertThat((Object)cachingJdbcClient.getTableStatistics(session, first)).isEqualTo((Object)NON_EMPTY_STATS));
        TestCachingJdbcClient.assertStatisticsCacheStats(cachingJdbcClient).hits(1L).afterRunning(() -> Assertions.assertThat((Object)cachingJdbcClient.getTableStatistics(session, first)).isEqualTo((Object)NON_EMPTY_STATS));
        cachingJdbcClient.flushCache();
        JdbcTableHandle secondFirst = this.createTable(new SchemaTableName(this.schema, "first"));
        TestCachingJdbcClient.assertStatisticsCacheStats(cachingJdbcClient).loads(1L).misses(1L).afterRunning(() -> Assertions.assertThat((Object)cachingJdbcClient.getTableStatistics(session, secondFirst)).isEqualTo((Object)NON_EMPTY_STATS));
        TestCachingJdbcClient.assertStatisticsCacheStats(cachingJdbcClient).hits(1L).afterRunning(() -> Assertions.assertThat((Object)cachingJdbcClient.getTableStatistics(session, secondFirst)).isEqualTo((Object)NON_EMPTY_STATS));
        this.jdbcClient.dropTable(SESSION, first);
    }

    @Test
    @Timeout(value=60L)
    public void testConcurrentSchemaCreateAndDrop() {
        CachingJdbcClient cachingJdbcClient = this.cachingClientBuilder().delegate(this.jdbcClientWithTableStats()).build();
        ConnectorSession session = TestCachingJdbcClient.createSession("asession");
        ArrayList<Future<Object>> futures = new ArrayList<Future<Object>>();
        for (int i = 0; i < 5; ++i) {
            futures.add(this.executor.submit(() -> {
                String schemaName = "schema_" + TestingNames.randomNameSuffix();
                Assertions.assertThat((Collection)cachingJdbcClient.getSchemaNames(session)).doesNotContain((Object[])new String[]{schemaName});
                cachingJdbcClient.createSchema(session, schemaName);
                Assertions.assertThat((Collection)cachingJdbcClient.getSchemaNames(session)).contains((Object[])new String[]{schemaName});
                cachingJdbcClient.dropSchema(session, schemaName, false);
                Assertions.assertThat((Collection)cachingJdbcClient.getSchemaNames(session)).doesNotContain((Object[])new String[]{schemaName});
                return null;
            }));
        }
        futures.forEach(Futures::getUnchecked);
    }

    @Test
    @Timeout(value=60L)
    public void testLoadFailureNotSharedWhenDisabled() throws Exception {
        final AtomicBoolean first = new AtomicBoolean(true);
        CyclicBarrier barrier = new CyclicBarrier(2);
        CachingJdbcClient cachingJdbcClient = this.cachingClientBuilder().delegate((JdbcClient)new ForwardingJdbcClient(this){
            private final JdbcClient delegate;
            final /* synthetic */ TestCachingJdbcClient this$0;
            {
                this.this$0 = this$0;
                this.delegate = this.this$0.database.getJdbcClient();
            }

            protected JdbcClient delegate() {
                return this.delegate;
            }

            public Optional<JdbcTableHandle> getTableHandle(ConnectorSession session, SchemaTableName schemaTableName) {
                if (first.compareAndSet(true, false)) {
                    try {
                        Thread.sleep(5L);
                    }
                    catch (InterruptedException e1) {
                        throw new RuntimeException(e1);
                    }
                    throw new RuntimeException("first attempt is poised to fail");
                }
                return super.getTableHandle(session, schemaTableName);
            }
        }).config(new BaseJdbcConfig().setMetadataCacheTtl(Duration.ZERO)).build();
        SchemaTableName tableName = new SchemaTableName(this.schema, "test_load_failure_not_shared");
        this.createTable(tableName);
        ConnectorSession session = TestCachingJdbcClient.createSession("session");
        ArrayList<Future<JdbcTableHandle>> futures = new ArrayList<Future<JdbcTableHandle>>();
        for (int i = 0; i < 2; ++i) {
            futures.add(this.executor.submit(() -> {
                barrier.await(10L, TimeUnit.SECONDS);
                return (JdbcTableHandle)cachingJdbcClient.getTableHandle(session, tableName).orElseThrow();
            }));
        }
        ArrayList<String> results = new ArrayList<String>();
        for (Future future : futures) {
            try {
                results.add(((JdbcTableHandle)future.get()).toString());
            }
            catch (ExecutionException e) {
                results.add(e.getCause().toString());
            }
        }
        Assertions.assertThat(results).containsExactlyInAnyOrder((Object[])new String[]{"example.test_load_failure_not_shared " + this.database.getDatabaseName() + ".EXAMPLE.TEST_LOAD_FAILURE_NOT_SHARED", "com.google.common.util.concurrent.UncheckedExecutionException: java.lang.RuntimeException: first attempt is poised to fail"});
    }

    @Test
    public void testSpecificSchemaAndTableCaches() {
        TestingTicker ticker = new TestingTicker();
        CachingJdbcClient cachingJdbcClient = this.cachingClientBuilder().ticker((Ticker)ticker).config(new BaseJdbcConfig().setMetadataCacheTtl(FOREVER).setSchemaNamesCacheTtl(new Duration(30.0, TimeUnit.SECONDS)).setTableNamesCacheTtl(new Duration(20.0, TimeUnit.SECONDS)).setCacheMissing(false)).build();
        String secondSchema = this.schema + "_two";
        SchemaTableName firstName = new SchemaTableName(this.schema, "first_table");
        SchemaTableName secondName = new SchemaTableName(secondSchema, "second_table");
        ConnectorSession session = TestCachingJdbcClient.createSession("asession");
        JdbcTableHandle first = this.createTable(firstName);
        TestCachingJdbcClient.assertSchemaNamesCache(cachingJdbcClient).loads(1L).misses(1L).afterRunning(() -> ((AbstractCollectionAssert)Assertions.assertThat((Collection)cachingJdbcClient.getSchemaNames(session)).contains((Object[])new String[]{this.schema})).doesNotContain((Object[])new String[]{secondSchema}));
        TestCachingJdbcClient.assertTableNamesCache(cachingJdbcClient).loads(1L).misses(1L).afterRunning(() -> ((ListAssert)Assertions.assertThat((List)cachingJdbcClient.getTableNames(session, Optional.empty())).contains((Object[])new SchemaTableName[]{firstName})).doesNotContain((Object[])new SchemaTableName[]{secondName}));
        TestCachingJdbcClient.assertTableHandleByNameCache(cachingJdbcClient).misses(1L).loads(1L).afterRunning(() -> Assertions.assertThat((Optional)cachingJdbcClient.getTableHandle(session, firstName)).isNotEmpty());
        TestCachingJdbcClient.assertTableHandleByNameCache(cachingJdbcClient).misses(1L).loads(1L).afterRunning(() -> Assertions.assertThat((Optional)cachingJdbcClient.getTableHandle(session, secondName)).isEmpty());
        this.jdbcClient.createSchema(SESSION, secondSchema);
        JdbcTableHandle second = this.createTable(secondName);
        TestCachingJdbcClient.assertSchemaNamesCache(cachingJdbcClient).hits(1L).afterRunning(() -> ((AbstractCollectionAssert)Assertions.assertThat((Collection)cachingJdbcClient.getSchemaNames(session)).contains((Object[])new String[]{this.schema})).doesNotContain((Object[])new String[]{secondSchema}));
        TestCachingJdbcClient.assertTableNamesCache(cachingJdbcClient).hits(1L).afterRunning(() -> ((ListAssert)Assertions.assertThat((List)cachingJdbcClient.getTableNames(session, Optional.empty())).contains((Object[])new SchemaTableName[]{firstName})).doesNotContain((Object[])new SchemaTableName[]{secondName}));
        TestCachingJdbcClient.assertTableHandleByNameCache(cachingJdbcClient).hits(1L).afterRunning(() -> Assertions.assertThat((Optional)cachingJdbcClient.getTableHandle(session, firstName)).isNotEmpty());
        TestCachingJdbcClient.assertTableHandleByNameCache(cachingJdbcClient).hits(1L).misses(1L).loads(1L).afterRunning(() -> Assertions.assertThat((Optional)cachingJdbcClient.getTableHandle(session, secondName)).isNotEmpty());
        ticker.increment(25L, TimeUnit.SECONDS);
        TestCachingJdbcClient.assertSchemaNamesCache(cachingJdbcClient).hits(1L).afterRunning(() -> ((AbstractCollectionAssert)Assertions.assertThat((Collection)cachingJdbcClient.getSchemaNames(session)).contains((Object[])new String[]{this.schema})).doesNotContain((Object[])new String[]{secondSchema}));
        TestCachingJdbcClient.assertTableNamesCache(cachingJdbcClient).loads(1L).misses(1L).afterRunning(() -> Assertions.assertThat((List)cachingJdbcClient.getTableNames(session, Optional.empty())).contains((Object[])new SchemaTableName[]{firstName, secondName}));
        TestCachingJdbcClient.assertTableHandleByNameCache(cachingJdbcClient).hits(1L).afterRunning(() -> Assertions.assertThat((Optional)cachingJdbcClient.getTableHandle(session, firstName)).isNotEmpty());
        TestCachingJdbcClient.assertTableHandleByNameCache(cachingJdbcClient).hits(1L).afterRunning(() -> Assertions.assertThat((Optional)cachingJdbcClient.getTableHandle(session, secondName)).isNotEmpty());
        ticker.increment(35L, TimeUnit.SECONDS);
        TestCachingJdbcClient.assertSchemaNamesCache(cachingJdbcClient).loads(1L).misses(1L).afterRunning(() -> Assertions.assertThat((Collection)cachingJdbcClient.getSchemaNames(session)).contains((Object[])new String[]{this.schema, secondSchema}));
        TestCachingJdbcClient.assertTableNamesCache(cachingJdbcClient).loads(1L).misses(1L).afterRunning(() -> Assertions.assertThat((List)cachingJdbcClient.getTableNames(session, Optional.empty())).contains((Object[])new SchemaTableName[]{firstName, secondName}));
        TestCachingJdbcClient.assertTableHandleByNameCache(cachingJdbcClient).hits(1L).afterRunning(() -> Assertions.assertThat((Optional)cachingJdbcClient.getTableHandle(session, firstName)).isNotEmpty());
        TestCachingJdbcClient.assertTableHandleByNameCache(cachingJdbcClient).hits(1L).afterRunning(() -> Assertions.assertThat((Optional)cachingJdbcClient.getTableHandle(session, secondName)).isNotEmpty());
        this.jdbcClient.dropTable(SESSION, first);
        this.jdbcClient.dropTable(SESSION, second);
        this.jdbcClient.dropSchema(SESSION, secondSchema, false);
    }

    @Test
    public void testCacheOnlyStatistics() throws Exception {
        CachingJdbcClient cachingJdbcClient = this.cachingClientBuilder().delegate(this.jdbcClientWithTableStats()).config(new BaseJdbcConfig().setMetadataCacheTtl(Duration.ZERO).setStatisticsCacheTtl(FOREVER)).build();
        SchemaTableName phantomTable = new SchemaTableName(this.schema, "phantom_table");
        this.createTable(phantomTable);
        JdbcTableHandle firstHandle = (JdbcTableHandle)TestCachingJdbcClient.assertTableHandleByNameCache(cachingJdbcClient).misses(2L).loads(1L).calling(() -> cachingJdbcClient.getTableHandle(SESSION, phantomTable)).orElseThrow();
        TestCachingJdbcClient.assertStatisticsCacheStats(cachingJdbcClient).misses(1L).loads(1L).calling(() -> cachingJdbcClient.getTableStatistics(SESSION, firstHandle));
        JdbcTableHandle secondHandle = (JdbcTableHandle)TestCachingJdbcClient.assertTableHandleByNameCache(cachingJdbcClient).misses(2L).loads(1L).calling(() -> cachingJdbcClient.getTableHandle(SESSION, phantomTable)).orElseThrow();
        TestCachingJdbcClient.assertStatisticsCacheStats(cachingJdbcClient).hits(1L).calling(() -> cachingJdbcClient.getTableStatistics(SESSION, secondHandle));
        this.dropTable(phantomTable);
    }

    private JdbcTableHandle getAnyTable(String schema) {
        SchemaTableName tableName = this.jdbcClient.getTableNames(SESSION, Optional.of(schema)).stream().filter(schemaTableName -> !"public".equals(schemaTableName.getTableName())).findAny().orElseThrow();
        return (JdbcTableHandle)this.jdbcClient.getTableHandle(SESSION, tableName).orElseThrow();
    }

    private JdbcColumnHandle addColumn(JdbcTableHandle tableHandle) {
        return this.addColumn(tableHandle, "phantom_column");
    }

    private JdbcColumnHandle addColumn(JdbcTableHandle tableHandle, String columnName) {
        return this.addColumn(this.jdbcClient, tableHandle, columnName);
    }

    private JdbcColumnHandle addColumn(JdbcClient client, JdbcTableHandle tableHandle, String columnName) {
        ColumnMetadata columnMetadata = new ColumnMetadata(columnName, (Type)IntegerType.INTEGER);
        client.addColumn(SESSION, tableHandle, columnMetadata);
        return (JdbcColumnHandle)client.getColumns(SESSION, tableHandle).stream().filter(jdbcColumnHandle -> jdbcColumnHandle.getColumnMetadata().equals((Object)columnMetadata)).collect(MoreCollectors.onlyElement());
    }

    private static ConnectorSession createSession(String sessionName) {
        return TestingConnectorSession.builder().setPropertyMetadata(PROPERTY_METADATA).setPropertyValues((Map)ImmutableMap.of((Object)"session_name", (Object)sessionName)).build();
    }

    private static ConnectorSession createUserSession(String userName) {
        return TestingConnectorSession.builder().setIdentity(ConnectorIdentity.forUser((String)userName).withExtraCredentials((Map)ImmutableMap.of((Object)"user", (Object)userName)).build()).build();
    }

    @Test
    public void testEverythingImplemented() {
        InterfaceTestUtils.assertAllMethodsOverridden(JdbcClient.class, CachingJdbcClient.class);
    }

    private CachingJdbcClient createCachingJdbcClient() {
        return this.cachingClientBuilder().build();
    }

    private CachingJdbcClientBuilder cachingClientBuilder() {
        return new CachingJdbcClientBuilder().ticker(Ticker.systemTicker()).delegate(this.database.getJdbcClient()).sessionPropertiesProviders(SESSION_PROPERTIES_PROVIDERS).identityCacheMapping((IdentityCacheMapping)new SingletonIdentityCacheMapping()).config(TestCachingJdbcClient.enableCache());
    }

    private JdbcClient jdbcClientWithTableStats() {
        return new ForwardingJdbcClient(){

            protected JdbcClient delegate() {
                return TestCachingJdbcClient.this.database.getJdbcClient();
            }

            public TableStatistics getTableStatistics(ConnectorSession session, JdbcTableHandle handle) {
                return NON_EMPTY_STATS;
            }
        };
    }

    private static BaseJdbcConfig enableCache() {
        return new BaseJdbcConfig().setMetadataCacheTtl(FOREVER);
    }

    private static SingleJdbcCacheStatsAssertions assertSchemaNamesCache(CachingJdbcClient client) {
        return TestCachingJdbcClient.assertCacheStats(client, CachingJdbcCache.SCHEMA_NAMES_CACHE);
    }

    private static SingleJdbcCacheStatsAssertions assertTableNamesCache(CachingJdbcClient client) {
        return TestCachingJdbcClient.assertCacheStats(client, CachingJdbcCache.TABLE_NAMES_CACHE);
    }

    private static SingleJdbcCacheStatsAssertions assertTableHandleByNameCache(CachingJdbcClient client) {
        return TestCachingJdbcClient.assertCacheStats(client, CachingJdbcCache.TABLE_HANDLES_BY_NAME_CACHE);
    }

    private static SingleJdbcCacheStatsAssertions assertTableHandleByQueryCache(CachingJdbcClient client) {
        return TestCachingJdbcClient.assertCacheStats(client, CachingJdbcCache.TABLE_HANDLES_BY_QUERY_CACHE);
    }

    private static SingleJdbcCacheStatsAssertions assertProcedureHandleByQueryCache(CachingJdbcClient client) {
        return TestCachingJdbcClient.assertCacheStats(client, CachingJdbcCache.PROCEDURE_HANDLES_BY_QUERY_CACHE);
    }

    private static SingleJdbcCacheStatsAssertions assertColumnCacheStats(CachingJdbcClient client) {
        return TestCachingJdbcClient.assertCacheStats(client, CachingJdbcCache.COLUMNS_CACHE);
    }

    private static SingleJdbcCacheStatsAssertions assertStatisticsCacheStats(CachingJdbcClient client) {
        return TestCachingJdbcClient.assertCacheStats(client, CachingJdbcCache.STATISTICS_CACHE);
    }

    private static SingleJdbcCacheStatsAssertions assertCacheStats(CachingJdbcClient client, CachingJdbcCache cache) {
        return new SingleJdbcCacheStatsAssertions(client, cache);
    }

    private static JdbcCacheStatsAssertions assertCacheStats(CachingJdbcClient client) {
        return new JdbcCacheStatsAssertions(client);
    }

    private static class SingleJdbcCacheStatsAssertions {
        private CachingJdbcCache chosenCache;
        private JdbcCacheStatsAssertions delegate;

        private SingleJdbcCacheStatsAssertions(CachingJdbcClient jdbcClient, CachingJdbcCache chosenCache) {
            this.chosenCache = Objects.requireNonNull(chosenCache, "chosenCache is null");
            this.delegate = new JdbcCacheStatsAssertions(jdbcClient);
        }

        public SingleJdbcCacheStatsAssertions loads(long value) {
            this.delegate.loads(this.chosenCache, value);
            return this;
        }

        public SingleJdbcCacheStatsAssertions hits(long value) {
            this.delegate.hits(this.chosenCache, value);
            return this;
        }

        public SingleJdbcCacheStatsAssertions misses(long value) {
            this.delegate.misses(this.chosenCache, value);
            return this;
        }

        public void afterRunning(Runnable runnable) {
            this.delegate.afterRunning(runnable);
        }

        public <T> T calling(Callable<T> callable) throws Exception {
            return this.delegate.calling(callable);
        }
    }

    private static class CachingJdbcClientBuilder {
        private Ticker ticker;
        private JdbcClient delegate;
        private Set<SessionPropertiesProvider> sessionPropertiesProviders;
        private IdentityCacheMapping identityCacheMapping;
        private BaseJdbcConfig config;

        private CachingJdbcClientBuilder() {
        }

        public CachingJdbcClientBuilder ticker(Ticker ticker) {
            this.ticker = ticker;
            return this;
        }

        public CachingJdbcClientBuilder delegate(JdbcClient delegate) {
            this.delegate = delegate;
            return this;
        }

        public CachingJdbcClientBuilder sessionPropertiesProviders(Set<SessionPropertiesProvider> sessionPropertiesProviders) {
            this.sessionPropertiesProviders = sessionPropertiesProviders;
            return this;
        }

        public CachingJdbcClientBuilder identityCacheMapping(IdentityCacheMapping identityCacheMapping) {
            this.identityCacheMapping = identityCacheMapping;
            return this;
        }

        public CachingJdbcClientBuilder config(BaseJdbcConfig config) {
            this.config = config;
            return this;
        }

        CachingJdbcClient build() {
            return new CachingJdbcClient(this.ticker, this.delegate, this.sessionPropertiesProviders, this.identityCacheMapping, this.config.getMetadataCacheTtl(), this.config.getSchemaNamesCacheTtl(), this.config.getTableNamesCacheTtl(), this.config.getStatisticsCacheTtl(), this.config.isCacheMissing(), this.config.getCacheMaximumSize());
        }
    }

    private static class JdbcCacheStatsAssertions {
        private final CachingJdbcClient jdbcClient;
        private final Map<CachingJdbcCache, Long> loads = new HashMap<CachingJdbcCache, Long>();
        private final Map<CachingJdbcCache, Long> hits = new HashMap<CachingJdbcCache, Long>();
        private final Map<CachingJdbcCache, Long> misses = new HashMap<CachingJdbcCache, Long>();

        public JdbcCacheStatsAssertions(CachingJdbcClient jdbcClient) {
            this.jdbcClient = Objects.requireNonNull(jdbcClient, "jdbcClient is null");
        }

        public JdbcCacheStatsAssertions loads(CachingJdbcCache cache, long value) {
            this.loads.put(cache, value);
            return this;
        }

        public JdbcCacheStatsAssertions hits(CachingJdbcCache cache, long value) {
            this.hits.put(cache, value);
            return this;
        }

        public JdbcCacheStatsAssertions misses(CachingJdbcCache cache, long value) {
            this.misses.put(cache, value);
            return this;
        }

        public void afterRunning(Runnable runnable) {
            try {
                this.calling(() -> {
                    runnable.run();
                    return null;
                });
            }
            catch (Exception e) {
                throw new RuntimeException(e);
            }
        }

        public <T> T calling(Callable<T> callable) throws Exception {
            Map beforeStats = (Map)Stream.of(CachingJdbcCache.values()).collect(ImmutableMap.toImmutableMap(Function.identity(), cache -> cache.statsGetter.apply(this.jdbcClient)));
            T value = callable.call();
            Map afterStats = (Map)Stream.of(CachingJdbcCache.values()).collect(ImmutableMap.toImmutableMap(Function.identity(), cache -> cache.statsGetter.apply(this.jdbcClient)));
            for (CachingJdbcCache cache2 : CachingJdbcCache.values()) {
                long loadDelta = ((CacheStats)afterStats.get((Object)cache2)).loadCount() - ((CacheStats)beforeStats.get((Object)cache2)).loadCount();
                long missesDelta = ((CacheStats)afterStats.get((Object)cache2)).missCount() - ((CacheStats)beforeStats.get((Object)cache2)).missCount();
                long hitsDelta = ((CacheStats)afterStats.get((Object)cache2)).hitCount() - ((CacheStats)beforeStats.get((Object)cache2)).hitCount();
                ((AbstractLongAssert)Assertions.assertThat((long)loadDelta).as(String.valueOf((Object)cache2) + " loads (delta)", new Object[0])).isEqualTo((Object)this.loads.getOrDefault((Object)cache2, 0L));
                ((AbstractLongAssert)Assertions.assertThat((long)hitsDelta).as(String.valueOf((Object)cache2) + " hits (delta)", new Object[0])).isEqualTo((Object)this.hits.getOrDefault((Object)cache2, 0L));
                ((AbstractLongAssert)Assertions.assertThat((long)missesDelta).as(String.valueOf((Object)cache2) + " misses (delta)", new Object[0])).isEqualTo((Object)this.misses.getOrDefault((Object)cache2, 0L));
            }
            return value;
        }
    }

    static enum CachingJdbcCache {
        SCHEMA_NAMES_CACHE(CachingJdbcClient::getSchemaNamesCacheStats),
        TABLE_NAMES_CACHE(CachingJdbcClient::getTableNamesCacheStats),
        TABLE_HANDLES_BY_NAME_CACHE(CachingJdbcClient::getTableHandlesByNameCacheStats),
        TABLE_HANDLES_BY_QUERY_CACHE(CachingJdbcClient::getTableHandlesByQueryCacheStats),
        PROCEDURE_HANDLES_BY_QUERY_CACHE(CachingJdbcClient::getProcedureHandlesByQueryCacheStats),
        COLUMNS_CACHE(CachingJdbcClient::getColumnsCacheStats),
        STATISTICS_CACHE(CachingJdbcClient::getStatisticsCacheStats);

        private final Function<CachingJdbcClient, CacheStats> statsGetter;

        private CachingJdbcCache(Function<CachingJdbcClient, CacheStats> statsGetter) {
            this.statsGetter = Objects.requireNonNull(statsGetter, "statsGetter is null");
        }
    }
}

