/*
 * Decompiled with CFR 0.152.
 */
package io.trino.plugin.bigquery;

import com.google.cloud.bigquery.BigQuery;
import com.google.cloud.bigquery.BigQueryException;
import com.google.cloud.bigquery.BigQueryOptions;
import com.google.cloud.bigquery.DatasetId;
import com.google.cloud.bigquery.DatasetInfo;
import com.google.cloud.bigquery.InsertAllRequest;
import com.google.cloud.bigquery.InsertAllResponse;
import com.google.cloud.bigquery.Job;
import com.google.cloud.bigquery.JobConfiguration;
import com.google.cloud.bigquery.JobInfo;
import com.google.cloud.bigquery.JobStatistics;
import com.google.cloud.bigquery.QueryJobConfiguration;
import com.google.cloud.bigquery.Schema;
import com.google.cloud.bigquery.Table;
import com.google.cloud.bigquery.TableDefinition;
import com.google.cloud.bigquery.TableId;
import com.google.cloud.bigquery.TableInfo;
import com.google.cloud.bigquery.TableResult;
import com.google.common.base.Preconditions;
import com.google.common.base.Verify;
import com.google.common.cache.CacheLoader;
import com.google.common.cache.LoadingCache;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.ImmutableSet;
import com.google.common.collect.Iterables;
import com.google.common.collect.Streams;
import io.airlift.log.Logger;
import io.airlift.units.Duration;
import io.trino.cache.EvictableCacheBuilder;
import io.trino.plugin.bigquery.BigQueryColumnHandle;
import io.trino.plugin.bigquery.BigQueryErrorCode;
import io.trino.plugin.bigquery.BigQueryLabelFactory;
import io.trino.plugin.bigquery.BigQuerySessionProperties;
import io.trino.plugin.bigquery.BigQueryTableHandle;
import io.trino.plugin.bigquery.BigQueryTypeManager;
import io.trino.plugin.bigquery.ViewMaterializationCache;
import io.trino.spi.ErrorCodeSupplier;
import io.trino.spi.TrinoException;
import io.trino.spi.connector.ConnectorSession;
import io.trino.spi.connector.SchemaTableName;
import io.trino.spi.connector.TableNotFoundException;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Locale;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.TimeUnit;
import java.util.function.Supplier;
import java.util.stream.Collectors;

public class BigQueryClient {
    private static final Logger log = Logger.get(BigQueryClient.class);
    static final Set<TableDefinition.Type> TABLE_TYPES = ImmutableSet.of((Object)TableDefinition.Type.TABLE, (Object)TableDefinition.Type.VIEW, (Object)TableDefinition.Type.MATERIALIZED_VIEW, (Object)TableDefinition.Type.EXTERNAL, (Object)TableDefinition.Type.SNAPSHOT);
    private final BigQuery bigQuery;
    private final BigQueryLabelFactory labelFactory;
    private final BigQueryTypeManager typeManager;
    private final ViewMaterializationCache materializationCache;
    private final boolean caseInsensitiveNameMatching;
    private final LoadingCache<String, List<DatasetId>> remoteDatasetIdCache;
    private final Optional<String> configProjectId;

    public BigQueryClient(BigQuery bigQuery, BigQueryLabelFactory labelFactory, BigQueryTypeManager typeManager, boolean caseInsensitiveNameMatching, ViewMaterializationCache materializationCache, Duration metadataCacheTtl, Optional<String> configProjectId) {
        this.bigQuery = Objects.requireNonNull(bigQuery, "bigQuery is null");
        this.labelFactory = Objects.requireNonNull(labelFactory, "labelFactory is null");
        this.typeManager = Objects.requireNonNull(typeManager, "typeManager is null");
        this.materializationCache = Objects.requireNonNull(materializationCache, "materializationCache is null");
        this.caseInsensitiveNameMatching = caseInsensitiveNameMatching;
        this.remoteDatasetIdCache = EvictableCacheBuilder.newBuilder().expireAfterWrite(metadataCacheTtl.toMillis(), TimeUnit.MILLISECONDS).shareNothingWhenDisabled().build(CacheLoader.from(this::listDatasetIdsFromBigQuery));
        this.configProjectId = Objects.requireNonNull(configProjectId, "projectId is null");
    }

    public Optional<RemoteDatabaseObject> toRemoteDataset(DatasetId datasetId) {
        return this.toRemoteDataset(datasetId.getProject(), datasetId.getDataset());
    }

    public Optional<RemoteDatabaseObject> toRemoteDataset(String projectId, String datasetName) {
        Objects.requireNonNull(projectId, "projectId is null");
        Objects.requireNonNull(datasetName, "datasetName is null");
        Verify.verify((boolean)datasetName.codePoints().noneMatch(Character::isUpperCase), (String)"Expected schema name from internal metadata to be lowercase: %s", (Object)datasetName);
        if (!this.caseInsensitiveNameMatching) {
            return Optional.of(RemoteDatabaseObject.of(datasetName));
        }
        HashMap mapping = new HashMap();
        for (DatasetId datasetId : this.listDatasetIds(projectId)) {
            mapping.merge(datasetId.getDataset().toLowerCase(Locale.ENGLISH), Optional.of(RemoteDatabaseObject.of(datasetId.getDataset())), (currentValue, collision) -> currentValue.map(current -> current.registerCollision(((RemoteDatabaseObject)collision.get()).getOnlyRemoteName())));
        }
        if (!mapping.containsKey(datasetName)) {
            mapping.put(datasetName, Optional.empty());
        }
        Verify.verify((boolean)mapping.containsKey(datasetName));
        return (Optional)mapping.get(datasetName);
    }

    public Optional<RemoteDatabaseObject> toRemoteTable(String projectId, String remoteDatasetName, String tableName) {
        return this.toRemoteTable(projectId, remoteDatasetName, tableName, () -> this.listTableIds(DatasetId.of((String)projectId, (String)remoteDatasetName)));
    }

    public Optional<RemoteDatabaseObject> toRemoteTable(String projectId, String remoteDatasetName, String tableName, Iterable<TableId> tableIds) {
        return this.toRemoteTable(projectId, remoteDatasetName, tableName, () -> tableIds);
    }

    private Optional<RemoteDatabaseObject> toRemoteTable(String projectId, String remoteDatasetName, String tableName, Supplier<Iterable<TableId>> tableIds) {
        Objects.requireNonNull(projectId, "projectId is null");
        Objects.requireNonNull(remoteDatasetName, "remoteDatasetName is null");
        Objects.requireNonNull(tableName, "tableName is null");
        Verify.verify((boolean)tableName.codePoints().noneMatch(Character::isUpperCase), (String)"Expected table name from internal metadata to be lowercase: %s", (Object)tableName);
        if (!this.caseInsensitiveNameMatching) {
            return Optional.of(RemoteDatabaseObject.of(tableName));
        }
        TableId cacheKey = TableId.of((String)projectId, (String)remoteDatasetName, (String)tableName);
        HashMap mapping = new HashMap();
        for (TableId tableId : tableIds.get()) {
            mapping.merge(BigQueryClient.tableIdToLowerCase(tableId), Optional.of(RemoteDatabaseObject.of(tableId.getTable())), (currentValue, collision) -> currentValue.map(current -> current.registerCollision(((RemoteDatabaseObject)collision.get()).getOnlyRemoteName())));
        }
        if (!mapping.containsKey(cacheKey)) {
            mapping.put(cacheKey, Optional.empty());
        }
        Verify.verify((boolean)mapping.containsKey(cacheKey));
        return (Optional)mapping.get(cacheKey);
    }

    private static TableId tableIdToLowerCase(TableId tableId) {
        return TableId.of((String)tableId.getProject(), (String)tableId.getDataset(), (String)tableId.getTable().toLowerCase(Locale.ENGLISH));
    }

    public DatasetInfo getDataset(DatasetId datasetId) {
        return this.bigQuery.getDataset(datasetId, new BigQuery.DatasetOption[0]);
    }

    public Optional<TableInfo> getTable(TableId remoteTableId) {
        return Optional.ofNullable(this.bigQuery.getTable(remoteTableId, new BigQuery.TableOption[0]));
    }

    public TableInfo getCachedTable(Duration viewExpiration, TableInfo remoteTableId, List<String> requiredColumns) {
        String query = this.selectSql(remoteTableId, requiredColumns);
        log.debug("query is %s", new Object[]{query});
        return this.materializationCache.getCachedTable(this, query, viewExpiration, remoteTableId);
    }

    public String getParentProjectId() {
        return ((BigQueryOptions)this.bigQuery.getOptions()).getProjectId();
    }

    public String getProjectId() {
        String projectId = this.configProjectId.orElseGet(() -> ((BigQueryOptions)this.bigQuery.getOptions()).getProjectId());
        Preconditions.checkState((boolean)projectId.toLowerCase(Locale.ENGLISH).equals(projectId), (Object)("projectId must be lowercase but it's " + projectId));
        return projectId;
    }

    protected DatasetId toDatasetId(String schemaName) {
        return DatasetId.of((String)this.getProjectId(), (String)schemaName);
    }

    protected String toSchemaName(DatasetId datasetId) {
        return datasetId.getDataset();
    }

    public Iterable<DatasetId> listDatasetIds(String projectId) {
        try {
            return (Iterable)this.remoteDatasetIdCache.get((Object)projectId);
        }
        catch (ExecutionException e) {
            throw new TrinoException((ErrorCodeSupplier)BigQueryErrorCode.BIGQUERY_LISTING_DATASET_ERROR, "Failed to retrieve datasets from BigQuery", (Throwable)e);
        }
    }

    private List<DatasetId> listDatasetIdsFromBigQuery(String projectId) {
        return (List)Streams.stream((Iterable)this.bigQuery.listDatasets(projectId, new BigQuery.DatasetListOption[0]).iterateAll()).map(DatasetInfo::getDatasetId).collect(ImmutableList.toImmutableList());
    }

    public Iterable<TableId> listTableIds(DatasetId remoteDatasetId) {
        Iterable allTables;
        try {
            allTables = this.bigQuery.listTables(remoteDatasetId, new BigQuery.TableListOption[0]).iterateAll();
        }
        catch (BigQueryException e) {
            throw new TrinoException((ErrorCodeSupplier)BigQueryErrorCode.BIGQUERY_LISTING_TABLE_ERROR, "Failed to retrieve tables from BigQuery", (Throwable)e);
        }
        return (Iterable)Streams.stream((Iterable)allTables).filter(table -> TABLE_TYPES.contains(table.getDefinition().getType())).map(TableInfo::getTableId).collect(ImmutableList.toImmutableList());
    }

    Table update(TableInfo table) {
        return this.bigQuery.update(table, new BigQuery.TableOption[0]);
    }

    public void createSchema(DatasetInfo datasetInfo) {
        this.bigQuery.create(datasetInfo, new BigQuery.DatasetOption[0]);
    }

    public void dropSchema(DatasetId datasetId, boolean cascade) {
        if (cascade) {
            this.bigQuery.delete(datasetId, new BigQuery.DatasetDeleteOption[]{BigQuery.DatasetDeleteOption.deleteContents()});
        } else {
            this.bigQuery.delete(datasetId, new BigQuery.DatasetDeleteOption[0]);
        }
    }

    public void createTable(TableInfo tableInfo) {
        this.bigQuery.create(tableInfo, new BigQuery.TableOption[0]);
    }

    public void dropTable(TableId tableId) {
        this.bigQuery.delete(tableId);
    }

    Job create(JobInfo jobInfo) {
        return this.bigQuery.create(jobInfo, new BigQuery.JobOption[0]);
    }

    public long executeUpdate(ConnectorSession session, QueryJobConfiguration job) {
        log.debug("Execute query: %s", new Object[]{job.getQuery()});
        return this.execute(session, job).getTotalRows();
    }

    public TableResult executeQuery(ConnectorSession session, String sql) {
        log.debug("Execute query: %s", new Object[]{sql});
        QueryJobConfiguration job = QueryJobConfiguration.newBuilder((String)sql).setUseQueryCache(Boolean.valueOf(BigQuerySessionProperties.isQueryResultsCacheEnabled(session))).setCreateDisposition(BigQuerySessionProperties.createDisposition(session)).build();
        return this.execute(session, job);
    }

    private TableResult execute(ConnectorSession session, QueryJobConfiguration job) {
        QueryJobConfiguration jobWithQueryLabel = job.toBuilder().setLabels(this.labelFactory.getLabels(session)).build();
        try {
            return this.bigQuery.query(jobWithQueryLabel, new BigQuery.JobOption[0]);
        }
        catch (InterruptedException e) {
            Thread.currentThread().interrupt();
            throw new BigQueryException(0, String.format("Failed to run the query [%s]", job.getQuery()), (Throwable)e);
        }
    }

    public Schema getSchema(String sql) {
        JobStatistics statistics;
        log.debug("Get schema from query: %s", new Object[]{sql});
        JobInfo jobInfo = JobInfo.of((JobConfiguration)QueryJobConfiguration.newBuilder((String)sql).setDryRun(Boolean.valueOf(true)).build());
        try {
            statistics = this.bigQuery.create(jobInfo, new BigQuery.JobOption[0]).getStatistics();
        }
        catch (BigQueryException e) {
            throw new TrinoException((ErrorCodeSupplier)BigQueryErrorCode.BIGQUERY_INVALID_STATEMENT, "Failed to get schema for query: " + sql, (Throwable)e);
        }
        JobStatistics.QueryStatistics queryStatistics = (JobStatistics.QueryStatistics)statistics;
        if (!queryStatistics.getStatementType().equals((Object)JobStatistics.QueryStatistics.StatementType.SELECT)) {
            throw new TrinoException((ErrorCodeSupplier)BigQueryErrorCode.BIGQUERY_INVALID_STATEMENT, "Unsupported statement type: " + String.valueOf(queryStatistics.getStatementType()));
        }
        return Objects.requireNonNull(queryStatistics.getSchema(), "Cannot determine schema for query");
    }

    public static String selectSql(TableId table, List<String> requiredColumns, Optional<String> filter) {
        String columns = requiredColumns.stream().map(column -> String.format("`%s`", column)).collect(Collectors.joining(","));
        return BigQueryClient.selectSql(table, columns, filter);
    }

    public static String selectSql(TableId table, String formattedColumns, Optional<String> filter) {
        String tableName = BigQueryClient.fullTableName(table);
        String query = String.format("SELECT %s FROM `%s`", formattedColumns, tableName);
        if (filter.isEmpty()) {
            return query;
        }
        return query + " WHERE " + filter.get();
    }

    private String selectSql(TableInfo remoteTable, List<String> requiredColumns) {
        String columns = requiredColumns.isEmpty() ? "*" : requiredColumns.stream().map(column -> String.format("`%s`", column)).collect(Collectors.joining(","));
        return this.selectSql(remoteTable.getTableId(), columns);
    }

    public String selectSql(TableId table, String formattedColumns) {
        String tableName = BigQueryClient.fullTableName(table);
        return String.format("SELECT %s FROM `%s`", formattedColumns, tableName);
    }

    public void insert(InsertAllRequest insertAllRequest) {
        InsertAllResponse response = this.bigQuery.insertAll(insertAllRequest);
        if (response.hasErrors()) {
            throw new TrinoException((ErrorCodeSupplier)BigQueryErrorCode.BIGQUERY_FAILED_TO_EXECUTE_QUERY, String.format("Failed to insert rows: %s", response.getInsertErrors()));
        }
    }

    private static String fullTableName(TableId remoteTableId) {
        String remoteSchemaName = remoteTableId.getDataset();
        String remoteTableName = remoteTableId.getTable();
        remoteTableId = TableId.of((String)remoteTableId.getProject(), (String)remoteSchemaName, (String)remoteTableName);
        return String.format("%s.%s.%s", remoteTableId.getProject(), remoteTableId.getDataset(), remoteTableId.getTable());
    }

    public List<BigQueryColumnHandle> getColumns(BigQueryTableHandle tableHandle) {
        if (tableHandle.getProjectedColumns().isPresent()) {
            return tableHandle.getProjectedColumns().get();
        }
        Preconditions.checkArgument((boolean)tableHandle.isNamedRelation(), (String)"Cannot get columns for %s", (Object)tableHandle);
        TableInfo tableInfo = this.getTable(tableHandle.asPlainTable().getRemoteTableName().toTableId()).orElseThrow(() -> new TableNotFoundException(tableHandle.asPlainTable().getSchemaTableName()));
        return this.buildColumnHandles(tableInfo);
    }

    public List<BigQueryColumnHandle> buildColumnHandles(TableInfo tableInfo) {
        Schema schema = tableInfo.getDefinition().getSchema();
        if (schema == null) {
            SchemaTableName schemaTableName = new SchemaTableName(tableInfo.getTableId().getDataset(), tableInfo.getTableId().getTable());
            throw new TableNotFoundException(schemaTableName, String.format("Table '%s' has no schema", schemaTableName));
        }
        return (List)schema.getFields().stream().filter(this.typeManager::isSupportedType).map(this.typeManager::toColumnHandle).collect(ImmutableList.toImmutableList());
    }

    public static final class RemoteDatabaseObject {
        private final Set<String> remoteNames;

        private RemoteDatabaseObject(Set<String> remoteNames) {
            this.remoteNames = ImmutableSet.copyOf(remoteNames);
        }

        public static RemoteDatabaseObject of(String remoteName) {
            return new RemoteDatabaseObject((Set<String>)ImmutableSet.of((Object)remoteName));
        }

        public RemoteDatabaseObject registerCollision(String ambiguousName) {
            return new RemoteDatabaseObject((Set<String>)ImmutableSet.builderWithExpectedSize((int)(this.remoteNames.size() + 1)).addAll(this.remoteNames).add((Object)ambiguousName).build());
        }

        public String getAnyRemoteName() {
            return Collections.min(this.remoteNames);
        }

        public String getOnlyRemoteName() {
            if (!this.isAmbiguous()) {
                return (String)Iterables.getOnlyElement(this.remoteNames);
            }
            throw new TrinoException((ErrorCodeSupplier)BigQueryErrorCode.BIGQUERY_AMBIGUOUS_OBJECT_NAME, "Found ambiguous names in BigQuery when looking up '" + this.getAnyRemoteName().toLowerCase(Locale.ENGLISH) + "': " + String.valueOf(this.remoteNames));
        }

        public boolean isAmbiguous() {
            return this.remoteNames.size() > 1;
        }
    }
}

