/*
 * Decompiled with CFR 0.152.
 */
package io.trino.plugin.bigquery;

import com.google.cloud.bigquery.BigQuery;
import com.google.cloud.bigquery.BigQueryException;
import com.google.cloud.bigquery.BigQueryOptions;
import com.google.cloud.bigquery.DatasetId;
import com.google.cloud.bigquery.DatasetInfo;
import com.google.cloud.bigquery.Field;
import com.google.cloud.bigquery.FieldValue;
import com.google.cloud.bigquery.Job;
import com.google.cloud.bigquery.JobConfiguration;
import com.google.cloud.bigquery.JobException;
import com.google.cloud.bigquery.JobInfo;
import com.google.cloud.bigquery.JobStatistics;
import com.google.cloud.bigquery.QueryJobConfiguration;
import com.google.cloud.bigquery.Schema;
import com.google.cloud.bigquery.Table;
import com.google.cloud.bigquery.TableDefinition;
import com.google.cloud.bigquery.TableId;
import com.google.cloud.bigquery.TableInfo;
import com.google.cloud.bigquery.TableResult;
import com.google.common.base.Joiner;
import com.google.common.base.MoreObjects;
import com.google.common.base.Preconditions;
import com.google.common.base.Verify;
import com.google.common.cache.Cache;
import com.google.common.cache.CacheLoader;
import com.google.common.cache.LoadingCache;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.ImmutableMap;
import com.google.common.collect.ImmutableSet;
import com.google.common.collect.Iterables;
import com.google.common.collect.Streams;
import com.google.common.util.concurrent.UncheckedExecutionException;
import io.airlift.log.Logger;
import io.airlift.units.Duration;
import io.trino.cache.EvictableCacheBuilder;
import io.trino.plugin.bigquery.BigQueryColumnHandle;
import io.trino.plugin.bigquery.BigQueryErrorCode;
import io.trino.plugin.bigquery.BigQueryLabelFactory;
import io.trino.plugin.bigquery.BigQuerySessionProperties;
import io.trino.plugin.bigquery.BigQueryTableHandle;
import io.trino.plugin.bigquery.BigQueryTypeManager;
import io.trino.plugin.bigquery.BigQueryUtil;
import io.trino.plugin.bigquery.ViewMaterializationCache;
import io.trino.spi.ErrorCodeSupplier;
import io.trino.spi.TrinoException;
import io.trino.spi.connector.ConnectorSession;
import io.trino.spi.connector.RelationCommentMetadata;
import io.trino.spi.connector.SchemaTableName;
import io.trino.spi.connector.TableNotFoundException;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.OptionalLong;
import java.util.Set;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.TimeUnit;
import java.util.function.Supplier;
import java.util.stream.Collectors;
import java.util.stream.Stream;

public class BigQueryClient {
    private static final Logger log = Logger.get(BigQueryClient.class);
    static final Map<TableDefinition.Type, String> TABLE_TYPES = ImmutableMap.builder().put((Object)TableDefinition.Type.TABLE, (Object)"BASE TABLE").put((Object)TableDefinition.Type.VIEW, (Object)"VIEW").put((Object)TableDefinition.Type.MATERIALIZED_VIEW, (Object)"MATERIALIZED VIEW").put((Object)TableDefinition.Type.EXTERNAL, (Object)"EXTERNAL").put((Object)TableDefinition.Type.SNAPSHOT, (Object)"SNAPSHOT").buildOrThrow();
    private final BigQuery bigQuery;
    private final BigQueryLabelFactory labelFactory;
    private final BigQueryTypeManager typeManager;
    private final ViewMaterializationCache materializationCache;
    private final boolean caseInsensitiveNameMatching;
    private final LoadingCache<String, List<DatasetId>> remoteDatasetIdCache;
    private final int metadataPageSize;
    private final Cache<DatasetId, RemoteDatabaseObject> remoteDatasetCaseInsensitiveCache;
    private final Cache<TableId, RemoteDatabaseObject> remoteTableCaseInsensitiveCache;
    private final Optional<String> configProjectId;

    public BigQueryClient(BigQuery bigQuery, BigQueryLabelFactory labelFactory, BigQueryTypeManager typeManager, boolean caseInsensitiveNameMatching, Duration caseInsensitiveNameMatchingCacheTtl, ViewMaterializationCache materializationCache, Duration metadataCacheTtl, int metadataPageSize, Optional<String> configProjectId) {
        this.bigQuery = Objects.requireNonNull(bigQuery, "bigQuery is null");
        this.labelFactory = Objects.requireNonNull(labelFactory, "labelFactory is null");
        this.typeManager = Objects.requireNonNull(typeManager, "typeManager is null");
        this.materializationCache = Objects.requireNonNull(materializationCache, "materializationCache is null");
        this.caseInsensitiveNameMatching = caseInsensitiveNameMatching;
        this.remoteDatasetIdCache = EvictableCacheBuilder.newBuilder().expireAfterWrite(metadataCacheTtl.toMillis(), TimeUnit.MILLISECONDS).shareNothingWhenDisabled().build(CacheLoader.from(this::listDatasetIdsFromBigQuery));
        this.metadataPageSize = metadataPageSize;
        this.remoteDatasetCaseInsensitiveCache = BigQueryClient.buildCache(caseInsensitiveNameMatchingCacheTtl);
        this.remoteTableCaseInsensitiveCache = BigQueryClient.buildCache(caseInsensitiveNameMatchingCacheTtl);
        this.configProjectId = Objects.requireNonNull(configProjectId, "projectId is null");
    }

    private static <K, V> Cache<K, V> buildCache(Duration cachingTtl) {
        return EvictableCacheBuilder.newBuilder().expireAfterWrite(cachingTtl.toMillis(), TimeUnit.MILLISECONDS).shareNothingWhenDisabled().build();
    }

    public Optional<RemoteDatabaseObject> toRemoteDataset(DatasetId datasetId) {
        return this.toRemoteDataset(datasetId.getProject(), datasetId.getDataset());
    }

    public Optional<RemoteDatabaseObject> toRemoteDataset(String projectId, String datasetName) {
        return this.toRemoteDataset(projectId, datasetName, () -> this.listDatasetIds(projectId));
    }

    public Optional<RemoteDatabaseObject> toRemoteDataset(String projectId, String datasetName, Supplier<List<DatasetId>> datasetIds) {
        Objects.requireNonNull(projectId, "projectId is null");
        Objects.requireNonNull(datasetName, "datasetName is null");
        Verify.verify((boolean)datasetName.codePoints().noneMatch(Character::isUpperCase), (String)"Expected schema name from internal metadata to be lowercase: %s", (Object)datasetName);
        if (!this.caseInsensitiveNameMatching) {
            return Optional.of(RemoteDatabaseObject.of(datasetName));
        }
        DatasetId cacheKey = DatasetId.of((String)projectId, (String)datasetName);
        Optional<RemoteDatabaseObject> remoteDataSetFromCache = Optional.ofNullable((RemoteDatabaseObject)this.remoteDatasetCaseInsensitiveCache.getIfPresent((Object)cacheKey));
        if (remoteDataSetFromCache.isPresent()) {
            return remoteDataSetFromCache;
        }
        HashMap<DatasetId, RemoteDatabaseObject> mapping = new HashMap<DatasetId, RemoteDatabaseObject>((Map<DatasetId, RemoteDatabaseObject>)this.remoteDatasetCaseInsensitiveCache.getAllPresent(this.remoteDatasetCaseInsensitiveCache.asMap().keySet()));
        for (DatasetId datasetId : datasetIds.get()) {
            DatasetId newCacheKey = BigQueryClient.datasetIdToLowerCase(datasetId);
            RemoteDatabaseObject newValue = RemoteDatabaseObject.of(datasetId.getDataset());
            mapping.merge(newCacheKey, newValue, (currentValue, collision) -> currentValue.registerCollision(collision.getOnlyRemoteName()));
            BigQueryClient.updateCache(this.remoteDatasetCaseInsensitiveCache, newCacheKey, newValue);
        }
        return Optional.ofNullable((RemoteDatabaseObject)mapping.get(cacheKey));
    }

    public Optional<RemoteDatabaseObject> toRemoteTable(ConnectorSession session, String projectId, String remoteDatasetName, String tableName) {
        return this.toRemoteTable(projectId, remoteDatasetName, tableName, () -> this.findTableIdsIgnoreCase(session, DatasetId.of((String)projectId, (String)remoteDatasetName), tableName));
    }

    public List<SchemaTableName> listNonAmbiguousSchemaTableNames(String projectId, String remoteDatasetName, Iterable<TableId> tableIds) {
        Preconditions.checkState((projectId != null ? 1 : 0) != 0, (Object)"projectId was not set");
        Preconditions.checkState((remoteDatasetName != null ? 1 : 0) != 0, (Object)"remoteDatasetName was not set");
        if (!this.caseInsensitiveNameMatching) {
            return Streams.stream(tableIds).map(table -> new SchemaTableName(this.toSchemaName(DatasetId.of((String)projectId, (String)table.getDataset())), table.getTable())).collect(Collectors.toList());
        }
        ImmutableList.Builder schemaTableNamesBuilder = ImmutableList.builder();
        HashMap<TableId, Set> collisionTracker = new HashMap<TableId, Set>();
        for (TableId tableId2 : tableIds) {
            String tableName = tableId2.getTable().toLowerCase(Locale.ENGLISH);
            TableId cacheKey = TableId.of((String)projectId, (String)remoteDatasetName, (String)tableName);
            RemoteDatabaseObject remoteTableFromCache = (RemoteDatabaseObject)this.remoteTableCaseInsensitiveCache.getIfPresent((Object)cacheKey);
            if (remoteTableFromCache != null && remoteTableFromCache.remoteNames.size() == 1) {
                schemaTableNamesBuilder.add((Object)new SchemaTableName(this.toSchemaName(DatasetId.of((String)projectId, (String)remoteDatasetName)), cacheKey.getTable()));
                continue;
            }
            collisionTracker.computeIfAbsent(cacheKey, tableId -> new HashSet()).add(tableId2.getTable());
        }
        for (Map.Entry entry : collisionTracker.entrySet()) {
            TableId cacheKey = (TableId)entry.getKey();
            Set remoteNames = (Set)entry.getValue();
            if (remoteNames.size() == 1) {
                String uniqueTableName = (String)Iterables.getOnlyElement((Iterable)remoteNames);
                RemoteDatabaseObject remoteTable = RemoteDatabaseObject.of(uniqueTableName);
                BigQueryClient.updateCache(this.remoteTableCaseInsensitiveCache, cacheKey, remoteTable);
                schemaTableNamesBuilder.add((Object)new SchemaTableName(this.toSchemaName(DatasetId.of((String)projectId, (String)remoteDatasetName)), uniqueTableName));
                continue;
            }
            log.debug("Filtered out [%s] due to ambiguous remote names: %s", new Object[]{cacheKey.getTable(), remoteNames});
        }
        return schemaTableNamesBuilder.build();
    }

    private Optional<RemoteDatabaseObject> toRemoteTable(String projectId, String remoteDatasetName, String tableName, Supplier<Iterable<TableId>> tableIds) {
        Objects.requireNonNull(projectId, "projectId is null");
        Objects.requireNonNull(remoteDatasetName, "remoteDatasetName is null");
        Objects.requireNonNull(tableName, "tableName is null");
        Verify.verify((boolean)tableName.codePoints().noneMatch(Character::isUpperCase), (String)"Expected table name from internal metadata to be lowercase: %s", (Object)tableName);
        if (!this.caseInsensitiveNameMatching) {
            return Optional.of(RemoteDatabaseObject.of(tableName));
        }
        TableId cacheKey = TableId.of((String)projectId, (String)remoteDatasetName, (String)tableName);
        Optional<RemoteDatabaseObject> remoteTableFromCache = Optional.ofNullable((RemoteDatabaseObject)this.remoteTableCaseInsensitiveCache.getIfPresent((Object)cacheKey));
        if (remoteTableFromCache.isPresent()) {
            return remoteTableFromCache;
        }
        HashMap<TableId, RemoteDatabaseObject> mapping = new HashMap<TableId, RemoteDatabaseObject>((Map<TableId, RemoteDatabaseObject>)this.remoteTableCaseInsensitiveCache.getAllPresent(this.remoteTableCaseInsensitiveCache.asMap().keySet()));
        for (TableId table : tableIds.get()) {
            TableId newCacheKey = BigQueryClient.tableIdToLowerCase(table);
            RemoteDatabaseObject newValue = RemoteDatabaseObject.of(table.getTable());
            mapping.merge(newCacheKey, newValue, (currentValue, collision) -> currentValue.registerCollision(collision.getOnlyRemoteName()));
            BigQueryClient.updateCache(this.remoteTableCaseInsensitiveCache, newCacheKey, newValue);
        }
        return Optional.ofNullable((RemoteDatabaseObject)mapping.get(cacheKey));
    }

    private static <T> void updateCache(Cache<T, RemoteDatabaseObject> caseInsensitiveCache, T newCacheKey, RemoteDatabaseObject newValue) {
        try {
            RemoteDatabaseObject currentCacheValue = (RemoteDatabaseObject)caseInsensitiveCache.getIfPresent(newCacheKey);
            if (currentCacheValue == null) {
                caseInsensitiveCache.get(newCacheKey, () -> newValue);
            } else if (!currentCacheValue.remoteNames.contains(newValue.getOnlyRemoteName())) {
                RemoteDatabaseObject mergedValue = currentCacheValue.registerCollision(newValue.getOnlyRemoteName());
                caseInsensitiveCache.invalidate(newCacheKey);
                caseInsensitiveCache.get(newCacheKey, () -> mergedValue);
            }
        }
        catch (ExecutionException e) {
            throw new UncheckedExecutionException((Throwable)e);
        }
    }

    private static DatasetId datasetIdToLowerCase(DatasetId datasetId) {
        return DatasetId.of((String)datasetId.getProject(), (String)datasetId.getDataset().toLowerCase(Locale.ENGLISH));
    }

    private static TableId tableIdToLowerCase(TableId tableId) {
        return TableId.of((String)tableId.getProject(), (String)tableId.getDataset(), (String)tableId.getTable().toLowerCase(Locale.ENGLISH));
    }

    public DatasetInfo getDataset(DatasetId datasetId) {
        return this.bigQuery.getDataset(datasetId, new BigQuery.DatasetOption[0]);
    }

    public Optional<TableInfo> getTable(TableId remoteTableId) {
        try {
            return Optional.ofNullable(this.bigQuery.getTable(remoteTableId, new BigQuery.TableOption[0]));
        }
        catch (BigQueryException e) {
            log.debug((Throwable)e, "Failed to get table '%s'", new Object[]{remoteTableId});
            return Optional.empty();
        }
    }

    public TableInfo getCachedTable(Duration viewExpiration, TableInfo remoteTableId, List<BigQueryColumnHandle> requiredColumns, Optional<String> filter) {
        String query = BigQueryClient.selectSql(remoteTableId.getTableId(), requiredColumns, filter, OptionalLong.empty());
        log.debug("query is %s", new Object[]{query});
        return this.materializationCache.getCachedTable(this, query, viewExpiration, remoteTableId);
    }

    public String getParentProjectId() {
        return ((BigQueryOptions)this.bigQuery.getOptions()).getProjectId();
    }

    public String getProjectId() {
        String projectId = this.configProjectId.orElseGet(() -> ((BigQueryOptions)this.bigQuery.getOptions()).getProjectId());
        Preconditions.checkState((boolean)projectId.toLowerCase(Locale.ENGLISH).equals(projectId), (Object)("projectId must be lowercase but it's " + projectId));
        return projectId;
    }

    protected DatasetId toDatasetId(String schemaName) {
        return DatasetId.of((String)this.getProjectId(), (String)schemaName);
    }

    protected String toSchemaName(DatasetId datasetId) {
        return datasetId.getDataset();
    }

    public List<DatasetId> listDatasetIds(String projectId) {
        try {
            return (List)this.remoteDatasetIdCache.get((Object)projectId);
        }
        catch (ExecutionException e) {
            throw new TrinoException((ErrorCodeSupplier)BigQueryErrorCode.BIGQUERY_LISTING_DATASET_ERROR, "Failed to retrieve datasets from BigQuery", (Throwable)e);
        }
    }

    private List<DatasetId> listDatasetIdsFromBigQuery(String projectId) {
        return (List)Streams.stream((Iterable)this.bigQuery.listDatasets(projectId, new BigQuery.DatasetListOption[]{BigQuery.DatasetListOption.pageSize((long)this.metadataPageSize)}).iterateAll()).map(DatasetInfo::getDatasetId).collect(ImmutableList.toImmutableList());
    }

    public Iterable<TableId> listTableIds(DatasetId remoteDatasetId) {
        Iterable allTables;
        try {
            allTables = this.bigQuery.listTables(remoteDatasetId, new BigQuery.TableListOption[]{BigQuery.TableListOption.pageSize((long)this.metadataPageSize)}).iterateAll();
        }
        catch (BigQueryException e) {
            throw new TrinoException((ErrorCodeSupplier)BigQueryErrorCode.BIGQUERY_LISTING_TABLE_ERROR, "Failed to retrieve tables from BigQuery", (Throwable)e);
        }
        return (Iterable)Streams.stream((Iterable)allTables).filter(table -> TABLE_TYPES.containsKey(table.getDefinition().getType())).map(TableInfo::getTableId).collect(ImmutableList.toImmutableList());
    }

    public Iterable<TableId> findTableIdsIgnoreCase(ConnectorSession session, DatasetId remoteDatasetId, String tableName) {
        try {
            TableResult tableNamesMatchingResults = this.executeQuery(session, "SELECT table_name\nFROM %s.%s.INFORMATION_SCHEMA.TABLES\nWHERE LOWER(table_name) = '%s' AND table_type IN (%s)".formatted(BigQueryUtil.quote(remoteDatasetId.getProject()), BigQueryUtil.quote(remoteDatasetId.getDataset()), tableName.toLowerCase(Locale.ENGLISH), TABLE_TYPES.values().stream().map(value -> "'" + value + "'").collect(Collectors.joining(","))));
            return (Iterable)tableNamesMatchingResults.streamAll().map(row -> TableId.of((String)remoteDatasetId.getProject(), (String)remoteDatasetId.getDataset(), (String)((FieldValue)row.getFirst()).getStringValue())).collect(ImmutableList.toImmutableList());
        }
        catch (TrinoException e) {
            if (e.getMessage().contains("Dataset %s:%s was not found".formatted(remoteDatasetId.getProject(), remoteDatasetId.getDataset()))) {
                return ImmutableList.of();
            }
            throw e;
        }
    }

    Table update(TableInfo table) {
        return this.bigQuery.update(table, new BigQuery.TableOption[0]);
    }

    public void createSchema(DatasetInfo datasetInfo) {
        this.bigQuery.create(datasetInfo, new BigQuery.DatasetOption[0]);
        this.remoteDatasetIdCache.invalidate((Object)datasetInfo.getDatasetId().getProject());
        this.remoteDatasetCaseInsensitiveCache.invalidate((Object)BigQueryClient.datasetIdToLowerCase(datasetInfo.getDatasetId()));
    }

    public void dropSchema(DatasetId datasetId, boolean cascade) {
        if (cascade) {
            this.bigQuery.delete(datasetId, new BigQuery.DatasetDeleteOption[]{BigQuery.DatasetDeleteOption.deleteContents()});
        } else {
            this.bigQuery.delete(datasetId, new BigQuery.DatasetDeleteOption[0]);
        }
        this.remoteDatasetIdCache.invalidate((Object)datasetId.getProject());
        this.remoteDatasetCaseInsensitiveCache.invalidate((Object)BigQueryClient.datasetIdToLowerCase(datasetId));
    }

    public void createTable(TableInfo tableInfo) {
        this.bigQuery.create(tableInfo, new BigQuery.TableOption[0]);
        this.remoteTableCaseInsensitiveCache.invalidate((Object)BigQueryClient.tableIdToLowerCase(tableInfo.getTableId()));
    }

    public void dropTable(TableId tableId) {
        this.bigQuery.delete(tableId);
        this.remoteTableCaseInsensitiveCache.invalidate((Object)BigQueryClient.tableIdToLowerCase(tableId));
    }

    Job create(JobInfo jobInfo) {
        return this.bigQuery.create(jobInfo, new BigQuery.JobOption[0]);
    }

    public long executeUpdate(ConnectorSession session, QueryJobConfiguration job) {
        log.debug("Execute query: %s", new Object[]{job.getQuery()});
        return this.execute(session, job).getTotalRows();
    }

    public TableResult executeQuery(ConnectorSession session, String sql) {
        return this.executeQuery(session, sql, null);
    }

    public TableResult executeQuery(ConnectorSession session, String sql, Long maxResults) {
        log.debug("Execute query: %s", new Object[]{sql});
        QueryJobConfiguration job = QueryJobConfiguration.newBuilder((String)sql).setUseQueryCache(Boolean.valueOf(BigQuerySessionProperties.isQueryResultsCacheEnabled(session))).setCreateDisposition(BigQuerySessionProperties.createDisposition(session)).setMaxResults(maxResults).build();
        return this.execute(session, job);
    }

    private TableResult execute(ConnectorSession session, QueryJobConfiguration job) {
        QueryJobConfiguration jobWithQueryLabel = job.toBuilder().setLabels(this.labelFactory.getLabels(session)).build();
        try {
            return this.bigQuery.query(jobWithQueryLabel, new BigQuery.JobOption[0]);
        }
        catch (BigQueryException | JobException e) {
            throw new TrinoException((ErrorCodeSupplier)BigQueryErrorCode.BIGQUERY_FAILED_TO_EXECUTE_QUERY, "Failed to run the query: " + String.valueOf(MoreObjects.firstNonNull((Object)e.getMessage(), (Object)e)), e);
        }
        catch (InterruptedException e) {
            Thread.currentThread().interrupt();
            throw new BigQueryException(0, String.format("Failed to run the query [%s]", job.getQuery()), (Throwable)e);
        }
    }

    public Schema getSchema(String sql) {
        JobStatistics statistics;
        log.debug("Get schema from query: %s", new Object[]{sql});
        JobInfo jobInfo = JobInfo.of((JobConfiguration)QueryJobConfiguration.newBuilder((String)sql).setDryRun(Boolean.valueOf(true)).build());
        try {
            statistics = this.bigQuery.create(jobInfo, new BigQuery.JobOption[0]).getStatistics();
        }
        catch (BigQueryException e) {
            throw new TrinoException((ErrorCodeSupplier)BigQueryErrorCode.BIGQUERY_INVALID_STATEMENT, "Failed to get schema for query: " + sql, (Throwable)e);
        }
        JobStatistics.QueryStatistics queryStatistics = (JobStatistics.QueryStatistics)statistics;
        if (!queryStatistics.getStatementType().equals((Object)JobStatistics.QueryStatistics.StatementType.SELECT)) {
            throw new TrinoException((ErrorCodeSupplier)BigQueryErrorCode.BIGQUERY_INVALID_STATEMENT, "Unsupported statement type: " + String.valueOf(queryStatistics.getStatementType()));
        }
        return Objects.requireNonNull(queryStatistics.getSchema(), "Cannot determine schema for query");
    }

    public boolean useStorageApi(String sql, TableId destinationTable) {
        block2: {
            JobInfo jobInfo = JobInfo.of((JobConfiguration)QueryJobConfiguration.newBuilder((String)sql).setDryRun(Boolean.valueOf(true)).setDestinationTable(destinationTable).build());
            try {
                this.bigQuery.create(jobInfo, new BigQuery.JobOption[0]);
            }
            catch (BigQueryException e) {
                if (!e.getMessage().startsWith("Duplicate column names in the result are not supported when a destination table is present.")) break block2;
                return false;
            }
        }
        return true;
    }

    public TableId getDestinationTable(String sql) {
        JobConfiguration jobConfiguration;
        log.debug("Get destination table from query: %s", new Object[]{sql});
        JobInfo jobInfo = JobInfo.of((JobConfiguration)QueryJobConfiguration.newBuilder((String)sql).setDryRun(Boolean.valueOf(true)).build());
        try {
            jobConfiguration = this.bigQuery.create(jobInfo, new BigQuery.JobOption[0]).getConfiguration();
        }
        catch (BigQueryException e) {
            throw new TrinoException((ErrorCodeSupplier)BigQueryErrorCode.BIGQUERY_INVALID_STATEMENT, "Failed to get destination table for query. " + String.valueOf(MoreObjects.firstNonNull((Object)e.getMessage(), (Object)((Object)e))), (Throwable)e);
        }
        return Objects.requireNonNull(((QueryJobConfiguration)jobConfiguration).getDestinationTable(), "Cannot determine destination table for query");
    }

    public static String selectSql(TableId table, List<BigQueryColumnHandle> requiredColumns, Optional<String> filter, OptionalLong limit) {
        return BigQueryClient.selectSql(table, requiredColumns.stream().map(column -> Joiner.on((char)'.').join((Iterable)ImmutableList.builder().add((Object)String.format("`%s`", column.name())).addAll((Iterable)column.dereferenceNames().stream().map(dereferenceName -> String.format("`%s`", dereferenceName)).collect(ImmutableList.toImmutableList())).build())).collect(Collectors.joining(",")), filter, limit);
    }

    public static String selectSql(TableId table, String formattedColumns, Optional<String> filter, OptionalLong limit) {
        String tableName = BigQueryClient.fullTableName(table);
        Object query = String.format("SELECT %s FROM `%s`", formattedColumns, tableName);
        if (filter.isPresent()) {
            query = (String)query + " WHERE " + filter.get();
        }
        if (limit.isPresent()) {
            query = (String)query + " LIMIT " + limit.getAsLong();
        }
        return query;
    }

    private static String fullTableName(TableId remoteTableId) {
        return String.format("%s.%s.%s", remoteTableId.getProject(), remoteTableId.getDataset(), remoteTableId.getTable());
    }

    public Stream<RelationCommentMetadata> listRelationCommentMetadata(ConnectorSession session, BigQueryClient client, String schemaName) {
        TableResult result = client.executeQuery(session, "SELECT tbls.table_name, options.option_value\nFROM %1$s.`INFORMATION_SCHEMA`.`TABLES` tbls\nLEFT JOIN %1$s.`INFORMATION_SCHEMA`.`TABLE_OPTIONS` options\nON tbls.table_schema = options.table_schema AND tbls.table_name = options.table_name AND options.option_name = 'description'\n".formatted(BigQueryUtil.quote(schemaName)));
        return result.streamValues().map(row -> {
            Optional comment = row.get(1).isNull() ? Optional.empty() : Optional.of(BigQueryClient.unquoteOptionValue(row.get(1).getStringValue()));
            return new RelationCommentMetadata(new SchemaTableName(schemaName, row.get(0).getStringValue()), false, comment);
        });
    }

    private static String unquoteOptionValue(String quoted) {
        return quoted.substring(1, quoted.length() - 1).replace("\"\"", "\"").replace("\\\\", "\\").replace("\\\"", "\"");
    }

    public List<BigQueryColumnHandle> getColumns(BigQueryTableHandle tableHandle) {
        if (tableHandle.projectedColumns().isPresent()) {
            return tableHandle.projectedColumns().get();
        }
        Preconditions.checkArgument((boolean)tableHandle.isNamedRelation(), (String)"Cannot get columns for %s", (Object)tableHandle);
        TableInfo tableInfo = this.getTable(tableHandle.asPlainTable().getRemoteTableName().toTableId()).orElseThrow(() -> new TableNotFoundException(tableHandle.asPlainTable().getSchemaTableName()));
        return this.buildColumnHandles(tableInfo, tableHandle.relationHandle().isUseStorageApi());
    }

    public List<BigQueryColumnHandle> buildColumnHandles(TableInfo tableInfo, boolean useStorageApi) {
        Schema schema = tableInfo.getDefinition().getSchema();
        if (schema == null) {
            SchemaTableName schemaTableName = new SchemaTableName(tableInfo.getTableId().getDataset(), tableInfo.getTableId().getTable());
            throw new TableNotFoundException(schemaTableName, String.format("Table '%s' has no schema", schemaTableName));
        }
        return (List)schema.getFields().stream().filter(field -> this.typeManager.isSupportedType((Field)field, useStorageApi)).map(field -> this.typeManager.toColumnHandle((Field)field, useStorageApi)).collect(ImmutableList.toImmutableList());
    }

    public static final class RemoteDatabaseObject {
        private final Set<String> remoteNames;

        private RemoteDatabaseObject(Set<String> remoteNames) {
            this.remoteNames = ImmutableSet.copyOf(remoteNames);
        }

        public static RemoteDatabaseObject of(String remoteName) {
            return new RemoteDatabaseObject((Set<String>)ImmutableSet.of((Object)remoteName));
        }

        public RemoteDatabaseObject registerCollision(String ambiguousName) {
            return new RemoteDatabaseObject((Set<String>)ImmutableSet.builderWithExpectedSize((int)(this.remoteNames.size() + 1)).addAll(this.remoteNames).add((Object)ambiguousName).build());
        }

        public String getAnyRemoteName() {
            return Collections.min(this.remoteNames);
        }

        public String getOnlyRemoteName() {
            if (!this.isAmbiguous()) {
                return (String)Iterables.getOnlyElement(this.remoteNames);
            }
            throw new TrinoException((ErrorCodeSupplier)BigQueryErrorCode.BIGQUERY_AMBIGUOUS_OBJECT_NAME, "Found ambiguous names in BigQuery when looking up '" + this.getAnyRemoteName().toLowerCase(Locale.ENGLISH) + "': " + String.valueOf(this.remoteNames));
        }

        public boolean isAmbiguous() {
            return this.remoteNames.size() > 1;
        }
    }
}

