/*
 * Decompiled with CFR 0.152.
 */
package io.trino.plugin.bigquery;

import com.google.cloud.bigquery.storage.v1.BigQueryReadClient;
import com.google.cloud.bigquery.storage.v1.ReadRowsResponse;
import com.google.common.base.MoreObjects;
import com.google.common.base.Preconditions;
import com.google.common.base.Verify;
import com.google.common.collect.ImmutableList;
import io.airlift.concurrent.MoreFutures;
import io.airlift.log.Logger;
import io.airlift.slice.Slice;
import io.airlift.slice.Slices;
import io.trino.plugin.bigquery.BigQueryColumnHandle;
import io.trino.plugin.bigquery.BigQuerySplit;
import io.trino.plugin.bigquery.BigQueryTypeManager;
import io.trino.plugin.bigquery.BigQueryUtil;
import io.trino.plugin.bigquery.ReadRowsHelper;
import io.trino.spi.ErrorCodeSupplier;
import io.trino.spi.Page;
import io.trino.spi.PageBuilder;
import io.trino.spi.StandardErrorCode;
import io.trino.spi.TrinoException;
import io.trino.spi.block.ArrayBlockBuilder;
import io.trino.spi.block.BlockBuilder;
import io.trino.spi.block.RowBlockBuilder;
import io.trino.spi.connector.ConnectorPageSource;
import io.trino.spi.connector.SourcePage;
import io.trino.spi.type.ArrayType;
import io.trino.spi.type.BigintType;
import io.trino.spi.type.DateType;
import io.trino.spi.type.DecimalType;
import io.trino.spi.type.Decimals;
import io.trino.spi.type.Int128;
import io.trino.spi.type.IntegerType;
import io.trino.spi.type.LongTimestampWithTimeZone;
import io.trino.spi.type.RowType;
import io.trino.spi.type.TimeType;
import io.trino.spi.type.TimeZoneKey;
import io.trino.spi.type.TimestampType;
import io.trino.spi.type.TimestampWithTimeZoneType;
import io.trino.spi.type.Type;
import io.trino.spi.type.VarbinaryType;
import io.trino.spi.type.VarcharType;
import java.io.IOException;
import java.io.UncheckedIOException;
import java.math.BigDecimal;
import java.nio.ByteBuffer;
import java.util.Iterator;
import java.util.List;
import java.util.NoSuchElementException;
import java.util.Objects;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.atomic.AtomicLong;
import org.apache.avro.Conversions;
import org.apache.avro.Schema;
import org.apache.avro.SchemaParseException;
import org.apache.avro.generic.GenericDatumReader;
import org.apache.avro.generic.GenericRecord;
import org.apache.avro.io.BinaryDecoder;
import org.apache.avro.io.Decoder;
import org.apache.avro.io.DecoderFactory;
import org.apache.avro.util.Utf8;

public class BigQueryStorageAvroPageSource
implements ConnectorPageSource {
    private static final Logger log = Logger.get(BigQueryStorageAvroPageSource.class);
    private static final AvroDecimalConverter DECIMAL_CONVERTER = new AvroDecimalConverter();
    private final BigQueryReadClient bigQueryReadClient;
    private final ExecutorService executor;
    private final BigQueryTypeManager typeManager;
    private final String streamName;
    private final Schema avroSchema;
    private final List<BigQueryColumnHandle> columns;
    private final AtomicLong readBytes = new AtomicLong();
    private final AtomicLong readTimeNanos = new AtomicLong();
    private final PageBuilder pageBuilder;
    private final Iterator<ReadRowsResponse> responses;
    private CompletableFuture<ReadRowsResponse> nextResponse;
    private boolean finished;

    public BigQueryStorageAvroPageSource(BigQueryReadClient bigQueryReadClient, ExecutorService executor, BigQueryTypeManager typeManager, int maxReadRowsRetries, BigQuerySplit split, List<BigQueryColumnHandle> columns) {
        this.bigQueryReadClient = Objects.requireNonNull(bigQueryReadClient, "bigQueryReadClient is null");
        this.executor = Objects.requireNonNull(executor, "executor is null");
        this.typeManager = Objects.requireNonNull(typeManager, "typeManager is null");
        Objects.requireNonNull(split, "split is null");
        this.streamName = split.streamName();
        this.avroSchema = this.parseSchema(split.schemaString());
        this.columns = Objects.requireNonNull(columns, "columns is null");
        this.pageBuilder = new PageBuilder((List)columns.stream().map(BigQueryColumnHandle::trinoType).collect(ImmutableList.toImmutableList()));
        log.debug("Starting to read from %s", new Object[]{this.streamName});
        this.responses = new ReadRowsHelper(bigQueryReadClient, this.streamName, maxReadRowsRetries).readRows();
        this.nextResponse = CompletableFuture.supplyAsync(this::getResponse, executor);
    }

    private Schema parseSchema(String schemaString) {
        try {
            return new Schema.Parser().parse(schemaString);
        }
        catch (SchemaParseException e) {
            throw new TrinoException((ErrorCodeSupplier)StandardErrorCode.GENERIC_INTERNAL_ERROR, "Invalid Avro schema: " + String.valueOf(MoreObjects.firstNonNull((Object)e.getMessage(), (Object)((Object)e))), (Throwable)e);
        }
    }

    public long getCompletedBytes() {
        return this.readBytes.get();
    }

    public long getReadTimeNanos() {
        return this.readTimeNanos.get();
    }

    public boolean isFinished() {
        return this.finished;
    }

    public SourcePage getNextSourcePage() {
        ReadRowsResponse response;
        Preconditions.checkState((boolean)this.pageBuilder.isEmpty(), (Object)"PageBuilder is not empty at the beginning of a new page");
        if (!this.nextResponse.isDone()) {
            return null;
        }
        try {
            response = (ReadRowsResponse)MoreFutures.getFutureValue(this.nextResponse);
        }
        catch (NoSuchElementException ignored) {
            this.finished = true;
            return null;
        }
        this.nextResponse = CompletableFuture.supplyAsync(this::getResponse, this.executor);
        long start = System.nanoTime();
        Iterable<GenericRecord> records = this.parse(response);
        for (GenericRecord record : records) {
            this.pageBuilder.declarePosition();
            for (int column = 0; column < this.columns.size(); ++column) {
                BlockBuilder output = this.pageBuilder.getBlockBuilder(column);
                BigQueryColumnHandle columnHandle = this.columns.get(column);
                this.appendTo(columnHandle.trinoType(), BigQueryStorageAvroPageSource.getValueRecord(record, columnHandle), output);
            }
        }
        Page page = this.pageBuilder.build();
        this.pageBuilder.reset();
        this.readTimeNanos.addAndGet(System.nanoTime() - start);
        return SourcePage.create((Page)page);
    }

    private static Object getValueRecord(GenericRecord record, BigQueryColumnHandle columnHandle) {
        Object valueRecord = record.get(BigQueryUtil.toBigQueryColumnName(columnHandle.name()));
        for (String dereferenceName : columnHandle.dereferenceNames()) {
            if (valueRecord == null) break;
            if (valueRecord instanceof GenericRecord) {
                GenericRecord genericRecord = (GenericRecord)valueRecord;
                valueRecord = genericRecord.get(dereferenceName);
                continue;
            }
            throw new TrinoException((ErrorCodeSupplier)StandardErrorCode.GENERIC_INTERNAL_ERROR, "Failed to extract dereference value from record");
        }
        return valueRecord;
    }

    private void appendTo(Type type, Object value, BlockBuilder output) {
        block17: {
            if (value == null) {
                output.appendNull();
                return;
            }
            Class javaType = type.getJavaType();
            try {
                if (javaType == Boolean.TYPE) {
                    type.writeBoolean(output, ((Boolean)value).booleanValue());
                    break block17;
                }
                if (javaType == Long.TYPE) {
                    if (type.equals((Object)BigintType.BIGINT)) {
                        type.writeLong(output, ((Number)value).longValue());
                        break block17;
                    }
                    if (type.equals((Object)IntegerType.INTEGER)) {
                        type.writeLong(output, (long)((Number)value).intValue());
                        break block17;
                    }
                    if (type instanceof DecimalType) {
                        DecimalType decimalType = (DecimalType)type;
                        Verify.verify((boolean)decimalType.isShort(), (String)"The type should be short decimal", (Object[])new Object[0]);
                        BigDecimal decimal = DECIMAL_CONVERTER.convert(decimalType.getPrecision(), decimalType.getScale(), value);
                        type.writeLong(output, Decimals.encodeShortScaledValue((BigDecimal)decimal, (int)decimalType.getScale()));
                        break block17;
                    }
                    if (type.equals((Object)DateType.DATE)) {
                        type.writeLong(output, (long)((Number)value).intValue());
                        break block17;
                    }
                    if (type.equals((Object)TimestampType.TIMESTAMP_MICROS)) {
                        type.writeLong(output, BigQueryTypeManager.toTrinoTimestamp(((Utf8)value).toString()));
                        break block17;
                    }
                    if (type.equals((Object)TimeType.TIME_MICROS)) {
                        type.writeLong(output, (Long)value * 1000000L);
                        break block17;
                    }
                    throw new TrinoException((ErrorCodeSupplier)StandardErrorCode.GENERIC_INTERNAL_ERROR, String.format("Unhandled type for %s: %s", javaType.getSimpleName(), type));
                }
                if (javaType == Double.TYPE) {
                    type.writeDouble(output, ((Number)value).doubleValue());
                    break block17;
                }
                if (type.getJavaType() == Int128.class) {
                    BigQueryStorageAvroPageSource.writeObject(output, type, value);
                    break block17;
                }
                if (javaType == Slice.class) {
                    this.writeSlice(output, type, value);
                    break block17;
                }
                if (javaType == LongTimestampWithTimeZone.class) {
                    Verify.verify((boolean)type.equals((Object)TimestampWithTimeZoneType.TIMESTAMP_TZ_MICROS));
                    long epochMicros = (Long)value;
                    int picosOfMillis = Math.toIntExact(Math.floorMod(epochMicros, 1000)) * 1000000;
                    type.writeObject(output, (Object)LongTimestampWithTimeZone.fromEpochMillisAndFraction((long)Math.floorDiv(epochMicros, 1000), (int)picosOfMillis, (TimeZoneKey)TimeZoneKey.UTC_KEY));
                    break block17;
                }
                if (type instanceof ArrayType) {
                    ArrayType arrayType = (ArrayType)type;
                    this.writeArray((ArrayBlockBuilder)output, (List)value, arrayType);
                    break block17;
                }
                if (type instanceof RowType) {
                    RowType rowType = (RowType)type;
                    this.writeRow((RowBlockBuilder)output, rowType, (GenericRecord)value);
                    break block17;
                }
                throw new TrinoException((ErrorCodeSupplier)StandardErrorCode.GENERIC_INTERNAL_ERROR, String.format("Unhandled type for %s: %s", javaType.getSimpleName(), type));
            }
            catch (ClassCastException ignore) {
                output.appendNull();
            }
        }
    }

    private void writeSlice(BlockBuilder output, Type type, Object value) {
        if (type instanceof VarcharType) {
            type.writeSlice(output, Slices.utf8Slice((String)((Utf8)value).toString()));
        } else if (type instanceof VarbinaryType) {
            if (value instanceof ByteBuffer) {
                ByteBuffer bytes = (ByteBuffer)value;
                type.writeSlice(output, Slices.wrappedHeapBuffer((ByteBuffer)bytes));
            } else {
                output.appendNull();
            }
        } else if (this.typeManager.isJsonType(type)) {
            type.writeSlice(output, Slices.utf8Slice((String)((Utf8)value).toString()));
        } else {
            throw new TrinoException((ErrorCodeSupplier)StandardErrorCode.GENERIC_INTERNAL_ERROR, "Unhandled type for Slice: " + String.valueOf(type.getTypeSignature()));
        }
    }

    private static void writeObject(BlockBuilder output, Type type, Object value) {
        if (!(type instanceof DecimalType)) {
            throw new TrinoException((ErrorCodeSupplier)StandardErrorCode.GENERIC_INTERNAL_ERROR, "Unhandled type for Object: " + String.valueOf(type.getTypeSignature()));
        }
        DecimalType decimalType = (DecimalType)type;
        Verify.verify((!decimalType.isShort() ? 1 : 0) != 0, (String)"The type should be long decimal", (Object[])new Object[0]);
        BigDecimal decimal = DECIMAL_CONVERTER.convert(decimalType.getPrecision(), decimalType.getScale(), value);
        type.writeObject(output, (Object)Decimals.encodeScaledValue((BigDecimal)decimal, (int)decimalType.getScale()));
    }

    private void writeArray(ArrayBlockBuilder output, List<?> value, ArrayType arrayType) {
        Type elementType = arrayType.getElementType();
        output.buildEntry(elementBuilder -> {
            for (Object element : value) {
                this.appendTo(elementType, element, elementBuilder);
            }
        });
    }

    private void writeRow(RowBlockBuilder output, RowType rowType, GenericRecord record) {
        List fields = rowType.getFields();
        output.buildEntry(fieldBuilders -> {
            for (int index = 0; index < fields.size(); ++index) {
                RowType.Field field = (RowType.Field)fields.get(index);
                this.appendTo(field.getType(), record.get((String)((Object)field.getName().orElse("field" + index))), (BlockBuilder)fieldBuilders.get(index));
            }
        });
    }

    public long getMemoryUsage() {
        return this.pageBuilder.getRetainedSizeInBytes();
    }

    public void close() {
        this.nextResponse.cancel(true);
        this.bigQueryReadClient.close();
    }

    public CompletableFuture<?> isBlocked() {
        return this.nextResponse;
    }

    private ReadRowsResponse getResponse() {
        long start = System.nanoTime();
        ReadRowsResponse response = this.responses.next();
        this.readTimeNanos.addAndGet(System.nanoTime() - start);
        return response;
    }

    Iterable<GenericRecord> parse(ReadRowsResponse response) {
        byte[] buffer = response.getAvroRows().getSerializedBinaryRows().toByteArray();
        this.readBytes.addAndGet(buffer.length);
        log.debug("Read %d bytes (total %d) from %s", new Object[]{buffer.length, this.readBytes.get(), this.streamName});
        return () -> new AvroBinaryIterator(this.avroSchema, buffer);
    }

    static class AvroDecimalConverter {
        private static final Conversions.DecimalConversion AVRO_DECIMAL_CONVERSION = new Conversions.DecimalConversion();

        AvroDecimalConverter() {
        }

        BigDecimal convert(int precision, int scale, Object value) {
            Schema schema = new Schema.Parser().parse(String.format("{\"type\":\"bytes\",\"logicalType\":\"decimal\",\"precision\":%d,\"scale\":%d}", precision, scale));
            return AVRO_DECIMAL_CONVERSION.fromBytes((ByteBuffer)value, schema, schema.getLogicalType());
        }
    }

    private static class AvroBinaryIterator
    implements Iterator<GenericRecord> {
        GenericDatumReader<GenericRecord> reader;
        BinaryDecoder in;

        AvroBinaryIterator(Schema avroSchema, byte[] buffer) {
            this.reader = new GenericDatumReader(avroSchema);
            this.in = new DecoderFactory().binaryDecoder(buffer, null);
        }

        @Override
        public boolean hasNext() {
            try {
                return !this.in.isEnd();
            }
            catch (IOException e) {
                throw new UncheckedIOException("Error determining the end of Avro buffer", e);
            }
        }

        @Override
        public GenericRecord next() {
            try {
                return (GenericRecord)this.reader.read(null, (Decoder)this.in);
            }
            catch (IOException e) {
                throw new UncheckedIOException("Error reading next Avro Record", e);
            }
        }
    }
}

