/*
 * Decompiled with CFR 0.152.
 */
package io.trino.plugin.bigquery;

import com.google.api.gax.rpc.ApiException;
import com.google.cloud.bigquery.TableDefinition;
import com.google.cloud.bigquery.TableId;
import com.google.cloud.bigquery.TableInfo;
import com.google.cloud.bigquery.storage.v1.ArrowSerializationOptions;
import com.google.cloud.bigquery.storage.v1.BigQueryReadClient;
import com.google.cloud.bigquery.storage.v1.CreateReadSessionRequest;
import com.google.cloud.bigquery.storage.v1.DataFormat;
import com.google.cloud.bigquery.storage.v1.ReadSession;
import com.google.common.base.MoreObjects;
import dev.failsafe.Failsafe;
import dev.failsafe.Policy;
import dev.failsafe.RetryPolicy;
import dev.failsafe.RetryPolicyBuilder;
import io.airlift.log.Logger;
import io.airlift.units.Duration;
import io.trino.plugin.bigquery.BigQueryClient;
import io.trino.plugin.bigquery.BigQueryClientFactory;
import io.trino.plugin.bigquery.BigQueryColumnHandle;
import io.trino.plugin.bigquery.BigQueryErrorCode;
import io.trino.plugin.bigquery.BigQueryReadClientFactory;
import io.trino.plugin.bigquery.BigQuerySessionProperties;
import io.trino.plugin.bigquery.BigQueryUtil;
import io.trino.spi.ErrorCodeSupplier;
import io.trino.spi.StandardErrorCode;
import io.trino.spi.TrinoException;
import io.trino.spi.connector.ConnectorSession;
import io.trino.spi.connector.SchemaTableName;
import io.trino.spi.connector.TableNotFoundException;
import java.time.temporal.ChronoUnit;
import java.util.List;
import java.util.Optional;
import java.util.stream.Collectors;

public class ReadSessionCreator {
    private static final Logger log = Logger.get(ReadSessionCreator.class);
    private final BigQueryClientFactory bigQueryClientFactory;
    private final BigQueryReadClientFactory bigQueryReadClientFactory;
    private final boolean viewEnabled;
    private final boolean arrowSerializationEnabled;
    private final Duration viewExpiration;
    private final int maxCreateReadSessionRetries;
    private final Optional<Integer> maxParallelism;

    public ReadSessionCreator(BigQueryClientFactory bigQueryClientFactory, BigQueryReadClientFactory bigQueryReadClientFactory, boolean viewEnabled, boolean arrowSerializationEnabled, Duration viewExpiration, int maxCreateReadSessionRetries, Optional<Integer> maxParallelism) {
        this.bigQueryClientFactory = bigQueryClientFactory;
        this.bigQueryReadClientFactory = bigQueryReadClientFactory;
        this.viewEnabled = viewEnabled;
        this.arrowSerializationEnabled = arrowSerializationEnabled;
        this.viewExpiration = viewExpiration;
        this.maxCreateReadSessionRetries = maxCreateReadSessionRetries;
        this.maxParallelism = maxParallelism;
    }

    public ReadSession create(ConnectorSession session, TableId remoteTable, List<BigQueryColumnHandle> selectedFields, Optional<String> filter, int currentWorkerCount) {
        BigQueryClient client = this.bigQueryClientFactory.create(session);
        TableInfo tableDetails = client.getTable(remoteTable).orElseThrow(() -> new TableNotFoundException(new SchemaTableName(remoteTable.getDataset(), remoteTable.getTable())));
        TableInfo actualTable = this.getActualTable(client, tableDetails, selectedFields, BigQuerySessionProperties.isViewMaterializationWithFilter(session) ? filter : Optional.empty());
        List filteredSelectedFields = selectedFields.stream().map(BigQueryColumnHandle::getQualifiedName).map(BigQueryUtil::toBigQueryColumnName).collect(Collectors.toList());
        try (BigQueryReadClient bigQueryReadClient = this.bigQueryReadClientFactory.create(session);){
            ReadSession.TableReadOptions.Builder readOptions = ReadSession.TableReadOptions.newBuilder().addAllSelectedFields(filteredSelectedFields);
            filter.ifPresent(arg_0 -> ((ReadSession.TableReadOptions.Builder)readOptions).setRowRestriction(arg_0));
            DataFormat format = DataFormat.AVRO;
            if (this.arrowSerializationEnabled) {
                format = DataFormat.ARROW;
                readOptions.setArrowSerializationOptions(ArrowSerializationOptions.newBuilder().setBufferCompression(ArrowSerializationOptions.CompressionCodec.ZSTD).build());
            }
            int desiredParallelism = Math.min(currentWorkerCount * 3, 100);
            CreateReadSessionRequest.Builder requestBuilder = CreateReadSessionRequest.newBuilder().setParent("projects/" + client.getParentProjectId()).setReadSession(ReadSession.newBuilder().setDataFormat(format).setTable(this.toTableResourceName(actualTable.getTableId())).setReadOptions(readOptions));
            if (this.maxParallelism.isPresent()) {
                int maxStreamCount = this.maxParallelism.get();
                requestBuilder.setMaxStreamCount(maxStreamCount);
                desiredParallelism = Math.min(desiredParallelism, maxStreamCount);
            }
            requestBuilder.setPreferredMinStreamCount(desiredParallelism);
            ReadSession readSession = (ReadSession)Failsafe.with((Policy)((RetryPolicyBuilder)RetryPolicy.builder().withMaxRetries(this.maxCreateReadSessionRetries).withBackoff(10L, 500L, ChronoUnit.MILLIS).onRetry(event -> log.debug("Request failed, retrying: %s", new Object[]{event.getLastException()})).handleIf(BigQueryUtil::isRetryable)).build(), (Policy[])new RetryPolicy[0]).get(() -> {
                try {
                    return bigQueryReadClient.createReadSession(requestBuilder.build());
                }
                catch (ApiException e) {
                    throw new TrinoException((ErrorCodeSupplier)BigQueryErrorCode.BIGQUERY_CREATE_READ_SESSION_ERROR, "Cannot create read session" + String.valueOf(MoreObjects.firstNonNull((Object)e.getMessage(), (Object)((Object)e))), (Throwable)e);
                }
            });
            return readSession;
        }
    }

    String toTableResourceName(TableId tableId) {
        return String.format("projects/%s/datasets/%s/tables/%s", tableId.getProject(), tableId.getDataset(), tableId.getTable());
    }

    private TableInfo getActualTable(BigQueryClient client, TableInfo remoteTable, List<BigQueryColumnHandle> requiredColumns, Optional<String> filter) {
        TableDefinition tableDefinition = remoteTable.getDefinition();
        TableDefinition.Type tableType = tableDefinition.getType();
        if (tableType == TableDefinition.Type.TABLE || tableType == TableDefinition.Type.SNAPSHOT || tableType == TableDefinition.Type.EXTERNAL) {
            return remoteTable;
        }
        if (tableType == TableDefinition.Type.VIEW || tableType == TableDefinition.Type.MATERIALIZED_VIEW) {
            if (!this.viewEnabled) {
                throw new TrinoException((ErrorCodeSupplier)StandardErrorCode.NOT_SUPPORTED, String.format("Views are not enabled. You can enable views by setting '%s' to true. Notice additional cost may occur.", "bigquery.views-enabled"));
            }
            return client.getCachedTable(this.viewExpiration, remoteTable, requiredColumns, filter);
        }
        throw new TrinoException((ErrorCodeSupplier)StandardErrorCode.NOT_SUPPORTED, String.format("Table type '%s' of table '%s.%s' is not supported", tableType, remoteTable.getTableId().getDataset(), remoteTable.getTableId().getTable()));
    }
}

