/*
 * Decompiled with CFR 0.152.
 */
package io.trino.plugin.hive;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import com.google.common.collect.ImmutableList;
import com.google.common.util.concurrent.Futures;
import com.google.common.util.concurrent.ListenableFuture;
import com.google.common.util.concurrent.MoreExecutors;
import io.airlift.concurrent.MoreFutures;
import io.airlift.log.Logger;
import io.airlift.stats.CounterStat;
import io.airlift.units.DataSize;
import io.trino.plugin.hive.HiveErrorCode;
import io.trino.plugin.hive.HivePartitionHandle;
import io.trino.plugin.hive.HiveSessionProperties;
import io.trino.plugin.hive.HiveSplit;
import io.trino.plugin.hive.HiveSplitLoader;
import io.trino.plugin.hive.HiveSplitWeightProvider;
import io.trino.plugin.hive.InternalHiveSplit;
import io.trino.plugin.hive.util.AsyncQueue;
import io.trino.plugin.hive.util.SizeBasedSplitWeightProvider;
import io.trino.plugin.hive.util.ThrottledAsyncQueue;
import io.trino.spi.ErrorCodeSupplier;
import io.trino.spi.TrinoException;
import io.trino.spi.connector.ConnectorPartitionHandle;
import io.trino.spi.connector.ConnectorSession;
import io.trino.spi.connector.ConnectorSplitSource;
import io.trino.spi.connector.NotPartitionedPartitionHandle;
import java.io.FileNotFoundException;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.OptionalInt;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.Executor;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.AtomicLong;
import java.util.concurrent.atomic.AtomicReference;
import java.util.function.Function;
import java.util.function.Predicate;

class HiveSplitSource
implements ConnectorSplitSource {
    private static final Logger log = Logger.get(HiveSplitSource.class);
    private final String queryId;
    private final String databaseName;
    private final String tableName;
    private final PerBucket queues;
    private final AtomicInteger bufferedInternalSplitCount = new AtomicInteger();
    private final long maxOutstandingSplitsBytes;
    private final DataSize maxSplitSize;
    private final DataSize maxInitialSplitSize;
    private final AtomicInteger remainingInitialSplits;
    private final AtomicLong numberOfProcessedSplits;
    private final HiveSplitLoader splitLoader;
    private final AtomicReference<State> stateReference;
    private final AtomicLong estimatedSplitSizeInBytes = new AtomicLong();
    private final CounterStat highMemorySplitSourceCounter;
    private final AtomicBoolean loggedHighMemoryWarning = new AtomicBoolean();
    private final HiveSplitWeightProvider splitWeightProvider;
    private final boolean recordScannedFiles;
    private final ImmutableList.Builder<Object> scannedFilePaths = ImmutableList.builder();

    private HiveSplitSource(ConnectorSession session, String databaseName, String tableName, PerBucket queues, int maxInitialSplits, DataSize maxOutstandingSplitsSize, HiveSplitLoader splitLoader, AtomicReference<State> stateReference, CounterStat highMemorySplitSourceCounter, boolean recordScannedFiles) {
        Objects.requireNonNull(session, "session is null");
        this.queryId = session.getQueryId();
        this.databaseName = Objects.requireNonNull(databaseName, "databaseName is null");
        this.tableName = Objects.requireNonNull(tableName, "tableName is null");
        this.queues = Objects.requireNonNull(queues, "queues is null");
        this.maxOutstandingSplitsBytes = Objects.requireNonNull(maxOutstandingSplitsSize, "maxOutstandingSplitsSize is null").toBytes();
        this.splitLoader = Objects.requireNonNull(splitLoader, "splitLoader is null");
        this.stateReference = Objects.requireNonNull(stateReference, "stateReference is null");
        this.highMemorySplitSourceCounter = Objects.requireNonNull(highMemorySplitSourceCounter, "highMemorySplitSourceCounter is null");
        this.maxSplitSize = HiveSessionProperties.getMaxSplitSize(session);
        this.maxInitialSplitSize = HiveSessionProperties.getMaxInitialSplitSize(session);
        this.remainingInitialSplits = new AtomicInteger(maxInitialSplits);
        this.numberOfProcessedSplits = new AtomicLong(0L);
        this.splitWeightProvider = HiveSessionProperties.isSizeBasedSplitWeightsEnabled(session) ? new SizeBasedSplitWeightProvider(HiveSessionProperties.getMinimumAssignedSplitWeight(session), this.maxSplitSize) : HiveSplitWeightProvider.uniformStandardWeightProvider();
        this.recordScannedFiles = recordScannedFiles;
    }

    public static HiveSplitSource allAtOnce(ConnectorSession session, String databaseName, String tableName, int maxInitialSplits, final int maxOutstandingSplits, DataSize maxOutstandingSplitsSize, final int maxSplitsPerSecond, HiveSplitLoader splitLoader, final Executor executor, CounterStat highMemorySplitSourceCounter, boolean recordScannedFiles) {
        AtomicReference<State> stateReference = new AtomicReference<State>(State.initial());
        return new HiveSplitSource(session, databaseName, tableName, new PerBucket(){
            private final AsyncQueue<InternalHiveSplit> queue;
            {
                this.queue = new ThrottledAsyncQueue<InternalHiveSplit>(maxSplitsPerSecond, maxOutstandingSplits, executor);
            }

            @Override
            public ListenableFuture<Void> offer(OptionalInt bucketNumber, InternalHiveSplit connectorSplit) {
                return this.queue.offer(connectorSplit);
            }

            @Override
            public <O> ListenableFuture<O> borrowBatchAsync(OptionalInt bucketNumber, int maxSize, Function<List<InternalHiveSplit>, AsyncQueue.BorrowResult<InternalHiveSplit, O>> function) {
                Preconditions.checkArgument((boolean)bucketNumber.isEmpty());
                return this.queue.borrowBatchAsync(maxSize, function);
            }

            @Override
            public void finish() {
                this.queue.finish();
            }

            @Override
            public boolean isFinished(OptionalInt bucketNumber) {
                Preconditions.checkArgument((boolean)bucketNumber.isEmpty());
                return this.queue.isFinished();
            }
        }, maxInitialSplits, maxOutstandingSplitsSize, splitLoader, stateReference, highMemorySplitSourceCounter, recordScannedFiles);
    }

    public static HiveSplitSource bucketed(ConnectorSession session, String databaseName, String tableName, final int estimatedOutstandingSplitsPerBucket, int maxInitialSplits, DataSize maxOutstandingSplitsSize, final int maxSplitsPerSecond, HiveSplitLoader splitLoader, final Executor executor, CounterStat highMemorySplitSourceCounter, boolean recordScannedFiles) {
        AtomicReference<State> stateReference = new AtomicReference<State>(State.initial());
        return new HiveSplitSource(session, databaseName, tableName, new PerBucket(){
            private final Map<Integer, AsyncQueue<InternalHiveSplit>> queues = new ConcurrentHashMap<Integer, AsyncQueue<InternalHiveSplit>>();
            private final AtomicBoolean finished = new AtomicBoolean();

            @Override
            public ListenableFuture<Void> offer(OptionalInt bucketNumber, InternalHiveSplit connectorSplit) {
                AsyncQueue<InternalHiveSplit> queue = this.queueFor(bucketNumber);
                queue.offer(connectorSplit);
                return Futures.immediateVoidFuture();
            }

            @Override
            public <O> ListenableFuture<O> borrowBatchAsync(OptionalInt bucketNumber, int maxSize, Function<List<InternalHiveSplit>, AsyncQueue.BorrowResult<InternalHiveSplit, O>> function) {
                return this.queueFor(bucketNumber).borrowBatchAsync(maxSize, function);
            }

            @Override
            public void finish() {
                if (this.finished.compareAndSet(false, true)) {
                    this.queues.values().forEach(AsyncQueue::finish);
                }
            }

            @Override
            public boolean isFinished(OptionalInt bucketNumber) {
                return this.queueFor(bucketNumber).isFinished();
            }

            public AsyncQueue<InternalHiveSplit> queueFor(OptionalInt bucketNumber) {
                Preconditions.checkArgument((boolean)bucketNumber.isPresent());
                AtomicBoolean isNew = new AtomicBoolean();
                AsyncQueue queue = this.queues.computeIfAbsent(bucketNumber.getAsInt(), ignored -> {
                    isNew.set(true);
                    return new ThrottledAsyncQueue(maxSplitsPerSecond, estimatedOutstandingSplitsPerBucket, executor);
                });
                if (isNew.get() && this.finished.get()) {
                    queue.finish();
                }
                return queue;
            }
        }, maxInitialSplits, maxOutstandingSplitsSize, splitLoader, stateReference, highMemorySplitSourceCounter, recordScannedFiles);
    }

    @VisibleForTesting
    int getBufferedInternalSplitCount() {
        return this.bufferedInternalSplitCount.get();
    }

    ListenableFuture<Void> addToQueue(List<? extends InternalHiveSplit> splits) {
        ListenableFuture<Void> lastResult = Futures.immediateVoidFuture();
        for (InternalHiveSplit internalHiveSplit : splits) {
            lastResult = this.addToQueue(internalHiveSplit);
        }
        return lastResult;
    }

    ListenableFuture<Void> addToQueue(InternalHiveSplit split) {
        if (this.stateReference.get().getKind() != StateKind.INITIAL) {
            return Futures.immediateVoidFuture();
        }
        if (this.estimatedSplitSizeInBytes.addAndGet(split.getEstimatedSizeInBytes()) > this.maxOutstandingSplitsBytes) {
            if (this.loggedHighMemoryWarning.compareAndSet(false, true)) {
                this.highMemorySplitSourceCounter.update(1L);
                log.warn("Split buffering for %s.%s in query %s exceeded memory limit (%s). %s splits are buffered.", new Object[]{this.databaseName, this.tableName, this.queryId, DataSize.succinctBytes((long)this.maxOutstandingSplitsBytes), this.getBufferedInternalSplitCount()});
            }
            throw new TrinoException((ErrorCodeSupplier)HiveErrorCode.HIVE_EXCEEDED_SPLIT_BUFFERING_LIMIT, String.format("Split buffering for %s.%s exceeded memory limit (%s). %s splits are buffered.", this.databaseName, this.tableName, DataSize.succinctBytes((long)this.maxOutstandingSplitsBytes), this.getBufferedInternalSplitCount()));
        }
        this.bufferedInternalSplitCount.incrementAndGet();
        OptionalInt bucketNumber = split.getReadBucketNumber();
        return this.queues.offer(bucketNumber, split);
    }

    void noMoreSplits() {
        if (HiveSplitSource.setIf(this.stateReference, State.noMoreSplits(), state -> state.getKind() == StateKind.INITIAL)) {
            this.splitLoader.stop();
            this.queues.finish();
        }
    }

    void fail(Throwable e) {
        if (HiveSplitSource.setIf(this.stateReference, State.failed(e), state -> state.getKind() == StateKind.INITIAL)) {
            this.splitLoader.stop();
            this.queues.finish();
        }
    }

    public CompletableFuture<ConnectorSplitSource.ConnectorSplitBatch> getNextBatch(ConnectorPartitionHandle partitionHandle, int maxSize) {
        boolean noMoreSplits;
        State state = this.stateReference.get();
        switch (state.getKind()) {
            case INITIAL: {
                noMoreSplits = false;
                break;
            }
            case NO_MORE_SPLITS: {
                noMoreSplits = true;
                break;
            }
            case FAILED: {
                return MoreFutures.failedFuture((Throwable)state.getThrowable());
            }
            case CLOSED: {
                throw new IllegalStateException("HiveSplitSource is already closed");
            }
            default: {
                throw new UnsupportedOperationException();
            }
        }
        OptionalInt bucketNumber = HiveSplitSource.toBucketNumber(partitionHandle);
        ListenableFuture future = this.queues.borrowBatchAsync(bucketNumber, maxSize, internalSplits -> {
            ImmutableList.Builder splitsToInsertBuilder = ImmutableList.builder();
            ImmutableList.Builder resultBuilder = ImmutableList.builder();
            int removedEstimatedSizeInBytes = 0;
            int removedSplitCount = 0;
            for (InternalHiveSplit internalSplit : internalSplits) {
                long remainingBlockBytes;
                if (!internalSplit.getPartitionMatchSupplier().getAsBoolean()) {
                    removedEstimatedSizeInBytes += internalSplit.getEstimatedSizeInBytes();
                    ++removedSplitCount;
                    continue;
                }
                long maxSplitBytes = this.maxSplitSize.toBytes();
                if (this.remainingInitialSplits.get() > 0 && this.remainingInitialSplits.getAndDecrement() > 0) {
                    maxSplitBytes = this.maxInitialSplitSize.toBytes();
                }
                InternalHiveSplit.InternalHiveBlock block = internalSplit.currentBlock();
                long splitBytes = internalSplit.isSplittable() ? ((remainingBlockBytes = block.getEnd() - internalSplit.getStart()) <= maxSplitBytes ? remainingBlockBytes : (maxSplitBytes * 2L >= remainingBlockBytes ? remainingBlockBytes / 2L : maxSplitBytes)) : internalSplit.getEnd() - internalSplit.getStart();
                resultBuilder.add((Object)new HiveSplit(this.databaseName, this.tableName, internalSplit.getPartitionName(), internalSplit.getPath(), internalSplit.getStart(), splitBytes, internalSplit.getEstimatedFileSize(), internalSplit.getFileModifiedTime(), internalSplit.getSchema(), internalSplit.getPartitionKeys(), block.getAddresses(), internalSplit.getReadBucketNumber(), internalSplit.getTableBucketNumber(), internalSplit.getStatementId(), internalSplit.isForceLocalScheduling(), internalSplit.getTableToPartitionMapping(), internalSplit.getBucketConversion(), internalSplit.getBucketValidation(), internalSplit.isS3SelectPushdownEnabled(), internalSplit.getAcidInfo(), this.numberOfProcessedSplits.getAndIncrement(), this.splitWeightProvider.weightForSplitSizeInBytes(splitBytes)));
                internalSplit.increaseStart(splitBytes);
                if (internalSplit.isDone()) {
                    removedEstimatedSizeInBytes += internalSplit.getEstimatedSizeInBytes();
                    ++removedSplitCount;
                    continue;
                }
                splitsToInsertBuilder.add((Object)internalSplit);
            }
            this.estimatedSplitSizeInBytes.addAndGet(-removedEstimatedSizeInBytes);
            this.bufferedInternalSplitCount.addAndGet(-removedSplitCount);
            ImmutableList splitsToInsert = splitsToInsertBuilder.build();
            ImmutableList result = resultBuilder.build();
            return new AsyncQueue.BorrowResult(splitsToInsert, result);
        });
        ListenableFuture transform = Futures.transform(future, splits -> {
            Objects.requireNonNull(splits, "splits is null");
            if (this.recordScannedFiles) {
                splits.forEach(split -> this.scannedFilePaths.add((Object)((HiveSplit)split).getPath()));
            }
            if (noMoreSplits) {
                return new ConnectorSplitSource.ConnectorSplitBatch(splits, splits.isEmpty() && this.queues.isFinished(bucketNumber));
            }
            return new ConnectorSplitSource.ConnectorSplitBatch(splits, false);
        }, (Executor)MoreExecutors.directExecutor());
        return MoreFutures.toCompletableFuture((ListenableFuture)transform);
    }

    public boolean isFinished() {
        State state = this.stateReference.get();
        switch (state.getKind()) {
            case INITIAL: {
                return false;
            }
            case NO_MORE_SPLITS: {
                return this.bufferedInternalSplitCount.get() == 0;
            }
            case FAILED: {
                throw HiveSplitSource.propagateTrinoException(state.getThrowable());
            }
            case CLOSED: {
                throw new IllegalStateException("HiveSplitSource is already closed");
            }
        }
        throw new UnsupportedOperationException();
    }

    public Optional<List<Object>> getTableExecuteSplitsInfo() {
        Preconditions.checkState((boolean)this.isFinished(), (Object)"HiveSplitSource must be finished before TableExecuteSplitsInfo is read");
        if (!this.recordScannedFiles) {
            return Optional.empty();
        }
        return Optional.of(this.scannedFilePaths.build());
    }

    public void close() {
        if (HiveSplitSource.setIf(this.stateReference, State.closed(), state -> state.getKind() == StateKind.INITIAL || state.getKind() == StateKind.NO_MORE_SPLITS)) {
            this.splitLoader.stop();
            this.queues.finish();
        }
    }

    private static OptionalInt toBucketNumber(ConnectorPartitionHandle partitionHandle) {
        if (partitionHandle == NotPartitionedPartitionHandle.NOT_PARTITIONED) {
            return OptionalInt.empty();
        }
        return OptionalInt.of(((HivePartitionHandle)partitionHandle).getBucket());
    }

    private static <T> boolean setIf(AtomicReference<T> atomicReference, T newValue, Predicate<T> predicate) {
        T current;
        do {
            if (predicate.test(current = atomicReference.get())) continue;
            return false;
        } while (!atomicReference.compareAndSet(current, newValue));
        return true;
    }

    private static RuntimeException propagateTrinoException(Throwable throwable) {
        if (throwable instanceof TrinoException) {
            throw (TrinoException)throwable;
        }
        if (throwable instanceof FileNotFoundException) {
            throw new TrinoException((ErrorCodeSupplier)HiveErrorCode.HIVE_FILE_NOT_FOUND, throwable);
        }
        throw new TrinoException((ErrorCodeSupplier)HiveErrorCode.HIVE_UNKNOWN_ERROR, throwable);
    }

    static enum StateKind {
        INITIAL,
        NO_MORE_SPLITS,
        FAILED,
        CLOSED;

    }

    static class State {
        private final StateKind kind;
        private final Throwable throwable;

        private State(StateKind kind, Throwable throwable) {
            this.kind = kind;
            this.throwable = throwable;
        }

        public StateKind getKind() {
            return this.kind;
        }

        public Throwable getThrowable() {
            Preconditions.checkState((this.throwable != null ? 1 : 0) != 0);
            return this.throwable;
        }

        public static State initial() {
            return new State(StateKind.INITIAL, null);
        }

        public static State noMoreSplits() {
            return new State(StateKind.NO_MORE_SPLITS, null);
        }

        public static State failed(Throwable throwable) {
            return new State(StateKind.FAILED, throwable);
        }

        public static State closed() {
            return new State(StateKind.CLOSED, null);
        }
    }

    static interface PerBucket {
        public ListenableFuture<Void> offer(OptionalInt var1, InternalHiveSplit var2);

        public <O> ListenableFuture<O> borrowBatchAsync(OptionalInt var1, int var2, Function<List<InternalHiveSplit>, AsyncQueue.BorrowResult<InternalHiveSplit, O>> var3);

        public void finish();

        public boolean isFinished(OptionalInt var1);
    }
}

