/*
 * Decompiled with CFR 0.152.
 */
package io.trino.plugin.deltalake.transactionlog.writer;

import com.fasterxml.jackson.annotation.JsonCreator;
import com.fasterxml.jackson.annotation.JsonProperty;
import com.google.common.collect.ImmutableList;
import io.airlift.json.JsonCodec;
import io.airlift.log.Logger;
import io.trino.filesystem.FileEntry;
import io.trino.filesystem.FileIterator;
import io.trino.filesystem.Locations;
import io.trino.filesystem.TrinoFileSystem;
import io.trino.filesystem.TrinoFileSystemFactory;
import io.trino.filesystem.TrinoInputStream;
import io.trino.filesystem.TrinoOutputFile;
import io.trino.plugin.deltalake.transactionlog.writer.TransactionConflictException;
import io.trino.plugin.deltalake.transactionlog.writer.TransactionLogSynchronizer;
import io.trino.spi.connector.ConnectorSession;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.OutputStream;
import java.io.UncheckedIOException;
import java.time.Duration;
import java.time.Instant;
import java.time.temporal.ChronoUnit;
import java.util.Base64;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import javax.inject.Inject;
import org.apache.parquet.Preconditions;

public class S3NativeTransactionLogSynchronizer
implements TransactionLogSynchronizer {
    public static final Logger LOG = Logger.get(S3NativeTransactionLogSynchronizer.class);
    private static final Duration EXPIRATION_DURATION = Duration.of(5L, ChronoUnit.MINUTES);
    private static final String LOCK_DIRECTORY = "_sb_lock";
    private static final String LOCK_INFIX = "sb-lock_";
    private static final Pattern LOCK_FILENAME_PATTERN = Pattern.compile("(.*)\\.sb-lock_.*");
    private final TrinoFileSystemFactory fileSystemFactory;
    private final JsonCodec<LockFileContents> lockFileContentsJsonCodec;

    @Inject
    public S3NativeTransactionLogSynchronizer(TrinoFileSystemFactory fileSystemFactory, JsonCodec<LockFileContents> lockFileContentesCodec) {
        this.fileSystemFactory = Objects.requireNonNull(fileSystemFactory, "fileSystemFactory is null");
        this.lockFileContentsJsonCodec = Objects.requireNonNull(lockFileContentesCodec, "lockFileContentesCodec is null");
    }

    @Override
    public boolean isUnsafe() {
        return true;
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    @Override
    public void write(ConnectorSession session, String clusterId, String newLogEntryPath, byte[] entryContents) {
        TrinoFileSystem fileSystem = this.fileSystemFactory.create(session);
        String locksDirectory = Locations.appendPath((String)Locations.getParent((String)newLogEntryPath), (String)LOCK_DIRECTORY);
        String newEntryFilename = Locations.getFileName((String)newLogEntryPath);
        Optional<Object> myLockInfo = Optional.empty();
        try {
            if (fileSystem.newInputFile(newLogEntryPath).exists()) {
                throw new TransactionConflictException(newLogEntryPath + " already exists");
            }
            List<LockInfo> lockInfos = this.listLockInfos(fileSystem, locksDirectory);
            Optional<Object> currentLock = Optional.empty();
            for (LockInfo lockInfo2 : lockInfos) {
                if (lockInfo2.getExpirationTime().isBefore(Instant.now())) {
                    S3NativeTransactionLogSynchronizer.deleteLock(fileSystem, locksDirectory, lockInfo2);
                    continue;
                }
                if (!lockInfo2.getEntryFilename().equals(newEntryFilename)) continue;
                if (currentLock.isPresent()) {
                    throw new IllegalStateException(String.format("Multiple live locks found for: %s; lock1: %s; lock2: %s", newLogEntryPath, ((LockInfo)currentLock.get()).getLockFilename(), lockInfo2.getLockFilename()));
                }
                currentLock = Optional.of(lockInfo2);
            }
            currentLock.ifPresent(lock -> {
                throw new TransactionConflictException(String.format("Transaction log locked(1); lockingCluster=%s; lockingQuery=%s; expires=%s", lock.getClusterId(), lock.getOwningQuery(), lock.getExpirationTime()));
            });
            myLockInfo = Optional.of(this.writeNewLockInfo(fileSystem, locksDirectory, newEntryFilename, clusterId, session.getQueryId()));
            lockInfos = this.listLockInfos(fileSystem, locksDirectory);
            String myLockFilename = ((LockInfo)myLockInfo.get()).getLockFilename();
            currentLock = lockInfos.stream().filter(lockInfo -> lockInfo.getEntryFilename().equals(newEntryFilename)).filter(lockInfo -> !lockInfo.getLockFilename().equals(myLockFilename)).findFirst();
            if (currentLock.isPresent()) {
                throw new TransactionConflictException(String.format("Transaction log locked(2); lockingCluster=%s; lockingQuery=%s; expires=%s", ((LockInfo)currentLock.get()).getClusterId(), ((LockInfo)currentLock.get()).getOwningQuery(), ((LockInfo)currentLock.get()).getExpirationTime()));
            }
            Preconditions.checkState((!fileSystem.newInputFile(newLogEntryPath).exists() ? 1 : 0) != 0, (String)String.format("Target file %s was created during locking", newLogEntryPath));
            try (OutputStream outputStream = fileSystem.newOutputFile(newLogEntryPath).create();){
                outputStream.write(entryContents);
            }
            if (!myLockInfo.isPresent()) return;
        }
        catch (IOException e) {
            try {
                throw new UncheckedIOException("Internal error while writing " + newLogEntryPath, e);
            }
            catch (Throwable throwable) {
                if (!myLockInfo.isPresent()) throw throwable;
                try {
                    S3NativeTransactionLogSynchronizer.deleteLock(fileSystem, locksDirectory, (LockInfo)myLockInfo.get());
                    throw throwable;
                }
                catch (IOException e2) {
                    LOG.warn((Throwable)e2, "Could not delete lockfile %s", new Object[]{((LockInfo)myLockInfo.get()).lockFilename});
                }
                throw throwable;
            }
        }
        try {
            S3NativeTransactionLogSynchronizer.deleteLock(fileSystem, locksDirectory, (LockInfo)myLockInfo.get());
            return;
        }
        catch (IOException e) {
            LOG.warn((Throwable)e, "Could not delete lockfile %s", new Object[]{((LockInfo)myLockInfo.get()).lockFilename});
            return;
        }
    }

    private LockInfo writeNewLockInfo(TrinoFileSystem fileSystem, String lockDirectory, String logEntryFilename, String clusterId, String queryId) throws IOException {
        String lockFilename = logEntryFilename + ".sb-lock_" + queryId;
        Instant expiration = Instant.now().plus(EXPIRATION_DURATION);
        LockFileContents contents = new LockFileContents(clusterId, queryId, expiration.toEpochMilli());
        String lockPath = Locations.appendPath((String)lockDirectory, (String)lockFilename);
        TrinoOutputFile lockFile = fileSystem.newOutputFile(lockPath);
        byte[] contentsBytes = this.lockFileContentsJsonCodec.toJsonBytes((Object)contents);
        try (OutputStream outputStream = lockFile.create();){
            outputStream.write(contentsBytes);
        }
        return new LockInfo(lockFilename, contents);
    }

    private static void deleteLock(TrinoFileSystem fileSystem, String lockDirectoryPath, LockInfo lockInfo) throws IOException {
        String lockPath = Locations.appendPath((String)lockDirectoryPath, (String)lockInfo.getLockFilename());
        fileSystem.deleteFile(lockPath);
    }

    private List<LockInfo> listLockInfos(TrinoFileSystem fileSystem, String lockDirectoryPath) throws IOException {
        FileIterator files = fileSystem.listFiles(lockDirectoryPath);
        ImmutableList.Builder lockInfos = ImmutableList.builder();
        while (files.hasNext()) {
            FileEntry entry = files.next();
            String name = entry.location().substring(entry.location().lastIndexOf(47) + 1);
            if (!LOCK_FILENAME_PATTERN.matcher(name).matches()) continue;
            Optional<LockInfo> lockInfo = this.parseLockFile(fileSystem, entry.location(), name);
            lockInfo.ifPresent(arg_0 -> ((ImmutableList.Builder)lockInfos).add(arg_0));
        }
        return lockInfos.build();
    }

    private Optional<LockInfo> parseLockFile(TrinoFileSystem fileSystem, String path, String name) throws IOException {
        Optional<LockInfo> optional;
        block10: {
            byte[] bytes = null;
            TrinoInputStream inputStream = fileSystem.newInputFile(path).newStream();
            try {
                bytes = inputStream.readAllBytes();
                LockFileContents lockFileContents = (LockFileContents)this.lockFileContentsJsonCodec.fromJson(bytes);
                optional = Optional.of(new LockInfo(name, lockFileContents));
                if (inputStream == null) break block10;
            }
            catch (Throwable lockFileContents) {
                try {
                    if (inputStream != null) {
                        try {
                            inputStream.close();
                        }
                        catch (Throwable throwable) {
                            lockFileContents.addSuppressed(throwable);
                        }
                    }
                    throw lockFileContents;
                }
                catch (IllegalArgumentException e) {
                    String content = null;
                    if (bytes != null) {
                        content = Base64.getEncoder().encodeToString(bytes);
                    }
                    LOG.warn((Throwable)e, "Could not parse lock file: %s; contents=%s", new Object[]{path, content});
                    return Optional.empty();
                }
                catch (FileNotFoundException e) {
                    return Optional.empty();
                }
            }
            inputStream.close();
        }
        return optional;
    }

    public static String parseEntryFilename(String lockFilename) {
        Matcher matcher = LOCK_FILENAME_PATTERN.matcher(lockFilename);
        if (!matcher.matches()) {
            throw new IllegalArgumentException("Lock filename " + lockFilename + " does not match expected pattern");
        }
        return matcher.group(1);
    }

    private static class LockInfo {
        private final String lockFilename;
        private final String entryFilename;
        private final LockFileContents contents;

        public LockInfo(String lockFilename, LockFileContents contents) {
            this.lockFilename = Objects.requireNonNull(lockFilename, "lockFilename is null");
            this.entryFilename = S3NativeTransactionLogSynchronizer.parseEntryFilename(lockFilename);
            this.contents = Objects.requireNonNull(contents, "contents is null");
        }

        public String getLockFilename() {
            return this.lockFilename;
        }

        public String getEntryFilename() {
            return this.entryFilename;
        }

        public String getClusterId() {
            return this.contents.getClusterId();
        }

        public String getOwningQuery() {
            return this.contents.getOwningQuery();
        }

        public Instant getExpirationTime() {
            return Instant.ofEpochMilli(this.contents.getExpirationEpochMillis());
        }
    }

    public static class LockFileContents {
        private final String clusterId;
        private final String owningQuery;
        private final long expirationEpochMillis;

        @JsonCreator
        public LockFileContents(@JsonProperty(value="clusterId") String clusterId, @JsonProperty(value="owningQuery") String owningQuery, @JsonProperty(value="expirationEpochMillis") long expirationEpochMillis) {
            this.clusterId = Objects.requireNonNull(clusterId, "clusterId is null");
            this.owningQuery = Objects.requireNonNull(owningQuery, "owningQuery is null");
            this.expirationEpochMillis = expirationEpochMillis;
        }

        @JsonProperty
        public String getClusterId() {
            return this.clusterId;
        }

        @JsonProperty
        public String getOwningQuery() {
            return this.owningQuery;
        }

        @JsonProperty
        public long getExpirationEpochMillis() {
            return this.expirationEpochMillis;
        }
    }
}

