/*
 * Decompiled with CFR 0.152.
 */
package io.trino.hive.formats;

import com.google.common.base.Preconditions;
import com.google.common.base.Verify;
import io.airlift.slice.SizeOf;
import io.airlift.slice.Slice;
import io.airlift.slice.Slices;
import io.trino.filesystem.TrinoInputStream;
import java.io.DataInput;
import java.io.EOFException;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.util.Objects;

public final class TrinoDataInputStream
extends InputStream
implements DataInput {
    private static final int INSTANCE_SIZE = SizeOf.instanceSize(TrinoDataInputStream.class);
    private static final int DEFAULT_BUFFER_SIZE = 4096;
    private static final int MINIMUM_CHUNK_SIZE = 1024;
    private final TrinoInputStream inputStream;
    private long readTimeNanos;
    private long readBytes;
    private final byte[] buffer;
    private final Slice slice;
    private long bufferOffset;
    private int bufferPosition;
    private int bufferFill;

    public TrinoDataInputStream(TrinoInputStream inputStream) {
        this(inputStream, 4096);
    }

    public TrinoDataInputStream(TrinoInputStream inputStream, int bufferSize) {
        Objects.requireNonNull(inputStream, "inputStream is null");
        Preconditions.checkArgument((bufferSize >= 1024 ? 1 : 0) != 0, (Object)"minimum buffer size of 1024 required");
        this.inputStream = inputStream;
        this.buffer = new byte[bufferSize];
        this.slice = Slices.wrappedBuffer((byte[])this.buffer);
    }

    public long getReadTimeNanos() {
        return this.readTimeNanos;
    }

    public long getReadBytes() {
        return this.readBytes;
    }

    public long getPos() throws IOException {
        return TrinoDataInputStream.checkedCast(this.bufferOffset + (long)this.bufferPosition);
    }

    public void seek(long newPos) throws IOException {
        this.bufferPosition = 0;
        this.bufferFill = 0;
        this.inputStream.seek(newPos);
        this.bufferOffset = newPos;
        Verify.verify((newPos == this.getPos() ? 1 : 0) != 0);
    }

    @Override
    public int available() throws IOException {
        if (this.bufferPosition < this.bufferFill) {
            return this.availableBytes();
        }
        return this.fillBuffer();
    }

    @Override
    public int skipBytes(int n) throws IOException {
        return (int)this.skip(n);
    }

    @Override
    public boolean readBoolean() throws IOException {
        return this.readByte() != 0;
    }

    @Override
    public byte readByte() throws IOException {
        this.ensureAvailable(1);
        byte v = this.slice.getByte(this.bufferPosition);
        ++this.bufferPosition;
        return v;
    }

    @Override
    public int readUnsignedByte() throws IOException {
        return this.readByte() & 0xFF;
    }

    @Override
    public short readShort() throws IOException {
        this.ensureAvailable(2);
        short v = this.slice.getShort(this.bufferPosition);
        this.bufferPosition += 2;
        return v;
    }

    @Override
    public int readUnsignedShort() throws IOException {
        return this.readShort() & 0xFFFF;
    }

    @Override
    public int readInt() throws IOException {
        this.ensureAvailable(4);
        int v = this.slice.getInt(this.bufferPosition);
        this.bufferPosition += 4;
        return v;
    }

    public long readUnsignedInt() throws IOException {
        return (long)this.readInt() & 0xFFFFFFFFL;
    }

    @Override
    public long readLong() throws IOException {
        this.ensureAvailable(8);
        long v = this.slice.getLong(this.bufferPosition);
        this.bufferPosition += 8;
        return v;
    }

    @Override
    public float readFloat() throws IOException {
        return Float.intBitsToFloat(this.readInt());
    }

    @Override
    public double readDouble() throws IOException {
        return Double.longBitsToDouble(this.readLong());
    }

    @Override
    public int read() throws IOException {
        if (this.available() == 0) {
            return -1;
        }
        Verify.verify((this.availableBytes() > 0 ? 1 : 0) != 0);
        int v = this.slice.getByte(this.bufferPosition) & 0xFF;
        ++this.bufferPosition;
        return v;
    }

    @Override
    public long skip(long length) throws IOException {
        int availableBytes = this.availableBytes();
        if ((long)availableBytes >= length) {
            this.bufferPosition = TrinoDataInputStream.checkedCast((long)this.bufferPosition + length);
            return length;
        }
        this.bufferPosition = this.bufferFill;
        long start = System.nanoTime();
        long inputStreamSkip = this.inputStream.skip(length - (long)availableBytes);
        this.readTimeNanos += System.nanoTime() - start;
        this.readBytes += inputStreamSkip;
        this.bufferOffset += inputStreamSkip;
        return (long)availableBytes + inputStreamSkip;
    }

    @Override
    public int read(byte[] destination) throws IOException {
        return this.read(destination, 0, destination.length);
    }

    @Override
    public int read(byte[] destination, int destinationIndex, int length) throws IOException {
        if (this.available() == 0) {
            return -1;
        }
        Verify.verify((this.availableBytes() > 0 ? 1 : 0) != 0);
        int batch = Math.min(this.availableBytes(), length);
        this.slice.getBytes(this.bufferPosition, destination, destinationIndex, batch);
        this.bufferPosition += batch;
        return batch;
    }

    @Override
    public void readFully(byte[] destination) throws IOException {
        this.readFully(destination, 0, destination.length);
    }

    @Override
    public void readFully(byte[] destination, int destinationIndex, int length) throws IOException {
        while (length > 0) {
            int batch = Math.min(this.availableBytes(), length);
            this.slice.getBytes(this.bufferPosition, destination, destinationIndex, batch);
            this.bufferPosition += batch;
            destinationIndex += batch;
            this.ensureAvailable(Math.min(length -= batch, 1024));
        }
    }

    public Slice readSlice(int length) throws IOException {
        if (length == 0) {
            return Slices.EMPTY_SLICE;
        }
        Slice newSlice = Slices.allocate((int)length);
        this.readFully(newSlice, 0, length);
        return newSlice;
    }

    public void readFully(Slice destination) throws IOException {
        this.readFully(destination, 0, destination.length());
    }

    public void readFully(Slice destination, int destinationIndex, int length) throws IOException {
        while (length > 0) {
            int batch = Math.min(this.availableBytes(), length);
            this.slice.getBytes(this.bufferPosition, destination, destinationIndex, batch);
            this.bufferPosition += batch;
            destinationIndex += batch;
            this.ensureAvailable(Math.min(length -= batch, 1024));
        }
    }

    public void readFully(OutputStream out, int length) throws IOException {
        while (length > 0) {
            int batch = Math.min(this.availableBytes(), length);
            out.write(this.buffer, this.bufferPosition, batch);
            this.bufferPosition += batch;
            this.ensureAvailable(Math.min(length -= batch, 1024));
        }
    }

    @Override
    public void close() throws IOException {
        this.inputStream.close();
    }

    public long getRetainedSize() {
        return (long)INSTANCE_SIZE + SizeOf.sizeOf((byte[])this.buffer);
    }

    private int availableBytes() {
        return this.bufferFill - this.bufferPosition;
    }

    private void ensureAvailable(int size) throws IOException {
        if (this.bufferPosition + size < this.bufferFill) {
            return;
        }
        if (this.fillBuffer() < size) {
            throw new EOFException("End of stream");
        }
    }

    private int fillBuffer() throws IOException {
        int bytesRead;
        int rest = this.bufferFill - this.bufferPosition;
        System.arraycopy(this.buffer, this.bufferPosition, this.buffer, 0, rest);
        this.bufferFill = rest;
        this.bufferOffset += (long)this.bufferPosition;
        this.bufferPosition = 0;
        long start = System.nanoTime();
        while (this.bufferFill < 1024 && (bytesRead = this.inputStream.read(this.buffer, this.bufferFill, this.buffer.length - this.bufferFill)) >= 0) {
            this.readBytes += (long)bytesRead;
            this.bufferFill += bytesRead;
        }
        this.readTimeNanos += System.nanoTime() - start;
        return this.bufferFill;
    }

    private static int checkedCast(long value) {
        int result = (int)value;
        Preconditions.checkArgument(((long)result == value ? 1 : 0) != 0, (Object)"Size is greater than maximum int value");
        return result;
    }

    @Override
    @Deprecated
    public void mark(int readLimit) {
        throw new UnsupportedOperationException();
    }

    @Override
    @Deprecated
    public void reset() {
        throw new UnsupportedOperationException();
    }

    @Override
    @Deprecated
    public boolean markSupported() {
        throw new UnsupportedOperationException();
    }

    @Override
    @Deprecated
    public char readChar() {
        throw new UnsupportedOperationException();
    }

    @Override
    @Deprecated
    public String readLine() {
        throw new UnsupportedOperationException();
    }

    @Override
    @Deprecated
    public String readUTF() {
        throw new UnsupportedOperationException();
    }
}

