/*
 * Decompiled with CFR 0.152.
 */
package io.trino.filesystem.gcs;

import com.google.cloud.WriteChannel;
import com.google.cloud.storage.Blob;
import com.google.cloud.storage.Storage;
import com.google.common.base.Preconditions;
import com.google.common.primitives.Ints;
import io.trino.filesystem.gcs.GcsLocation;
import io.trino.memory.context.AggregatedMemoryContext;
import io.trino.memory.context.LocalMemoryContext;
import java.io.IOException;
import java.io.OutputStream;
import java.nio.ByteBuffer;
import java.util.Objects;

public class GcsOutputStream
extends OutputStream {
    private static final int BUFFER_SIZE = 8192;
    private final GcsLocation location;
    private final long writeBlockSizeBytes;
    private final LocalMemoryContext memoryContext;
    private final WriteChannel writeChannel;
    private final ByteBuffer buffer = ByteBuffer.allocate(8192);
    private long writtenBytes;
    private boolean closed;

    public GcsOutputStream(GcsLocation location, Blob blob, AggregatedMemoryContext memoryContext, long writeBlockSizeBytes) {
        this.location = Objects.requireNonNull(location, "location is null");
        Preconditions.checkArgument((writeBlockSizeBytes >= 0L ? 1 : 0) != 0, (Object)"writeBlockSizeBytes is negative");
        this.writeBlockSizeBytes = writeBlockSizeBytes;
        this.memoryContext = memoryContext.newLocalMemoryContext(GcsOutputStream.class.getSimpleName());
        this.writeChannel = blob.writer(new Storage.BlobWriteOption[0]);
        this.writeChannel.setChunkSize(Ints.saturatedCast((long)writeBlockSizeBytes));
    }

    @Override
    public void write(int b) throws IOException {
        this.ensureOpen();
        if (!this.buffer.hasRemaining()) {
            this.flush();
        }
        this.buffer.put((byte)b);
        this.recordBytesWritten(1);
    }

    @Override
    public void write(byte[] buffer, int offset, int length) throws IOException {
        this.ensureOpen();
        if (length > 8192) {
            this.writeDirect(ByteBuffer.wrap(buffer, offset, length));
        } else {
            if (length > this.buffer.remaining()) {
                this.flush();
            }
            this.buffer.put(buffer, offset, length);
            this.recordBytesWritten(length);
        }
    }

    private void writeDirect(ByteBuffer buffer) throws IOException {
        this.flush();
        int bytesWritten = 0;
        try {
            bytesWritten = this.writeChannel.write(buffer);
            if (buffer.remaining() != 0) {
                throw new IOException("Unexpected partial write (written=%s, remaining=%s)".formatted(bytesWritten, buffer.remaining()));
            }
        }
        catch (IOException e) {
            throw new IOException("Error writing file: " + this.location, e);
        }
        this.recordBytesWritten(bytesWritten);
    }

    private void ensureOpen() throws IOException {
        if (this.closed) {
            throw new IOException("Output stream closed: " + this.location);
        }
    }

    @Override
    public void flush() throws IOException {
        this.ensureOpen();
        if (this.buffer.position() > 0) {
            this.buffer.flip();
            while (this.buffer.hasRemaining()) {
                try {
                    this.writeChannel.write(this.buffer);
                }
                catch (IOException e) {
                    throw new IOException("Error writing file: " + this.location, e);
                }
            }
            this.buffer.clear();
        }
    }

    @Override
    public void close() throws IOException {
        if (!this.closed) {
            this.flush();
            this.closed = true;
            try {
                this.writeChannel.close();
            }
            catch (IOException e) {
                throw new IOException("Error closing file: " + this.location, e);
            }
            finally {
                this.memoryContext.close();
            }
        }
    }

    private void recordBytesWritten(int size) {
        if (this.writtenBytes < this.writeBlockSizeBytes) {
            this.memoryContext.setBytes(8192L + Math.min(this.writtenBytes + (long)size, this.writeBlockSizeBytes));
        }
        this.writtenBytes += (long)size;
    }
}

