/*
 * Decompiled with CFR 0.152.
 */
package io.trino.plugin.hive;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import com.google.common.base.Stopwatch;
import com.google.common.collect.ArrayListMultimap;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.ListMultimap;
import com.google.common.collect.Multimaps;
import com.google.common.collect.Streams;
import com.google.common.io.CharStreams;
import com.google.common.util.concurrent.Futures;
import com.google.common.util.concurrent.ListenableFuture;
import com.google.common.util.concurrent.MoreExecutors;
import io.airlift.concurrent.MoreFutures;
import io.airlift.units.Duration;
import io.trino.filesystem.FileEntry;
import io.trino.filesystem.FileIterator;
import io.trino.filesystem.Location;
import io.trino.filesystem.TrinoFileSystem;
import io.trino.filesystem.TrinoFileSystemFactory;
import io.trino.plugin.hive.AcidInfo;
import io.trino.plugin.hive.ConcurrentLazyQueue;
import io.trino.plugin.hive.HiveBucketHandle;
import io.trino.plugin.hive.HiveBucketProperty;
import io.trino.plugin.hive.HiveColumnHandle;
import io.trino.plugin.hive.HiveErrorCode;
import io.trino.plugin.hive.HivePartition;
import io.trino.plugin.hive.HivePartitionKey;
import io.trino.plugin.hive.HivePartitionMetadata;
import io.trino.plugin.hive.HiveSessionProperties;
import io.trino.plugin.hive.HiveSplit;
import io.trino.plugin.hive.HiveSplitLoader;
import io.trino.plugin.hive.HiveSplitSource;
import io.trino.plugin.hive.HiveStorageFormat;
import io.trino.plugin.hive.HiveType;
import io.trino.plugin.hive.InternalHiveSplit;
import io.trino.plugin.hive.fs.DirectoryLister;
import io.trino.plugin.hive.fs.HiveFileIterator;
import io.trino.plugin.hive.fs.TrinoFileStatus;
import io.trino.plugin.hive.metastore.Column;
import io.trino.plugin.hive.metastore.MetastoreUtil;
import io.trino.plugin.hive.metastore.Partition;
import io.trino.plugin.hive.metastore.StorageFormat;
import io.trino.plugin.hive.metastore.Table;
import io.trino.plugin.hive.util.AcidTables;
import io.trino.plugin.hive.util.HiveBucketing;
import io.trino.plugin.hive.util.HiveUtil;
import io.trino.plugin.hive.util.InternalHiveSplitFactory;
import io.trino.plugin.hive.util.PartitionMatchSupplier;
import io.trino.plugin.hive.util.ResumableTask;
import io.trino.plugin.hive.util.ResumableTasks;
import io.trino.plugin.hive.util.ValidWriteIdList;
import io.trino.spi.ErrorCodeSupplier;
import io.trino.spi.StandardErrorCode;
import io.trino.spi.TrinoException;
import io.trino.spi.connector.ColumnHandle;
import io.trino.spi.connector.ConnectorSession;
import io.trino.spi.connector.DynamicFilter;
import io.trino.spi.predicate.TupleDomain;
import io.trino.spi.type.TypeManager;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.nio.charset.StandardCharsets;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.Deque;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.OptionalInt;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ConcurrentLinkedDeque;
import java.util.concurrent.Executor;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.locks.ReadWriteLock;
import java.util.concurrent.locks.ReentrantLock;
import java.util.concurrent.locks.ReentrantReadWriteLock;
import java.util.function.BooleanSupplier;
import java.util.function.Function;
import java.util.function.IntPredicate;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.stream.Stream;

public class BackgroundHiveSplitLoader
implements HiveSplitLoader {
    private static final Pattern BUCKET_WITH_OPTIONAL_ATTEMPT_ID_PATTERN = Pattern.compile("bucket_(\\d+)(_\\d+)?$");
    private static final Iterable<Pattern> BUCKET_PATTERNS = ImmutableList.of((Object)Pattern.compile("\\d{8}_\\d{6}_\\d{5}_[a-z0-9]{5}_bucket-(\\d+)(?:[-_.].*)?"), (Object)Pattern.compile("(\\d+)_\\d+.*"), (Object)BUCKET_WITH_OPTIONAL_ATTEMPT_ID_PATTERN);
    private static final ListenableFuture<Void> COMPLETED_FUTURE = Futures.immediateVoidFuture();
    private final Table table;
    private final TupleDomain<? extends ColumnHandle> compactEffectivePredicate;
    private final DynamicFilter dynamicFilter;
    private final long dynamicFilteringWaitTimeoutMillis;
    private final TypeManager typeManager;
    private final Optional<BucketSplitInfo> tableBucketInfo;
    private final DirectoryLister directoryLister;
    private final TrinoFileSystemFactory fileSystemFactory;
    private final int loaderConcurrency;
    private final boolean recursiveDirWalkerEnabled;
    private final boolean ignoreAbsentPartitions;
    private final Executor executor;
    private final ConnectorSession session;
    private final ConcurrentLazyQueue<HivePartitionMetadata> partitions;
    private final Deque<Iterator<InternalHiveSplit>> fileIterators = new ConcurrentLinkedDeque<Iterator<InternalHiveSplit>>();
    private final Optional<ValidWriteIdList> validWriteIds;
    private final Optional<Long> maxSplitFileSize;
    private final int maxPartitions;
    private final ReadWriteLock taskExecutionLock = new ReentrantReadWriteLock();
    private HiveSplitSource hiveSplitSource;
    private Stopwatch stopwatch;
    private volatile boolean stopped;
    private final AtomicInteger activeLoaderCount = new AtomicInteger();
    private final AtomicInteger partitionCount = new AtomicInteger();

    public BackgroundHiveSplitLoader(Table table, Iterator<HivePartitionMetadata> partitions, TupleDomain<? extends ColumnHandle> compactEffectivePredicate, DynamicFilter dynamicFilter, Duration dynamicFilteringWaitTimeout, TypeManager typeManager, Optional<BucketSplitInfo> tableBucketInfo, ConnectorSession session, TrinoFileSystemFactory fileSystemFactory, DirectoryLister directoryLister, Executor executor, int loaderConcurrency, boolean recursiveDirWalkerEnabled, boolean ignoreAbsentPartitions, Optional<ValidWriteIdList> validWriteIds, Optional<Long> maxSplitFileSize, int maxPartitions) {
        this.table = table;
        this.compactEffectivePredicate = compactEffectivePredicate;
        this.dynamicFilter = dynamicFilter;
        this.dynamicFilteringWaitTimeoutMillis = dynamicFilteringWaitTimeout.toMillis();
        this.typeManager = typeManager;
        this.tableBucketInfo = tableBucketInfo;
        this.loaderConcurrency = loaderConcurrency;
        Preconditions.checkArgument((loaderConcurrency > 0 ? 1 : 0) != 0, (String)"loaderConcurrency must be > 0, found: %s", (int)loaderConcurrency);
        this.session = session;
        this.fileSystemFactory = Objects.requireNonNull(fileSystemFactory, "fileSystemFactory is null");
        this.directoryLister = directoryLister;
        this.recursiveDirWalkerEnabled = recursiveDirWalkerEnabled;
        this.ignoreAbsentPartitions = ignoreAbsentPartitions;
        Objects.requireNonNull(executor, "executor is null");
        BackgroundHiveSplitLoader.checkExecutorIsNotDirectExecutor(executor);
        this.executor = executor;
        this.partitions = new ConcurrentLazyQueue<HivePartitionMetadata>(partitions);
        this.validWriteIds = Objects.requireNonNull(validWriteIds, "validWriteIds is null");
        this.maxSplitFileSize = Objects.requireNonNull(maxSplitFileSize, "maxSplitFileSize is null");
        this.maxPartitions = maxPartitions;
    }

    @Override
    public void start(HiveSplitSource splitSource) {
        this.hiveSplitSource = splitSource;
        this.stopwatch = Stopwatch.createStarted();
        this.addLoaderIfNecessary();
    }

    private void addLoaderIfNecessary() {
        if (this.activeLoaderCount.get() >= this.loaderConcurrency) {
            return;
        }
        if (this.activeLoaderCount.incrementAndGet() > this.loaderConcurrency) {
            return;
        }
        ListenableFuture<Void> future = ResumableTasks.submit(this.executor, new HiveSplitLoaderTask());
        MoreFutures.addExceptionCallback(future, this.hiveSplitSource::fail);
    }

    @Override
    public void stop() {
        this.stopped = true;
    }

    private void invokeNoMoreSplitsIfNecessary() {
        this.taskExecutionLock.readLock().lock();
        try {
            if (!this.partitions.isEmpty() || !this.fileIterators.isEmpty()) {
                return;
            }
        }
        catch (Exception e) {
            this.hiveSplitSource.fail(e);
            Preconditions.checkState((boolean)this.stopped, (Object)"Task is not marked as stopped even though it failed");
            return;
        }
        finally {
            this.taskExecutionLock.readLock().unlock();
        }
        this.taskExecutionLock.writeLock().lock();
        try {
            if (this.partitions.isEmpty() && this.fileIterators.isEmpty()) {
                this.hiveSplitSource.noMoreSplits();
            }
        }
        catch (Exception e) {
            this.hiveSplitSource.fail(e);
            Preconditions.checkState((boolean)this.stopped, (Object)"Task is not marked as stopped even though it failed");
        }
        finally {
            this.taskExecutionLock.writeLock().unlock();
        }
    }

    private static <T> ListenableFuture<Void> asVoid(ListenableFuture<T> future) {
        return Futures.transform(future, v -> null, (Executor)MoreExecutors.directExecutor());
    }

    private ListenableFuture<Void> loadSplits() throws IOException {
        Iterator<InternalHiveSplit> splits = this.fileIterators.poll();
        if (splits == null) {
            HivePartitionMetadata partition = this.partitions.poll();
            if (partition == null) {
                return COMPLETED_FUTURE;
            }
            if (this.partitionCount.incrementAndGet() > this.maxPartitions) {
                throw new TrinoException((ErrorCodeSupplier)HiveErrorCode.HIVE_EXCEEDED_PARTITION_LIMIT, String.format("Query over table '%s' can potentially read more than %s partitions", partition.getHivePartition().getTableName(), this.maxPartitions));
            }
            if (!this.partitions.isEmpty()) {
                this.addLoaderIfNecessary();
            }
            return this.loadPartition(partition);
        }
        if (!this.fileIterators.isEmpty()) {
            this.addLoaderIfNecessary();
        }
        while (splits.hasNext() && !this.stopped) {
            ListenableFuture<Void> future = this.hiveSplitSource.addToQueue(splits.next());
            if (future.isDone()) continue;
            this.fileIterators.addFirst(splits);
            return future;
        }
        return COMPLETED_FUTURE;
    }

    private ListenableFuture<Void> loadPartition(HivePartitionMetadata partition) throws IOException {
        boolean splittable;
        HivePartition hivePartition = partition.getHivePartition();
        String partitionName = hivePartition.getPartitionId();
        Map schema = partition.getPartition().map(value -> MetastoreUtil.getHiveSchema(value, this.table)).orElseGet(() -> MetastoreUtil.getHiveSchema(this.table));
        List<HivePartitionKey> partitionKeys = BackgroundHiveSplitLoader.getPartitionKeys(this.table, partition.getPartition());
        TupleDomain effectivePredicate = this.compactEffectivePredicate.transformKeys(HiveColumnHandle.class::cast);
        BooleanSupplier partitionMatchSupplier = PartitionMatchSupplier.createPartitionMatchSupplier(this.dynamicFilter, hivePartition, HiveUtil.getPartitionKeyColumnHandles(this.table, this.typeManager));
        if (!partitionMatchSupplier.getAsBoolean()) {
            return COMPLETED_FUTURE;
        }
        Location location = Location.of((String)MetastoreUtil.getPartitionLocation(this.table, partition.getPartition()));
        boolean bl = splittable = HiveUtil.getFooterCount(schema) == 0 && HiveUtil.getHeaderCount(schema) <= 1;
        if ("org.apache.hadoop.hive.ql.io.SymlinkTextInputFormat".equals(HiveUtil.getInputFormatName(schema).orElse(null))) {
            if (this.tableBucketInfo.isPresent()) {
                throw new TrinoException((ErrorCodeSupplier)StandardErrorCode.NOT_SUPPORTED, "Bucketed table in SymlinkTextInputFormat is not yet supported");
            }
            HiveStorageFormat targetStorageFormat = BackgroundHiveSplitLoader.getSymlinkStorageFormat(HiveUtil.getDeserializerClassName(schema));
            ListMultimap<Location, Location> targets = this.getTargetLocationsByParentFromSymlink(location);
            InternalHiveSplitFactory splitFactory = new InternalHiveSplitFactory(partitionName, targetStorageFormat, schema, partitionKeys, (TupleDomain<HiveColumnHandle>)effectivePredicate, partitionMatchSupplier, partition.getHiveColumnCoercions(), Optional.empty(), Optional.empty(), HiveSessionProperties.getMaxInitialSplitSize(this.session), HiveSessionProperties.isForceLocalScheduling(this.session), this.maxSplitFileSize);
            for (Map.Entry entry : Multimaps.asMap(targets).entrySet()) {
                this.fileIterators.addLast(this.buildManifestFileIterator(splitFactory, (Location)entry.getKey(), (List)entry.getValue(), splittable));
            }
            return COMPLETED_FUTURE;
        }
        StorageFormat rawStorageFormat = partition.getPartition().map(Partition::getStorage).orElseGet(this.table::getStorage).getStorageFormat();
        HiveStorageFormat storageFormat = HiveStorageFormat.getHiveStorageFormat(rawStorageFormat).orElseThrow(() -> new TrinoException((ErrorCodeSupplier)HiveErrorCode.HIVE_INVALID_METADATA, "Unsupported storage format: %s %s".formatted(hivePartition, rawStorageFormat)));
        Optional<HiveSplit.BucketConversion> bucketConversion = Optional.empty();
        boolean bucketConversionRequiresWorkerParticipation = false;
        if (partition.getPartition().isPresent()) {
            Optional<HiveBucketProperty> partitionBucketProperty = partition.getPartition().get().getStorage().getBucketProperty();
            if (this.tableBucketInfo.isPresent() && partitionBucketProperty.isPresent()) {
                int tableBucketCount = this.tableBucketInfo.get().getTableBucketCount();
                HiveBucketing.BucketingVersion bucketingVersion = partitionBucketProperty.get().getBucketingVersion();
                int partitionBucketCount = partitionBucketProperty.get().getBucketCount();
                if (tableBucketCount != partitionBucketCount) {
                    bucketConversion = Optional.of(new HiveSplit.BucketConversion(bucketingVersion, tableBucketCount, partitionBucketCount, this.tableBucketInfo.get().getBucketColumns()));
                    if (tableBucketCount > partitionBucketCount) {
                        bucketConversionRequiresWorkerParticipation = true;
                    }
                }
            }
        }
        Optional<HiveSplit.BucketValidation> bucketValidation = Optional.empty();
        if (HiveSessionProperties.isValidateBucketing(this.session) && this.tableBucketInfo.isPresent()) {
            BucketSplitInfo info = this.tableBucketInfo.get();
            bucketValidation = Optional.of(new HiveSplit.BucketValidation(info.getBucketingVersion(), info.getTableBucketCount(), info.getBucketColumns()));
        }
        InternalHiveSplitFactory splitFactory = new InternalHiveSplitFactory(partitionName, storageFormat, schema, partitionKeys, (TupleDomain<HiveColumnHandle>)effectivePredicate, partitionMatchSupplier, partition.getHiveColumnCoercions(), bucketConversionRequiresWorkerParticipation ? bucketConversion : Optional.empty(), bucketValidation, HiveSessionProperties.getMaxInitialSplitSize(this.session), HiveSessionProperties.isForceLocalScheduling(this.session), this.maxSplitFileSize);
        if (AcidTables.isTransactionalTable(this.table.getParameters())) {
            return this.getTransactionalSplits(location, splittable, bucketConversion, splitFactory);
        }
        TrinoFileSystem trinoFileSystem = this.fileSystemFactory.create(this.session);
        if (this.tableBucketInfo.isPresent()) {
            List<TrinoFileStatus> files = this.listBucketFiles(trinoFileSystem, location, splitFactory.getPartitionName());
            return this.hiveSplitSource.addToQueue(this.getBucketedSplits(files, splitFactory, this.tableBucketInfo.get(), bucketConversion, splittable, Optional.empty()));
        }
        this.fileIterators.addLast(this.createInternalHiveSplitIterator(trinoFileSystem, location, splitFactory, splittable, Optional.empty()));
        return COMPLETED_FUTURE;
    }

    private List<TrinoFileStatus> listBucketFiles(TrinoFileSystem fs, Location location, String partitionName) {
        try {
            HiveFileIterator fileIterator = new HiveFileIterator(this.table, location, fs, this.directoryLister, HiveFileIterator.NestedDirectoryPolicy.FAIL);
            if (!fileIterator.hasNext() && !this.ignoreAbsentPartitions) {
                BackgroundHiveSplitLoader.checkPartitionLocationExists(fs, location);
            }
            return ImmutableList.copyOf((Iterator)((Object)fileIterator));
        }
        catch (HiveFileIterator.NestedDirectoryNotAllowedException e) {
            throw new TrinoException((ErrorCodeSupplier)HiveErrorCode.HIVE_INVALID_BUCKET_FILES, "Hive table '%s' is corrupt. Found sub-directory '%s' in bucket directory for partition: %s".formatted(this.table.getSchemaTableName(), e.getNestedDirectoryPath(), partitionName));
        }
    }

    @VisibleForTesting
    Iterator<InternalHiveSplit> buildManifestFileIterator(InternalHiveSplitFactory splitFactory, Location location, List<Location> paths, boolean splittable) {
        TrinoFileSystem trinoFileSystem = this.fileSystemFactory.create(this.session);
        HashMap fileStatuses = new HashMap();
        HiveFileIterator fileStatusIterator = new HiveFileIterator(this.table, location, trinoFileSystem, this.directoryLister, HiveFileIterator.NestedDirectoryPolicy.RECURSE);
        if (!fileStatusIterator.hasNext()) {
            BackgroundHiveSplitLoader.checkPartitionLocationExists(trinoFileSystem, location);
        }
        fileStatusIterator.forEachRemaining(status -> fileStatuses.put(Location.of((String)status.getPath()).path(), status));
        Stream<TrinoFileStatus> fileStream = paths.stream().map(path -> {
            TrinoFileStatus status = (TrinoFileStatus)fileStatuses.get(path.path());
            if (status == null) {
                throw new TrinoException((ErrorCodeSupplier)HiveErrorCode.HIVE_FILE_NOT_FOUND, "Manifest file from the location [%s] contains non-existent path: %s".formatted(location, path));
            }
            return status;
        });
        return BackgroundHiveSplitLoader.createInternalHiveSplitIterator(splitFactory, splittable, Optional.empty(), fileStream);
    }

    private ListenableFuture<Void> getTransactionalSplits(Location path, boolean splittable, Optional<HiveSplit.BucketConversion> bucketConversion, InternalHiveSplitFactory splitFactory) throws IOException {
        Optional<Location> baseOrDeltaPath;
        TrinoFileSystem fileSystem = this.fileSystemFactory.create(this.session);
        ValidWriteIdList writeIds = this.validWriteIds.orElseThrow(() -> new IllegalStateException("No validWriteIds present"));
        AcidTables.AcidState acidState = AcidTables.getAcidState(fileSystem, path, writeIds);
        boolean fullAcid = AcidTables.isFullAcidTable(this.table.getParameters());
        AcidInfo.Builder acidInfoBuilder = AcidInfo.builder(path);
        if (fullAcid && (baseOrDeltaPath = acidState.baseDirectory().or(() -> acidState.deltas().stream().findFirst().map(delta -> Location.of((String)delta.path())))).isPresent() && AcidTables.readAcidVersionFile(fileSystem, baseOrDeltaPath.get()) >= 2) {
            acidInfoBuilder.setOrcAcidVersionValidated(true);
        }
        ArrayList<TrinoFileStatus> acidFiles = new ArrayList<TrinoFileStatus>();
        for (FileEntry fileEntry : acidState.baseFiles()) {
            acidFiles.add(new TrinoFileStatus(fileEntry));
        }
        for (AcidTables.ParsedDelta parsedDelta : acidState.deltas()) {
            if (parsedDelta.deleteDelta()) {
                if (!fullAcid) {
                    throw new TrinoException((ErrorCodeSupplier)HiveErrorCode.HIVE_BAD_DATA, "Unexpected delete delta for a non full ACID table '%s'. Would be ignored by the reader: %s".formatted(this.table.getSchemaTableName(), parsedDelta.path()));
                }
                acidInfoBuilder.addDeleteDelta(Location.of((String)parsedDelta.path()));
                continue;
            }
            for (FileEntry file : parsedDelta.files()) {
                acidFiles.add(new TrinoFileStatus(file));
            }
        }
        for (FileEntry fileEntry : acidState.originalFiles()) {
            acidInfoBuilder.addOriginalFile(fileEntry.location(), fileEntry.length(), BackgroundHiveSplitLoader.getRequiredBucketNumber(fileEntry.location()));
        }
        if (this.tableBucketInfo.isPresent()) {
            BucketSplitInfo bucketInfo = this.tableBucketInfo.get();
            for (FileEntry entry : acidState.originalFiles()) {
                ImmutableList fileStatuses = ImmutableList.of((Object)new TrinoFileStatus(entry));
                Optional<AcidInfo> acidInfo = BackgroundHiveSplitLoader.acidInfoForOriginalFiles(fullAcid, acidInfoBuilder, entry.location());
                this.hiveSplitSource.addToQueue(this.getBucketedSplits((List<TrinoFileStatus>)fileStatuses, splitFactory, bucketInfo, bucketConversion, splittable, acidInfo));
            }
            Optional<AcidInfo> optional = BackgroundHiveSplitLoader.acidInfo(fullAcid, acidInfoBuilder);
            return this.hiveSplitSource.addToQueue(this.getBucketedSplits(acidFiles, splitFactory, bucketInfo, bucketConversion, splittable, optional));
        }
        Optional<AcidInfo> acidInfo = BackgroundHiveSplitLoader.acidInfo(fullAcid, acidInfoBuilder);
        this.fileIterators.addLast(BackgroundHiveSplitLoader.createInternalHiveSplitIterator(splitFactory, splittable, acidInfo, acidFiles.stream()));
        this.fileIterators.addLast(BackgroundHiveSplitLoader.generateOriginalFilesSplits(splitFactory, acidState.originalFiles(), splittable, acidInfoBuilder, fullAcid));
        return COMPLETED_FUTURE;
    }

    private static Iterator<InternalHiveSplit> generateOriginalFilesSplits(InternalHiveSplitFactory splitFactory, List<FileEntry> originalFileLocations, boolean splittable, AcidInfo.Builder acidInfoBuilder, boolean fullAcid) {
        return originalFileLocations.stream().map(entry -> BackgroundHiveSplitLoader.createInternalHiveSplit(splitFactory, splittable, BackgroundHiveSplitLoader.acidInfoForOriginalFiles(fullAcid, acidInfoBuilder, entry.location()), new TrinoFileStatus((FileEntry)entry))).flatMap(Optional::stream).iterator();
    }

    private static Optional<AcidInfo> acidInfo(boolean fullAcid, AcidInfo.Builder builder) {
        return fullAcid ? builder.build() : Optional.empty();
    }

    private static Optional<AcidInfo> acidInfoForOriginalFiles(boolean fullAcid, AcidInfo.Builder builder, Location location) {
        return fullAcid ? Optional.of(builder.buildWithRequiredOriginalFiles(BackgroundHiveSplitLoader.getRequiredBucketNumber(location))) : Optional.empty();
    }

    private Iterator<InternalHiveSplit> createInternalHiveSplitIterator(TrinoFileSystem fileSystem, Location location, InternalHiveSplitFactory splitFactory, boolean splittable, Optional<AcidInfo> acidInfo) {
        HiveFileIterator iterator = new HiveFileIterator(this.table, location, fileSystem, this.directoryLister, this.recursiveDirWalkerEnabled ? HiveFileIterator.NestedDirectoryPolicy.RECURSE : HiveFileIterator.NestedDirectoryPolicy.IGNORED);
        if (!iterator.hasNext() && !this.ignoreAbsentPartitions) {
            BackgroundHiveSplitLoader.checkPartitionLocationExists(fileSystem, location);
        }
        return BackgroundHiveSplitLoader.createInternalHiveSplitIterator(splitFactory, splittable, acidInfo, Streams.stream((Iterator)((Object)iterator)));
    }

    private static void checkPartitionLocationExists(TrinoFileSystem fileSystem, Location location) {
        try {
            if (!fileSystem.directoryExists(location).orElse(true).booleanValue()) {
                throw new TrinoException((ErrorCodeSupplier)HiveErrorCode.HIVE_FILE_NOT_FOUND, "Partition location does not exist: " + location);
            }
        }
        catch (IOException e) {
            throw new TrinoException((ErrorCodeSupplier)HiveErrorCode.HIVE_FILESYSTEM_ERROR, "Failed checking directory path:" + location, (Throwable)e);
        }
    }

    private static Iterator<InternalHiveSplit> createInternalHiveSplitIterator(InternalHiveSplitFactory splitFactory, boolean splittable, Optional<AcidInfo> acidInfo, Stream<TrinoFileStatus> fileStream) {
        return fileStream.map(file -> BackgroundHiveSplitLoader.createInternalHiveSplit(splitFactory, splittable, acidInfo, file)).flatMap(Optional::stream).iterator();
    }

    private static Optional<InternalHiveSplit> createInternalHiveSplit(InternalHiveSplitFactory splitFactory, boolean splittable, Optional<AcidInfo> acidInfo, TrinoFileStatus file) {
        return splitFactory.createInternalHiveSplit(file, OptionalInt.empty(), OptionalInt.empty(), splittable, acidInfo);
    }

    private List<InternalHiveSplit> getBucketedSplits(List<TrinoFileStatus> files, InternalHiveSplitFactory splitFactory, BucketSplitInfo bucketSplitInfo, Optional<HiveSplit.BucketConversion> bucketConversion, boolean splittable, Optional<AcidInfo> acidInfo) {
        int readBucketCount = bucketSplitInfo.getReadBucketCount();
        int tableBucketCount = bucketSplitInfo.getTableBucketCount();
        int partitionBucketCount = bucketConversion.map(HiveSplit.BucketConversion::partitionBucketCount).orElse(tableBucketCount);
        int bucketCount = Math.max(readBucketCount, partitionBucketCount);
        Preconditions.checkState((readBucketCount <= tableBucketCount ? 1 : 0) != 0, (String)"readBucketCount(%s) should be less than or equal to tableBucketCount(%s)", (int)readBucketCount, (int)tableBucketCount);
        ArrayListMultimap bucketFiles = ArrayListMultimap.create();
        for (TrinoFileStatus file : files) {
            String fileName = Location.of((String)file.getPath()).fileName();
            OptionalInt bucket = BackgroundHiveSplitLoader.getBucketNumber(fileName);
            if (bucket.isPresent()) {
                bucketFiles.put((Object)bucket.getAsInt(), (Object)file);
                continue;
            }
            if (files.size() != partitionBucketCount) {
                throw new TrinoException((ErrorCodeSupplier)HiveErrorCode.HIVE_INVALID_BUCKET_FILES, String.format("Hive table '%s' is corrupt. File '%s' does not match the standard naming pattern, and the number of files in the directory (%s) does not match the declared bucket count (%s) for partition: %s", this.table.getSchemaTableName(), fileName, files.size(), partitionBucketCount, splitFactory.getPartitionName()));
            }
            files = files.stream().sorted().toList();
            bucketFiles.clear();
            for (int i = 0; i < files.size(); ++i) {
                bucketFiles.put((Object)i, (Object)files.get(i));
            }
        }
        BackgroundHiveSplitLoader.validateFileBuckets((ListMultimap<Integer, TrinoFileStatus>)bucketFiles, partitionBucketCount, this.table.getSchemaTableName().toString(), splitFactory.getPartitionName());
        ArrayList<InternalHiveSplit> splitList = new ArrayList<InternalHiveSplit>();
        for (int bucketNumber = 0; bucketNumber < bucketCount; ++bucketNumber) {
            int partitionBucketNumber = bucketNumber % partitionBucketCount;
            int readBucketNumber = bucketNumber % readBucketCount;
            boolean containsIneligibleTableBucket = false;
            ArrayList<Integer> eligibleTableBucketNumbers = new ArrayList<Integer>();
            for (int tableBucketNumber2 = bucketNumber % tableBucketCount; tableBucketNumber2 < tableBucketCount; tableBucketNumber2 += bucketCount) {
                if (bucketSplitInfo.isTableBucketEnabled(tableBucketNumber2)) {
                    eligibleTableBucketNumbers.add(tableBucketNumber2);
                    continue;
                }
                containsIneligibleTableBucket = true;
            }
            if (!eligibleTableBucketNumbers.isEmpty() && containsIneligibleTableBucket) {
                throw new TrinoException((ErrorCodeSupplier)StandardErrorCode.NOT_SUPPORTED, "The bucket filter cannot be satisfied. There are restrictions on the bucket filter when all the following is true: 1. a table has a different buckets count as at least one of its partitions that is read in this query; 2. the table has a different but compatible bucket number with another table in the query; 3. some buckets of the table is filtered out from the query, most likely using a filter on \"$bucket\". (table name: " + this.table.getTableName() + ", table bucket count: " + tableBucketCount + ", partition bucket count: " + partitionBucketCount + ", effective reading bucket count: " + readBucketCount + ")");
            }
            if (eligibleTableBucketNumbers.isEmpty()) continue;
            for (TrinoFileStatus file : bucketFiles.get((Object)partitionBucketNumber)) {
                eligibleTableBucketNumbers.stream().map(tableBucketNumber -> splitFactory.createInternalHiveSplit(file, OptionalInt.of(readBucketNumber), OptionalInt.of(tableBucketNumber), splittable, acidInfo)).flatMap(Optional::stream).forEach(splitList::add);
            }
        }
        return splitList;
    }

    @VisibleForTesting
    static void validateFileBuckets(ListMultimap<Integer, TrinoFileStatus> bucketFiles, int partitionBucketCount, String tableName, String partitionName) {
        if (bucketFiles.isEmpty()) {
            return;
        }
        int highestBucketNumber = (Integer)Collections.max(bucketFiles.keySet());
        if (highestBucketNumber >= partitionBucketCount) {
            throw new TrinoException((ErrorCodeSupplier)HiveErrorCode.HIVE_INVALID_BUCKET_FILES, String.format("Hive table '%s' is corrupt. The highest bucket number in the directory (%s) exceeds the bucket number range defined by the declared bucket count (%s) for partition: %s", tableName, highestBucketNumber, partitionBucketCount, partitionName));
        }
    }

    private static int getRequiredBucketNumber(Location location) {
        return BackgroundHiveSplitLoader.getBucketNumber(location.fileName()).orElseThrow(() -> new IllegalStateException("Cannot get bucket number from location: " + location));
    }

    @VisibleForTesting
    static OptionalInt getBucketNumber(String name) {
        for (Pattern pattern : BUCKET_PATTERNS) {
            Matcher matcher = pattern.matcher(name);
            if (!matcher.matches()) continue;
            return OptionalInt.of(Integer.parseInt(matcher.group(1)));
        }
        return OptionalInt.empty();
    }

    public static boolean hasAttemptId(String bucketFilename) {
        Matcher matcher = BUCKET_WITH_OPTIONAL_ATTEMPT_ID_PATTERN.matcher(bucketFilename);
        return matcher.matches() && matcher.group(2) != null;
    }

    private static HiveStorageFormat getSymlinkStorageFormat(String serde) {
        if (serde.equals(HiveStorageFormat.TEXTFILE.getSerde())) {
            return HiveStorageFormat.TEXTFILE;
        }
        return Arrays.stream(HiveStorageFormat.values()).filter(format -> serde.equals(format.getSerde())).findFirst().orElseThrow(() -> new TrinoException((ErrorCodeSupplier)HiveErrorCode.HIVE_UNSUPPORTED_FORMAT, "Unknown SerDe for SymlinkTextInputFormat: " + serde));
    }

    private ListMultimap<Location, Location> getTargetLocationsByParentFromSymlink(Location symlinkDir) {
        TrinoFileSystem fileSystem = this.fileSystemFactory.create(this.session);
        try {
            ArrayListMultimap targets = ArrayListMultimap.create();
            FileIterator iterator = fileSystem.listFiles(symlinkDir);
            while (iterator.hasNext()) {
                Location location = iterator.next().location();
                String name = location.fileName();
                if (name.startsWith("_") || name.startsWith(".")) continue;
                try (InputStreamReader reader = new InputStreamReader((InputStream)fileSystem.newInputFile(location).newStream(), StandardCharsets.UTF_8);){
                    CharStreams.readLines((Readable)reader).stream().map(Location::of).forEach(arg_0 -> BackgroundHiveSplitLoader.lambda$getTargetLocationsByParentFromSymlink$15((ListMultimap)targets, arg_0));
                }
            }
            return targets;
        }
        catch (IOException | IllegalArgumentException e) {
            throw new TrinoException((ErrorCodeSupplier)HiveErrorCode.HIVE_BAD_DATA, "Error parsing symlinks from: " + symlinkDir, (Throwable)e);
        }
    }

    private static List<HivePartitionKey> getPartitionKeys(Table table, Optional<Partition> partition) {
        if (partition.isEmpty()) {
            return ImmutableList.of();
        }
        ImmutableList.Builder partitionKeys = ImmutableList.builder();
        List<Column> keys = table.getPartitionColumns();
        List<String> values = partition.get().getValues();
        HiveUtil.checkCondition(keys.size() == values.size(), HiveErrorCode.HIVE_INVALID_METADATA, "Expected %s partition key values, but got %s", keys.size(), values.size());
        for (int i = 0; i < keys.size(); ++i) {
            String name = keys.get(i).getName();
            HiveType hiveType = keys.get(i).getType();
            if (!hiveType.isSupportedType(table.getStorage().getStorageFormat())) {
                throw new TrinoException((ErrorCodeSupplier)StandardErrorCode.NOT_SUPPORTED, String.format("Unsupported Hive type %s found in partition keys of table %s.%s", hiveType, table.getDatabaseName(), table.getTableName()));
            }
            String value = values.get(i);
            HiveUtil.checkCondition(value != null, HiveErrorCode.HIVE_INVALID_PARTITION_VALUE, "partition key value cannot be null for field: %s", name);
            partitionKeys.add((Object)new HivePartitionKey(name, value));
        }
        return partitionKeys.build();
    }

    private static void checkExecutorIsNotDirectExecutor(Executor executor) {
        ReentrantLock lock = new ReentrantLock();
        lock.lock();
        try {
            executor.execute(() -> Preconditions.checkState((!lock.isHeldByCurrentThread() ? 1 : 0) != 0, (Object)"executor is a direct executor"));
        }
        finally {
            lock.unlock();
        }
    }

    private static /* synthetic */ void lambda$getTargetLocationsByParentFromSymlink$15(ListMultimap targets, Location target) {
        targets.put((Object)target.parentDirectory(), (Object)target);
    }

    private class HiveSplitLoaderTask
    implements ResumableTask {
        private HiveSplitLoaderTask() {
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        @Override
        public ResumableTask.TaskStatus process() {
            ListenableFuture<Void> future;
            do {
                if (BackgroundHiveSplitLoader.this.stopped) {
                    return ResumableTask.TaskStatus.finished();
                }
                long timeLeft = BackgroundHiveSplitLoader.this.dynamicFilteringWaitTimeoutMillis - BackgroundHiveSplitLoader.this.stopwatch.elapsed(TimeUnit.MILLISECONDS);
                if (timeLeft > 0L && BackgroundHiveSplitLoader.this.dynamicFilter.isAwaitable()) {
                    future = BackgroundHiveSplitLoader.asVoid(MoreFutures.toListenableFuture(((CompletableFuture)BackgroundHiveSplitLoader.this.dynamicFilter.isBlocked().thenApply(Function.identity())).orTimeout(timeLeft, TimeUnit.MILLISECONDS)));
                    return ResumableTask.TaskStatus.continueOn(future);
                }
                BackgroundHiveSplitLoader.this.taskExecutionLock.readLock().lock();
                try {
                    future = BackgroundHiveSplitLoader.this.loadSplits();
                }
                catch (Throwable e2) {
                    TrinoException e2;
                    if (e2 instanceof IOException) {
                        e2 = new TrinoException((ErrorCodeSupplier)HiveErrorCode.HIVE_FILESYSTEM_ERROR, e2);
                    } else if (!(e2 instanceof TrinoException)) {
                        e2 = new TrinoException((ErrorCodeSupplier)HiveErrorCode.HIVE_UNKNOWN_ERROR, e2);
                    }
                    BackgroundHiveSplitLoader.this.hiveSplitSource.fail(e2);
                    Preconditions.checkState((boolean)BackgroundHiveSplitLoader.this.stopped);
                    ResumableTask.TaskStatus taskStatus = ResumableTask.TaskStatus.finished();
                    return taskStatus;
                }
                finally {
                    BackgroundHiveSplitLoader.this.taskExecutionLock.readLock().unlock();
                }
                BackgroundHiveSplitLoader.this.invokeNoMoreSplitsIfNecessary();
            } while (future.isDone());
            return ResumableTask.TaskStatus.continueOn(future);
        }
    }

    public static class BucketSplitInfo {
        private final HiveBucketing.BucketingVersion bucketingVersion;
        private final List<HiveColumnHandle> bucketColumns;
        private final int tableBucketCount;
        private final int readBucketCount;
        private final IntPredicate bucketFilter;

        public static Optional<BucketSplitInfo> createBucketSplitInfo(Optional<HiveBucketHandle> bucketHandle, Optional<HiveBucketing.HiveBucketFilter> bucketFilter) {
            Objects.requireNonNull(bucketHandle, "bucketHandle is null");
            Objects.requireNonNull(bucketFilter, "bucketFilter is null");
            if (bucketHandle.isEmpty()) {
                Preconditions.checkArgument((boolean)bucketFilter.isEmpty(), (Object)"bucketHandle must be present if bucketFilter is present");
                return Optional.empty();
            }
            HiveBucketing.BucketingVersion bucketingVersion = bucketHandle.get().getBucketingVersion();
            int tableBucketCount = bucketHandle.get().getTableBucketCount();
            int readBucketCount = bucketHandle.get().getReadBucketCount();
            List<HiveColumnHandle> bucketColumns = bucketHandle.get().getColumns();
            IntPredicate predicate = bucketFilter.map(filter -> filter.getBucketsToKeep()::contains).orElse(bucket -> true);
            return Optional.of(new BucketSplitInfo(bucketingVersion, bucketColumns, tableBucketCount, readBucketCount, predicate));
        }

        private BucketSplitInfo(HiveBucketing.BucketingVersion bucketingVersion, List<HiveColumnHandle> bucketColumns, int tableBucketCount, int readBucketCount, IntPredicate bucketFilter) {
            this.bucketingVersion = Objects.requireNonNull(bucketingVersion, "bucketingVersion is null");
            this.bucketColumns = ImmutableList.copyOf((Collection)Objects.requireNonNull(bucketColumns, "bucketColumns is null"));
            this.tableBucketCount = tableBucketCount;
            this.readBucketCount = readBucketCount;
            this.bucketFilter = Objects.requireNonNull(bucketFilter, "bucketFilter is null");
        }

        public HiveBucketing.BucketingVersion getBucketingVersion() {
            return this.bucketingVersion;
        }

        public List<HiveColumnHandle> getBucketColumns() {
            return this.bucketColumns;
        }

        public int getTableBucketCount() {
            return this.tableBucketCount;
        }

        public int getReadBucketCount() {
            return this.readBucketCount;
        }

        public boolean isTableBucketEnabled(int tableBucketNumber) {
            return this.bucketFilter.test(tableBucketNumber);
        }
    }
}

