/*
 * Decompiled with CFR 0.152.
 */
package alluxio.security.authentication;

import alluxio.conf.AlluxioConfiguration;
import alluxio.conf.PropertyKey;
import alluxio.exception.status.UnauthenticatedException;
import alluxio.grpc.ChannelAuthenticationScheme;
import alluxio.grpc.SaslAuthenticationServiceGrpc;
import alluxio.grpc.SaslMessage;
import alluxio.security.authentication.AuthType;
import alluxio.security.authentication.AuthenticatedChannelServerDriver;
import alluxio.security.authentication.AuthenticatedUserInfo;
import alluxio.security.authentication.AuthenticationServer;
import alluxio.security.authentication.ImpersonationAuthenticator;
import alluxio.security.authentication.SaslServerHandler;
import alluxio.security.authentication.plain.SaslServerHandlerPlain;
import alluxio.util.ThreadFactoryUtils;
import io.grpc.Status;
import io.grpc.StatusRuntimeException;
import io.grpc.stub.StreamObserver;
import java.time.LocalDateTime;
import java.time.LocalTime;
import java.util.ArrayList;
import java.util.Map;
import java.util.UUID;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import javax.annotation.concurrent.ThreadSafe;
import javax.security.sasl.SaslException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@ThreadSafe
public class DefaultAuthenticationServer
extends SaslAuthenticationServiceGrpc.SaslAuthenticationServiceImplBase
implements AuthenticationServer {
    private static final Logger LOG = LoggerFactory.getLogger(DefaultAuthenticationServer.class);
    protected final ConcurrentHashMap<UUID, AuthenticatedChannelInfo> mChannels;
    protected final ScheduledExecutorService mScheduler;
    protected final String mHostName;
    protected final long mCleanupIntervalMs;
    protected final AlluxioConfiguration mConfiguration;
    private final ImpersonationAuthenticator mImpersonationAuthenticator;

    public DefaultAuthenticationServer(String hostName, AlluxioConfiguration conf) {
        this.mHostName = hostName;
        this.mConfiguration = conf;
        this.mCleanupIntervalMs = conf.getMs(PropertyKey.AUTHENTICATION_INACTIVE_CHANNEL_REAUTHENTICATE_PERIOD);
        this.checkSupported(conf.getEnum(PropertyKey.SECURITY_AUTHENTICATION_TYPE, AuthType.class));
        this.mChannels = new ConcurrentHashMap();
        this.mScheduler = Executors.newScheduledThreadPool(1, ThreadFactoryUtils.build("auth-cleanup", true));
        this.mScheduler.scheduleAtFixedRate(this::cleanupStaleClients, this.mCleanupIntervalMs, this.mCleanupIntervalMs, TimeUnit.MILLISECONDS);
        this.mImpersonationAuthenticator = new ImpersonationAuthenticator(conf);
    }

    protected ImpersonationAuthenticator getImpersonationAuthenticator() {
        return this.mImpersonationAuthenticator;
    }

    public StreamObserver<SaslMessage> authenticate(StreamObserver<SaslMessage> responseObserver) {
        AuthenticatedChannelServerDriver driver = new AuthenticatedChannelServerDriver(this);
        driver.setClientObserver(responseObserver);
        return driver;
    }

    @Override
    public void registerChannel(UUID channelId, AuthenticatedUserInfo userInfo, AuthenticatedChannelServerDriver serverDriver) {
        LOG.debug("Registering new channel:{} for user:{}", (Object)channelId, (Object)userInfo);
        if (null != this.mChannels.putIfAbsent(channelId, new AuthenticatedChannelInfo(userInfo, serverDriver))) {
            AuthenticatedChannelInfo existingInfo = this.mChannels.remove(channelId);
            throw new RuntimeException(String.format("Channel: %s already exists in authentication registry for user: %s.", channelId.toString(), existingInfo.getUserInfo()));
        }
    }

    @Override
    public AuthenticatedUserInfo getUserInfoForChannel(UUID channelId) throws UnauthenticatedException {
        AuthenticatedChannelInfo clientInfo = this.mChannels.get(channelId);
        if (clientInfo != null) {
            return clientInfo.getUserInfo();
        }
        throw new UnauthenticatedException(String.format("Channel:%s needs to be authenticated", channelId.toString()));
    }

    @Override
    public void unregisterChannel(UUID channelId) {
        LOG.debug("Unregistering channel: {}", (Object)channelId);
        this.mChannels.remove(channelId);
    }

    @Override
    public SaslServerHandler createSaslHandler(ChannelAuthenticationScheme authScheme) throws SaslException {
        switch (authScheme) {
            case SIMPLE: 
            case CUSTOM: {
                return new SaslServerHandlerPlain(this.mHostName, this.mConfiguration, this.mImpersonationAuthenticator);
            }
        }
        throw new StatusRuntimeException(Status.UNAUTHENTICATED.augmentDescription(String.format("Authentication scheme:%s is not supported", authScheme)));
    }

    @Override
    public void close() {
        for (Map.Entry<UUID, AuthenticatedChannelInfo> entry : this.mChannels.entrySet()) {
            try {
                entry.getValue().getSaslServerDriver().close();
            }
            catch (Exception exc) {
                LOG.debug("Failed closing authentication session for channel:{}. Error:{}", (Object)entry.getKey(), (Object)exc);
            }
        }
    }

    private void cleanupStaleClients() {
        LocalDateTime cleanupTime = LocalDateTime.now();
        LOG.debug("Starting cleanup authentication registry at {}", (Object)cleanupTime);
        ArrayList<UUID> staleChannels = new ArrayList<UUID>();
        for (Map.Entry<UUID, AuthenticatedChannelInfo> clientEntry : this.mChannels.entrySet()) {
            LocalDateTime lat = clientEntry.getValue().getLastAccessTime();
            if (!lat.plusSeconds(this.mCleanupIntervalMs / 1000L).isBefore(cleanupTime)) continue;
            staleChannels.add(clientEntry.getKey());
        }
        LOG.debug("Found {} stale channels for cleanup.", (Object)staleChannels.size());
        for (UUID clientId : staleChannels) {
            this.mChannels.remove(clientId).getSaslServerDriver().close();
        }
        LOG.debug("Finished state channel cleanup at {}", (Object)LocalTime.now());
    }

    protected void checkSupported(AuthType authType) {
        switch (authType) {
            case NOSASL: 
            case SIMPLE: 
            case CUSTOM: {
                return;
            }
        }
        throw new RuntimeException("Authentication type not supported:" + authType.name());
    }

    class AuthenticatedChannelInfo {
        private LocalDateTime mLastAccessTime;
        private AuthenticatedUserInfo mUserInfo;
        private AuthenticatedChannelServerDriver mSaslServerDriver;

        public AuthenticatedChannelInfo(AuthenticatedUserInfo userInfo, AuthenticatedChannelServerDriver saslServerDriver) {
            this.mUserInfo = userInfo;
            this.mSaslServerDriver = saslServerDriver;
            this.mLastAccessTime = LocalDateTime.now();
        }

        private synchronized void updateLastAccessTime() {
            this.mLastAccessTime = LocalDateTime.now();
        }

        public synchronized LocalDateTime getLastAccessTime() {
            return this.mLastAccessTime;
        }

        public AuthenticatedUserInfo getUserInfo() {
            this.updateLastAccessTime();
            return this.mUserInfo;
        }

        public AuthenticatedChannelServerDriver getSaslServerDriver() {
            this.updateLastAccessTime();
            return this.mSaslServerDriver;
        }
    }
}

