/*
 * Decompiled with CFR 0.152.
 */
package io.trino.plugin.deltalake;

import com.google.common.base.Verify;
import com.google.common.io.MoreFiles;
import com.google.common.io.RecursiveDeleteOption;
import io.airlift.testing.Closeables;
import io.airlift.units.DataSize;
import io.trino.Session;
import io.trino.filesystem.TrinoFileSystem;
import io.trino.filesystem.hdfs.HdfsFileSystemFactory;
import io.trino.plugin.deltalake.DeltaLakeConfig;
import io.trino.plugin.deltalake.DeltaLakeQueryRunner;
import io.trino.plugin.deltalake.DeltaTestingConnectorSession;
import io.trino.plugin.deltalake.TestingDeltaLakeUtils;
import io.trino.plugin.deltalake.transactionlog.AddFileEntry;
import io.trino.plugin.deltalake.transactionlog.DeltaLakeTransactionLogEntry;
import io.trino.plugin.deltalake.transactionlog.TransactionLogEntries;
import io.trino.plugin.deltalake.transactionlog.checkpoint.TransactionLogTail;
import io.trino.plugin.deltalake.transactionlog.statistics.DeltaLakeFileStatistics;
import io.trino.plugin.hive.HiveTestUtils;
import io.trino.plugin.tpcds.TpcdsPlugin;
import io.trino.spi.Plugin;
import io.trino.testing.AbstractTestQueryFramework;
import io.trino.testing.DistributedQueryRunner;
import io.trino.testing.QueryRunner;
import io.trino.testing.TestingAccessControlManager;
import io.trino.testing.TestingNames;
import io.trino.testing.sql.TestTable;
import java.io.IOException;
import java.net.URI;
import java.net.URISyntaxException;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.attribute.FileAttribute;
import java.time.Instant;
import java.time.ZoneOffset;
import java.time.format.DateTimeFormatter;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import org.assertj.core.api.Assertions;
import org.junit.jupiter.api.Test;

public class TestDeltaLakeAnalyze
extends AbstractTestQueryFramework {
    private static final TrinoFileSystem FILE_SYSTEM = new HdfsFileSystemFactory(HiveTestUtils.HDFS_ENVIRONMENT, HiveTestUtils.HDFS_FILE_SYSTEM_STATS).create(DeltaTestingConnectorSession.SESSION);

    protected QueryRunner createQueryRunner() throws Exception {
        DistributedQueryRunner queryRunner = DeltaLakeQueryRunner.builder().addDeltaProperty("delta.enable-non-concurrent-writes", "true").addDeltaProperty("delta.register-table-procedure.enabled", "true").build();
        try {
            queryRunner.installPlugin((Plugin)new TpcdsPlugin());
            queryRunner.createCatalog("tpcds", "tpcds");
            return queryRunner;
        }
        catch (Exception e) {
            Closeables.closeAllSuppress((Throwable)e, (AutoCloseable[])new AutoCloseable[]{queryRunner});
            throw e;
        }
    }

    @Test
    public void testAnalyze() {
        this.testAnalyze(Optional.empty());
    }

    @Test
    public void testAnalyzeWithCheckpoints() {
        this.testAnalyze(Optional.of(1));
    }

    private void testAnalyze(Optional<Integer> checkpointInterval) {
        String tableName = "test_analyze_" + TestingNames.randomNameSuffix();
        this.assertUpdate("CREATE TABLE " + tableName + (checkpointInterval.isPresent() ? String.format(" WITH (checkpoint_interval = %s)", checkpointInterval.get()) : "") + " AS SELECT * FROM tpch.sf1.nation", 25L);
        this.assertQuery("SHOW STATS FOR " + tableName, "VALUES ('nationkey', null, 25.0, 0.0, null, 0, 24),('regionkey', null, 5.0, 0.0, null, 0, 4),('comment', 1857.0, 25.0, 0.0, null, null, null),('name', 177.0, 25.0, 0.0, null, null, null),(null, null, null, null, 25.0, null, null)");
        this.assertUpdate("ANALYZE " + tableName);
        String expectedStats = "VALUES ('nationkey', null, 25.0, 0.0, null, 0, 24),('regionkey', null, 5.0, 0.0, null, 0, 4),('comment', 1857.0, 25.0, 0.0, null, null, null),('name', 177.0, 25.0, 0.0, null, null, null),(null, null, null, null, 25.0, null, null)";
        this.assertQuery("SHOW STATS FOR " + tableName, expectedStats);
        this.assertUpdate("ANALYZE " + tableName + " WITH(mode = 'incremental')");
        this.assertQuery("SHOW STATS FOR " + tableName, expectedStats);
        this.assertUpdate("INSERT INTO " + tableName + " SELECT * FROM tpch.sf1.nation", 25L);
        this.assertQuery("SHOW STATS FOR " + tableName, "VALUES ('nationkey', null, 25.0, 0.0, null, 0, 24),('regionkey', null, 5.0, 0.0, null, 0, 4),('comment', 3714.0, 25.0, 0.0, null, null, null),('name', 354.0, 25.0, 0.0, null, null, null),(null, null, null, null, 50.0, null, null)");
        this.assertUpdate("ANALYZE " + tableName);
        this.assertQuery("SHOW STATS FOR " + tableName, "VALUES ('nationkey', null, 25.0, 0.0, null, 0, 24),('regionkey', null, 5.0, 0.0, null, 0, 4),('comment', 3714.0, 25.0, 0.0, null, null, null),('name', 354.0, 25.0, 0.0, null, null, null),(null, null, null, null, 50.0, null, null)");
        this.assertUpdate("INSERT INTO " + tableName + " SELECT nationkey + 25, reverse(name), regionkey + 5, reverse(comment) FROM tpch.sf1.nation", 25L);
        this.assertQuery("SHOW STATS FOR " + tableName, "VALUES ('nationkey', null, 50.0, 0.0, null, 0, 49),('regionkey', null, 10.0, 0.0, null, 0, 9),('comment', 5571.0, 50.0, 0.0, null, null, null),('name', 531.0, 50.0, 0.0, null, null, null),(null, null, null, null, 75.0, null, null)");
        this.assertUpdate("DROP TABLE " + tableName);
    }

    @Test
    public void testAnalyzePartitioned() {
        String tableName = "test_analyze_" + TestingNames.randomNameSuffix();
        this.assertUpdate("CREATE TABLE " + tableName + " WITH (   partitioned_by = ARRAY['regionkey'])AS SELECT * FROM tpch.sf1.nation", 25L);
        this.assertQuery("SHOW STATS FOR " + tableName, "VALUES ('nationkey', null, 25.0, 0.0, null, 0, 24),('regionkey', null, 5.0, 0.0, null, null, null),('comment', 1857.0, 25.0, 0.0, null, null, null),('name', 177.0, 25.0, 0.0, null, null, null),(null, null, null, null, 25.0, null, null)");
        this.assertUpdate("ANALYZE " + tableName);
        this.assertQuery("SHOW STATS FOR " + tableName, "VALUES ('nationkey', null, 25.0, 0.0, null, 0, 24),('regionkey', null, 5.0, 0.0, null, null, null),('comment', 1857.0, 25.0, 0.0, null, null, null),('name', 177.0, 25.0, 0.0, null, null, null),(null, null, null, null, 25.0, null, null)");
        this.assertUpdate("INSERT INTO " + tableName + " SELECT * FROM tpch.sf1.nation", 25L);
        this.assertQuery("SHOW STATS FOR " + tableName, "VALUES ('nationkey', null, 25.0, 0.0, null, 0, 24),('regionkey', null, 5.0, 0.0, null, null, null),('comment', 3714.0, 25.0, 0.0, null, null, null),('name', 354.0, 25.0, 0.0, null, null, null),(null, null, null, null, 50.0, null, null)");
        this.assertUpdate("ANALYZE " + tableName);
        this.assertQuery("SHOW STATS FOR " + tableName, "VALUES ('nationkey', null, 25.0, 0.0, null, 0, 24),('regionkey', null, 5.0, 0.0, null, null, null),('comment', 3714.0, 25.0, 0.0, null, null, null),('name', 354.0, 25.0, 0.0, null, null, null),(null, null, null, null, 50.0, null, null)");
        this.assertUpdate("INSERT INTO " + tableName + " SELECT nationkey + 25, reverse(name), regionkey + 5, reverse(comment) FROM tpch.sf1.nation", 25L);
        this.assertQuery("SHOW STATS FOR " + tableName, "VALUES ('nationkey', null, 50.0, 0.0, null, 0, 49),('regionkey', null, 10.0, 0.0, null, null, null),('comment', 5571.0, 50.0, 0.0, null, null, null),('name', 531.0, 50.0, 0.0, null, null, null),(null, null, null, null, 75.0, null, null)");
        this.assertUpdate("DROP TABLE " + tableName);
    }

    @Test
    public void testAnalyzeEmpty() {
        String tableName = "test_analyze_empty_" + TestingNames.randomNameSuffix();
        this.assertUpdate("CREATE TABLE " + tableName + " AS SELECT * FROM tpch.sf1.nation WHERE false", 0L);
        this.assertQuery("SHOW STATS FOR " + tableName, "VALUES ('nationkey', 0.0, 0.0, 1.0, null, null, null),('regionkey', 0.0, 0.0, 1.0, null, null, null),('comment', 0.0, 0.0, 1.0, null, null, null),('name', 0.0, 0.0, 1.0, null, null, null),(null, null, null, null, 0, null, null)");
        this.assertUpdate("ANALYZE " + tableName);
        this.assertQuery("SHOW STATS FOR " + tableName, "VALUES ('nationkey', 0.0, 0.0, 1.0, null, null, null),('regionkey', 0.0, 0.0, 1.0, null, null, null),('comment', 0.0, 0.0, 1.0, null, null, null),('name', 0.0, 0.0, 1.0, null, null, null),(null, null, null, null, 0, null, null)");
        this.assertUpdate("INSERT INTO " + tableName + " SELECT * FROM tpch.sf1.nation", 25L);
        this.assertUpdate("ANALYZE " + tableName);
        this.assertQuery("SHOW STATS FOR " + tableName, "VALUES ('nationkey', null, 25.0, 0.0, null, 0, 24),('regionkey', null, 5.0, 0.0, null, 0, 4),('comment', 1857.0, 25.0, 0.0, null, null, null),('name', 177.0, 25.0, 0.0, null, null, null),(null, null, null, null, 25.0, null, null)");
        this.assertUpdate("DROP TABLE " + tableName);
    }

    @Test
    public void testAnalyzeExtendedStatisticsDisabled() {
        String tableName = "test_analyze_extended_stats_disabled" + TestingNames.randomNameSuffix();
        this.assertUpdate("CREATE TABLE " + tableName + " AS SELECT * FROM tpch.sf1.nation", 25L);
        Session extendedStatisticsDisabled = Session.builder((Session)this.getSession()).setCatalogSessionProperty((String)this.getSession().getCatalog().orElseThrow(), "extended_statistics_enabled", "false").build();
        this.assertQueryFails(extendedStatisticsDisabled, "ANALYZE " + tableName, "ANALYZE not supported if extended statistics are disabled. Enable via delta.extended-statistics.enabled config property or extended_statistics_enabled session property.");
        this.assertUpdate("DROP TABLE " + tableName);
    }

    @Test
    public void testAnalyzeWithFilesModifiedAfter() throws InterruptedException {
        String tableName = "test_analyze_" + TestingNames.randomNameSuffix();
        this.assertUpdate(this.withStatsOnWrite(false), "CREATE TABLE " + tableName + " AS SELECT * FROM tpch.sf1.nation", 25L);
        Thread.sleep(100L);
        Instant afterInitialDataIngestion = Instant.now();
        Thread.sleep(100L);
        this.assertUpdate("INSERT INTO " + tableName + " SELECT * FROM tpch.sf1.nation WHERE nationkey < 5", 5L);
        DateTimeFormatter formatter = DateTimeFormatter.ofPattern("'TIMESTAMP '''yyyy-MM-dd HH:mm:ss.SSS VV''").withZone(ZoneOffset.UTC);
        this.getDistributedQueryRunner().executeWithPlan(this.getSession(), String.format("ANALYZE %s WITH(files_modified_after = %s)", tableName, formatter.format(afterInitialDataIngestion)));
        this.assertQuery("SHOW STATS FOR " + tableName, "VALUES ('nationkey', null, 5.0, 0.0, null, 0, 24),('regionkey', null, 3.0, 0.0, null, 0, 4),('comment', 434.0, 5.0, 0.0, null, null, null),('name', 33.0, 5.0, 0.0, null, null, null),(null, null, null, null, 30.0, null, null)");
        this.assertUpdate("DROP TABLE " + tableName);
    }

    @Test
    public void testAnalyzeSomeColumns() {
        String tableName = "test_analyze_some_columns" + TestingNames.randomNameSuffix();
        this.assertUpdate("CREATE TABLE " + tableName + " AS SELECT * FROM tpch.sf1.nation", 25L);
        this.assertQueryFails(String.format("ANALYZE %s WITH(columns = ARRAY[])", tableName), "Cannot specify empty list of columns for analysis");
        this.assertQueryFails(String.format("ANALYZE %s WITH(columns = ARRAY['nationkey', 'blah'])", tableName), "\\QInvalid columns specified for analysis: [blah]\\E");
        this.assertUpdate(String.format("ANALYZE %s WITH(columns = ARRAY['nationkey', 'regionkey'])", tableName));
        this.assertQuery("SHOW STATS FOR " + tableName, "VALUES ('nationkey', null, 25.0, 0.0, null, 0, 24),('regionkey', null, 5.0, 0.0, null, 0, 4),('comment', null, null, 0.0, null, null, null),('name', null, null, 0.0, null, null, null),(null, null, null, null, 25.0, null, null)");
        this.assertQueryFails(String.format("ANALYZE %s WITH(columns = ARRAY['nationkey', 'regionkey', 'name'])", tableName), "List of columns to be analyzed must be a subset of previously used: \\[nationkey, regionkey\\]. To extend list of analyzed columns drop table statistics");
        this.assertQueryFails("ANALYZE " + tableName, "List of columns to be analyzed must be a subset of previously used: \\[nationkey, regionkey\\]. To extend list of analyzed columns drop table statistics");
        this.assertUpdate("INSERT INTO " + tableName + " SELECT nationkey + 25, concat(name, '1'), regionkey + 5, concat(comment, '21') FROM tpch.sf1.nation", 25L);
        this.assertQuery("SHOW STATS FOR " + tableName, "VALUES ('nationkey', null, 50.0, 0.0, null, 0, 49),('regionkey', null, 10.0, 0.0, null, 0, 9),('comment', null, null, 0.0, null, null, null),('name', null, null, 0.0, null, null, null),(null, null, null, null, 50.0, null, null)");
        this.assertQueryFails(String.format("ANALYZE %s WITH(columns = ARRAY['nationkey', 'regionkey', 'name'])", tableName), "List of columns to be analyzed must be a subset of previously used: \\[nationkey, regionkey\\]. To extend list of analyzed columns drop table statistics");
        this.assertUpdate(String.format("ANALYZE %s WITH(columns = ARRAY['nationkey', 'regionkey'])", tableName));
        this.assertQuery("SHOW STATS FOR " + tableName, "VALUES ('nationkey', null, 50.0, 0.0, null, 0, 49),('regionkey', null, 10.0, 0.0, null, 0, 9),('comment', null, null, 0.0, null, null, null),('name', null, null, 0.0, null, null, null),(null, null, null, null, 50.0, null, null)");
        this.assertUpdate(String.format("ANALYZE %s WITH(mode = 'full_refresh', columns = ARRAY['nationkey', 'regionkey', 'name'])", tableName), 50L);
        this.assertQuery("SHOW STATS FOR " + tableName, "VALUES ('nationkey', null, 50.0, 0.0, null, 0, 49),('regionkey', null, 10.0, 0.0, null, 0, 9),('comment', null, null, 0.0, null, null, null),('name', 379.0, 50.0, 0.0, null, null, null),(null, null, null, null, 50.0, null, null)");
        String expectedFullStats = "VALUES ('nationkey', null, 50.0, 0.0, null, 0, 49),('regionkey', null, 10.0, 0.0, null, 0, 9),('comment', 3764.0, 50.0, 0.0, null, null, null),('name', 379.0, 50.0, 0.0, null, null, null),(null, null, null, null, 50.0, null, null)";
        this.assertUpdate(String.format("ANALYZE %s WITH(mode = 'full_refresh')", tableName), 50L);
        this.assertQuery("SHOW STATS FOR " + tableName, expectedFullStats);
        this.assertUpdate(String.format("CALL system.drop_extended_stats(CURRENT_SCHEMA, '%s')", tableName));
        this.assertUpdate(String.format("ANALYZE %s", tableName), 50L);
        this.assertQuery("SHOW STATS FOR " + tableName, expectedFullStats);
        this.assertUpdate(String.format("ANALYZE %s WITH(columns = ARRAY['nationkey', 'regionkey'])", tableName));
        this.assertQuery("SHOW STATS FOR " + tableName, "VALUES ('nationkey', null, 50.0, 0.0, null, 0, 49),('regionkey', null, 10.0, 0.0, null, 0, 9),('comment', null, null, 0.0, null, null, null),('name', null, null, 0.0, null, null, null),(null, null, null, null, 50.0, null, null)");
        this.assertUpdate(String.format("ANALYZE %s WITH(columns = ARRAY['nationkey'])", tableName));
        this.assertQuery("SHOW STATS FOR " + tableName, "VALUES ('nationkey', null, 50.0, 0.0, null, 0, 49),('regionkey', null, null, 0.0, null, 0, 9),('comment', null, null, 0.0, null, null, null),('name', null, null, 0.0, null, null, null),(null, null, null, null, 50.0, null, null)");
        this.assertUpdate("DROP TABLE " + tableName);
    }

    @Test
    public void testDropExtendedStats() {
        try (TestTable table = this.newTrinoTable("test_drop_extended_stats", "AS SELECT * FROM tpch.sf1.nation");){
            String query = "SHOW STATS FOR " + table.getName();
            String baseStats = "VALUES('nationkey', null, null,  0.0, null,    0,   24),('regionkey', null, null,  0.0, null,    0,    4),('comment',   null, null,  0.0, null, null, null),('name',      null, null,  0.0, null, null, null),(null,        null, null, null, 25.0, null, null)";
            String extendedStats = "VALUES('nationkey', null, 25.0,  0.0, null,    0,   24),('regionkey', null,  5.0,  0.0, null,    0,    4),('comment', 1857.0, 25.0,  0.0, null, null, null),('name',     177.0, 25.0,  0.0, null, null, null),(null,        null, null, null, 25.0, null, null)";
            this.assertQuery(query, extendedStats);
            this.assertUpdate(String.format("CALL system.drop_extended_stats(CURRENT_SCHEMA, '%s')", table.getName()));
            this.assertQuery(query, baseStats);
            this.assertUpdate("ANALYZE " + table.getName(), 25L);
            this.assertQuery(query, extendedStats);
        }
    }

    @Test
    public void testDropMissingStats() {
        try (TestTable table = this.newTrinoTable("test_drop_missing_stats", "AS SELECT * FROM tpch.sf1.nation");){
            this.assertUpdate(String.format("CALL system.drop_extended_stats(CURRENT_SCHEMA, '%s')", table.getName()));
            this.assertQuery("SHOW STATS FOR " + table.getName(), "VALUES('nationkey', null, null,  0.0, null,    0,   24),('regionkey', null, null,  0.0, null,    0,    4),('comment',   null, null,  0.0, null, null, null),('name',      null, null,  0.0, null, null, null),(null,        null, null, null, 25.0, null, null)");
        }
    }

    @Test
    public void testDropStatsAccessControl() {
        try (TestTable table = this.newTrinoTable("test_deny_drop_stats", "AS SELECT * FROM tpch.sf1.nation");){
            this.assertAccessDenied(String.format("CALL system.drop_extended_stats(CURRENT_SCHEMA, '%s')", table.getName()), "Cannot insert into table .*", new TestingAccessControlManager.TestingPrivilege[]{TestingAccessControlManager.privilege((String)table.getName(), (TestingAccessControlManager.TestingPrivilegeType)TestingAccessControlManager.TestingPrivilegeType.INSERT_TABLE)});
        }
    }

    @Test
    public void testStatsOnTpcDsData() {
        try (TestTable table = this.newTrinoTable("test_old_date_stats", "AS SELECT d_date FROM tpcds.tiny.date_dim");){
            this.assertUpdate("ANALYZE " + table.getName());
            this.assertQuery("SHOW STATS FOR " + table.getName(), "VALUES('d_date', null, 72713.0, 0.0,  null,    '1900-01-02', '2100-01-01'),(null,     null, null,    null, 73049.0, null,         null)");
        }
    }

    @Test
    public void testCreateTableStatisticsWhenCollectionOnWriteDisabled() {
        String tableName = "test_statistics_" + TestingNames.randomNameSuffix();
        this.assertUpdate(this.withStatsOnWrite(false), "CREATE TABLE " + tableName + " AS SELECT * FROM tpch.sf1.nation", 25L);
        this.assertQuery("SHOW STATS FOR " + tableName, "VALUES ('nationkey', null, null, 0.0, null, 0, 24),('regionkey', null, null, 0.0, null, 0, 4),('comment', null, null, 0.0, null, null, null),('name', null, null, 0.0, null, null, null),(null, null, null, null, 25.0, null, null)");
        this.assertUpdate("ANALYZE " + tableName, 25L);
        this.assertQuery("SHOW STATS FOR " + tableName, "VALUES ('nationkey', null, 25.0, 0.0, null, 0, 24),('regionkey', null, 5.0, 0.0, null, 0, 4),('comment', 1857.0, 25.0, 0.0, null, null, null),('name', 177.0, 25.0, 0.0, null, null, null),(null, null, null, null, 25.0, null, null)");
        this.assertUpdate("DROP TABLE " + tableName);
    }

    @Test
    public void testCreatePartitionedTableStatisticsWhenCollectionOnWriteDisabled() {
        String tableName = "test_statistics_" + TestingNames.randomNameSuffix();
        this.assertUpdate(this.withStatsOnWrite(false), "CREATE TABLE " + tableName + " WITH (   partitioned_by = ARRAY['regionkey'])AS SELECT * FROM tpch.sf1.nation", 25L);
        this.assertQuery("SHOW STATS FOR " + tableName, "VALUES ('nationkey', null, null, 0.0, null, 0, 24),('regionkey', null, 5.0, 0.0, null, null, null),('comment', null, null, 0.0, null, null, null),('name', null, null, 0.0, null, null, null),(null, null, null, null, 25.0, null, null)");
        this.assertUpdate("ANALYZE " + tableName, 25L);
        this.assertQuery("SHOW STATS FOR " + tableName, "VALUES ('nationkey', null, 25.0, 0.0, null, 0, 24),('regionkey', null, 5.0, 0.0, null, null, null),('comment', 1857.0, 25.0, 0.0, null, null, null),('name', 177.0, 25.0, 0.0, null, null, null),(null, null, null, null, 25.0, null, null)");
        this.assertUpdate("DROP TABLE " + tableName);
    }

    @Test
    public void testStatisticsOnInsertWhenStatsNotCollectedBefore() {
        String tableName = "test_statistics_on_insert_when_stats_not_collected_before_" + TestingNames.randomNameSuffix();
        this.assertUpdate(this.withStatsOnWrite(false), "CREATE TABLE " + tableName + " AS SELECT * FROM tpch.sf1.nation", 25L);
        this.assertQuery("SHOW STATS FOR " + tableName, "VALUES ('nationkey', null, null, 0.0, null, 0, 24),('regionkey', null, null, 0.0, null, 0, 4),('comment', null, null, 0.0, null, null, null),('name', null, null, 0.0, null, null, null),(null, null, null, null, 25.0, null, null)");
        this.assertUpdate("INSERT INTO " + tableName + " VALUES (111, 'a', 333, 'b')", 1L);
        this.assertQuery("SHOW STATS FOR " + tableName, "VALUES ('nationkey', null, 1.0, 0.0, null, 0, 111),('regionkey', null, 1.0, 0.0, null, 0, 333),('comment', 1.0, 1.0, 0.0, null, null, null),('name', 1.0, 1.0, 0.0, null, null, null),(null, null, null, null, 26.0, null, null)");
        this.assertUpdate("DROP TABLE " + tableName);
    }

    @Test
    public void testStatisticsOnInsertWhenCollectionOnWriteDisabled() {
        String tableName = "test_statistics_on_insert_when_collection_on_write_disabled_" + TestingNames.randomNameSuffix();
        this.assertUpdate("CREATE TABLE " + tableName + " AS SELECT * FROM tpch.sf1.nation", 25L);
        this.assertQuery("SHOW STATS FOR " + tableName, "VALUES ('nationkey', null, 25.0, 0.0, null, 0, 24),('regionkey', null, 5.0, 0.0, null, 0, 4),('comment', 1857.0, 25.0, 0.0, null, null, null),('name', 177.0, 25.0, 0.0, null, null, null),(null, null, null, null, 25.0, null, null)");
        this.assertUpdate(this.withStatsOnWrite(false), "INSERT INTO " + tableName + " SELECT nationkey + 25, reverse(name), regionkey + 5, reverse(comment) FROM tpch.sf1.nation", 25L);
        this.assertQuery("SHOW STATS FOR " + tableName, "VALUES ('nationkey', null, 25.0, 0.0, null, 0, 49),('regionkey', null, 5.0, 0.0, null, 0, 9),('comment', 1857.0, 25.0, 0.0, null, null, null),('name', 177.0, 25.0, 0.0, null, null, null),(null, null, null, null, 50.0, null, null)");
        this.assertUpdate("ANALYZE " + tableName);
        this.assertQuery("SHOW STATS FOR " + tableName, "VALUES ('nationkey', null, 50.0, 0.0, null, 0, 49),('regionkey', null, 10.0, 0.0, null, 0, 9),('comment', 3714.0, 50.0, 0.0, null, null, null),('name', 354.0, 50.0, 0.0, null, null, null),(null, null, null, null, 50.0, null, null)");
        this.assertUpdate("DROP TABLE " + tableName);
    }

    @Test
    public void testPartitionedStatisticsOnInsertWhenCollectionOnWriteDisabled() {
        String tableName = "test_partitioned_statistics_on_insert_when_collection_on_write_disabled_" + TestingNames.randomNameSuffix();
        this.assertUpdate("CREATE TABLE " + tableName + " WITH (   partitioned_by = ARRAY['regionkey'])AS SELECT * FROM tpch.sf1.nation", 25L);
        this.assertQuery("SHOW STATS FOR " + tableName, "VALUES ('nationkey', null, 25.0, 0.0, null, 0, 24),('regionkey', null, 5.0, 0.0, null,  null, null),('comment', 1857.0, 25.0, 0.0, null, null, null),('name', 177.0, 25.0, 0.0, null, null, null),(null, null, null, null, 25.0, null, null)");
        this.assertUpdate(this.withStatsOnWrite(false), "INSERT INTO " + tableName + " SELECT nationkey + 25, reverse(name), regionkey + 5, reverse(comment) FROM tpch.sf1.nation", 25L);
        this.assertQuery("SHOW STATS FOR " + tableName, "VALUES ('nationkey', null, 25.0, 0.0, null, 0, 49),('regionkey', null, 10.0, 0.0, null,  null, null),('comment', 1857.0, 25.0, 0.0, null, null, null),('name', 177.0, 25.0, 0.0, null, null, null),(null, null, null, null, 50.0, null, null)");
        this.assertUpdate("ANALYZE " + tableName);
        this.assertQuery("SHOW STATS FOR " + tableName, "VALUES ('nationkey', null, 50.0, 0.0, null, 0, 49),('regionkey', null, 10.0, 0.0, null,  null, null),('comment', 3714.0, 50.0, 0.0, null, null, null),('name', 354.0, 50.0, 0.0, null, null, null),(null, null, null, null, 50.0, null, null)");
        this.assertUpdate("DROP TABLE " + tableName);
    }

    @Test
    public void testIncrementalStatisticsUpdateOnInsert() {
        String tableName = "test_incremental_statistics_update_on_insert_" + TestingNames.randomNameSuffix();
        this.assertUpdate("CREATE TABLE " + tableName + " AS SELECT * FROM tpch.sf1.nation", 25L);
        this.assertQuery("SHOW STATS FOR " + tableName, "VALUES ('nationkey', null, 25.0, 0.0, null, 0, 24),('regionkey', null, 5.0, 0.0, null, 0, 4),('comment', 1857.0, 25.0, 0.0, null, null, null),('name', 177.0, 25.0, 0.0, null, null, null),(null, null, null, null, 25.0, null, null)");
        this.assertUpdate("INSERT INTO " + tableName + " SELECT nationkey + 25, reverse(name), regionkey + 5, reverse(comment) FROM tpch.sf1.nation", 25L);
        this.assertQuery("SHOW STATS FOR " + tableName, "VALUES ('nationkey', null, 50.0, 0.0, null, 0, 49),('regionkey', null, 10.0, 0.0, null, 0, 9),('comment', 3714.0, 50.0, 0.0, null, null, null),('name', 354.0, 50.0, 0.0, null, null, null),(null, null, null, null, 50.0, null, null)");
        this.assertUpdate("INSERT INTO " + tableName + " SELECT nationkey + 50, reverse(name), regionkey + 5, reverse(comment) FROM tpch.sf1.nation", 25L);
        this.assertQuery("SHOW STATS FOR " + tableName, "VALUES ('nationkey', null, 75.0, 0.0, null, 0, 74),('regionkey', null, 10.0, 0.0, null, 0, 9),('comment', 5571.0, 50.0, 0.0, null, null, null),('name', 531.0, 50.0, 0.0, null, null, null),(null, null, null, null, 75.0, null, null)");
        this.assertUpdate("DROP TABLE " + tableName);
    }

    @Test
    public void testCollectStatsAfterColumnAdded() {
        this.testCollectStatsAfterColumnAdded(false);
        this.testCollectStatsAfterColumnAdded(true);
    }

    private void testCollectStatsAfterColumnAdded(boolean collectOnWrite) {
        String tableName = "test_collect_stats_after_column_added_" + TestingNames.randomNameSuffix();
        this.assertUpdate("CREATE TABLE " + tableName + " (col_int_1 bigint, col_varchar_1 varchar)");
        this.assertUpdate("INSERT INTO " + tableName + " VALUES (11, 'aa')", 1L);
        this.assertUpdate("ALTER TABLE " + tableName + " ADD COLUMN col_int_2 bigint");
        this.assertUpdate("ALTER TABLE " + tableName + " ADD COLUMN col_varchar_2 varchar");
        this.assertUpdate(this.withStatsOnWrite(collectOnWrite), "INSERT INTO " + tableName + " VALUES (12, 'ab', 21, 'ba'), (13, 'ac', 22, 'bb')", 2L);
        if (!collectOnWrite) {
            this.assertQuery("SHOW STATS FOR " + tableName, "VALUES\n('col_int_1', null, 1.0, 0.0, null, 11, 13),\n('col_varchar_1', 2.0, 1.0, 0.0, null, null, null),\n('col_int_2', null, null, null, null, 21, 22),\n('col_varchar_2', null, null, null, null, null, null),\n(null, null, null, null, 3.0, null, null)\n");
            this.assertUpdate("ANALYZE " + tableName);
        }
        this.assertQuery("SHOW STATS FOR " + tableName, "VALUES\n('col_int_1', null, 3.0, 0.0, null, 11, 13),\n('col_varchar_1', 6.0, 3.0, 0.0, null, null, null),\n('col_int_2', null, 2.0, 0.1, null, 21, 22),\n('col_varchar_2', 4.0, 2.0, 0.1, null, null, null),\n(null, null, null, null, 3.0, null, null)\n");
        this.assertUpdate("DROP TABLE " + tableName);
    }

    @Test
    public void testForceRecalculateStatsWithDeleteAndUpdate() {
        String tableName = "test_recalculate_all_stats_with_delete_and_update_" + TestingNames.randomNameSuffix();
        this.assertUpdate("CREATE TABLE " + tableName + " AS SELECT * FROM tpch.sf1.nation", 25L);
        this.assertQuery("SHOW STATS FOR " + tableName, "VALUES ('nationkey', null, 25.0, 0.0, null, 0, 24),('regionkey', null, 5.0, 0.0, null, 0, 4),('comment', 1857.0, 25.0, 0.0, null, null, null),('name', 177.0, 25.0, 0.0, null, null, null),(null, null, null, null, 25.0, null, null)");
        this.assertUpdate("ANALYZE " + tableName);
        this.assertQuery("SHOW STATS FOR " + tableName, "VALUES ('nationkey', null, 25.0, 0.0, null, 0, 24),('regionkey', null, 5.0, 0.0, null, 0, 4),('comment', 1857.0, 25.0, 0.0, null, null, null),('name', 177.0, 25.0, 0.0, null, null, null),(null, null, null, null, 25.0, null, null)");
        this.assertUpdate("DELETE FROM " + tableName + " WHERE nationkey = 1", 1L);
        this.assertQuery("SHOW STATS FOR " + tableName, "VALUES ('nationkey', null, 24.0, 0.0, null, 0, 24),('regionkey', null, 5.0, 0.0, null, 0, 4),('comment', 1857.0, 24.0, 0.0, null, null, null),('name', 177.0, 24.0, 0.0, null, null, null),(null, null, null, null, 24.0, null, null)");
        this.assertUpdate("UPDATE " + tableName + " SET name = null WHERE nationkey = 2", 1L);
        this.assertQuery("SHOW STATS FOR " + tableName, "VALUES ('nationkey', null, 24.0, 0.0, null, 0, 24),('regionkey', null, 5.0, 0.0, null, 0, 4),('comment', 1857.0, 24.0, 0.0, null, null, null),('name', 180.84782608695653, 23.5, 0.02083333333333337, null, null, null),(null, null, null, null, 24.0, null, null)");
        this.assertUpdate(String.format("ANALYZE %s", tableName));
        this.assertQuery("SHOW STATS FOR " + tableName, "VALUES ('nationkey', null, 24.0, 0.0, null, 0, 24),('regionkey', null, 5.0, 0.0, null, 0, 4),('comment', 3638.0, 24.0, 0.0, null, null, null),('name', 346.3695652173913, 23.5, 0.02083333333333337, null, null, null),(null, null, null, null, 24.0, null, null)");
        this.assertUpdate(String.format("ANALYZE %s WITH(mode = 'full_refresh')", tableName), 24L);
        this.assertQuery("SHOW STATS FOR " + tableName, "VALUES ('nationkey', null, 24.0, 0.0, null, 0, 24),('regionkey', null, 5.0, 0.0, null, 0, 4),('comment', 1781.0, 24.0, 0.0, null, null, null),('name', 162.0, 23.0, 0.041666666666666664, null, null, null),(null, null, null, null, 24.0, null, null)");
        this.assertUpdate("DROP TABLE " + tableName);
    }

    @Test
    public void testForceRecalculateAllStats() {
        String tableName = "test_recalculate_all_stats_" + TestingNames.randomNameSuffix();
        this.assertUpdate(this.withStatsOnWrite(false), "CREATE TABLE " + tableName + " AS SELECT nationkey, regionkey, name  FROM tpch.sf1.nation", 25L);
        this.assertUpdate(this.withStatsOnWrite(true), "INSERT INTO " + tableName + " VALUES(27, 1, 'name1')", 1L);
        this.assertQuery("SHOW STATS FOR " + tableName, "VALUES ('nationkey', null, 1.0, 0.0, null, 0, 27),('regionkey', null, 1.0, 0.0, null, 0, 4),('name', 5.0, 1.0, 0.0, null, null, null),(null, null, null, null, 26.0, null, null)");
        this.assertUpdate("ANALYZE " + tableName);
        this.assertQuery("SHOW STATS FOR " + tableName, "VALUES ('nationkey', null, 1.0, 0.0, null, 0, 27),('regionkey', null, 1.0, 0.0, null, 0, 4),('name', 5.0, 1.0, 0.0, null, null, null),(null, null, null, null, 26.0, null, null)");
        this.assertUpdate(String.format("ANALYZE %s WITH(mode = 'full_refresh')", tableName), 26L);
        this.assertQuery("SHOW STATS FOR " + tableName, "VALUES ('nationkey', null, 26.0, 0.0, null, 0, 27),('regionkey', null, 5.0, 0.0, null, 0, 4),('name', 182.0, 26.0, 0.0, null, null, null),(null, null, null, null, 26.0, null, null)");
        this.assertUpdate("DROP TABLE " + tableName);
    }

    @Test
    public void testNoStats() throws Exception {
        String tableName = this.copyResourcesAndRegisterTable("no_stats", "trino410/no_stats");
        String expectedData = "VALUES (42, 'foo'), (12, 'ab'), (null, null), (15, 'cd'), (15, 'bar')";
        this.assertQuery("SELECT * FROM " + tableName, expectedData);
        this.assertQuery("SHOW STATS FOR " + tableName, "VALUES\n('c_int', null, null, null, null, null, null),\n('c_str', null, null, null, null, null, null),\n(null, null, null, null, null, null, null)\n");
        this.assertUpdate("ANALYZE " + tableName, 5L);
        this.assertQuery("SHOW STATS FOR " + tableName, "VALUES\n('c_int', null, 3.0, 0.2, null, 12, 42),\n('c_str', 10.0, 4.0, 0.2, null, null, null),\n(null, null, null, null, 5.0, null, null)\n");
        this.assertQuery("SELECT * FROM " + tableName, expectedData);
        this.cleanExternalTable(tableName);
    }

    @Test
    public void testNoColumnStats() throws Exception {
        String tableName = this.copyResourcesAndRegisterTable("no_column_stats", "databricks73/no_column_stats");
        this.assertQuery("SELECT * FROM " + tableName, "VALUES (42, 'foo')");
        this.assertUpdate("ANALYZE " + tableName, 1L);
        this.assertQuery("SHOW STATS FOR " + tableName, "VALUES\n('c_int', null, 1.0, 0.0, null, 42, 42),\n('c_str', 3.0, 1.0, 0.0, null, null, null),\n(null, null, null, null, 1.0, null, null)\n");
        this.cleanExternalTable(tableName);
    }

    @Test
    public void testNoColumnStatsMixedCase() throws Exception {
        String tableName = this.copyResourcesAndRegisterTable("no_column_stats_mixed_case", "databricks104/no_column_stats_mixed_case");
        String tableLocation = this.getTableLocation(tableName);
        this.assertQuery("SELECT * FROM " + tableName, "VALUES (11, 'a'), (2, 'b'), (null, null)");
        this.assertUpdate("ANALYZE " + tableName, 3L);
        this.assertQuery("SHOW STATS FOR " + tableName, "VALUES\n('c_int', null, 2.0, 0.33333333, null, 2, 11),\n('c_str', 2.0, 2.0, 0.33333333, null, null, null),\n(null, null, null, null, 3.0, null, null)\n");
        List transactionLogAfterUpdate = ((TransactionLogEntries)TransactionLogTail.getEntriesFromJson((long)3L, (String)(tableLocation + "/_delta_log"), (TrinoFileSystem)FILE_SYSTEM, (DataSize)DeltaLakeConfig.DEFAULT_TRANSACTION_LOG_MAX_CACHED_SIZE).orElseThrow()).getEntriesList(FILE_SYSTEM);
        Assertions.assertThat((List)transactionLogAfterUpdate).hasSize(2);
        AddFileEntry updateAddFileEntry = ((DeltaLakeTransactionLogEntry)transactionLogAfterUpdate.get(1)).getAdd();
        DeltaLakeFileStatistics updateStats = (DeltaLakeFileStatistics)updateAddFileEntry.getStats().orElseThrow();
        Assertions.assertThat((Map)((Map)updateStats.getMinValues().orElseThrow())).containsEntry((Object)"c_Int", (Object)2);
        Assertions.assertThat((Map)((Map)updateStats.getMaxValues().orElseThrow())).containsEntry((Object)"c_Int", (Object)11);
        Assertions.assertThat((Long)((Long)updateStats.getNullCount("c_Int").orElseThrow())).isEqualTo(1L);
        Assertions.assertThat((Long)((Long)updateStats.getNullCount("c_Str").orElseThrow())).isEqualTo(1L);
        this.cleanExternalTable(tableName);
    }

    @Test
    public void testPartiallyNoStats() throws Exception {
        String tableName = this.copyResourcesAndRegisterTable("no_stats", "trino410/no_stats");
        this.assertUpdate("INSERT INTO " + tableName + " VALUES (1,'a'), (12,'b')", 2L);
        this.assertQuery("SELECT * FROM " + tableName, " VALUES (42, 'foo'), (12, 'ab'), (null, null), (15, 'cd'), (15, 'bar'), (1, 'a'), (12, 'b')");
        this.assertUpdate(String.format("CALL system.drop_extended_stats(CURRENT_SCHEMA, '%s')", tableName));
        this.assertUpdate("ANALYZE " + tableName, 7L);
        this.assertQuery("SHOW STATS FOR " + tableName, "VALUES\n('c_int', null, 4.0, 0.14285714285714285, null, 1, 42),\n('c_str', 12.0, 6.0, 0.14285714285714285, null, null, null),\n(null, null, null, null, 7.0, null, null)\n");
        this.cleanExternalTable(tableName);
    }

    @Test
    public void testNoStatsPartitionedTable() throws Exception {
        String tableName = this.copyResourcesAndRegisterTable("no_stats_partitions", "trino410/no_stats_partitions");
        this.assertQuery("SELECT * FROM " + tableName, "VALUES\n('p?p', 42, 'foo'),\n('p?p', 12, 'ab'),\n(null, null, null),\n('ppp', 15, 'cd'),\n('ppp', 15, 'bar')\n");
        this.assertUpdate("ANALYZE " + tableName, 5L);
        this.assertQuery("SHOW STATS FOR " + tableName, "VALUES\n('p_str', null, 2.0, 0.2, null, null, null),\n('c_int', null, 3.0, 0.2, null, 12, 42),\n('c_str', 10.0, 4.0, 0.2, null, null, null),\n(null, null, null, null, 5.0, null, null)\n");
        this.cleanExternalTable(tableName);
    }

    @Test
    public void testNoStatsVariousTypes() throws Exception {
        String tableName = this.copyResourcesAndRegisterTable("no_stats_various_types", "trino410/no_stats_various_types");
        this.assertQuery("SELECT c_boolean, c_tinyint, c_smallint, c_integer, c_bigint, c_real, c_double, c_decimal1, c_decimal2, c_date1, CAST(c_timestamp AS TIMESTAMP), c_varchar1, c_varchar2, c_varbinary FROM " + tableName, "VALUES\n(false, 37, 32123, 1274942432, 312739231274942432, 567.123, 1234567890123.123, 12.345, 123456789012.345, '1999-01-01', '2020-02-12 14:03:00', 'ab', 'de',  X'12ab3f'),\n(true, 127, 32767, 2147483647, 9223372036854775807, 999999.999, 9999999999999.999, 99.999, 999999999999.99, '2028-10-04', '2199-12-31 22:59:59.999', 'zzz', 'zzz',  X'ffffffffffffffffffff'),\n(null,null,null,null,null,null,null,null,null,null,null,null,null,null),\n(null,null,null,null,null,null,null,null,null,null,null,null,null,null)\n");
        this.assertUpdate("ANALYZE " + tableName, 4L);
        this.assertQuery("SHOW STATS FOR " + tableName, "VALUES\n('c_boolean', null, 2.0, 0.5, null, null, null),\n('c_tinyint', null, 2.0, 0.5, null, '37', '127'),\n('c_smallint', null, 2.0, 0.5, null, '32123', '32767'),\n('c_integer', null, 2.0, 0.5, null, '1274942432', '2147483647'),\n('c_bigint', null, 2.0, 0.5, null, '312739231274942464', '9223372036854775807'),\n('c_real', null, 2.0, 0.5, null, '567.123', '1000000.0'),\n('c_double', null, 2.0, 0.5, null, '1.234567890123123E12', '9.999999999999998E12'),\n('c_decimal1', null, 2.0, 0.5, null, '12.345', '99.999'),\n('c_decimal2', null, 2.0, 0.5, null, '1.23456789012345E11', '9.9999999999999E11'),\n('c_date1', null, 2.0, 0.5, null, '1999-01-01', '2028-10-04'),\n('c_timestamp', null, 2.0, 0.5, null, '2020-02-12 14:03:00.000 UTC', '2199-12-31 22:59:59.999 UTC'),\n('c_varchar1', 5.0, 2.0, 0.5, null, null, null),\n('c_varchar2', 5.0, 2.0, 0.5, null, null, null),\n('c_varbinary', 13.0, 2.0, 0.5, null, null, null),\n(null, null, null, null, 4.0, null, null)\n");
        this.cleanExternalTable(tableName);
    }

    @Test
    public void testNoStatsWithColumnMappingModeId() throws Exception {
        String tableName = this.copyResourcesAndRegisterTable("no_stats_column_mapping_id", "databricks104/no_stats_column_mapping_id");
        this.assertQuery("SELECT * FROM " + tableName, " VALUES (42, 'foo'), (1, 'a'), (2, 'b'), (null, null)");
        this.assertUpdate("ANALYZE " + tableName, 4L);
        this.assertQuery("SHOW STATS FOR " + tableName, "VALUES\n('c_int', null, 3.0, 0.25, null, 1, 42),\n('c_str', 5.0, 3.0, 0.25, null, null, null),\n(null, null, null, null, 4.0, null, null)\n");
        this.cleanExternalTable(tableName);
    }

    private String copyResourcesAndRegisterTable(String resourceTable, String resourcePath) throws IOException, URISyntaxException {
        Path tableLocation = Files.createTempDirectory(null, new FileAttribute[0]);
        String tableName = resourceTable + TestingNames.randomNameSuffix();
        URI resourcesLocation = ((Object)((Object)this)).getClass().getClassLoader().getResource(resourcePath).toURI();
        TestingDeltaLakeUtils.copyDirectoryContents(Path.of(resourcesLocation), tableLocation);
        this.assertUpdate(String.format("CALL system.register_table(CURRENT_SCHEMA, '%s', '%s')", tableName, tableLocation.toUri()));
        return tableName;
    }

    private Session withStatsOnWrite(boolean value) {
        Session session = this.getSession();
        return Session.builder((Session)session).setCatalogSessionProperty((String)session.getCatalog().orElseThrow(), "extended_statistics_collect_on_write", Boolean.toString(value)).build();
    }

    private String getTableLocation(String tableName) {
        Pattern locationPattern = Pattern.compile(".*location = '(.*?)'.*", 32);
        Matcher m = locationPattern.matcher((String)this.computeActual("SHOW CREATE TABLE " + tableName).getOnlyValue());
        if (m.find()) {
            String location = m.group(1);
            Verify.verify((!m.find() ? 1 : 0) != 0, (String)"Unexpected second match", (Object[])new Object[0]);
            return location;
        }
        throw new IllegalStateException("Location not found in SHOW CREATE TABLE result");
    }

    private void cleanExternalTable(String tableName) throws Exception {
        String tableLocation = this.getTableLocation(tableName);
        this.assertUpdate("DROP TABLE " + tableName);
        MoreFiles.deleteRecursively((Path)Path.of(new URI(tableLocation).getPath(), new String[0]), (RecursiveDeleteOption[])new RecursiveDeleteOption[]{RecursiveDeleteOption.ALLOW_INSECURE});
    }
}

