/*
 * Decompiled with CFR 0.152.
 */
package io.trino.plugin.deltalake;

import com.google.common.base.Predicates;
import com.google.common.collect.HashMultiset;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.ImmutableMap;
import com.google.common.collect.ImmutableMultiset;
import com.google.common.collect.ImmutableSet;
import com.google.common.collect.MoreCollectors;
import com.google.common.collect.Multiset;
import com.google.common.collect.Sets;
import com.google.common.util.concurrent.MoreExecutors;
import io.airlift.slice.Slices;
import io.airlift.units.Duration;
import io.opentelemetry.sdk.trace.data.SpanData;
import io.trino.filesystem.TrinoFileSystemFactory;
import io.trino.filesystem.hdfs.HdfsFileSystemFactory;
import io.trino.filesystem.tracing.FileSystemAttributes;
import io.trino.filesystem.tracing.TracingFileSystemFactory;
import io.trino.plugin.base.metrics.FileFormatDataSourceStats;
import io.trino.plugin.deltalake.DeltaLakeColumnHandle;
import io.trino.plugin.deltalake.DeltaLakeColumnType;
import io.trino.plugin.deltalake.DeltaLakeConfig;
import io.trino.plugin.deltalake.DeltaLakeSessionProperties;
import io.trino.plugin.deltalake.DeltaLakeTableHandle;
import io.trino.plugin.deltalake.DeltaTestingConnectorSession;
import io.trino.plugin.deltalake.transactionlog.AddFileEntry;
import io.trino.plugin.deltalake.transactionlog.DeltaLakeSchemaSupport;
import io.trino.plugin.deltalake.transactionlog.MetadataEntry;
import io.trino.plugin.deltalake.transactionlog.ProtocolEntry;
import io.trino.plugin.deltalake.transactionlog.TableSnapshot;
import io.trino.plugin.deltalake.transactionlog.TransactionLogAccess;
import io.trino.plugin.deltalake.transactionlog.checkpoint.CheckpointSchemaManager;
import io.trino.plugin.deltalake.transactionlog.checkpoint.MetadataAndProtocolEntries;
import io.trino.plugin.deltalake.transactionlog.statistics.DeltaLakeFileStatistics;
import io.trino.plugin.hive.HiveTestUtils;
import io.trino.plugin.hive.parquet.ParquetReaderConfig;
import io.trino.plugin.hive.parquet.ParquetWriterConfig;
import io.trino.spi.connector.ColumnMetadata;
import io.trino.spi.connector.ConnectorSession;
import io.trino.spi.connector.SchemaTableName;
import io.trino.spi.predicate.TupleDomain;
import io.trino.spi.type.DateTimeEncoding;
import io.trino.spi.type.Decimals;
import io.trino.spi.type.IntegerType;
import io.trino.spi.type.TimeZoneKey;
import io.trino.spi.type.Type;
import io.trino.spi.type.TypeManager;
import io.trino.testing.MultisetAssertions;
import io.trino.testing.TestingConnectorContext;
import io.trino.testing.TestingConnectorSession;
import io.trino.testing.TestingTelemetry;
import io.trino.type.InternalTypeManager;
import java.io.File;
import java.io.IOException;
import java.math.BigDecimal;
import java.nio.file.CopyOption;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.attribute.FileAttribute;
import java.time.LocalDate;
import java.time.LocalDateTime;
import java.time.ZoneOffset;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.OptionalInt;
import java.util.Set;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.function.Predicate;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import org.assertj.core.api.Assertions;
import org.assertj.core.api.MapAssert;
import org.junit.jupiter.api.Test;
import org.junit.jupiter.api.parallel.Execution;
import org.junit.jupiter.api.parallel.ExecutionMode;

@Execution(value=ExecutionMode.SAME_THREAD)
public class TestTransactionLogAccess {
    private static final Set<String> EXPECTED_ADD_FILE_PATHS = ImmutableSet.of((Object)"age=42/part-00000-b26c891a-7288-4d96-9d3b-bef648f12a34.c000.snappy.parquet", (Object)"age=25/part-00001-aceaf062-1cd1-45cb-8f83-277ffebe995c.c000.snappy.parquet", (Object)"age=25/part-00000-b7fbbe31-c7f9-44ed-8757-5c47d10c3e81.c000.snappy.parquet", (Object)"age=42/part-00003-0f53cae3-3e34-4876-b651-e1db9584dbc3.c000.snappy.parquet", (Object)"age=30/part-00000-63c2205d-84a3-4a66-bd7c-f69f5af55bbc.c000.snappy.parquet", (Object)"age=25/part-00000-22a101a1-8f09-425e-847e-cbbe4f894eea.c000.snappy.parquet", (Object[])new String[]{"age=21/part-00001-290f0f26-19cf-4772-821e-36d55d9b7872.c000.snappy.parquet", "age=30/part-00002-5800be2e-2373-47d8-8b86-776a8ea9d69f.c000.snappy.parquet", "age=21/part-00000-3d546786-bedc-407f-b9f7-e97aa12cce0f.c000.snappy.parquet", "age=30/part-00000-37ccfcd3-b44b-4d04-a1e6-d2837da75f7a.c000.snappy.parquet", "age=28/part-00000-40dd1707-1d42-4328-a59a-21f5c945fe60.c000.snappy.parquet", "age=29/part-00000-3794c463-cb0c-4beb-8d07-7cc1e3b5920f.c000.snappy.parquet"});
    private final TestingTelemetry testingTelemetry = TestingTelemetry.create((String)"transaction-log-access");
    private final TracingFileSystemFactory tracingFileSystemFactory = new TracingFileSystemFactory(this.testingTelemetry.getTracer(), (TrinoFileSystemFactory)new HdfsFileSystemFactory(HiveTestUtils.HDFS_ENVIRONMENT, HiveTestUtils.HDFS_FILE_SYSTEM_STATS));
    private TransactionLogAccess transactionLogAccess;
    private TableSnapshot tableSnapshot;

    private void setupTransactionLogAccessFromResources(String tableName, String resourcePath) throws Exception {
        this.setupTransactionLogAccess(tableName, this.getClass().getClassLoader().getResource(resourcePath).toString());
    }

    private void setupTransactionLogAccess(String tableName, String tableLocation) throws IOException {
        this.setupTransactionLogAccess(tableName, tableLocation, new DeltaLakeConfig(), Optional.empty());
    }

    private void setupTransactionLogAccess(String tableName, String tableLocation, DeltaLakeConfig deltaLakeConfig, Optional<Long> endVersion) throws IOException {
        TestingConnectorContext context = new TestingConnectorContext();
        TypeManager typeManager = context.getTypeManager();
        FileFormatDataSourceStats fileFormatDataSourceStats = new FileFormatDataSourceStats();
        this.transactionLogAccess = new TransactionLogAccess(typeManager, new CheckpointSchemaManager(typeManager), deltaLakeConfig, fileFormatDataSourceStats, (TrinoFileSystemFactory)this.tracingFileSystemFactory, new ParquetReaderConfig(), (ExecutorService)MoreExecutors.newDirectExecutorService());
        DeltaLakeTableHandle tableHandle = new DeltaLakeTableHandle("schema", tableName, true, "location", new MetadataEntry("id", "test", "description", null, "", (List)ImmutableList.of(), (Map)ImmutableMap.of(), 0L), new ProtocolEntry(1, 2, Optional.empty(), Optional.empty()), TupleDomain.none(), TupleDomain.none(), Optional.empty(), Optional.empty(), Optional.empty(), Optional.empty(), Optional.empty(), 0L, false);
        this.tableSnapshot = this.transactionLogAccess.loadSnapshot(DeltaTestingConnectorSession.SESSION, tableHandle.getSchemaTableName(), tableLocation, endVersion);
    }

    @Test
    public void testGetMetadataEntry() throws Exception {
        this.setupTransactionLogAccessFromResources("person", "databricks73/person");
        MetadataEntry metadataEntry = this.transactionLogAccess.getMetadataEntry(DeltaTestingConnectorSession.SESSION, this.tableSnapshot);
        Assertions.assertThat((long)metadataEntry.getCreatedTime()).isEqualTo(1579190100722L);
        Assertions.assertThat((String)metadataEntry.getId()).isEqualTo("b6aeffad-da73-4dde-b68e-937e468b1fdf");
        Assertions.assertThat((List)metadataEntry.getOriginalPartitionColumns()).containsOnly((Object[])new String[]{"age"});
        Assertions.assertThat((List)metadataEntry.getLowercasePartitionColumns()).containsOnly((Object[])new String[]{"age"});
        MetadataEntry.Format format = metadataEntry.getFormat();
        Assertions.assertThat(format.options().keySet()).isEmpty();
        Assertions.assertThat((String)format.provider()).isEqualTo("parquet");
        Assertions.assertThat((Optional)this.tableSnapshot.getCachedMetadata()).isEqualTo(Optional.of(metadataEntry));
    }

    @Test
    public void testGetMetadataEntryUppercase() throws Exception {
        this.setupTransactionLogAccessFromResources("uppercase_columns", "databricks73/uppercase_columns");
        MetadataEntry metadataEntry = this.transactionLogAccess.getMetadataEntry(DeltaTestingConnectorSession.SESSION, this.tableSnapshot);
        Assertions.assertThat((List)metadataEntry.getOriginalPartitionColumns()).containsOnly((Object[])new String[]{"ALA"});
        Assertions.assertThat((List)metadataEntry.getLowercasePartitionColumns()).containsOnly((Object[])new String[]{"ala"});
        Assertions.assertThat((Optional)this.tableSnapshot.getCachedMetadata()).isEqualTo(Optional.of(metadataEntry));
    }

    @Test
    public void testGetActiveAddEntries() throws Exception {
        List addFileEntries;
        this.setupTransactionLogAccessFromResources("person", "databricks73/person");
        MetadataEntry metadataEntry = this.transactionLogAccess.getMetadataEntry(DeltaTestingConnectorSession.SESSION, this.tableSnapshot);
        ProtocolEntry protocolEntry = this.transactionLogAccess.getProtocolEntry(DeltaTestingConnectorSession.SESSION, this.tableSnapshot);
        try (Stream addFileEntriesStream = this.transactionLogAccess.getActiveFiles(DeltaTestingConnectorSession.SESSION, this.tableSnapshot, metadataEntry, protocolEntry, TupleDomain.all(), (Predicate)Predicates.alwaysTrue());){
            addFileEntries = (List)addFileEntriesStream.collect(ImmutableList.toImmutableList());
        }
        Set paths = addFileEntries.stream().map(AddFileEntry::getPath).collect(Collectors.toSet());
        Assertions.assertThat(paths).isEqualTo(EXPECTED_ADD_FILE_PATHS);
        try (Stream checkpointFilteredAddFileEntriesStream = this.transactionLogAccess.getActiveFiles(DeltaTestingConnectorSession.SESSION, this.tableSnapshot, metadataEntry, protocolEntry, TupleDomain.all(), (Predicate)Predicates.alwaysTrue());){
            Assertions.assertThat((List)((List)checkpointFilteredAddFileEntriesStream.collect(ImmutableList.toImmutableList()))).isEqualTo((Object)addFileEntries);
        }
        AddFileEntry addFileEntry = (AddFileEntry)addFileEntries.stream().filter(entry -> entry.getPath().equals("age=42/part-00000-b26c891a-7288-4d96-9d3b-bef648f12a34.c000.snappy.parquet")).collect(MoreCollectors.onlyElement());
        ((MapAssert)Assertions.assertThat((Map)addFileEntry.getPartitionValues()).hasSize(1)).containsEntry((Object)"age", (Object)"42");
        ((MapAssert)Assertions.assertThat((Map)addFileEntry.getCanonicalPartitionValues()).hasSize(1)).containsEntry((Object)"age", Optional.of("42"));
        Assertions.assertThat((long)addFileEntry.getSize()).isEqualTo(2687L);
        Assertions.assertThat((long)addFileEntry.getModificationTime()).isEqualTo(1579190188000L);
        Assertions.assertThat((boolean)addFileEntry.isDataChange()).isFalse();
    }

    @Test
    public void testAddFileEntryUppercase() throws Exception {
        this.setupTransactionLogAccessFromResources("uppercase_columns", "databricks73/uppercase_columns");
        MetadataEntry metadataEntry = this.transactionLogAccess.getMetadataEntry(DeltaTestingConnectorSession.SESSION, this.tableSnapshot);
        ProtocolEntry protocolEntry = this.transactionLogAccess.getProtocolEntry(DeltaTestingConnectorSession.SESSION, this.tableSnapshot);
        try (Stream addFileEntries = this.transactionLogAccess.getActiveFiles(DeltaTestingConnectorSession.SESSION, this.tableSnapshot, metadataEntry, protocolEntry, TupleDomain.all(), (Predicate)Predicates.alwaysTrue());){
            AddFileEntry addFileEntry = (AddFileEntry)addFileEntries.filter(entry -> entry.getPath().equals("ALA=1/part-00000-20a863e0-890d-4776-8825-f9dccc8973ba.c000.snappy.parquet")).collect(MoreCollectors.onlyElement());
            ((MapAssert)Assertions.assertThat((Map)addFileEntry.getPartitionValues()).hasSize(1)).containsEntry((Object)"ALA", (Object)"1");
            ((MapAssert)Assertions.assertThat((Map)addFileEntry.getCanonicalPartitionValues()).hasSize(1)).containsEntry((Object)"ALA", Optional.of("1"));
        }
    }

    @Test
    public void testAddEntryPruning() throws Exception {
        this.setupTransactionLogAccessFromResources("person_test_pruning", "databricks73/person_test_pruning");
        MetadataEntry metadataEntry = this.transactionLogAccess.getMetadataEntry(DeltaTestingConnectorSession.SESSION, this.tableSnapshot);
        ProtocolEntry protocolEntry = this.transactionLogAccess.getProtocolEntry(DeltaTestingConnectorSession.SESSION, this.tableSnapshot);
        try (Stream addFileEntries = this.transactionLogAccess.getActiveFiles(DeltaTestingConnectorSession.SESSION, this.tableSnapshot, metadataEntry, protocolEntry, TupleDomain.all(), (Predicate)Predicates.alwaysTrue());){
            Set paths = addFileEntries.map(AddFileEntry::getPath).collect(Collectors.toSet());
            Assertions.assertThat((boolean)paths.contains("age=25/part-00001-aceaf062-1cd1-45cb-8f83-277ffebe995c.c000.snappy.parquet")).isFalse();
            Assertions.assertThat((boolean)paths.contains("age=29/part-00000-3794c463-cb0c-4beb-8d07-7cc1e3b5920f.c000.snappy.parquet")).isFalse();
        }
    }

    @Test
    public void testAddEntryOverrides() throws Exception {
        List addFileEntries;
        this.setupTransactionLogAccessFromResources("person_test_pruning", "databricks73/person_test_pruning");
        MetadataEntry metadataEntry = this.transactionLogAccess.getMetadataEntry(DeltaTestingConnectorSession.SESSION, this.tableSnapshot);
        ProtocolEntry protocolEntry = this.transactionLogAccess.getProtocolEntry(DeltaTestingConnectorSession.SESSION, this.tableSnapshot);
        try (Stream addFileEntryStream = this.transactionLogAccess.getActiveFiles(DeltaTestingConnectorSession.SESSION, this.tableSnapshot, metadataEntry, protocolEntry, TupleDomain.all(), (Predicate)Predicates.alwaysTrue());){
            addFileEntries = (List)addFileEntryStream.collect(ImmutableList.toImmutableList());
        }
        ImmutableList overwrittenPaths = ImmutableList.of((Object)"age=42/part-00000-b26c891a-7288-4d96-9d3b-bef648f12a34.c000.snappy.parquet", (Object)"age=28/part-00000-40dd1707-1d42-4328-a59a-21f5c945fe60.c000.snappy.parquet");
        for (String path : overwrittenPaths) {
            List<AddFileEntry> activeEntries = addFileEntries.stream().filter(addFileEntry -> addFileEntry.getPath().equals(path)).toList();
            Assertions.assertThat(activeEntries).hasSize(1);
            Assertions.assertThat((long)activeEntries.get(0).getModificationTime()).isEqualTo(9999999L);
        }
    }

    @Test
    public void testAddRemoveAdd() throws Exception {
        this.setupTransactionLogAccessFromResources("person_test_pruning", "databricks73/person_test_pruning");
        MetadataEntry metadataEntry = this.transactionLogAccess.getMetadataEntry(DeltaTestingConnectorSession.SESSION, this.tableSnapshot);
        ProtocolEntry protocolEntry = this.transactionLogAccess.getProtocolEntry(DeltaTestingConnectorSession.SESSION, this.tableSnapshot);
        try (Stream addFileEntries = this.transactionLogAccess.getActiveFiles(DeltaTestingConnectorSession.SESSION, this.tableSnapshot, metadataEntry, protocolEntry, TupleDomain.all(), (Predicate)Predicates.alwaysTrue());){
            List<AddFileEntry> activeEntries = addFileEntries.filter(addFileEntry -> addFileEntry.getPath().equals("age=30/part-00002-5800be2e-2373-47d8-8b86-776a8ea9d69f.c000.snappy.parquet")).toList();
            Assertions.assertThat(activeEntries).hasSize(1);
            Assertions.assertThat((long)activeEntries.get(0).getModificationTime()).isEqualTo(9999999L);
        }
    }

    @Test
    public void testAllGetMetadataEntry() throws Exception {
        this.testAllGetMetadataEntry("person", "databricks73/person");
        this.testAllGetMetadataEntry("person_without_last_checkpoint", "databricks73/person_without_last_checkpoint");
        this.testAllGetMetadataEntry("person_without_old_jsons", "databricks73/person_without_old_jsons");
        this.testAllGetMetadataEntry("person_without_checkpoints", "databricks73/person_without_checkpoints");
    }

    private void testAllGetMetadataEntry(String tableName, String resourcePath) throws Exception {
        this.setupTransactionLogAccessFromResources(tableName, resourcePath);
        this.transactionLogAccess.getMetadataEntry(DeltaTestingConnectorSession.SESSION, this.tableSnapshot);
        MetadataEntry metadataEntry = this.transactionLogAccess.getMetadataEntry(DeltaTestingConnectorSession.SESSION, this.tableSnapshot);
        Assertions.assertThat((List)metadataEntry.getOriginalPartitionColumns()).containsOnly((Object[])new String[]{"age"});
        MetadataEntry.Format format = metadataEntry.getFormat();
        Assertions.assertThat(format.options().keySet()).isEmpty();
        Assertions.assertThat((String)format.provider()).isEqualTo("parquet");
    }

    @Test
    void testGetMetadataAndProtocolEntry() throws Exception {
        this.testGetMetadataAndProtocolEntry("person", "databricks73/person");
        this.testGetMetadataAndProtocolEntry("person_without_last_checkpoint", "databricks73/person_without_last_checkpoint");
        this.testGetMetadataAndProtocolEntry("person_without_old_jsons", "databricks73/person_without_old_jsons");
        this.testGetMetadataAndProtocolEntry("person_without_checkpoints", "databricks73/person_without_checkpoints");
    }

    private void testGetMetadataAndProtocolEntry(String tableName, String resourcePath) throws Exception {
        this.setupTransactionLogAccessFromResources(tableName, resourcePath);
        this.transactionLogAccess.getMetadataAndProtocolEntry(DeltaTestingConnectorSession.SESSION, this.tableSnapshot);
        MetadataAndProtocolEntries logEntries = this.transactionLogAccess.getMetadataAndProtocolEntry(DeltaTestingConnectorSession.SESSION, this.tableSnapshot);
        MetadataEntry metadataEntry = (MetadataEntry)logEntries.metadata().orElseThrow();
        Assertions.assertThat((List)metadataEntry.getOriginalPartitionColumns()).containsOnly((Object[])new String[]{"age"});
        MetadataEntry.Format format = metadataEntry.getFormat();
        Assertions.assertThat(format.options().keySet()).isEmpty();
        Assertions.assertThat((String)format.provider()).isEqualTo("parquet");
        ProtocolEntry protocolEntry = (ProtocolEntry)logEntries.protocol().orElseThrow();
        Assertions.assertThat((int)protocolEntry.minReaderVersion()).isEqualTo(1);
        Assertions.assertThat((int)protocolEntry.minWriterVersion()).isEqualTo(2);
    }

    @Test
    public void testAllGetActiveAddEntries() throws Exception {
        this.testAllGetActiveAddEntries("person", "databricks73/person");
        this.testAllGetActiveAddEntries("person_without_last_checkpoint", "databricks73/person_without_last_checkpoint");
        this.testAllGetActiveAddEntries("person_without_old_jsons", "databricks73/person_without_old_jsons");
        this.testAllGetActiveAddEntries("person_without_checkpoints", "databricks73/person_without_checkpoints");
    }

    private void testAllGetActiveAddEntries(String tableName, String resourcePath) throws Exception {
        this.setupTransactionLogAccessFromResources(tableName, resourcePath);
        MetadataEntry metadataEntry = this.transactionLogAccess.getMetadataEntry(DeltaTestingConnectorSession.SESSION, this.tableSnapshot);
        ProtocolEntry protocolEntry = this.transactionLogAccess.getProtocolEntry(DeltaTestingConnectorSession.SESSION, this.tableSnapshot);
        try (Stream addFileEntries = this.transactionLogAccess.getActiveFiles(DeltaTestingConnectorSession.SESSION, this.tableSnapshot, metadataEntry, protocolEntry, TupleDomain.all(), (Predicate)Predicates.alwaysTrue());){
            Set paths = addFileEntries.map(AddFileEntry::getPath).collect(Collectors.toSet());
            Assertions.assertThat(paths).isEqualTo(EXPECTED_ADD_FILE_PATHS);
        }
    }

    @Test
    public void testGetProtocolEntry() throws Exception {
        this.testGetProtocolEntry("person", "databricks73/person");
        this.testGetProtocolEntry("person_without_last_checkpoint", "databricks73/person_without_last_checkpoint");
        this.testGetProtocolEntry("person_without_old_jsons", "databricks73/person_without_old_jsons");
        this.testGetProtocolEntry("person_without_checkpoints", "databricks73/person_without_checkpoints");
    }

    private void testGetProtocolEntry(String tableName, String resourcePath) throws Exception {
        this.setupTransactionLogAccessFromResources(tableName, resourcePath);
        ProtocolEntry protocolEntry = this.transactionLogAccess.getProtocolEntry(DeltaTestingConnectorSession.SESSION, this.tableSnapshot);
        Assertions.assertThat((int)protocolEntry.minReaderVersion()).isEqualTo(1);
        Assertions.assertThat((int)protocolEntry.minWriterVersion()).isEqualTo(2);
    }

    @Test
    public void testMetadataCacheUpdates() throws Exception {
        String tableName = "person";
        File tempDir = Files.createTempDirectory(null, new FileAttribute[0]).toFile();
        File tableDir = new File(tempDir, tableName);
        File transactionLogDir = new File(tableDir, "_delta_log");
        transactionLogDir.mkdirs();
        Path resourceDir = Paths.get(this.getClass().getClassLoader().getResource("databricks73/person/_delta_log").toURI());
        for (int i = 0; i < 12; ++i) {
            String extension = i == 10 ? ".checkpoint.parquet" : ".json";
            String fileName = String.format("%020d%s", i, extension);
            Files.copy(resourceDir.resolve(fileName), new File(transactionLogDir, fileName).toPath(), new CopyOption[0]);
        }
        Files.copy(resourceDir.resolve("_last_checkpoint"), new File(transactionLogDir, "_last_checkpoint").toPath(), new CopyOption[0]);
        this.setupTransactionLogAccess(tableName, tableDir.toURI().toString());
        Assertions.assertThat((long)this.tableSnapshot.getVersion()).isEqualTo(11L);
        String lastTransactionName = String.format("%020d.json", 12);
        Files.copy(resourceDir.resolve(lastTransactionName), new File(transactionLogDir, lastTransactionName).toPath(), new CopyOption[0]);
        TableSnapshot updatedSnapshot = this.transactionLogAccess.loadSnapshot(DeltaTestingConnectorSession.SESSION, new SchemaTableName("schema", tableName), tableDir.toURI().toString(), Optional.empty());
        Assertions.assertThat((long)updatedSnapshot.getVersion()).isEqualTo(12L);
    }

    @Test
    public void testUpdatingTailEntriesNoCheckpoint() throws Exception {
        String tableName = "person";
        File tempDir = Files.createTempDirectory(null, new FileAttribute[0]).toFile();
        File tableDir = new File(tempDir, tableName);
        File transactionLogDir = new File(tableDir, "_delta_log");
        transactionLogDir.mkdirs();
        File resourceDir = new File(this.getClass().getClassLoader().getResource("databricks73/person/_delta_log").toURI());
        this.copyTransactionLogEntry(0, 7, resourceDir, transactionLogDir);
        this.setupTransactionLogAccess(tableName, tableDir.toURI().toString());
        MetadataEntry metadataEntry = this.transactionLogAccess.getMetadataEntry(DeltaTestingConnectorSession.SESSION, this.tableSnapshot);
        ProtocolEntry protocolEntry = this.transactionLogAccess.getProtocolEntry(DeltaTestingConnectorSession.SESSION, this.tableSnapshot);
        try (Stream activeDataFiles = this.transactionLogAccess.getActiveFiles(DeltaTestingConnectorSession.SESSION, this.tableSnapshot, metadataEntry, protocolEntry, TupleDomain.all(), (Predicate)Predicates.alwaysTrue());){
            ImmutableSet dataFiles = ImmutableSet.of((Object)"age=42/part-00000-b82d8859-84a0-4f05-872c-206b07dd54f0.c000.snappy.parquet", (Object)"age=30/part-00000-72a56c23-01ba-483a-9062-dd0accc86599.c000.snappy.parquet", (Object)"age=25/part-00000-609e34b1-5466-4dbc-a780-2708166e7adb.c000.snappy.parquet", (Object)"age=30/part-00000-7e43a3c3-ea26-4ae7-8eac-8f60cbb4df03.c000.snappy.parquet", (Object)"age=21/part-00000-3d546786-bedc-407f-b9f7-e97aa12cce0f.c000.snappy.parquet", (Object)"age=21/part-00001-290f0f26-19cf-4772-821e-36d55d9b7872.c000.snappy.parquet", (Object[])new String[0]);
            Assertions.assertThat((Collection)activeDataFiles.map(AddFileEntry::getPath).collect(Collectors.toSet())).containsExactlyInAnyOrderElementsOf((Iterable)dataFiles);
        }
        this.copyTransactionLogEntry(7, 9, resourceDir, transactionLogDir);
        TableSnapshot updatedSnapshot = this.transactionLogAccess.loadSnapshot(DeltaTestingConnectorSession.SESSION, new SchemaTableName("schema", tableName), tableDir.toURI().toString(), Optional.empty());
        try (Stream activeDataFiles = this.transactionLogAccess.getActiveFiles(DeltaTestingConnectorSession.SESSION, updatedSnapshot, metadataEntry, protocolEntry, TupleDomain.all(), (Predicate)Predicates.alwaysTrue());){
            ImmutableSet dataFiles = ImmutableSet.of((Object)"age=21/part-00000-3d546786-bedc-407f-b9f7-e97aa12cce0f.c000.snappy.parquet", (Object)"age=21/part-00001-290f0f26-19cf-4772-821e-36d55d9b7872.c000.snappy.parquet", (Object)"age=30/part-00000-63c2205d-84a3-4a66-bd7c-f69f5af55bbc.c000.snappy.parquet", (Object)"age=25/part-00001-aceaf062-1cd1-45cb-8f83-277ffebe995c.c000.snappy.parquet", (Object)"age=30/part-00002-5800be2e-2373-47d8-8b86-776a8ea9d69f.c000.snappy.parquet", (Object)"age=42/part-00003-0f53cae3-3e34-4876-b651-e1db9584dbc3.c000.snappy.parquet", (Object[])new String[]{"age=25/part-00000-b7fbbe31-c7f9-44ed-8757-5c47d10c3e81.c000.snappy.parquet"});
            Assertions.assertThat((Collection)activeDataFiles.map(AddFileEntry::getPath).collect(Collectors.toSet())).containsExactlyInAnyOrderElementsOf((Iterable)dataFiles);
        }
    }

    @Test
    public void testLoadingTailEntriesPastCheckpoint() throws Exception {
        String tableName = "person";
        File tempDir = Files.createTempDirectory(null, new FileAttribute[0]).toFile();
        File tableDir = new File(tempDir, tableName);
        File transactionLogDir = new File(tableDir, "_delta_log");
        transactionLogDir.mkdirs();
        File resourceDir = new File(this.getClass().getClassLoader().getResource("databricks73/person/_delta_log").toURI());
        this.copyTransactionLogEntry(0, 8, resourceDir, transactionLogDir);
        this.setupTransactionLogAccess(tableName, tableDir.toURI().toString());
        MetadataEntry metadataEntry = this.transactionLogAccess.getMetadataEntry(DeltaTestingConnectorSession.SESSION, this.tableSnapshot);
        ProtocolEntry protocolEntry = this.transactionLogAccess.getProtocolEntry(DeltaTestingConnectorSession.SESSION, this.tableSnapshot);
        try (Stream activeDataFiles = this.transactionLogAccess.getActiveFiles(DeltaTestingConnectorSession.SESSION, this.tableSnapshot, metadataEntry, protocolEntry, TupleDomain.all(), (Predicate)Predicates.alwaysTrue());){
            ImmutableSet dataFiles = ImmutableSet.of((Object)"age=21/part-00000-3d546786-bedc-407f-b9f7-e97aa12cce0f.c000.snappy.parquet", (Object)"age=21/part-00001-290f0f26-19cf-4772-821e-36d55d9b7872.c000.snappy.parquet", (Object)"age=30/part-00000-63c2205d-84a3-4a66-bd7c-f69f5af55bbc.c000.snappy.parquet", (Object)"age=25/part-00001-aceaf062-1cd1-45cb-8f83-277ffebe995c.c000.snappy.parquet", (Object)"age=30/part-00002-5800be2e-2373-47d8-8b86-776a8ea9d69f.c000.snappy.parquet", (Object)"age=42/part-00003-0f53cae3-3e34-4876-b651-e1db9584dbc3.c000.snappy.parquet", (Object[])new String[0]);
            Assertions.assertThat((Collection)activeDataFiles.map(AddFileEntry::getPath).collect(Collectors.toSet())).containsExactlyInAnyOrderElementsOf((Iterable)dataFiles);
        }
        this.copyTransactionLogEntry(8, 12, resourceDir, transactionLogDir);
        Files.copy(new File(resourceDir, "_last_checkpoint").toPath(), new File(transactionLogDir, "_last_checkpoint").toPath(), new CopyOption[0]);
        TableSnapshot updatedSnapshot = this.transactionLogAccess.loadSnapshot(DeltaTestingConnectorSession.SESSION, new SchemaTableName("schema", tableName), tableDir.toURI().toString(), Optional.empty());
        try (Stream activeDataFiles = this.transactionLogAccess.getActiveFiles(DeltaTestingConnectorSession.SESSION, updatedSnapshot, metadataEntry, protocolEntry, TupleDomain.all(), (Predicate)Predicates.alwaysTrue());){
            ImmutableSet dataFiles = ImmutableSet.of((Object)"age=21/part-00000-3d546786-bedc-407f-b9f7-e97aa12cce0f.c000.snappy.parquet", (Object)"age=21/part-00001-290f0f26-19cf-4772-821e-36d55d9b7872.c000.snappy.parquet", (Object)"age=30/part-00000-63c2205d-84a3-4a66-bd7c-f69f5af55bbc.c000.snappy.parquet", (Object)"age=25/part-00001-aceaf062-1cd1-45cb-8f83-277ffebe995c.c000.snappy.parquet", (Object)"age=30/part-00002-5800be2e-2373-47d8-8b86-776a8ea9d69f.c000.snappy.parquet", (Object)"age=42/part-00003-0f53cae3-3e34-4876-b651-e1db9584dbc3.c000.snappy.parquet", (Object[])new String[]{"age=25/part-00000-b7fbbe31-c7f9-44ed-8757-5c47d10c3e81.c000.snappy.parquet", "age=25/part-00000-22a101a1-8f09-425e-847e-cbbe4f894eea.c000.snappy.parquet", "age=42/part-00000-b26c891a-7288-4d96-9d3b-bef648f12a34.c000.snappy.parquet", "age=30/part-00000-37ccfcd3-b44b-4d04-a1e6-d2837da75f7a.c000.snappy.parquet"});
            Assertions.assertThat((Collection)activeDataFiles.map(AddFileEntry::getPath).collect(Collectors.toSet())).containsExactlyInAnyOrderElementsOf((Iterable)dataFiles);
        }
    }

    @Test
    public void testIncrementalCacheUpdates() throws Exception {
        this.setupTransactionLogAccessFromResources("person", "databricks73/person");
        String tableName = "person";
        File tempDir = Files.createTempDirectory(null, new FileAttribute[0]).toFile();
        File tableDir = new File(tempDir, tableName);
        File transactionLogDir = new File(tableDir, "_delta_log");
        transactionLogDir.mkdirs();
        File resourceDir = new File(this.getClass().getClassLoader().getResource("databricks73/person/_delta_log").toURI());
        this.copyTransactionLogEntry(0, 12, resourceDir, transactionLogDir);
        Files.copy(new File(resourceDir, "_last_checkpoint").toPath(), new File(transactionLogDir, "_last_checkpoint").toPath(), new CopyOption[0]);
        TestingConnectorSession activeDataFileCacheSession = TestingConnectorSession.builder().setPropertyMetadata(new DeltaLakeSessionProperties(new DeltaLakeConfig().setCheckpointFilteringEnabled(false), new ParquetReaderConfig(), new ParquetWriterConfig()).getSessionProperties()).build();
        MetadataEntry metadataEntry = this.transactionLogAccess.getMetadataEntry((ConnectorSession)activeDataFileCacheSession, this.tableSnapshot);
        ProtocolEntry protocolEntry = this.transactionLogAccess.getProtocolEntry((ConnectorSession)activeDataFileCacheSession, this.tableSnapshot);
        ImmutableSet originalDataFiles = ImmutableSet.of((Object)"age=42/part-00000-b26c891a-7288-4d96-9d3b-bef648f12a34.c000.snappy.parquet", (Object)"age=25/part-00001-aceaf062-1cd1-45cb-8f83-277ffebe995c.c000.snappy.parquet", (Object)"age=25/part-00000-b7fbbe31-c7f9-44ed-8757-5c47d10c3e81.c000.snappy.parquet", (Object)"age=42/part-00003-0f53cae3-3e34-4876-b651-e1db9584dbc3.c000.snappy.parquet", (Object)"age=30/part-00000-63c2205d-84a3-4a66-bd7c-f69f5af55bbc.c000.snappy.parquet", (Object)"age=25/part-00000-22a101a1-8f09-425e-847e-cbbe4f894eea.c000.snappy.parquet", (Object[])new String[]{"age=21/part-00001-290f0f26-19cf-4772-821e-36d55d9b7872.c000.snappy.parquet", "age=30/part-00002-5800be2e-2373-47d8-8b86-776a8ea9d69f.c000.snappy.parquet", "age=21/part-00000-3d546786-bedc-407f-b9f7-e97aa12cce0f.c000.snappy.parquet", "age=30/part-00000-37ccfcd3-b44b-4d04-a1e6-d2837da75f7a.c000.snappy.parquet"});
        this.assertFileSystemAccesses(() -> this.lambda$testIncrementalCacheUpdates$0(tableName, tableDir, (ConnectorSession)activeDataFileCacheSession, metadataEntry, protocolEntry, (Set)originalDataFiles), (Multiset<FileOperation>)ImmutableMultiset.builder().add((Object)new FileOperation("_last_checkpoint", "InputFile.newStream")).add((Object)new FileOperation("00000000000000000010.checkpoint.parquet", "InputFile.length")).add((Object)new FileOperation("00000000000000000010.checkpoint.parquet", "InputFile.newInput")).add((Object)new FileOperation("00000000000000000011.json", "InputFile.newStream")).add((Object)new FileOperation("00000000000000000011.json", "InputFile.length")).add((Object)new FileOperation("00000000000000000012.json", "InputFile.length")).build());
        this.copyTransactionLogEntry(12, 14, resourceDir, transactionLogDir);
        ImmutableSet newDataFiles = ImmutableSet.of((Object)"age=28/part-00000-40dd1707-1d42-4328-a59a-21f5c945fe60.c000.snappy.parquet", (Object)"age=29/part-00000-3794c463-cb0c-4beb-8d07-7cc1e3b5920f.c000.snappy.parquet");
        this.assertFileSystemAccesses(() -> this.lambda$testIncrementalCacheUpdates$1((ConnectorSession)activeDataFileCacheSession, tableName, tableDir, metadataEntry, protocolEntry, (Set)originalDataFiles, (Set)newDataFiles), (Multiset<FileOperation>)ImmutableMultiset.builder().add((Object)new FileOperation("_last_checkpoint", "InputFile.newStream")).addCopies((Object)new FileOperation("00000000000000000012.json", "InputFile.newStream"), 2).addCopies((Object)new FileOperation("00000000000000000013.json", "InputFile.newStream"), 2).addCopies((Object)new FileOperation("00000000000000000012.json", "InputFile.length"), 2).addCopies((Object)new FileOperation("00000000000000000013.json", "InputFile.length"), 2).add((Object)new FileOperation("00000000000000000014.json", "InputFile.length")).build());
    }

    @Test
    public void testSnapshotsAreConsistent() throws Exception {
        List dataFilesWithFixedVersion;
        List allDataFiles;
        List expectedDataFiles;
        String tableName = "person";
        File tempDir = Files.createTempDirectory(null, new FileAttribute[0]).toFile();
        File tableDir = new File(tempDir, tableName);
        File transactionLogDir = new File(tableDir, "_delta_log");
        transactionLogDir.mkdirs();
        File resourceDir = new File(this.getClass().getClassLoader().getResource("databricks73/person/_delta_log").toURI());
        this.copyTransactionLogEntry(0, 12, resourceDir, transactionLogDir);
        Files.copy(new File(resourceDir, "_last_checkpoint").toPath(), new File(transactionLogDir, "_last_checkpoint").toPath(), new CopyOption[0]);
        this.setupTransactionLogAccess(tableName, tableDir.toURI().toString());
        MetadataEntry metadataEntry = this.transactionLogAccess.getMetadataEntry(DeltaTestingConnectorSession.SESSION, this.tableSnapshot);
        ProtocolEntry protocolEntry = this.transactionLogAccess.getProtocolEntry(DeltaTestingConnectorSession.SESSION, this.tableSnapshot);
        try (Stream addFileEntryStream = this.transactionLogAccess.getActiveFiles(DeltaTestingConnectorSession.SESSION, this.tableSnapshot, metadataEntry, protocolEntry, TupleDomain.all(), (Predicate)Predicates.alwaysTrue());){
            expectedDataFiles = (List)addFileEntryStream.collect(ImmutableList.toImmutableList());
        }
        this.copyTransactionLogEntry(12, 14, resourceDir, transactionLogDir);
        ImmutableSet newDataFiles = ImmutableSet.of((Object)"age=28/part-00000-40dd1707-1d42-4328-a59a-21f5c945fe60.c000.snappy.parquet", (Object)"age=29/part-00000-3794c463-cb0c-4beb-8d07-7cc1e3b5920f.c000.snappy.parquet");
        TableSnapshot updatedTableSnapshot = this.transactionLogAccess.loadSnapshot(DeltaTestingConnectorSession.SESSION, new SchemaTableName("schema", tableName), tableDir.toURI().toString(), Optional.empty());
        try (Stream addFileEntryStream = this.transactionLogAccess.getActiveFiles(DeltaTestingConnectorSession.SESSION, updatedTableSnapshot, metadataEntry, protocolEntry, TupleDomain.all(), (Predicate)Predicates.alwaysTrue());){
            allDataFiles = (List)addFileEntryStream.collect(ImmutableList.toImmutableList());
        }
        try (Stream addFileEntryStream = this.transactionLogAccess.getActiveFiles(DeltaTestingConnectorSession.SESSION, this.tableSnapshot, metadataEntry, protocolEntry, TupleDomain.all(), (Predicate)Predicates.alwaysTrue());){
            dataFilesWithFixedVersion = (List)addFileEntryStream.collect(ImmutableList.toImmutableList());
        }
        for (String newFilePath : newDataFiles) {
            Assertions.assertThat((boolean)allDataFiles.stream().anyMatch(entry -> entry.getPath().equals(newFilePath))).isTrue();
            Assertions.assertThat((boolean)dataFilesWithFixedVersion.stream().noneMatch(entry -> entry.getPath().equals(newFilePath))).isTrue();
        }
        Assertions.assertThat((List)expectedDataFiles).hasSize(dataFilesWithFixedVersion.size());
        List columns = DeltaLakeSchemaSupport.extractColumnMetadata((MetadataEntry)this.transactionLogAccess.getMetadataEntry(DeltaTestingConnectorSession.SESSION, this.tableSnapshot), (ProtocolEntry)this.transactionLogAccess.getProtocolEntry(DeltaTestingConnectorSession.SESSION, this.tableSnapshot), (TypeManager)InternalTypeManager.TESTING_TYPE_MANAGER);
        for (int i = 0; i < expectedDataFiles.size(); ++i) {
            AddFileEntry expected = (AddFileEntry)expectedDataFiles.get(i);
            AddFileEntry actual = (AddFileEntry)dataFilesWithFixedVersion.get(i);
            Assertions.assertThat((String)expected.getPath()).isEqualTo(actual.getPath());
            Assertions.assertThat((Map)expected.getPartitionValues()).isEqualTo((Object)actual.getPartitionValues());
            Assertions.assertThat((long)expected.getSize()).isEqualTo(actual.getSize());
            Assertions.assertThat((long)expected.getModificationTime()).isEqualTo(actual.getModificationTime());
            Assertions.assertThat((boolean)expected.isDataChange()).isEqualTo(actual.isDataChange());
            Assertions.assertThat((Map)expected.getTags()).isEqualTo((Object)actual.getTags());
            Assertions.assertThat((Optional)expected.getStats()).isPresent();
            Assertions.assertThat((Optional)actual.getStats()).isPresent();
            for (ColumnMetadata column : columns) {
                DeltaLakeColumnHandle columnHandle = new DeltaLakeColumnHandle(column.getName(), column.getType(), OptionalInt.empty(), column.getName(), column.getType(), DeltaLakeColumnType.REGULAR, Optional.empty());
                Assertions.assertThat((Optional)((DeltaLakeFileStatistics)expected.getStats().get()).getMinColumnValue(columnHandle)).isEqualTo((Object)((DeltaLakeFileStatistics)actual.getStats().get()).getMinColumnValue(columnHandle));
                Assertions.assertThat((Optional)((DeltaLakeFileStatistics)expected.getStats().get()).getMaxColumnValue(columnHandle)).isEqualTo((Object)((DeltaLakeFileStatistics)actual.getStats().get()).getMaxColumnValue(columnHandle));
                Assertions.assertThat((Optional)((DeltaLakeFileStatistics)expected.getStats().get()).getNullCount(columnHandle.baseColumnName())).isEqualTo((Object)((DeltaLakeFileStatistics)actual.getStats().get()).getNullCount(columnHandle.baseColumnName()));
                Assertions.assertThat((Optional)((DeltaLakeFileStatistics)expected.getStats().get()).getNumRecords()).isEqualTo((Object)((DeltaLakeFileStatistics)actual.getStats().get()).getNumRecords());
            }
        }
    }

    @Test
    public void testAddNewTransactionLogs() throws Exception {
        String tableName = "person";
        File tempDir = Files.createTempDirectory(null, new FileAttribute[0]).toFile();
        File tableDir = new File(tempDir, tableName);
        File transactionLogDir = new File(tableDir, "_delta_log");
        transactionLogDir.mkdirs();
        String tableLocation = tableDir.toURI().toString();
        SchemaTableName schemaTableName = new SchemaTableName("schema", tableName);
        File resourceDir = new File(this.getClass().getClassLoader().getResource("databricks73/person/_delta_log").toURI());
        this.copyTransactionLogEntry(0, 1, resourceDir, transactionLogDir);
        this.setupTransactionLogAccess(tableName, tableLocation);
        Assertions.assertThat((long)this.tableSnapshot.getVersion()).isEqualTo(0L);
        this.copyTransactionLogEntry(1, 2, resourceDir, transactionLogDir);
        TableSnapshot firstUpdate = this.transactionLogAccess.loadSnapshot(DeltaTestingConnectorSession.SESSION, schemaTableName, tableLocation, Optional.empty());
        Assertions.assertThat((long)firstUpdate.getVersion()).isEqualTo(1L);
        this.copyTransactionLogEntry(2, 3, resourceDir, transactionLogDir);
        TableSnapshot secondUpdate = this.transactionLogAccess.loadSnapshot(DeltaTestingConnectorSession.SESSION, schemaTableName, tableLocation, Optional.empty());
        Assertions.assertThat((long)secondUpdate.getVersion()).isEqualTo(2L);
    }

    @Test
    public void testParquetStructStatistics() throws Exception {
        List addFileEntries;
        String tableName = "parquet_struct_statistics";
        this.setupTransactionLogAccess(tableName, this.getClass().getClassLoader().getResource("databricks73/pruning/" + tableName).toURI().toString());
        MetadataEntry metadataEntry = this.transactionLogAccess.getMetadataEntry(DeltaTestingConnectorSession.SESSION, this.tableSnapshot);
        ProtocolEntry protocolEntry = this.transactionLogAccess.getProtocolEntry(DeltaTestingConnectorSession.SESSION, this.tableSnapshot);
        try (Stream addFileEntryStream = this.transactionLogAccess.getActiveFiles(DeltaTestingConnectorSession.SESSION, this.tableSnapshot, metadataEntry, protocolEntry, TupleDomain.all(), (Predicate)Predicates.alwaysTrue());){
            addFileEntries = (List)addFileEntryStream.collect(ImmutableList.toImmutableList());
        }
        AddFileEntry addFileEntry = (AddFileEntry)addFileEntries.stream().filter(entry -> entry.getPath().equalsIgnoreCase("part-00000-0e22455f-5650-442f-a094-e1a8b7ed2271-c000.snappy.parquet")).collect(MoreCollectors.onlyElement());
        Assertions.assertThat((Optional)addFileEntry.getStats()).isPresent();
        DeltaLakeFileStatistics fileStats = (DeltaLakeFileStatistics)addFileEntry.getStats().get();
        BigDecimal decValue = BigDecimal.valueOf(999999999999123L).movePointLeft(3);
        ImmutableMap statsValues = ImmutableMap.builder().put((Object)"ts", (Object)DateTimeEncoding.packDateTimeWithZone((long)LocalDateTime.parse("2960-10-31T01:00:00").toInstant(ZoneOffset.UTC).toEpochMilli(), (TimeZoneKey)TimeZoneKey.UTC_KEY)).put((Object)"str", (Object)Slices.utf8Slice((String)"a")).put((Object)"dec_short", (Object)101L).put((Object)"dec_long", (Object)Decimals.valueOf((BigDecimal)decValue)).put((Object)"l", (Object)10000000L).put((Object)"in", (Object)20000000L).put((Object)"sh", (Object)123L).put((Object)"byt", (Object)42L).put((Object)"fl", (Object)Float.floatToIntBits(0.123f)).put((Object)"dou", (Object)0.321).put((Object)"dat", (Object)LocalDate.parse("5000-01-01").toEpochDay()).buildOrThrow();
        for (String columnName : statsValues.keySet()) {
            Assertions.assertThat((Optional)fileStats.getMinColumnValue(new DeltaLakeColumnHandle(columnName, (Type)IntegerType.INTEGER, OptionalInt.empty(), columnName, (Type)IntegerType.INTEGER, DeltaLakeColumnType.REGULAR, Optional.empty()))).isEqualTo(Optional.of(statsValues.get(columnName)));
            Assertions.assertThat((Optional)fileStats.getMaxColumnValue(new DeltaLakeColumnHandle(columnName, (Type)IntegerType.INTEGER, OptionalInt.empty(), columnName, (Type)IntegerType.INTEGER, DeltaLakeColumnType.REGULAR, Optional.empty()))).isEqualTo(Optional.of(statsValues.get(columnName)));
        }
    }

    @Test
    public void testTableSnapshotsCacheDisabled() throws Exception {
        String tableName = "person";
        String tableDir = this.getClass().getClassLoader().getResource("databricks73/" + tableName).toURI().toString();
        DeltaLakeConfig cacheDisabledConfig = new DeltaLakeConfig();
        cacheDisabledConfig.setMetadataCacheTtl(new Duration(0.0, TimeUnit.SECONDS));
        this.assertFileSystemAccesses(() -> this.setupTransactionLogAccess(tableName, tableDir, cacheDisabledConfig, Optional.empty()), (Multiset<FileOperation>)ImmutableMultiset.builder().add((Object)new FileOperation("_last_checkpoint", "InputFile.newStream")).add((Object)new FileOperation("00000000000000000011.json", "InputFile.newStream")).add((Object)new FileOperation("00000000000000000012.json", "InputFile.newStream")).add((Object)new FileOperation("00000000000000000013.json", "InputFile.newStream")).add((Object)new FileOperation("00000000000000000011.json", "InputFile.length")).add((Object)new FileOperation("00000000000000000012.json", "InputFile.length")).add((Object)new FileOperation("00000000000000000013.json", "InputFile.length")).add((Object)new FileOperation("00000000000000000014.json", "InputFile.length")).build());
        this.assertFileSystemAccesses(() -> this.transactionLogAccess.loadSnapshot(DeltaTestingConnectorSession.SESSION, new SchemaTableName("schema", tableName), tableDir, Optional.empty()), (Multiset<FileOperation>)ImmutableMultiset.builder().add((Object)new FileOperation("_last_checkpoint", "InputFile.newStream")).add((Object)new FileOperation("00000000000000000011.json", "InputFile.newStream")).add((Object)new FileOperation("00000000000000000012.json", "InputFile.newStream")).add((Object)new FileOperation("00000000000000000013.json", "InputFile.newStream")).add((Object)new FileOperation("00000000000000000011.json", "InputFile.length")).add((Object)new FileOperation("00000000000000000012.json", "InputFile.length")).add((Object)new FileOperation("00000000000000000013.json", "InputFile.length")).add((Object)new FileOperation("00000000000000000014.json", "InputFile.length")).build());
    }

    @Test
    public void testTableSnapshotsActiveDataFilesCache() throws Exception {
        this.setupTransactionLogAccessFromResources("person", "databricks73/person");
        String tableName = "person";
        String tableDir = this.getClass().getClassLoader().getResource("databricks73/" + tableName).toURI().toString();
        DeltaLakeConfig shortLivedActiveDataFilesCacheConfig = new DeltaLakeConfig();
        shortLivedActiveDataFilesCacheConfig.setDataFileCacheTtl(new Duration(10.0, TimeUnit.MINUTES));
        MetadataEntry metadataEntry = this.transactionLogAccess.getMetadataEntry(DeltaTestingConnectorSession.SESSION, this.tableSnapshot);
        ProtocolEntry protocolEntry = this.transactionLogAccess.getProtocolEntry(DeltaTestingConnectorSession.SESSION, this.tableSnapshot);
        TestingConnectorSession activeDataFileCacheSession = TestingConnectorSession.builder().setPropertyMetadata(new DeltaLakeSessionProperties(new DeltaLakeConfig().setCheckpointFilteringEnabled(false), new ParquetReaderConfig(), new ParquetWriterConfig()).getSessionProperties()).build();
        this.assertFileSystemAccesses(() -> this.lambda$testTableSnapshotsActiveDataFilesCache$0(tableName, tableDir, shortLivedActiveDataFilesCacheConfig, (ConnectorSession)activeDataFileCacheSession, metadataEntry, protocolEntry), (Multiset<FileOperation>)ImmutableMultiset.builder().add((Object)new FileOperation("_last_checkpoint", "InputFile.newStream")).add((Object)new FileOperation("00000000000000000011.json", "InputFile.newStream")).add((Object)new FileOperation("00000000000000000012.json", "InputFile.newStream")).add((Object)new FileOperation("00000000000000000013.json", "InputFile.newStream")).add((Object)new FileOperation("00000000000000000011.json", "InputFile.length")).add((Object)new FileOperation("00000000000000000012.json", "InputFile.length")).add((Object)new FileOperation("00000000000000000013.json", "InputFile.length")).add((Object)new FileOperation("00000000000000000014.json", "InputFile.length")).add((Object)new FileOperation("00000000000000000010.checkpoint.parquet", "InputFile.length")).add((Object)new FileOperation("00000000000000000010.checkpoint.parquet", "InputFile.newInput")).build());
        this.assertFileSystemAccesses(() -> this.lambda$testTableSnapshotsActiveDataFilesCache$1((ConnectorSession)activeDataFileCacheSession, metadataEntry, protocolEntry), (Multiset<FileOperation>)ImmutableMultiset.of());
    }

    @Test
    public void testFlushSnapshotAndActiveFileCache() throws Exception {
        this.setupTransactionLogAccessFromResources("person", "databricks73/person");
        String tableName = "person";
        String tableDir = this.getClass().getClassLoader().getResource("databricks73/" + tableName).toURI().toString();
        DeltaLakeConfig shortLivedActiveDataFilesCacheConfig = new DeltaLakeConfig();
        shortLivedActiveDataFilesCacheConfig.setDataFileCacheTtl(new Duration(10.0, TimeUnit.MINUTES));
        MetadataEntry metadataEntry = this.transactionLogAccess.getMetadataEntry(DeltaTestingConnectorSession.SESSION, this.tableSnapshot);
        ProtocolEntry protocolEntry = this.transactionLogAccess.getProtocolEntry(DeltaTestingConnectorSession.SESSION, this.tableSnapshot);
        this.assertFileSystemAccesses(() -> {
            this.setupTransactionLogAccess(tableName, tableDir, shortLivedActiveDataFilesCacheConfig, Optional.empty());
            try (Stream addFileEntries = this.transactionLogAccess.getActiveFiles(DeltaTestingConnectorSession.SESSION, this.tableSnapshot, metadataEntry, protocolEntry, TupleDomain.all(), (Predicate)Predicates.alwaysTrue());){
                Assertions.assertThat((long)addFileEntries.count()).isEqualTo(12L);
            }
        }, (Multiset<FileOperation>)ImmutableMultiset.builder().add((Object)new FileOperation("_last_checkpoint", "InputFile.newStream")).add((Object)new FileOperation("00000000000000000011.json", "InputFile.newStream")).add((Object)new FileOperation("00000000000000000012.json", "InputFile.newStream")).add((Object)new FileOperation("00000000000000000013.json", "InputFile.newStream")).add((Object)new FileOperation("00000000000000000011.json", "InputFile.length")).add((Object)new FileOperation("00000000000000000012.json", "InputFile.length")).add((Object)new FileOperation("00000000000000000013.json", "InputFile.length")).add((Object)new FileOperation("00000000000000000014.json", "InputFile.length")).add((Object)new FileOperation("00000000000000000010.checkpoint.parquet", "InputFile.length")).add((Object)new FileOperation("00000000000000000010.checkpoint.parquet", "InputFile.newInput")).build());
        this.transactionLogAccess.flushCache();
        this.assertFileSystemAccesses(() -> {
            this.transactionLogAccess.loadSnapshot(DeltaTestingConnectorSession.SESSION, new SchemaTableName("schema", tableName), tableDir, Optional.empty());
            try (Stream addFileEntries = this.transactionLogAccess.getActiveFiles(DeltaTestingConnectorSession.SESSION, this.tableSnapshot, metadataEntry, protocolEntry, TupleDomain.all(), (Predicate)Predicates.alwaysTrue());){
                Assertions.assertThat((long)addFileEntries.count()).isEqualTo(12L);
            }
        }, (Multiset<FileOperation>)ImmutableMultiset.builder().add((Object)new FileOperation("_last_checkpoint", "InputFile.newStream")).add((Object)new FileOperation("00000000000000000011.json", "InputFile.newStream")).add((Object)new FileOperation("00000000000000000012.json", "InputFile.newStream")).add((Object)new FileOperation("00000000000000000013.json", "InputFile.newStream")).add((Object)new FileOperation("00000000000000000011.json", "InputFile.length")).add((Object)new FileOperation("00000000000000000012.json", "InputFile.length")).add((Object)new FileOperation("00000000000000000013.json", "InputFile.length")).add((Object)new FileOperation("00000000000000000014.json", "InputFile.length")).add((Object)new FileOperation("00000000000000000010.checkpoint.parquet", "InputFile.length")).add((Object)new FileOperation("00000000000000000010.checkpoint.parquet", "InputFile.newInput")).build());
    }

    @Test
    public void testTableSnapshotsActiveDataFilesCacheDisabled() throws Exception {
        this.setupTransactionLogAccessFromResources("person", "databricks73/person");
        String tableName = "person";
        String tableDir = this.getClass().getClassLoader().getResource("databricks73/" + tableName).toURI().toString();
        DeltaLakeConfig shortLivedActiveDataFilesCacheConfig = new DeltaLakeConfig();
        shortLivedActiveDataFilesCacheConfig.setDataFileCacheTtl(new Duration(0.0, TimeUnit.SECONDS));
        MetadataEntry metadataEntry = this.transactionLogAccess.getMetadataEntry(DeltaTestingConnectorSession.SESSION, this.tableSnapshot);
        ProtocolEntry protocolEntry = this.transactionLogAccess.getProtocolEntry(DeltaTestingConnectorSession.SESSION, this.tableSnapshot);
        this.assertFileSystemAccesses(() -> {
            this.setupTransactionLogAccess(tableName, tableDir, shortLivedActiveDataFilesCacheConfig, Optional.empty());
            try (Stream addFileEntries = this.transactionLogAccess.getActiveFiles(DeltaTestingConnectorSession.SESSION, this.tableSnapshot, metadataEntry, protocolEntry, TupleDomain.all(), (Predicate)Predicates.alwaysTrue());){
                Assertions.assertThat((long)addFileEntries.count()).isEqualTo(12L);
            }
        }, (Multiset<FileOperation>)ImmutableMultiset.builder().add((Object)new FileOperation("_last_checkpoint", "InputFile.newStream")).add((Object)new FileOperation("00000000000000000011.json", "InputFile.newStream")).add((Object)new FileOperation("00000000000000000012.json", "InputFile.newStream")).add((Object)new FileOperation("00000000000000000013.json", "InputFile.newStream")).add((Object)new FileOperation("00000000000000000011.json", "InputFile.length")).add((Object)new FileOperation("00000000000000000012.json", "InputFile.length")).add((Object)new FileOperation("00000000000000000013.json", "InputFile.length")).add((Object)new FileOperation("00000000000000000014.json", "InputFile.length")).add((Object)new FileOperation("00000000000000000010.checkpoint.parquet", "InputFile.length")).add((Object)new FileOperation("00000000000000000010.checkpoint.parquet", "InputFile.newInput")).build());
        this.assertFileSystemAccesses(() -> {
            try (Stream addFileEntries = this.transactionLogAccess.getActiveFiles(DeltaTestingConnectorSession.SESSION, this.tableSnapshot, metadataEntry, protocolEntry, TupleDomain.all(), (Predicate)Predicates.alwaysTrue());){
                Assertions.assertThat((long)addFileEntries.count()).isEqualTo(12L);
            }
        }, (Multiset<FileOperation>)ImmutableMultiset.builder().add((Object)new FileOperation("00000000000000000010.checkpoint.parquet", "InputFile.length")).add((Object)new FileOperation("00000000000000000010.checkpoint.parquet", "InputFile.newInput")).build());
    }

    @Test
    public void testLoadSnapshotWithEndVersion() throws Exception {
        String tableName = "person";
        String tableDir = this.getClass().getClassLoader().getResource("databricks73/" + tableName).toURI().toString();
        DeltaLakeConfig shortLivedActiveDataFilesCacheConfig = new DeltaLakeConfig();
        shortLivedActiveDataFilesCacheConfig.setDataFileCacheTtl(new Duration(10.0, TimeUnit.MINUTES));
        this.setupTransactionLogAccess("person", this.getClass().getClassLoader().getResource("databricks73/person").toString(), new DeltaLakeConfig(), Optional.of(9L));
        MetadataEntry metadataEntry = this.transactionLogAccess.getMetadataEntry(DeltaTestingConnectorSession.SESSION, this.tableSnapshot);
        ProtocolEntry protocolEntry = this.transactionLogAccess.getProtocolEntry(DeltaTestingConnectorSession.SESSION, this.tableSnapshot);
        this.transactionLogAccess.flushCache();
        this.assertFileSystemAccesses(() -> {
            this.transactionLogAccess.loadSnapshot(DeltaTestingConnectorSession.SESSION, new SchemaTableName("schema", tableName), tableDir, Optional.of(9L));
            try (Stream addFileEntries = this.transactionLogAccess.getActiveFiles(DeltaTestingConnectorSession.SESSION, this.tableSnapshot, metadataEntry, protocolEntry, TupleDomain.all(), (Predicate)Predicates.alwaysTrue());){
                Assertions.assertThat((long)addFileEntries.count()).isEqualTo(8L);
            }
        }, (Multiset<FileOperation>)ImmutableMultiset.builder().add((Object)new FileOperation("_last_checkpoint", "InputFile.newStream")).add((Object)new FileOperation("00000000000000000000.json", "InputFile.newStream")).add((Object)new FileOperation("00000000000000000001.json", "InputFile.newStream")).add((Object)new FileOperation("00000000000000000002.json", "InputFile.newStream")).add((Object)new FileOperation("00000000000000000003.json", "InputFile.newStream")).add((Object)new FileOperation("00000000000000000004.json", "InputFile.newStream")).add((Object)new FileOperation("00000000000000000005.json", "InputFile.newStream")).add((Object)new FileOperation("00000000000000000006.json", "InputFile.newStream")).add((Object)new FileOperation("00000000000000000007.json", "InputFile.newStream")).add((Object)new FileOperation("00000000000000000008.json", "InputFile.newStream")).add((Object)new FileOperation("00000000000000000009.json", "InputFile.newStream")).add((Object)new FileOperation("00000000000000000000.json", "InputFile.length")).add((Object)new FileOperation("00000000000000000001.json", "InputFile.length")).add((Object)new FileOperation("00000000000000000002.json", "InputFile.length")).add((Object)new FileOperation("00000000000000000003.json", "InputFile.length")).add((Object)new FileOperation("00000000000000000004.json", "InputFile.length")).add((Object)new FileOperation("00000000000000000005.json", "InputFile.length")).add((Object)new FileOperation("00000000000000000006.json", "InputFile.length")).add((Object)new FileOperation("00000000000000000007.json", "InputFile.length")).add((Object)new FileOperation("00000000000000000008.json", "InputFile.length")).add((Object)new FileOperation("00000000000000000009.json", "InputFile.length")).build());
        this.setupTransactionLogAccess("person", this.getClass().getClassLoader().getResource("databricks73/person").toString(), new DeltaLakeConfig(), Optional.of(10L));
        this.transactionLogAccess.flushCache();
        this.assertFileSystemAccesses(() -> {
            this.transactionLogAccess.loadSnapshot(DeltaTestingConnectorSession.SESSION, new SchemaTableName("schema", tableName), tableDir, Optional.of(10L));
            try (Stream addFileEntries = this.transactionLogAccess.getActiveFiles(DeltaTestingConnectorSession.SESSION, this.tableSnapshot, metadataEntry, protocolEntry, TupleDomain.all(), (Predicate)Predicates.alwaysTrue());){
                Assertions.assertThat((long)addFileEntries.count()).isEqualTo(9L);
            }
        }, (Multiset<FileOperation>)ImmutableMultiset.builder().add((Object)new FileOperation("_last_checkpoint", "InputFile.newStream")).add((Object)new FileOperation("00000000000000000010.checkpoint.parquet", "InputFile.length")).add((Object)new FileOperation("00000000000000000010.checkpoint.parquet", "InputFile.newInput")).build());
        this.setupTransactionLogAccess("person", this.getClass().getClassLoader().getResource("databricks73/person").toString(), new DeltaLakeConfig(), Optional.of(11L));
        this.transactionLogAccess.flushCache();
        this.assertFileSystemAccesses(() -> {
            this.transactionLogAccess.loadSnapshot(DeltaTestingConnectorSession.SESSION, new SchemaTableName("schema", tableName), tableDir, Optional.of(11L));
            try (Stream addFileEntries = this.transactionLogAccess.getActiveFiles(DeltaTestingConnectorSession.SESSION, this.tableSnapshot, metadataEntry, protocolEntry, TupleDomain.all(), (Predicate)Predicates.alwaysTrue());){
                Assertions.assertThat((long)addFileEntries.count()).isEqualTo(10L);
            }
        }, (Multiset<FileOperation>)ImmutableMultiset.builder().add((Object)new FileOperation("_last_checkpoint", "InputFile.newStream")).add((Object)new FileOperation("00000000000000000010.checkpoint.parquet", "InputFile.length")).add((Object)new FileOperation("00000000000000000010.checkpoint.parquet", "InputFile.newInput")).add((Object)new FileOperation("00000000000000000011.json", "InputFile.newStream")).add((Object)new FileOperation("00000000000000000011.json", "InputFile.length")).build());
    }

    private void copyTransactionLogEntry(int startVersion, int endVersion, File sourceDir, File targetDir) throws IOException {
        for (int i = startVersion; i < endVersion; ++i) {
            if (i % 10 == 0 && i != 0) {
                String checkpointFileName = String.format("%020d.checkpoint.parquet", i);
                Files.copy(new File(sourceDir, checkpointFileName).toPath(), new File(targetDir, checkpointFileName).toPath(), new CopyOption[0]);
            }
            String lastTransactionName = String.format("%020d.json", i);
            Files.copy(new File(sourceDir, lastTransactionName).toPath(), new File(targetDir, lastTransactionName).toPath(), new CopyOption[0]);
        }
    }

    private void assertFileSystemAccesses(TestingTelemetry.CheckedRunnable<?> callback, Multiset<FileOperation> expectedAccesses) throws Exception {
        List spanData = this.testingTelemetry.captureSpans(() -> callback.run());
        MultisetAssertions.assertMultisetsEqual(this.getOperations(spanData), expectedAccesses);
    }

    private Multiset<FileOperation> getOperations(List<SpanData> spans) {
        return (Multiset)spans.stream().filter(span -> span.getName().startsWith("InputFile.")).map(span -> new FileOperation(((String)span.getAttributes().get(FileSystemAttributes.FILE_LOCATION)).replaceFirst(".*/_delta_log/", ""), span.getName())).collect(Collectors.toCollection(HashMultiset::create));
    }

    private /* synthetic */ void lambda$testTableSnapshotsActiveDataFilesCache$1(ConnectorSession activeDataFileCacheSession, MetadataEntry metadataEntry, ProtocolEntry protocolEntry) throws Exception {
        try (Stream addFileEntries = this.transactionLogAccess.getActiveFiles(activeDataFileCacheSession, this.tableSnapshot, metadataEntry, protocolEntry, TupleDomain.all(), (Predicate)Predicates.alwaysTrue());){
            Assertions.assertThat((long)addFileEntries.count()).isEqualTo(12L);
        }
    }

    private /* synthetic */ void lambda$testTableSnapshotsActiveDataFilesCache$0(String tableName, String tableDir, DeltaLakeConfig shortLivedActiveDataFilesCacheConfig, ConnectorSession activeDataFileCacheSession, MetadataEntry metadataEntry, ProtocolEntry protocolEntry) throws Exception {
        this.setupTransactionLogAccess(tableName, tableDir, shortLivedActiveDataFilesCacheConfig, Optional.empty());
        try (Stream addFileEntries = this.transactionLogAccess.getActiveFiles(activeDataFileCacheSession, this.tableSnapshot, metadataEntry, protocolEntry, TupleDomain.all(), (Predicate)Predicates.alwaysTrue());){
            Assertions.assertThat((long)addFileEntries.count()).isEqualTo(12L);
        }
    }

    private /* synthetic */ void lambda$testIncrementalCacheUpdates$1(ConnectorSession activeDataFileCacheSession, String tableName, File tableDir, MetadataEntry metadataEntry, ProtocolEntry protocolEntry, Set originalDataFiles, Set newDataFiles) throws Exception {
        TableSnapshot updatedTableSnapshot = this.transactionLogAccess.loadSnapshot(activeDataFileCacheSession, new SchemaTableName("schema", tableName), tableDir.toURI().toString(), Optional.empty());
        try (Stream activeDataFiles = this.transactionLogAccess.getActiveFiles(activeDataFileCacheSession, updatedTableSnapshot, metadataEntry, protocolEntry, TupleDomain.all(), (Predicate)Predicates.alwaysTrue());){
            Assertions.assertThat((Collection)activeDataFiles.map(AddFileEntry::getPath).collect(Collectors.toSet())).containsExactlyInAnyOrderElementsOf((Iterable)Sets.union((Set)originalDataFiles, (Set)newDataFiles));
        }
    }

    private /* synthetic */ void lambda$testIncrementalCacheUpdates$0(String tableName, File tableDir, ConnectorSession activeDataFileCacheSession, MetadataEntry metadataEntry, ProtocolEntry protocolEntry, Set originalDataFiles) throws Exception {
        this.setupTransactionLogAccess(tableName, tableDir.toURI().toString());
        try (Stream activeDataFiles = this.transactionLogAccess.getActiveFiles(activeDataFileCacheSession, this.tableSnapshot, metadataEntry, protocolEntry, TupleDomain.all(), (Predicate)Predicates.alwaysTrue());){
            Assertions.assertThat((Collection)activeDataFiles.map(AddFileEntry::getPath).collect(Collectors.toSet())).containsExactlyInAnyOrderElementsOf((Iterable)originalDataFiles);
        }
    }

    private record FileOperation(String path, String operationType) {
        FileOperation {
            Objects.requireNonNull(path, "path is null");
            Objects.requireNonNull(operationType, "operationType is null");
        }
    }
}

