/*
 * Decompiled with CFR 0.152.
 */
package io.trino.plugin.exchange.filesystem.azure;

import com.azure.core.credential.TokenCredential;
import com.azure.core.http.rest.PagedResponse;
import com.azure.core.util.BinaryData;
import com.azure.identity.DefaultAzureCredentialBuilder;
import com.azure.storage.blob.BlobContainerAsyncClient;
import com.azure.storage.blob.BlobServiceAsyncClient;
import com.azure.storage.blob.BlobServiceClientBuilder;
import com.azure.storage.blob.batch.BlobBatchAsyncClient;
import com.azure.storage.blob.batch.BlobBatchClientBuilder;
import com.azure.storage.blob.models.BlobItem;
import com.azure.storage.blob.models.BlobRange;
import com.azure.storage.blob.models.DeleteSnapshotsOptionType;
import com.azure.storage.blob.models.ListBlobsOptions;
import com.azure.storage.blob.specialized.BlockBlobAsyncClient;
import com.azure.storage.common.policy.RequestRetryOptions;
import com.azure.storage.common.policy.RetryPolicyType;
import com.google.common.base.Preconditions;
import com.google.common.base.Strings;
import com.google.common.base.Verify;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.ImmutableMultimap;
import com.google.common.collect.Lists;
import com.google.common.util.concurrent.FluentFuture;
import com.google.common.util.concurrent.FutureCallback;
import com.google.common.util.concurrent.Futures;
import com.google.common.util.concurrent.ListenableFuture;
import com.google.common.util.concurrent.MoreExecutors;
import com.google.errorprone.annotations.ThreadSafe;
import com.google.errorprone.annotations.concurrent.GuardedBy;
import com.google.inject.Inject;
import io.airlift.concurrent.MoreFutures;
import io.airlift.slice.SizeOf;
import io.airlift.slice.Slice;
import io.airlift.slice.SliceInput;
import io.airlift.slice.Slices;
import io.trino.annotation.NotThreadSafe;
import io.trino.plugin.exchange.filesystem.ExchangeSourceFile;
import io.trino.plugin.exchange.filesystem.ExchangeStorageReader;
import io.trino.plugin.exchange.filesystem.ExchangeStorageWriter;
import io.trino.plugin.exchange.filesystem.FileStatus;
import io.trino.plugin.exchange.filesystem.FileSystemExchangeFutures;
import io.trino.plugin.exchange.filesystem.FileSystemExchangeStorage;
import io.trino.plugin.exchange.filesystem.MetricsBuilder;
import io.trino.plugin.exchange.filesystem.azure.ExchangeAzureConfig;
import jakarta.annotation.PreDestroy;
import java.io.IOException;
import java.net.URI;
import java.net.URISyntaxException;
import java.nio.ByteBuffer;
import java.nio.charset.StandardCharsets;
import java.util.ArrayDeque;
import java.util.ArrayList;
import java.util.Base64;
import java.util.Collection;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.Queue;
import java.util.UUID;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.Executor;
import reactor.core.publisher.Flux;

public class AzureBlobFileSystemExchangeStorage
implements FileSystemExchangeStorage {
    private final int blockSize;
    private final BlobServiceAsyncClient blobServiceAsyncClient;

    @Inject
    public AzureBlobFileSystemExchangeStorage(ExchangeAzureConfig config) {
        this.blockSize = Math.toIntExact(config.getAzureStorageBlockSize().toBytes());
        BlobServiceClientBuilder blobServiceClientBuilder = new BlobServiceClientBuilder().retryOptions(new RequestRetryOptions(RetryPolicyType.EXPONENTIAL, Integer.valueOf(config.getMaxErrorRetries()), (Integer)null, null, null, null));
        Optional<String> connectionString = config.getAzureStorageConnectionString();
        Optional<String> endpoint = config.getAzureStorageEndpoint();
        if (connectionString.isEmpty() && endpoint.isEmpty() || connectionString.isPresent() && endpoint.isPresent()) {
            throw new IllegalArgumentException("Exactly one of exchange.azure.endpoint or exchange.azure.connection-string must be provided");
        }
        if (connectionString.isPresent()) {
            blobServiceClientBuilder.connectionString(connectionString.get());
        } else {
            blobServiceClientBuilder.endpoint(endpoint.get());
            blobServiceClientBuilder.credential((TokenCredential)new DefaultAzureCredentialBuilder().build());
        }
        this.blobServiceAsyncClient = blobServiceClientBuilder.buildAsyncClient();
    }

    @Override
    public void createDirectories(URI dir) throws IOException {
    }

    @Override
    public ExchangeStorageReader createExchangeStorageReader(List<ExchangeSourceFile> sourceFiles, int maxPageStorageSize, MetricsBuilder metricsBuilder) {
        return new AzureExchangeStorageReader(this.blobServiceAsyncClient, sourceFiles, metricsBuilder, this.blockSize, maxPageStorageSize);
    }

    @Override
    public ExchangeStorageWriter createExchangeStorageWriter(URI file) {
        String containerName = AzureBlobFileSystemExchangeStorage.getContainerName(file);
        String blobName = AzureBlobFileSystemExchangeStorage.getPath(file);
        BlockBlobAsyncClient blockBlobAsyncClient = this.blobServiceAsyncClient.getBlobContainerAsyncClient(containerName).getBlobAsyncClient(blobName).getBlockBlobAsyncClient();
        return new AzureExchangeStorageWriter(blockBlobAsyncClient, this.blockSize);
    }

    @Override
    public ListenableFuture<Void> createEmptyFile(URI file) {
        String containerName = AzureBlobFileSystemExchangeStorage.getContainerName(file);
        String blobName = AzureBlobFileSystemExchangeStorage.getPath(file);
        return FileSystemExchangeFutures.translateFailures(MoreFutures.toListenableFuture((CompletableFuture)this.blobServiceAsyncClient.getBlobContainerAsyncClient(containerName).getBlobAsyncClient(blobName).upload(BinaryData.fromString((String)"")).toFuture()));
    }

    @Override
    public ListenableFuture<Void> deleteRecursively(List<URI> directories) {
        ImmutableMultimap.Builder containerToListObjectsFuturesBuilder = ImmutableMultimap.builder();
        directories.forEach(dir -> containerToListObjectsFuturesBuilder.put((Object)AzureBlobFileSystemExchangeStorage.getContainerName(dir), this.listObjectsRecursively((URI)dir)));
        ImmutableMultimap containerToListObjectsFutures = containerToListObjectsFuturesBuilder.build();
        ImmutableList.Builder deleteObjectsFutures = ImmutableList.builder();
        for (String containerName : containerToListObjectsFutures.keySet()) {
            BlobContainerAsyncClient blobContainerAsyncClient = this.blobServiceAsyncClient.getBlobContainerAsyncClient(containerName);
            deleteObjectsFutures.add((Object)Futures.transformAsync((ListenableFuture)Futures.allAsList((Iterable)containerToListObjectsFutures.get((Object)containerName)), nestedPagedResponseList -> {
                ImmutableList.Builder blobUrls = ImmutableList.builder();
                for (List pagedResponseList : nestedPagedResponseList) {
                    for (PagedResponse pagedResponse : pagedResponseList) {
                        pagedResponse.getValue().forEach(blobItem -> blobUrls.add((Object)blobContainerAsyncClient.getBlobAsyncClient(blobItem.getName()).getBlobUrl()));
                    }
                }
                return this.deleteObjects((List<String>)blobUrls.build());
            }, (Executor)MoreExecutors.directExecutor()));
        }
        return FileSystemExchangeFutures.translateFailures(Futures.allAsList((Iterable)deleteObjectsFutures.build()));
    }

    @Override
    public ListenableFuture<List<FileStatus>> listFilesRecursively(URI dir) {
        return Futures.transform(this.listObjectsRecursively(dir), pagedResponseList -> {
            ImmutableList.Builder fileStatuses = ImmutableList.builder();
            for (PagedResponse pagedResponse : pagedResponseList) {
                for (BlobItem blobItem : pagedResponse.getValue()) {
                    URI uri;
                    if (blobItem.isPrefix().equals(Boolean.TRUE)) continue;
                    try {
                        uri = new URI(dir.getScheme(), dir.getUserInfo(), dir.getHost(), -1, "/" + blobItem.getName(), null, dir.getFragment());
                    }
                    catch (URISyntaxException e) {
                        throw new IllegalArgumentException(e);
                    }
                    fileStatuses.add((Object)new FileStatus(uri.toString(), blobItem.getProperties().getContentLength()));
                }
            }
            return fileStatuses.build();
        }, (Executor)MoreExecutors.directExecutor());
    }

    @Override
    public int getWriteBufferSize() {
        return this.blockSize;
    }

    @Override
    @PreDestroy
    public void close() throws IOException {
    }

    private ListenableFuture<List<PagedResponse<BlobItem>>> listObjectsRecursively(URI dir) {
        Preconditions.checkArgument((boolean)AzureBlobFileSystemExchangeStorage.isDirectory(dir), (String)"listObjectsRecursively called on file uri %s", (Object)dir);
        String containerName = AzureBlobFileSystemExchangeStorage.getContainerName(dir);
        String directoryPath = AzureBlobFileSystemExchangeStorage.getPath(dir);
        return MoreFutures.toListenableFuture((CompletableFuture)this.blobServiceAsyncClient.getBlobContainerAsyncClient(containerName).listBlobsByHierarchy(null, new ListBlobsOptions().setPrefix(directoryPath)).byPage().collectList().toFuture());
    }

    private ListenableFuture<List<Void>> deleteObjects(List<String> blobUrls) {
        BlobBatchAsyncClient blobBatchAsyncClient = new BlobBatchClientBuilder(this.blobServiceAsyncClient).buildAsyncClient();
        return Futures.allAsList((Iterable)((Iterable)Lists.partition(blobUrls, (int)256).stream().map(list -> MoreFutures.toListenableFuture((CompletableFuture)blobBatchAsyncClient.deleteBlobs(list, DeleteSnapshotsOptionType.INCLUDE).then().toFuture())).collect(ImmutableList.toImmutableList())));
    }

    private static String getContainerName(URI uri) {
        return uri.getUserInfo();
    }

    private static String getPath(URI uri) {
        Preconditions.checkArgument((boolean)uri.isAbsolute(), (String)"Uri is not absolute: %s", (Object)uri);
        String blobName = Strings.nullToEmpty((String)uri.getPath());
        if (blobName.startsWith("/")) {
            blobName = blobName.substring("/".length());
        }
        if (blobName.endsWith("/")) {
            blobName = blobName.substring(0, blobName.length() - "/".length());
        }
        return blobName;
    }

    private static boolean isDirectory(URI uri) {
        return uri.toString().endsWith("/");
    }

    @ThreadSafe
    private static class AzureExchangeStorageReader
    implements ExchangeStorageReader {
        private static final int INSTANCE_SIZE = SizeOf.instanceSize(AzureExchangeStorageReader.class);
        private final BlobServiceAsyncClient blobServiceAsyncClient;
        @GuardedBy(value="this")
        private final Queue<ExchangeSourceFile> sourceFiles;
        private final int blockSize;
        private final int bufferSize;
        MetricsBuilder.CounterMetricBuilder sourceFilesProcessedMetric;
        @GuardedBy(value="this")
        private ExchangeSourceFile currentFile;
        @GuardedBy(value="this")
        private long fileOffset;
        @GuardedBy(value="this")
        private SliceInput sliceInput;
        @GuardedBy(value="this")
        private int sliceSize = -1;
        private volatile boolean closed;
        private volatile long bufferRetainedSize;
        private volatile ListenableFuture<Void> inProgressReadFuture = Futures.immediateVoidFuture();

        public AzureExchangeStorageReader(BlobServiceAsyncClient blobServiceAsyncClient, List<ExchangeSourceFile> sourceFiles, MetricsBuilder metricsBuilder, int blockSize, int maxPageStorageSize) {
            this.blobServiceAsyncClient = Objects.requireNonNull(blobServiceAsyncClient, "blobServiceAsyncClient is null");
            this.sourceFiles = new ArrayDeque<ExchangeSourceFile>((Collection)Objects.requireNonNull(sourceFiles, "sourceFiles is null"));
            Objects.requireNonNull(metricsBuilder, "metricsBuilder is null");
            this.sourceFilesProcessedMetric = metricsBuilder.getCounterMetric("FileSystemExchangeSource.filesProcessed");
            this.blockSize = blockSize;
            this.bufferSize = maxPageStorageSize + blockSize;
            this.fillBuffer();
        }

        @Override
        public synchronized Slice read() throws IOException {
            if (this.closed || !this.inProgressReadFuture.isDone()) {
                return null;
            }
            try {
                MoreFutures.getFutureValue(this.inProgressReadFuture);
            }
            catch (RuntimeException e) {
                throw new IOException(e);
            }
            if (this.sliceSize < 0) {
                this.sliceSize = this.sliceInput.readInt();
            }
            Slice data = this.sliceInput.readSlice(this.sliceSize);
            if (this.sliceInput.available() > 4) {
                this.sliceSize = this.sliceInput.readInt();
                if (this.sliceInput.available() < this.sliceSize) {
                    this.fillBuffer();
                }
            } else {
                this.sliceSize = -1;
                this.fillBuffer();
            }
            return data;
        }

        @Override
        public ListenableFuture<Void> isBlocked() {
            return this.inProgressReadFuture;
        }

        @Override
        public synchronized long getRetainedSize() {
            return (long)INSTANCE_SIZE + this.bufferRetainedSize;
        }

        @Override
        public boolean isFinished() {
            return this.closed;
        }

        @Override
        public synchronized void close() {
            if (this.closed) {
                return;
            }
            this.closed = true;
            this.currentFile = null;
            this.sliceInput = null;
            this.bufferRetainedSize = 0L;
            this.inProgressReadFuture.cancel(true);
            this.inProgressReadFuture = Futures.immediateVoidFuture();
        }

        @GuardedBy(value="this")
        private void fillBuffer() {
            if (this.currentFile == null || this.fileOffset == this.currentFile.getFileSize()) {
                this.currentFile = this.sourceFiles.poll();
                if (this.currentFile == null) {
                    this.close();
                    return;
                }
                this.fileOffset = 0L;
            }
            byte[] buffer = new byte[this.bufferSize];
            int bufferFill = 0;
            if (this.sliceInput != null) {
                int length = this.sliceInput.available();
                this.sliceInput.readBytes(buffer, 0, length);
                bufferFill += length;
            }
            ImmutableList.Builder downloadFutures = ImmutableList.builder();
            while (true) {
                long fileSize = this.currentFile.getFileSize();
                int readableBlocks = (buffer.length - bufferFill) / this.blockSize;
                if (readableBlocks == 0) {
                    if ((long)(buffer.length - bufferFill) < fileSize - this.fileOffset) break;
                    readableBlocks = 1;
                }
                BlockBlobAsyncClient blockBlobAsyncClient = this.blobServiceAsyncClient.getBlobContainerAsyncClient(AzureBlobFileSystemExchangeStorage.getContainerName(this.currentFile.getFileUri())).getBlobAsyncClient(AzureBlobFileSystemExchangeStorage.getPath(this.currentFile.getFileUri())).getBlockBlobAsyncClient();
                for (int i = 0; i < readableBlocks && this.fileOffset < fileSize; ++i) {
                    int length = (int)Math.min((long)this.blockSize, fileSize - this.fileOffset);
                    int finalBufferFill = bufferFill;
                    FluentFuture downloadFuture = FluentFuture.from((ListenableFuture)MoreFutures.toListenableFuture((CompletableFuture)blockBlobAsyncClient.downloadStreamWithResponse(new BlobRange(this.fileOffset, Long.valueOf(length)), null, null, false).toFuture())).transformAsync(response -> MoreFutures.toListenableFuture((CompletableFuture)((Flux)response.getValue()).collectList().toFuture()), MoreExecutors.directExecutor()).transform(byteBuffers -> {
                        int offset = finalBufferFill;
                        for (ByteBuffer byteBuffer : byteBuffers) {
                            int readableBytes = byteBuffer.remaining();
                            if (byteBuffer.hasArray()) {
                                System.arraycopy(byteBuffer.array(), byteBuffer.arrayOffset() + byteBuffer.position(), buffer, offset, readableBytes);
                            } else {
                                byteBuffer.asReadOnlyBuffer().get(buffer, offset, readableBytes);
                            }
                            offset += readableBytes;
                        }
                        return null;
                    }, MoreExecutors.directExecutor());
                    downloadFutures.add((Object)downloadFuture);
                    bufferFill += length;
                    this.fileOffset += (long)length;
                }
                if (this.fileOffset != fileSize) continue;
                this.sourceFilesProcessedMetric.increment();
                this.currentFile = this.sourceFiles.poll();
                if (this.currentFile == null) break;
                this.fileOffset = 0L;
            }
            this.inProgressReadFuture = MoreFutures.asVoid((ListenableFuture)Futures.allAsList((Iterable)downloadFutures.build()));
            this.sliceInput = Slices.wrappedBuffer((byte[])buffer, (int)0, (int)bufferFill).getInput();
            this.bufferRetainedSize = this.sliceInput.getRetainedSize();
        }
    }

    @NotThreadSafe
    private static class AzureExchangeStorageWriter
    implements ExchangeStorageWriter {
        private static final int INSTANCE_SIZE = SizeOf.instanceSize(AzureExchangeStorageWriter.class);
        private final BlockBlobAsyncClient blockBlobAsyncClient;
        private final int blockSize;
        private ListenableFuture<Void> directUploadFuture;
        private final List<ListenableFuture<Void>> multiPartUploadFutures = new ArrayList<ListenableFuture<Void>>();
        private final List<String> blockIds = new ArrayList<String>();
        private volatile boolean closed;

        public AzureExchangeStorageWriter(BlockBlobAsyncClient blockBlobAsyncClient, int blockSize) {
            this.blockBlobAsyncClient = Objects.requireNonNull(blockBlobAsyncClient, "blockBlobAsyncClient is null");
            this.blockSize = blockSize;
        }

        @Override
        public ListenableFuture<Void> write(Slice slice) {
            Preconditions.checkState((this.directUploadFuture == null ? 1 : 0) != 0, (Object)"Direct upload already started");
            if (this.closed) {
                return Futures.immediateVoidFuture();
            }
            if (slice.length() < this.blockSize && this.multiPartUploadFutures.isEmpty()) {
                this.directUploadFuture = FileSystemExchangeFutures.translateFailures(MoreFutures.toListenableFuture((CompletableFuture)this.blockBlobAsyncClient.upload(Flux.just((Object)slice.toByteBuffer()), (long)slice.length()).toFuture()));
                return this.directUploadFuture;
            }
            String blockId = Base64.getEncoder().encodeToString(UUID.randomUUID().toString().getBytes(StandardCharsets.UTF_8));
            ListenableFuture uploadFuture = MoreFutures.toListenableFuture((CompletableFuture)this.blockBlobAsyncClient.stageBlock(blockId, Flux.just((Object)slice.toByteBuffer()), (long)slice.length()).toFuture());
            this.multiPartUploadFutures.add((ListenableFuture<Void>)uploadFuture);
            this.blockIds.add(blockId);
            return FileSystemExchangeFutures.translateFailures(uploadFuture);
        }

        @Override
        public ListenableFuture<Void> finish() {
            if (this.closed) {
                return Futures.immediateVoidFuture();
            }
            if (this.multiPartUploadFutures.isEmpty()) {
                return Objects.requireNonNullElseGet(this.directUploadFuture, Futures::immediateVoidFuture);
            }
            ListenableFuture<Void> finishFuture = FileSystemExchangeFutures.translateFailures(Futures.transformAsync((ListenableFuture)Futures.allAsList(this.multiPartUploadFutures), list -> MoreFutures.toListenableFuture((CompletableFuture)this.blockBlobAsyncClient.commitBlockList(this.blockIds).toFuture()), (Executor)MoreExecutors.directExecutor()));
            Futures.addCallback(finishFuture, (FutureCallback)new FutureCallback<Void>(){

                public void onSuccess(Void result) {
                    closed = true;
                }

                public void onFailure(Throwable ignored) {
                }
            }, (Executor)MoreExecutors.directExecutor());
            return finishFuture;
        }

        @Override
        public ListenableFuture<Void> abort() {
            if (this.closed) {
                return Futures.immediateVoidFuture();
            }
            this.closed = true;
            if (this.multiPartUploadFutures.isEmpty()) {
                if (this.directUploadFuture != null) {
                    this.directUploadFuture.cancel(true);
                }
                return Futures.immediateVoidFuture();
            }
            Verify.verify((this.directUploadFuture == null ? 1 : 0) != 0);
            this.multiPartUploadFutures.forEach(future -> future.cancel(true));
            return Futures.immediateVoidFuture();
        }

        @Override
        public long getRetainedSize() {
            return (long)INSTANCE_SIZE + SizeOf.estimatedSizeOf(this.blockIds, SizeOf::estimatedSizeOf);
        }
    }
}

