/*
 * Decompiled with CFR 0.152.
 */
package io.trino.execution;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.Ordering;
import com.google.inject.Inject;
import io.airlift.log.Logger;
import io.trino.FeaturesConfig;
import io.trino.execution.SqlTask;
import io.trino.execution.SqlTaskManager;
import io.trino.execution.TaskManagementExecutor;
import io.trino.execution.TaskState;
import io.trino.memory.LocalMemoryManager;
import io.trino.memory.MemoryPool;
import io.trino.memory.MemoryPoolListener;
import io.trino.memory.TraversingQueryContextVisitor;
import io.trino.memory.VoidTraversingQueryContextVisitor;
import io.trino.operator.OperatorContext;
import io.trino.operator.PipelineContext;
import io.trino.operator.TaskContext;
import jakarta.annotation.Nullable;
import jakarta.annotation.PostConstruct;
import jakarta.annotation.PreDestroy;
import java.util.Collection;
import java.util.Comparator;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicLong;
import java.util.function.Supplier;

public class MemoryRevokingScheduler {
    private static final Logger log = Logger.get(MemoryRevokingScheduler.class);
    private static final Ordering<SqlTask> ORDER_BY_CREATE_TIME = Ordering.natural().onResultOf(SqlTask::getTaskCreatedTime);
    private final MemoryPool memoryPool;
    private final Supplier<? extends Collection<SqlTask>> currentTasksSupplier;
    private final ScheduledExecutorService taskManagementExecutor;
    private final double memoryRevokingThreshold;
    private final double memoryRevokingTarget;
    private final MemoryPoolListener memoryPoolListener = MemoryPoolListener.onMemoryReserved(this::onMemoryReserved);
    @Nullable
    private ScheduledFuture<?> scheduledFuture;
    private final AtomicBoolean checkPending = new AtomicBoolean();

    @Inject
    public MemoryRevokingScheduler(LocalMemoryManager localMemoryManager, SqlTaskManager sqlTaskManager, TaskManagementExecutor taskManagementExecutor, FeaturesConfig config) {
        this(localMemoryManager.getMemoryPool(), sqlTaskManager::getAllTasks, taskManagementExecutor.getExecutor(), config.getMemoryRevokingThreshold(), config.getMemoryRevokingTarget());
    }

    @VisibleForTesting
    MemoryRevokingScheduler(MemoryPool memoryPool, Supplier<? extends Collection<SqlTask>> currentTasksSupplier, ScheduledExecutorService taskManagementExecutor, double memoryRevokingThreshold, double memoryRevokingTarget) {
        this.memoryPool = Objects.requireNonNull(memoryPool, "memoryPool is null");
        this.currentTasksSupplier = Objects.requireNonNull(currentTasksSupplier, "currentTasksSupplier is null");
        this.taskManagementExecutor = Objects.requireNonNull(taskManagementExecutor, "taskManagementExecutor is null");
        this.memoryRevokingThreshold = MemoryRevokingScheduler.checkFraction(memoryRevokingThreshold, "memoryRevokingThreshold");
        this.memoryRevokingTarget = MemoryRevokingScheduler.checkFraction(memoryRevokingTarget, "memoryRevokingTarget");
        Preconditions.checkArgument((memoryRevokingTarget <= memoryRevokingThreshold ? 1 : 0) != 0, (String)"memoryRevokingTarget should be less than or equal memoryRevokingThreshold, but got %s and %s respectively", (Object)memoryRevokingTarget, (Object)memoryRevokingThreshold);
    }

    private static double checkFraction(double value, String valueName) {
        Objects.requireNonNull(valueName, "valueName is null");
        Preconditions.checkArgument((0.0 <= value && value <= 1.0 ? 1 : 0) != 0, (String)"%s should be within [0, 1] range, got %s", (Object)valueName, (Object)value);
        return value;
    }

    @PostConstruct
    public void start() {
        this.registerPeriodicCheck();
        this.registerPoolListeners();
    }

    private void registerPeriodicCheck() {
        this.scheduledFuture = this.taskManagementExecutor.scheduleWithFixedDelay(() -> {
            try {
                this.requestMemoryRevokingIfNeeded();
            }
            catch (Throwable e) {
                log.error(e, "Error requesting system memory revoking");
            }
        }, 1L, 1L, TimeUnit.SECONDS);
    }

    @PreDestroy
    public void stop() {
        if (this.scheduledFuture != null) {
            this.scheduledFuture.cancel(true);
            this.scheduledFuture = null;
        }
        this.memoryPool.removeListener(this.memoryPoolListener);
    }

    @VisibleForTesting
    void registerPoolListeners() {
        this.memoryPool.addListener(this.memoryPoolListener);
    }

    private void onMemoryReserved(MemoryPool memoryPool) {
        try {
            if (!this.memoryRevokingNeeded(memoryPool)) {
                return;
            }
            if (this.checkPending.compareAndSet(false, true)) {
                log.debug("Scheduling check for %s", new Object[]{memoryPool});
                this.scheduleRevoking();
            }
        }
        catch (Throwable e) {
            log.error(e, "Error when acting on memory pool reservation");
        }
    }

    @VisibleForTesting
    void requestMemoryRevokingIfNeeded() {
        if (this.checkPending.compareAndSet(false, true)) {
            this.runMemoryRevoking();
        }
    }

    private void scheduleRevoking() {
        this.taskManagementExecutor.execute(() -> {
            try {
                this.runMemoryRevoking();
            }
            catch (Throwable e) {
                log.error(e, "Error requesting memory revoking");
            }
        });
    }

    private synchronized void runMemoryRevoking() {
        if (this.checkPending.getAndSet(false)) {
            if (!this.memoryRevokingNeeded(this.memoryPool)) {
                return;
            }
            this.requestMemoryRevoking(this.memoryPool, Objects.requireNonNull(this.currentTasksSupplier.get()));
        }
    }

    private void requestMemoryRevoking(MemoryPool memoryPool, Collection<SqlTask> allTasks) {
        long remainingBytesToRevoke = (long)((double)(-memoryPool.getFreeBytes()) + (double)memoryPool.getMaxBytes() * (1.0 - this.memoryRevokingTarget));
        List<SqlTask> runningTasksInPool = MemoryRevokingScheduler.findRunningTasksInMemoryPool(allTasks, memoryPool);
        if ((remainingBytesToRevoke -= this.getMemoryAlreadyBeingRevoked(runningTasksInPool, remainingBytesToRevoke)) > 0L) {
            this.requestRevoking(runningTasksInPool, remainingBytesToRevoke);
        }
    }

    private boolean memoryRevokingNeeded(MemoryPool memoryPool) {
        return memoryPool.getReservedRevocableBytes() > 0L && (double)memoryPool.getFreeBytes() <= (double)memoryPool.getMaxBytes() * (1.0 - this.memoryRevokingThreshold);
    }

    private long getMemoryAlreadyBeingRevoked(List<SqlTask> sqlTasks, long targetRevokingLimit) {
        TraversingQueryContextVisitor<Void, Long> visitor = new TraversingQueryContextVisitor<Void, Long>(this){

            @Override
            public Long visitOperatorContext(OperatorContext operatorContext, Void context) {
                if (operatorContext.isMemoryRevokingRequested()) {
                    return operatorContext.getReservedRevocableBytes();
                }
                return 0L;
            }

            @Override
            public Long mergeResults(List<Long> childrenResults) {
                return childrenResults.stream().mapToLong(i -> i).sum();
            }
        };
        long currentRevoking = 0L;
        for (SqlTask task : sqlTasks) {
            Optional<TaskContext> taskContext = task.getTaskContext();
            if (!taskContext.isPresent() || (currentRevoking += taskContext.get().accept(visitor, null).longValue()) < targetRevokingLimit) continue;
            return currentRevoking;
        }
        return currentRevoking;
    }

    private void requestRevoking(List<SqlTask> sqlTasks, long remainingBytesToRevoke) {
        VoidTraversingQueryContextVisitor<AtomicLong> visitor = new VoidTraversingQueryContextVisitor<AtomicLong>(this){

            @Override
            public Void visitPipelineContext(PipelineContext pipelineContext, AtomicLong remainingBytesToRevoke) {
                if (remainingBytesToRevoke.get() <= 0L) {
                    return null;
                }
                return (Void)super.visitPipelineContext(pipelineContext, remainingBytesToRevoke);
            }

            @Override
            public Void visitOperatorContext(OperatorContext operatorContext, AtomicLong remainingBytesToRevoke) {
                long revokedBytes;
                if (remainingBytesToRevoke.get() > 0L && (revokedBytes = operatorContext.requestMemoryRevoking()) > 0L) {
                    remainingBytesToRevoke.addAndGet(-revokedBytes);
                    log.debug("requested revoking %s; remaining %s", new Object[]{revokedBytes, remainingBytesToRevoke.get()});
                }
                return null;
            }
        };
        AtomicLong remainingBytesToRevokeAtomic = new AtomicLong(remainingBytesToRevoke);
        for (SqlTask task : sqlTasks) {
            Optional<TaskContext> taskContext = task.getTaskContext();
            if (!taskContext.isPresent()) continue;
            taskContext.get().accept(visitor, remainingBytesToRevokeAtomic);
            if (remainingBytesToRevokeAtomic.get() > 0L) continue;
            return;
        }
    }

    private static List<SqlTask> findRunningTasksInMemoryPool(Collection<SqlTask> allCurrentTasks, MemoryPool memoryPool) {
        return (List)allCurrentTasks.stream().filter(task -> task.getTaskState() == TaskState.RUNNING && task.getQueryContext().getMemoryPool() == memoryPool).sorted((Comparator<SqlTask>)ORDER_BY_CREATE_TIME).collect(ImmutableList.toImmutableList());
    }
}

