/*
 * Decompiled with CFR 0.152.
 */
package io.trino.filesystem.alluxio;

import alluxio.AlluxioURI;
import alluxio.client.file.FileSystem;
import alluxio.client.file.URIStatus;
import alluxio.exception.AlluxioException;
import alluxio.exception.FileDoesNotExistException;
import alluxio.exception.runtime.NotFoundRuntimeException;
import alluxio.grpc.CreateDirectoryPOptions;
import alluxio.grpc.DeletePOptions;
import alluxio.grpc.ListStatusPOptions;
import io.trino.filesystem.FileIterator;
import io.trino.filesystem.Location;
import io.trino.filesystem.TrinoFileSystem;
import io.trino.filesystem.TrinoInputFile;
import io.trino.filesystem.TrinoOutputFile;
import io.trino.filesystem.alluxio.AlluxioFileIterator;
import io.trino.filesystem.alluxio.AlluxioFileSystemInputFile;
import io.trino.filesystem.alluxio.AlluxioFileSystemOutputFile;
import io.trino.filesystem.alluxio.AlluxioUtils;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.time.Instant;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.UUID;
import java.util.stream.Collectors;

public class AlluxioFileSystem
implements TrinoFileSystem {
    private final FileSystem alluxioClient;
    private final String mountRoot;
    private final Location rootLocation;

    public AlluxioFileSystem(FileSystem alluxioClient) {
        this(alluxioClient, "/", Location.of((String)"alluxio:///"));
    }

    public AlluxioFileSystem(FileSystem alluxioClient, String mountRoot, Location rootLocation) {
        this.alluxioClient = Objects.requireNonNull(alluxioClient, "filesystem is null");
        this.mountRoot = mountRoot;
        this.rootLocation = Objects.requireNonNull(rootLocation, "rootLocation is null");
    }

    public String getMountRoot() {
        return this.mountRoot;
    }

    public TrinoInputFile newInputFile(Location location) {
        this.ensureNotRootLocation(location);
        this.ensureNotEndWithSlash(location);
        return new AlluxioFileSystemInputFile(location, null, this.alluxioClient, this.mountRoot, Optional.empty());
    }

    public TrinoInputFile newInputFile(Location location, long length) {
        this.ensureNotRootLocation(location);
        this.ensureNotEndWithSlash(location);
        return new AlluxioFileSystemInputFile(location, length, this.alluxioClient, this.mountRoot, Optional.empty());
    }

    public TrinoInputFile newInputFile(Location location, long length, Instant lastModified) {
        this.ensureNotRootLocation(location);
        this.ensureNotEndWithSlash(location);
        return new AlluxioFileSystemInputFile(location, length, this.alluxioClient, this.mountRoot, Optional.of(lastModified));
    }

    public TrinoOutputFile newOutputFile(Location location) {
        this.ensureNotRootLocation(location);
        this.ensureNotEndWithSlash(location);
        return new AlluxioFileSystemOutputFile(this.rootLocation, location, this.alluxioClient, this.mountRoot);
    }

    public void deleteFile(Location location) throws IOException {
        this.ensureNotRootLocation(location);
        this.ensureNotEndWithSlash(location);
        try {
            this.alluxioClient.delete(AlluxioUtils.convertToAlluxioURI(location, this.mountRoot));
        }
        catch (FileDoesNotExistException fileDoesNotExistException) {
        }
        catch (AlluxioException e) {
            throw new IOException("Error deleteFile %s".formatted(location), e);
        }
    }

    public void deleteDirectory(Location location) throws IOException {
        try {
            AlluxioURI uri = AlluxioUtils.convertToAlluxioURI(location, this.mountRoot);
            URIStatus status = this.alluxioClient.getStatus(uri);
            if (status == null) {
                return;
            }
            if (!status.isFolder()) {
                throw new IOException("delete directory cannot be called on a file %s".formatted(location));
            }
            DeletePOptions deletePOptions = DeletePOptions.newBuilder().setRecursive(true).build();
            if (location.path().isEmpty() || location.path().equals(this.mountRoot)) {
                for (URIStatus uriStatus : this.alluxioClient.listStatus(uri)) {
                    this.alluxioClient.delete(new AlluxioURI(uriStatus.getPath()), deletePOptions);
                }
            } else {
                this.alluxioClient.delete(uri, deletePOptions);
            }
        }
        catch (FileDoesNotExistException | NotFoundRuntimeException uri) {
        }
        catch (AlluxioException e) {
            throw new IOException("Error deleteDirectory %s".formatted(location), e);
        }
    }

    public void renameFile(Location source, Location target) throws IOException {
        try {
            this.ensureNotRootLocation(source);
            this.ensureNotEndWithSlash(source);
            this.ensureNotRootLocation(target);
            this.ensureNotEndWithSlash(target);
        }
        catch (IllegalStateException e) {
            throw new IllegalStateException("Cannot rename file from %s to %s as one of them is root location".formatted(source, target), e);
        }
        AlluxioURI sourceUri = AlluxioUtils.convertToAlluxioURI(source, this.mountRoot);
        AlluxioURI targetUri = AlluxioUtils.convertToAlluxioURI(target, this.mountRoot);
        try {
            if (!this.alluxioClient.exists(sourceUri)) {
                throw new IOException("Cannot rename file %s to %s as file %s doesn't exist".formatted(source, target, source));
            }
            if (this.alluxioClient.exists(targetUri)) {
                throw new IOException("Cannot rename file %s to %s as file %s already exists".formatted(source, target, target));
            }
            URIStatus status = this.alluxioClient.getStatus(sourceUri);
            if (status.isFolder()) {
                throw new IOException("Cannot rename file %s to %s as %s is a directory".formatted(source, target, source));
            }
            this.alluxioClient.rename(AlluxioUtils.convertToAlluxioURI(source, this.mountRoot), AlluxioUtils.convertToAlluxioURI(target, this.mountRoot));
        }
        catch (AlluxioException e) {
            throw new IOException("Error renameFile from %s to %s".formatted(source, target), e);
        }
    }

    public FileIterator listFiles(Location location) throws IOException {
        try {
            URIStatus status2 = this.alluxioClient.getStatus(AlluxioUtils.convertToAlluxioURI(location, this.mountRoot));
            if (status2 == null) {
                new AlluxioFileIterator(Collections.emptyList(), AlluxioUtils.getAlluxioBase(location.toString()));
            }
            if (!status2.isFolder()) {
                throw new IOException("Location is not a directory: %s".formatted(location));
            }
        }
        catch (AlluxioException | NotFoundRuntimeException e) {
            return new AlluxioFileIterator(Collections.emptyList(), AlluxioUtils.getAlluxioBase(location.toString()));
        }
        try {
            List filesStatus = this.alluxioClient.listStatus(AlluxioUtils.convertToAlluxioURI(location, this.mountRoot), ListStatusPOptions.newBuilder().setRecursive(true).build());
            return new AlluxioFileIterator(filesStatus.stream().filter(status -> !status.isFolder() & status.isCompleted()).toList(), AlluxioUtils.getAlluxioBase(location.toString()));
        }
        catch (AlluxioException e) {
            throw new IOException("Error listFiles %s".formatted(location), e);
        }
    }

    public Optional<Boolean> directoryExists(Location location) throws IOException {
        if (location.path().isEmpty()) {
            return Optional.of(true);
        }
        try {
            URIStatus status = this.alluxioClient.getStatus(AlluxioUtils.convertToAlluxioURI(location, this.mountRoot));
            if (status != null && status.isFolder()) {
                return Optional.of(true);
            }
            return Optional.of(false);
        }
        catch (FileDoesNotExistException | NotFoundRuntimeException | FileNotFoundException e) {
            return Optional.of(false);
        }
        catch (AlluxioException e) {
            throw new IOException("Error directoryExists %s".formatted(location), e);
        }
    }

    public void createDirectory(Location location) throws IOException {
        try {
            URIStatus status;
            AlluxioURI locationUri = AlluxioUtils.convertToAlluxioURI(location, this.mountRoot);
            if (this.alluxioClient.exists(locationUri) && !(status = this.alluxioClient.getStatus(locationUri)).isFolder()) {
                throw new IOException("Cannot create a directory for an existing file location %s".formatted(location));
            }
            this.alluxioClient.createDirectory(locationUri, CreateDirectoryPOptions.newBuilder().setAllowExists(true).setRecursive(true).build());
        }
        catch (AlluxioException e) {
            throw new IOException("Error createDirectory %s".formatted(location), e);
        }
    }

    public void renameDirectory(Location source, Location target) throws IOException {
        try {
            this.ensureNotRootLocation(source);
            this.ensureNotRootLocation(target);
        }
        catch (IllegalStateException e) {
            throw new IOException("Cannot rename directory from %s to %s as one of them is root location".formatted(source, target), e);
        }
        try {
            if (this.alluxioClient.exists(AlluxioUtils.convertToAlluxioURI(target, this.mountRoot))) {
                throw new IOException("Cannot rename %s to %s as file %s already exists".formatted(source, target, target));
            }
            this.alluxioClient.rename(AlluxioUtils.convertToAlluxioURI(source, this.mountRoot), AlluxioUtils.convertToAlluxioURI(target, this.mountRoot));
        }
        catch (AlluxioException e) {
            throw new IOException("Error renameDirectory from %s to %s".formatted(source, target), e);
        }
    }

    public Set<Location> listDirectories(Location location) throws IOException {
        try {
            if (this.isFile(location)) {
                throw new IOException("Cannot list directories for a file %s".formatted(location));
            }
            List filesStatus = this.alluxioClient.listStatus(AlluxioUtils.convertToAlluxioURI(location, this.mountRoot));
            return filesStatus.stream().filter(URIStatus::isFolder).map(fileStatus -> AlluxioUtils.convertToLocation(fileStatus.getPath(), this.mountRoot)).map(loc -> {
                if (!loc.toString().endsWith("/")) {
                    return Location.of((String)(String.valueOf(loc) + "/"));
                }
                return loc;
            }).collect(Collectors.toSet());
        }
        catch (FileDoesNotExistException | NotFoundRuntimeException | FileNotFoundException e) {
            return Collections.emptySet();
        }
        catch (AlluxioException e) {
            throw new IOException("Error listDirectories %s".formatted(location), e);
        }
    }

    public Optional<Location> createTemporaryDirectory(Location targetPath, String temporaryPrefix, String relativePrefix) throws IOException {
        Location temporary;
        if (temporaryPrefix.startsWith("/")) {
            String prefix = temporaryPrefix;
            while (prefix.startsWith("/")) {
                prefix = prefix.substring(1);
            }
            temporary = targetPath.appendPath(prefix);
        } else {
            temporary = targetPath.appendPath(temporaryPrefix);
        }
        temporary = temporary.appendPath(UUID.randomUUID().toString());
        this.createDirectory(temporary);
        return Optional.of(temporary);
    }

    private void ensureNotRootLocation(Location location) {
        String locationPath = location.path();
        while (locationPath.endsWith("/")) {
            locationPath = locationPath.substring(0, locationPath.length() - 1);
        }
        String rootLocationPath = this.rootLocation.path();
        while (rootLocationPath.endsWith("/")) {
            rootLocationPath = rootLocationPath.substring(0, rootLocationPath.length() - 1);
        }
        if (rootLocationPath.equals(locationPath)) {
            throw new IllegalStateException("Illegal operation on %s".formatted(location));
        }
    }

    private void ensureNotEndWithSlash(Location location) {
        String locationPath = location.path();
        if (locationPath.endsWith("/")) {
            throw new IllegalStateException("Illegal operation on %s".formatted(location));
        }
    }

    private boolean isFile(Location location) {
        try {
            URIStatus status = this.alluxioClient.getStatus(AlluxioUtils.convertToAlluxioURI(location, this.mountRoot));
            if (status == null) {
                return false;
            }
            return !status.isFolder();
        }
        catch (AlluxioException | NotFoundRuntimeException | IOException e) {
            return false;
        }
    }
}

