/*
 * Decompiled with CFR 0.152.
 */
package io.trino.filesystem.alluxio;

import alluxio.client.file.cache.PageId;
import alluxio.client.file.cache.PageStore;
import alluxio.client.file.cache.store.PageStoreOptions;
import alluxio.conf.AlluxioConfiguration;
import com.google.common.base.MoreObjects;
import com.google.common.collect.HashMultiset;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.ImmutableMultiset;
import com.google.common.collect.Iterables;
import com.google.common.collect.Multiset;
import com.google.common.io.MoreFiles;
import com.google.common.io.RecursiveDeleteOption;
import io.airlift.slice.Slices;
import io.airlift.units.DataSize;
import io.opentelemetry.api.common.Attributes;
import io.opentelemetry.sdk.trace.data.SpanData;
import io.trino.filesystem.Location;
import io.trino.filesystem.TrinoFileSystemFactory;
import io.trino.filesystem.TrinoInput;
import io.trino.filesystem.TrinoInputFile;
import io.trino.filesystem.alluxio.AlluxioCacheStats;
import io.trino.filesystem.alluxio.AlluxioConfigurationFactory;
import io.trino.filesystem.alluxio.AlluxioFileSystemCache;
import io.trino.filesystem.alluxio.AlluxioFileSystemCacheConfig;
import io.trino.filesystem.alluxio.TestingCacheKeyProvider;
import io.trino.filesystem.cache.CacheFileSystem;
import io.trino.filesystem.cache.CacheKeyProvider;
import io.trino.filesystem.cache.TrinoFileSystemCache;
import io.trino.filesystem.memory.MemoryFileSystemFactory;
import io.trino.filesystem.tracing.CacheSystemAttributes;
import io.trino.filesystem.tracing.TracingFileSystemCache;
import io.trino.filesystem.tracing.TracingFileSystemFactory;
import io.trino.spi.security.ConnectorIdentity;
import io.trino.testing.MultisetAssertions;
import io.trino.testing.TestingTelemetry;
import java.io.IOException;
import java.io.OutputStream;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.attribute.FileAttribute;
import java.util.Arrays;
import java.util.List;
import java.util.stream.Collectors;
import org.assertj.core.api.Assertions;
import org.junit.jupiter.api.AfterAll;
import org.junit.jupiter.api.AfterEach;
import org.junit.jupiter.api.BeforeAll;
import org.junit.jupiter.api.Test;
import org.junit.jupiter.api.TestInstance;

@TestInstance(value=TestInstance.Lifecycle.PER_CLASS)
public class TestAlluxioCacheFileSystemAccessOperations {
    private static final int CACHE_SIZE = 1024;
    private static final int PAGE_SIZE = 128;
    private final TestingTelemetry testingTelemetry = TestingTelemetry.create((String)"alluxio-cache");
    private final TestingCacheKeyProvider cacheKeyProvider = new TestingCacheKeyProvider();
    private TracingFileSystemFactory tracingFileSystemFactory;
    private AlluxioFileSystemCache alluxioCache;
    private CacheFileSystem fileSystem;
    private Path tempDirectory;
    private PageStore pageStore;

    @BeforeAll
    public void setUp() throws IOException {
        this.tempDirectory = Files.createTempDirectory("test", new FileAttribute[0]);
        Path cacheDirectory = Files.createDirectory(this.tempDirectory.resolve("cache"), new FileAttribute[0]);
        AlluxioFileSystemCacheConfig configuration = new AlluxioFileSystemCacheConfig().setCacheDirectories((List)ImmutableList.of((Object)cacheDirectory.toAbsolutePath().toString())).disableTTL().setCachePageSize(DataSize.ofBytes((long)128L)).setMaxCacheSizes((List)ImmutableList.of((Object)DataSize.ofBytes((long)1024L)));
        this.tracingFileSystemFactory = new TracingFileSystemFactory(this.testingTelemetry.getTracer(), (TrinoFileSystemFactory)new MemoryFileSystemFactory());
        this.alluxioCache = new AlluxioFileSystemCache(this.testingTelemetry.getTracer(), configuration, new AlluxioCacheStats());
        this.fileSystem = new CacheFileSystem(this.tracingFileSystemFactory.create(ConnectorIdentity.ofUser((String)"hello")), (TrinoFileSystemCache)new TracingFileSystemCache(this.testingTelemetry.getTracer(), (TrinoFileSystemCache)this.alluxioCache), (CacheKeyProvider)this.cacheKeyProvider);
        this.pageStore = PageStore.create((PageStoreOptions)((PageStoreOptions)Iterables.getOnlyElement((Iterable)PageStoreOptions.create((AlluxioConfiguration)AlluxioConfigurationFactory.create((AlluxioFileSystemCacheConfig)configuration)))));
    }

    @AfterAll
    public void tearDown() throws Exception {
        this.tracingFileSystemFactory = null;
        this.fileSystem = null;
        MoreFiles.deleteRecursively((Path)this.tempDirectory, (RecursiveDeleteOption[])new RecursiveDeleteOption[]{RecursiveDeleteOption.ALLOW_INSECURE});
        this.tempDirectory = null;
    }

    @AfterEach
    public void nextCacheId() {
        this.cacheKeyProvider.increaseCacheVersion();
    }

    @Test
    public void testCache() throws IOException {
        Location location = this.getRootLocation().appendPath("hello");
        byte[] content = "hello world".getBytes(StandardCharsets.UTF_8);
        try (OutputStream output = this.fileSystem.newOutputFile(location).create();){
            output.write(content);
        }
        int readTimes = 3;
        this.assertCacheOperations(location, content, readTimes, (Multiset<CacheOperationSpan>)ImmutableMultiset.builder().addCopies((Object)new CacheOperationSpan("Alluxio.readCached", location.toString(), 11L), readTimes).addCopies((Object)new CacheOperationSpan("AlluxioCacheManager.get", TestAlluxioCacheFileSystemAccessOperations.cacheKey(location, this.cacheKeyProvider.currentCacheVersion()), 0L, 11L), readTimes).add((Object)new CacheOperationSpan("AlluxioCacheManager.put", TestAlluxioCacheFileSystemAccessOperations.cacheKey(location, this.cacheKeyProvider.currentCacheVersion()), 0L, 11L)).add((Object)new CacheOperationSpan("Alluxio.writeCache", location.toString(), 11L)).add((Object)new CacheOperationSpan("Alluxio.readExternal", location.toString(), 11L)).build());
        byte[] modifiedContent = "modified content".getBytes(StandardCharsets.UTF_8);
        this.fileSystem.newOutputFile(location).createOrOverwrite(modifiedContent);
        this.cacheKeyProvider.increaseCacheVersion();
        readTimes = 7;
        this.assertCacheOperations(location, modifiedContent, readTimes, (Multiset<CacheOperationSpan>)ImmutableMultiset.builder().add((Object)new CacheOperationSpan("Alluxio.readExternal", location.toString(), 16L)).add((Object)new CacheOperationSpan("Alluxio.writeCache", location.toString(), 16L)).add((Object)new CacheOperationSpan("AlluxioCacheManager.put", TestAlluxioCacheFileSystemAccessOperations.cacheKey(location, this.cacheKeyProvider.currentCacheVersion()), 0L, 16L)).addCopies((Object)new CacheOperationSpan("AlluxioCacheManager.get", TestAlluxioCacheFileSystemAccessOperations.cacheKey(location, this.cacheKeyProvider.currentCacheVersion()), 0L, 16L), readTimes).addCopies((Object)new CacheOperationSpan("Alluxio.readCached", location.toString(), 16L), readTimes).build());
    }

    @Test
    public void testPartialCacheHits() throws IOException {
        Location location = this.getRootLocation().appendPath("partial");
        byte[] content = new byte[256];
        for (int i = 0; i < content.length; ++i) {
            content[i] = (byte)i;
        }
        try (OutputStream output = this.fileSystem.newOutputFile(location).create();){
            output.write(content);
        }
        this.assertCacheOperations(location, Arrays.copyOf(content, 128), (Multiset<CacheOperationSpan>)ImmutableMultiset.builder().add((Object)new CacheOperationSpan("Alluxio.readCached", "memory:///partial", 0L, 128L)).add((Object)new CacheOperationSpan("AlluxioCacheManager.get", TestAlluxioCacheFileSystemAccessOperations.cacheKey(location, this.cacheKeyProvider.currentCacheVersion()), 0L, 128L)).add((Object)new CacheOperationSpan("Alluxio.readExternal", location.toString(), 0L, 128L)).add((Object)new CacheOperationSpan("Alluxio.writeCache", location.toString(), 0L, 128L)).add((Object)new CacheOperationSpan("AlluxioCacheManager.put", TestAlluxioCacheFileSystemAccessOperations.cacheKey(location, this.cacheKeyProvider.currentCacheVersion()), 0L, 128L)).build());
        this.assertCacheOperations(location, Arrays.copyOf(content, 138), (Multiset<CacheOperationSpan>)ImmutableMultiset.builder().add((Object)new CacheOperationSpan("Alluxio.readCached", location.toString(), 0L, 138L)).add((Object)new CacheOperationSpan("AlluxioCacheManager.get", TestAlluxioCacheFileSystemAccessOperations.cacheKey(location, this.cacheKeyProvider.currentCacheVersion()), 0L, 128L)).add((Object)new CacheOperationSpan("AlluxioCacheManager.get", TestAlluxioCacheFileSystemAccessOperations.cacheKey(location, this.cacheKeyProvider.currentCacheVersion()), 128L, 128L)).add((Object)new CacheOperationSpan("Alluxio.readExternal", location.toString(), 128L, 10L)).add((Object)new CacheOperationSpan("Alluxio.writeCache", location.toString(), 128L, 128L)).add((Object)new CacheOperationSpan("AlluxioCacheManager.put", TestAlluxioCacheFileSystemAccessOperations.cacheKey(location, this.cacheKeyProvider.currentCacheVersion()), 128L, 128L)).build());
        this.assertCacheOperations(location, Arrays.copyOf(content, 138), (Multiset<CacheOperationSpan>)ImmutableMultiset.builder().add((Object)new CacheOperationSpan("AlluxioCacheManager.get", TestAlluxioCacheFileSystemAccessOperations.cacheKey(location, this.cacheKeyProvider.currentCacheVersion()), 0L, 128L)).add((Object)new CacheOperationSpan("AlluxioCacheManager.get", TestAlluxioCacheFileSystemAccessOperations.cacheKey(location, this.cacheKeyProvider.currentCacheVersion()), 128L, 128L)).add((Object)new CacheOperationSpan("Alluxio.readCached", location.toString(), 138L)).build());
        this.assertCacheOperations(location, content, (Multiset<CacheOperationSpan>)ImmutableMultiset.builder().add((Object)new CacheOperationSpan("AlluxioCacheManager.get", TestAlluxioCacheFileSystemAccessOperations.cacheKey(location, this.cacheKeyProvider.currentCacheVersion()), 0L, 128L)).add((Object)new CacheOperationSpan("AlluxioCacheManager.get", TestAlluxioCacheFileSystemAccessOperations.cacheKey(location, this.cacheKeyProvider.currentCacheVersion()), 128L, 128L)).add((Object)new CacheOperationSpan("Alluxio.readCached", location.toString(), 0L, 256L)).build());
    }

    @Test
    public void testMultiPageExternalsReads() throws IOException {
        Location location = this.getRootLocation().appendPath("multipage");
        byte[] content = new byte[256];
        for (int i = 0; i < content.length; ++i) {
            content[i] = (byte)i;
        }
        try (OutputStream output = this.fileSystem.newOutputFile(location).create();){
            output.write(content);
        }
        this.assertCacheOperations(location, Arrays.copyOf(content, 129), (Multiset<CacheOperationSpan>)ImmutableMultiset.builder().add((Object)new CacheOperationSpan("AlluxioCacheManager.get", TestAlluxioCacheFileSystemAccessOperations.cacheKey(location, this.cacheKeyProvider.currentCacheVersion()), 0L, 128L)).add((Object)new CacheOperationSpan("AlluxioCacheManager.put", TestAlluxioCacheFileSystemAccessOperations.cacheKey(location, this.cacheKeyProvider.currentCacheVersion()), 0L, 128L)).add((Object)new CacheOperationSpan("AlluxioCacheManager.put", TestAlluxioCacheFileSystemAccessOperations.cacheKey(location, this.cacheKeyProvider.currentCacheVersion()), 128L, 128L)).add((Object)new CacheOperationSpan("Alluxio.readCached", location.toString(), 129L)).add((Object)new CacheOperationSpan("Alluxio.readExternal", location.toString(), 129L)).add((Object)new CacheOperationSpan("Alluxio.writeCache", location.toString(), 256L)).build());
        this.cacheKeyProvider.increaseCacheVersion();
        this.assertCacheOperations(location, Arrays.copyOf(content, 256), (Multiset<CacheOperationSpan>)ImmutableMultiset.builder().add((Object)new CacheOperationSpan("AlluxioCacheManager.get", TestAlluxioCacheFileSystemAccessOperations.cacheKey(location, this.cacheKeyProvider.currentCacheVersion()), 0L, 128L)).add((Object)new CacheOperationSpan("AlluxioCacheManager.put", TestAlluxioCacheFileSystemAccessOperations.cacheKey(location, this.cacheKeyProvider.currentCacheVersion()), 0L, 128L)).add((Object)new CacheOperationSpan("AlluxioCacheManager.put", TestAlluxioCacheFileSystemAccessOperations.cacheKey(location, this.cacheKeyProvider.currentCacheVersion()), 128L, 128L)).add((Object)new CacheOperationSpan("Alluxio.readExternal", location.toString(), 256L)).add((Object)new CacheOperationSpan("Alluxio.readCached", location.toString(), 256L)).add((Object)new CacheOperationSpan("Alluxio.writeCache", location.toString(), 256L)).build());
    }

    @Test
    public void testCacheInvalidation() throws IOException {
        Location aLocation = this.createFile("a", 768);
        Location bLocation = this.createFile("b", 640);
        Location cLocation = this.createFile("c", 512);
        Location dLocation = this.createFile("d", 384);
        this.assertUnCachedRead(aLocation, 768);
        this.assertCachedRead(aLocation, 768);
        this.assertUnCachedRead(bLocation, 640);
        this.assertUnCachedRead(aLocation, 768);
        this.assertCachedRead(aLocation, 768);
        this.assertCachedRead(aLocation, 768);
        this.assertUnCachedRead(bLocation, 640);
        this.assertCachedRead(bLocation, 640);
        this.assertUnCachedRead(cLocation, 512);
        this.assertUnCachedRead(dLocation, 384);
        this.assertCachedRead(cLocation, 512);
        this.assertCachedRead(dLocation, 384);
        this.assertUnCachedRead(bLocation, 640);
        this.assertCachedRead(bLocation, 640);
        this.assertUnCachedRead(cLocation, 512);
        this.assertUnCachedRead(dLocation, 384);
    }

    @Test
    public void testCacheWithMissingPage() throws Exception {
        Location location = this.getRootLocation().appendPath("missing_page");
        byte[] content = "missing page".getBytes(StandardCharsets.UTF_8);
        try (OutputStream output = this.fileSystem.newOutputFile(location).create();){
            output.write(content);
        }
        int readTimes = 3;
        this.assertCacheOperations(location, content, readTimes, (Multiset<CacheOperationSpan>)ImmutableMultiset.builder().addCopies((Object)new CacheOperationSpan("Alluxio.readCached", location.toString(), 12L), readTimes).addCopies((Object)new CacheOperationSpan("AlluxioCacheManager.get", TestAlluxioCacheFileSystemAccessOperations.cacheKey(location, this.cacheKeyProvider.currentCacheVersion()), 0L, 12L), readTimes).add((Object)new CacheOperationSpan("AlluxioCacheManager.put", TestAlluxioCacheFileSystemAccessOperations.cacheKey(location, this.cacheKeyProvider.currentCacheVersion()), 0L, 12L)).add((Object)new CacheOperationSpan("Alluxio.writeCache", location.toString(), 12L)).add((Object)new CacheOperationSpan("Alluxio.readExternal", location.toString(), 12L)).build());
        TrinoInputFile inputFile = this.fileSystem.newInputFile(location);
        String fileId = this.alluxioCache.uriStatus(inputFile, this.cacheKeyProvider.getCacheKey(inputFile).get()).getCacheContext().getCacheIdentifier();
        this.pageStore.delete(new PageId(fileId, 0L));
        this.assertCacheOperations(location, content, (Multiset<CacheOperationSpan>)ImmutableMultiset.builder().add((Object)new CacheOperationSpan("Alluxio.readCached", location.toString(), 12L)).add((Object)new CacheOperationSpan("AlluxioCacheManager.get", TestAlluxioCacheFileSystemAccessOperations.cacheKey(location, this.cacheKeyProvider.currentCacheVersion()), 12L)).add((Object)new CacheOperationSpan("Alluxio.readExternal", location.toString(), 12L)).add((Object)new CacheOperationSpan("Alluxio.writeCache", location.toString(), 12L)).add((Object)new CacheOperationSpan("AlluxioCacheManager.put", TestAlluxioCacheFileSystemAccessOperations.cacheKey(location, this.cacheKeyProvider.currentCacheVersion()), 0L, 12L)).build());
    }

    @Test
    public void testCacheWithCorruptedPage() throws Exception {
        Location location = this.getRootLocation().appendPath("corrupted_page");
        byte[] content = "corrupted page".getBytes(StandardCharsets.UTF_8);
        try (OutputStream output = this.fileSystem.newOutputFile(location).create();){
            output.write(content);
        }
        int readTimes = 3;
        this.assertCacheOperations(location, content, readTimes, (Multiset<CacheOperationSpan>)ImmutableMultiset.builder().addCopies((Object)new CacheOperationSpan("Alluxio.readCached", location.toString(), 14L), readTimes).addCopies((Object)new CacheOperationSpan("AlluxioCacheManager.get", TestAlluxioCacheFileSystemAccessOperations.cacheKey(location, this.cacheKeyProvider.currentCacheVersion()), 0L, 14L), readTimes).add((Object)new CacheOperationSpan("AlluxioCacheManager.put", TestAlluxioCacheFileSystemAccessOperations.cacheKey(location, this.cacheKeyProvider.currentCacheVersion()), 0L, 14L)).add((Object)new CacheOperationSpan("Alluxio.writeCache", location.toString(), 14L)).add((Object)new CacheOperationSpan("Alluxio.readExternal", location.toString(), 14L)).build());
        TrinoInputFile inputFile = this.fileSystem.newInputFile(location);
        String fileId = this.alluxioCache.uriStatus(inputFile, this.cacheKeyProvider.getCacheKey(inputFile).get()).getCacheContext().getCacheIdentifier();
        this.pageStore.put(new PageId(fileId, 0L), new byte[0]);
        this.assertCacheOperations(location, content, (Multiset<CacheOperationSpan>)ImmutableMultiset.builder().add((Object)new CacheOperationSpan("Alluxio.readCached", location.toString(), 14L)).add((Object)new CacheOperationSpan("AlluxioCacheManager.get", TestAlluxioCacheFileSystemAccessOperations.cacheKey(location, this.cacheKeyProvider.currentCacheVersion()), 14L)).add((Object)new CacheOperationSpan("Alluxio.readExternal", location.toString(), 14L)).add((Object)new CacheOperationSpan("Alluxio.writeCache", location.toString(), 14L)).add((Object)new CacheOperationSpan("AlluxioCacheManager.put", TestAlluxioCacheFileSystemAccessOperations.cacheKey(location, this.cacheKeyProvider.currentCacheVersion()), 0L, 14L)).build());
    }

    private Location createFile(String name, int size) throws IOException {
        Location location = this.getRootLocation().appendPath(name);
        try (OutputStream output = this.fileSystem.newOutputFile(location).create();){
            output.write("a".repeat(size).getBytes(StandardCharsets.UTF_8));
        }
        return location;
    }

    private Location getRootLocation() {
        return Location.of((String)"memory://");
    }

    private void assertCachedRead(Location location, int fileSize) throws IOException {
        ImmutableMultiset.Builder builder = ImmutableMultiset.builder().add((Object)new CacheOperationSpan("Alluxio.readCached", location.toString(), 0L, fileSize));
        for (int offset = 0; offset < fileSize; offset += 128) {
            builder.add((Object)new CacheOperationSpan("AlluxioCacheManager.get", TestAlluxioCacheFileSystemAccessOperations.cacheKey(location, this.cacheKeyProvider.currentCacheVersion()), offset, 128L));
        }
        this.assertCacheOperations(location, (Multiset<CacheOperationSpan>)builder.build());
    }

    private void assertUnCachedRead(Location location, int fileSize) throws IOException {
        ImmutableMultiset.Builder builder = ImmutableMultiset.builder().add((Object)new CacheOperationSpan("Alluxio.readCached", location.toString(), fileSize)).add((Object)new CacheOperationSpan("Alluxio.writeCache", location.toString(), fileSize)).add((Object)new CacheOperationSpan("Alluxio.readExternal", location.toString(), fileSize));
        for (int offset = 0; offset < fileSize; offset += 128) {
            builder.add((Object)new CacheOperationSpan("AlluxioCacheManager.put", TestAlluxioCacheFileSystemAccessOperations.cacheKey(location, this.cacheKeyProvider.currentCacheVersion()), offset, 128L));
        }
        builder.add((Object)new CacheOperationSpan("AlluxioCacheManager.get", TestAlluxioCacheFileSystemAccessOperations.cacheKey(location, this.cacheKeyProvider.currentCacheVersion()), 0L, 128L));
        this.assertCacheOperations(location, (Multiset<CacheOperationSpan>)builder.build());
    }

    private void assertCacheOperations(Location location, Multiset<CacheOperationSpan> cacheOperations) throws IOException {
        List spans = this.testingTelemetry.captureSpans(() -> {
            TrinoInputFile file = this.fileSystem.newInputFile(location);
            int length = (int)file.length();
            try (TrinoInput input = file.newInput();){
                input.readFully(0L, length);
            }
        });
        MultisetAssertions.assertMultisetsEqual(this.getCacheOperations(spans), cacheOperations);
    }

    private void assertCacheOperations(Location location, byte[] content, Multiset<CacheOperationSpan> cacheOperations) throws IOException {
        this.assertCacheOperations(location, content, 1, cacheOperations);
    }

    private void assertCacheOperations(Location location, byte[] content, int readTimes, Multiset<CacheOperationSpan> cacheOperations) throws IOException {
        List spans = this.testingTelemetry.captureSpans(() -> {
            TrinoInputFile file = this.fileSystem.newInputFile(location);
            int length = content.length;
            try (TrinoInput input = file.newInput();){
                for (int i = 0; i < readTimes; ++i) {
                    Assertions.assertThat((Comparable)input.readFully(0L, length)).isEqualTo((Object)Slices.wrappedBuffer((byte[])content));
                }
            }
        });
        MultisetAssertions.assertMultisetsEqual(this.getCacheOperations(spans), cacheOperations);
    }

    private Multiset<CacheOperationSpan> getCacheOperations(List<SpanData> spans) {
        return (Multiset)spans.stream().filter(span -> span.getName().startsWith("Alluxio")).map(CacheOperationSpan::create).collect(Collectors.toCollection(HashMultiset::create));
    }

    private static String cacheKey(Location location, int cacheVersion) {
        return TestingCacheKeyProvider.testingCacheKeyForLocation(location, cacheVersion);
    }

    private record CacheOperationSpan(String spanName, String location, long position, long length) {
        public CacheOperationSpan(String spanName, String location, long length) {
            this(spanName, location, 0L, length);
        }

        public static CacheOperationSpan create(SpanData span) {
            Attributes attributes = span.getAttributes();
            long length = switch (span.getName()) {
                case "Alluxio.readCached", "Alluxio.readExternal", "AlluxioCacheManager.get" -> (Long)attributes.get(CacheSystemAttributes.CACHE_FILE_READ_SIZE);
                case "Alluxio.writeCache", "AlluxioCacheManager.put" -> (Long)attributes.get(CacheSystemAttributes.CACHE_FILE_WRITE_SIZE);
                default -> throw new IllegalArgumentException("Unrecognized span " + span.getName() + " [" + String.valueOf(span.getAttributes()) + "]");
            };
            long position = switch (span.getName()) {
                case "Alluxio.readCached", "Alluxio.readExternal", "AlluxioCacheManager.get" -> (Long)attributes.get(CacheSystemAttributes.CACHE_FILE_READ_POSITION);
                case "Alluxio.writeCache", "AlluxioCacheManager.put" -> (Long)attributes.get(CacheSystemAttributes.CACHE_FILE_WRITE_POSITION);
                default -> throw new IllegalArgumentException("Unrecognized span  " + span.getName() + " [" + String.valueOf(span.getAttributes()) + "]");
            };
            return new CacheOperationSpan(span.getName(), (String)MoreObjects.firstNonNull((Object)((String)attributes.get(CacheSystemAttributes.CACHE_FILE_LOCATION)), (Object)((String)attributes.get(CacheSystemAttributes.CACHE_KEY))), position, length);
        }

        @Override
        public String toString() {
            return "CacheOperationSpan(\"%s\", \"%s\", %d, %d)".formatted(this.spanName, this.location, this.position, this.length);
        }
    }
}

