/*
 * Decompiled with CFR 0.152.
 */
package io.trino.filesystem.alluxio;

import alluxio.client.file.CacheContext;
import alluxio.client.file.URIStatus;
import alluxio.client.file.cache.CacheManager;
import alluxio.client.file.cache.PageId;
import alluxio.conf.AlluxioConfiguration;
import alluxio.conf.PropertyKey;
import com.google.common.base.Preconditions;
import com.google.common.primitives.Ints;
import io.opentelemetry.api.trace.Span;
import io.opentelemetry.api.trace.Tracer;
import io.trino.filesystem.Location;
import io.trino.filesystem.alluxio.AlluxioCacheStats;
import io.trino.filesystem.tracing.CacheSystemAttributes;
import io.trino.filesystem.tracing.Tracing;
import java.nio.ByteBuffer;
import java.util.Objects;

public class AlluxioInputHelper {
    private final Tracer tracer;
    private final URIStatus status;
    private final String cacheKey;
    private final CacheManager cacheManager;
    private final AlluxioCacheStats statistics;
    private final Location location;
    private final int pageSize;
    private final long fileLength;
    private final int bufferSize;
    private final byte[] readBuffer;
    private long bufferStartPosition;
    private long bufferEndPosition;

    public AlluxioInputHelper(Tracer tracer, Location location, String cacheKey, URIStatus status, CacheManager cacheManager, AlluxioConfiguration configuration, AlluxioCacheStats statistics) {
        this.tracer = Objects.requireNonNull(tracer, "tracer is null");
        this.status = Objects.requireNonNull(status, "status is null");
        this.cacheKey = Objects.requireNonNull(cacheKey, "cacheKey is null");
        this.fileLength = status.getLength();
        this.cacheManager = Objects.requireNonNull(cacheManager, "cacheManager is null");
        this.pageSize = (int)Objects.requireNonNull(configuration, "configuration is null").getBytes(PropertyKey.USER_CLIENT_CACHE_PAGE_SIZE);
        this.statistics = Objects.requireNonNull(statistics, "statistics is null");
        this.location = Objects.requireNonNull(location, "location is null");
        this.bufferSize = this.pageSize;
        this.readBuffer = new byte[this.bufferSize];
    }

    public int doCacheRead(long position, byte[] bytes, int offset, int length) {
        Span span = this.tracer.spanBuilder("Alluxio.readCached").setAttribute(CacheSystemAttributes.CACHE_KEY, (Object)this.cacheKey).setAttribute(CacheSystemAttributes.CACHE_FILE_LOCATION, (Object)this.location.toString()).setAttribute(CacheSystemAttributes.CACHE_FILE_READ_SIZE, (Object)length).setAttribute(CacheSystemAttributes.CACHE_FILE_READ_POSITION, (Object)position).startSpan();
        return (Integer)Tracing.withTracing((Span)span, () -> {
            int bytesRead = this.doBufferRead(position, bytes, offset, length);
            return Math.addExact(bytesRead, this.doInternalCacheRead(position + (long)bytesRead, bytes, offset + bytesRead, length - bytesRead));
        });
    }

    private int doBufferRead(long position, byte[] bytes, int offset, int length) {
        if (length == 0) {
            return 0;
        }
        if (position < this.bufferStartPosition || position >= this.bufferEndPosition) {
            return 0;
        }
        int bytesToCopy = Math.min(length, Ints.saturatedCast((long)(this.bufferEndPosition - position)));
        System.arraycopy(this.readBuffer, Ints.saturatedCast((long)(position - this.bufferStartPosition)), bytes, offset, bytesToCopy);
        return bytesToCopy;
    }

    private int doInternalCacheRead(long position, byte[] bytes, int offset, int length) {
        int bytesReadFromCache;
        if (length == 0) {
            return 0;
        }
        int remainingLength = length;
        while (remainingLength > 0 && (bytesReadFromCache = this.readPageFromCache(position, bytes, offset, remainingLength)) > 0) {
            position += (long)bytesReadFromCache;
            remainingLength -= bytesReadFromCache;
            offset += bytesReadFromCache;
        }
        int bytesRead = length - remainingLength;
        this.statistics.recordCacheRead(bytesRead);
        return bytesRead;
    }

    private int readPageFromCache(long position, byte[] buffer, int offset, int length) {
        long currentPage = position / (long)this.pageSize;
        int currentPageOffset = (int)(position % (long)this.pageSize);
        int bytesLeftInPage = (int)Math.min((long)(this.pageSize - currentPageOffset), this.fileLength - position);
        int bytesToReadInPage = Math.min(bytesLeftInPage, length);
        if (bytesToReadInPage == 0) {
            return 0;
        }
        CacheContext cacheContext = this.status.getCacheContext();
        PageId pageId = new PageId(cacheContext.getCacheIdentifier(), currentPage);
        if (bytesLeftInPage > length && this.bufferSize > length) {
            int putBytes = this.putBuffer(currentPageOffset, pageId, cacheContext);
            if (putBytes <= 0) {
                return putBytes;
            }
            return this.doBufferRead(position, buffer, offset, length);
        }
        return this.cacheManager.get(pageId, currentPageOffset, bytesToReadInPage, buffer, offset, cacheContext);
    }

    private int putBuffer(int pageOffset, PageId pageId, CacheContext cacheContext) {
        long bufferStart;
        int bytesToReadInPage;
        int bytesRead = this.cacheManager.get(pageId, pageOffset = Math.min(pageOffset, Integer.max(this.pageSize - this.bufferSize, 0)), Math.min(bytesToReadInPage = Ints.saturatedCast((long)Math.min((long)(this.pageSize - pageOffset), this.fileLength - (bufferStart = (long)pageOffset + pageId.getPageIndex() * (long)this.pageSize))), this.bufferSize), this.readBuffer, 0, cacheContext);
        if (bytesRead < 0) {
            this.bufferStartPosition = 0L;
            this.bufferEndPosition = 0L;
            return bytesRead;
        }
        if (bytesRead == 0) {
            return bytesRead;
        }
        this.bufferStartPosition = (long)pageOffset + pageId.getPageIndex() * (long)this.pageSize;
        this.bufferEndPosition = this.bufferStartPosition + (long)bytesRead;
        return bytesRead;
    }

    public PageAlignedRead alignRead(long position, long length) {
        long pageStart = position - position % (long)this.pageSize;
        int pageOffset = (int)(position % (long)this.pageSize);
        long readEnd = position + length;
        long alignedReadEnd = readEnd + ((long)this.pageSize - readEnd % (long)this.pageSize) % (long)this.pageSize;
        long pageEnd = Math.min(alignedReadEnd, this.fileLength);
        return new PageAlignedRead(pageStart, pageEnd, pageOffset);
    }

    public void putCache(long pageStart, long pageEnd, byte[] readBuffer, int length) {
        Span span = this.tracer.spanBuilder("Alluxio.writeCache").setAttribute(CacheSystemAttributes.CACHE_KEY, (Object)this.cacheKey).setAttribute(CacheSystemAttributes.CACHE_FILE_LOCATION, (Object)this.location.toString()).setAttribute(CacheSystemAttributes.CACHE_FILE_WRITE_SIZE, (Object)length).setAttribute(CacheSystemAttributes.CACHE_FILE_WRITE_POSITION, (Object)pageStart).startSpan();
        Tracing.withTracing((Span)span, () -> {
            Preconditions.checkArgument((pageStart + (long)length <= pageEnd ? 1 : 0) != 0);
            long end = pageEnd;
            if (pageStart + (long)length < pageEnd) {
                end = pageStart + (long)length - (long)(length % this.pageSize);
            }
            int offset = 0;
            long currentPageStart = pageStart;
            while (currentPageStart < end) {
                long currentPage = currentPageStart / (long)this.pageSize;
                int currentPageSize = (int)Math.min((long)this.pageSize, pageEnd - currentPageStart);
                PageId pageId = new PageId(this.status.getCacheContext().getCacheIdentifier(), currentPage);
                this.cacheManager.put(pageId, ByteBuffer.wrap(readBuffer, offset, currentPageSize));
                currentPageStart += (long)currentPageSize;
                offset += this.pageSize;
            }
        });
    }

    public record PageAlignedRead(long pageStart, long pageEnd, int pageOffset) {
        public int length() {
            return (int)(this.pageEnd - this.pageStart);
        }
    }
}

