/*
 * Decompiled with CFR 0.152.
 */
package io.trino.filesystem.local;

import com.google.common.base.Preconditions;
import io.trino.filesystem.FileIterator;
import io.trino.filesystem.ParsedLocation;
import io.trino.filesystem.TrinoFileSystem;
import io.trino.filesystem.TrinoInputFile;
import io.trino.filesystem.TrinoOutputFile;
import io.trino.filesystem.local.LocalFileIterator;
import io.trino.filesystem.local.LocalInputFile;
import io.trino.filesystem.local.LocalOutputFile;
import io.trino.filesystem.local.LocalUtils;
import java.io.IOException;
import java.nio.file.CopyOption;
import java.nio.file.FileVisitResult;
import java.nio.file.FileVisitor;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.nio.file.SimpleFileVisitor;
import java.nio.file.attribute.BasicFileAttributes;
import java.nio.file.attribute.FileAttribute;

public class LocalFileSystem
implements TrinoFileSystem {
    private final Path rootPath;

    public LocalFileSystem(Path rootPath) {
        this.rootPath = rootPath;
        Preconditions.checkArgument((boolean)Files.isDirectory(rootPath, new LinkOption[0]), (Object)"root is not a directory");
    }

    @Override
    public TrinoInputFile newInputFile(String location) {
        return new LocalInputFile(location, this.toFilePath(location));
    }

    @Override
    public TrinoInputFile newInputFile(String location, long length) {
        return new LocalInputFile(location, this.toFilePath(location), length);
    }

    @Override
    public TrinoOutputFile newOutputFile(String location) {
        return new LocalOutputFile(location, this.toFilePath(location));
    }

    @Override
    public void deleteFile(String location) throws IOException {
        Path filePath = this.toFilePath(location);
        try {
            Files.delete(filePath);
        }
        catch (IOException e) {
            throw LocalUtils.handleException(location, e);
        }
    }

    @Override
    public void deleteDirectory(String location) throws IOException {
        Path directoryPath = this.toDirectoryPath(location);
        if (!Files.exists(directoryPath, new LinkOption[0])) {
            return;
        }
        if (!Files.isDirectory(directoryPath, new LinkOption[0])) {
            throw new IOException("Location is not a directory: " + location);
        }
        try {
            Files.walkFileTree(directoryPath, (FileVisitor<? super Path>)new SimpleFileVisitor<Path>(){

                @Override
                public FileVisitResult visitFile(Path file, BasicFileAttributes attrs) throws IOException {
                    Files.delete(file);
                    return FileVisitResult.CONTINUE;
                }

                @Override
                public FileVisitResult postVisitDirectory(Path directory, IOException exception) throws IOException {
                    if (exception != null) {
                        throw exception;
                    }
                    if (!directory.equals(LocalFileSystem.this.rootPath)) {
                        Files.delete(directory);
                    }
                    return FileVisitResult.CONTINUE;
                }
            });
        }
        catch (IOException e) {
            throw LocalUtils.handleException(location, e);
        }
    }

    @Override
    public void renameFile(String source, String target) throws IOException {
        Path sourcePath = this.toFilePath(source);
        Path targetPath = this.toFilePath(target);
        try {
            if (!Files.exists(sourcePath, new LinkOption[0])) {
                throw new IOException("Source does not exist: " + source);
            }
            if (!Files.isRegularFile(sourcePath, new LinkOption[0])) {
                throw new IOException("Source is not a file: " + source);
            }
            Files.createDirectories(targetPath.getParent(), new FileAttribute[0]);
            Files.move(sourcePath, targetPath, new CopyOption[0]);
        }
        catch (IOException e) {
            throw new IOException("File rename from %s to %s failed: %s".formatted(source, target, e.getMessage()), e);
        }
    }

    @Override
    public FileIterator listFiles(String location) throws IOException {
        return new LocalFileIterator(location, this.rootPath, this.toDirectoryPath(location));
    }

    private Path toFilePath(String fileLocation) {
        ParsedLocation parsedLocation = LocalFileSystem.parseLocalLocation(fileLocation);
        parsedLocation.verifyValidFileLocation();
        Path localPath = this.toPath(fileLocation, parsedLocation);
        Preconditions.checkArgument((!localPath.equals(this.rootPath) ? 1 : 0) != 0, (String)"Local file location must contain a path: %s", (Object)fileLocation);
        return localPath;
    }

    private Path toDirectoryPath(String directoryLocation) {
        ParsedLocation parsedLocation = LocalFileSystem.parseLocalLocation(directoryLocation);
        Path localPath = this.toPath(directoryLocation, parsedLocation);
        return localPath;
    }

    private static ParsedLocation parseLocalLocation(String location) {
        ParsedLocation parsedLocation = ParsedLocation.parseLocation(location);
        Preconditions.checkArgument((boolean)"local".equals(parsedLocation.scheme()), (String)"Only 'local' scheme is supported: %s", (Object)location);
        Preconditions.checkArgument((boolean)parsedLocation.userInfo().isEmpty(), (String)"Local location cannot contain user info: %s", (Object)location);
        Preconditions.checkArgument((boolean)parsedLocation.host().isEmpty(), (String)"Local location cannot contain a host: %s", (Object)location);
        return parsedLocation;
    }

    private Path toPath(String location, ParsedLocation parsedLocation) {
        Path localPath = this.rootPath.resolve(parsedLocation.path()).normalize();
        Preconditions.checkArgument((boolean)localPath.startsWith(this.rootPath), (String)"Location references data outside of the root: %s", (Object)location);
        return localPath;
    }
}

