/*
 * Decompiled with CFR 0.152.
 */
package io.trino.filesystem.memory;

import com.google.common.base.Preconditions;
import io.airlift.slice.Slice;
import io.trino.filesystem.FileEntry;
import io.trino.filesystem.FileIterator;
import io.trino.filesystem.ParsedLocation;
import io.trino.filesystem.TrinoFileSystem;
import io.trino.filesystem.TrinoInputFile;
import io.trino.filesystem.TrinoOutputFile;
import io.trino.filesystem.memory.MemoryBlob;
import io.trino.filesystem.memory.MemoryInputFile;
import io.trino.filesystem.memory.MemoryOutputFile;
import java.io.IOException;
import java.nio.file.FileAlreadyExistsException;
import java.nio.file.NoSuchFileException;
import java.util.Iterator;
import java.util.Optional;
import java.util.OptionalLong;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;

public class MemoryFileSystem
implements TrinoFileSystem {
    private final ConcurrentMap<String, MemoryBlob> blobs = new ConcurrentHashMap<String, MemoryBlob>();

    boolean isEmpty() {
        return this.blobs.isEmpty();
    }

    @Override
    public TrinoInputFile newInputFile(String location) {
        String key = MemoryFileSystem.toBlobKey(location);
        return new MemoryInputFile(location, () -> (MemoryBlob)this.blobs.get(key), OptionalLong.empty());
    }

    @Override
    public TrinoInputFile newInputFile(String location, long length) {
        String key = MemoryFileSystem.toBlobKey(location);
        return new MemoryInputFile(location, () -> (MemoryBlob)this.blobs.get(key), OptionalLong.of(length));
    }

    @Override
    public TrinoOutputFile newOutputFile(final String location) {
        final String key = MemoryFileSystem.toBlobKey(location);
        MemoryOutputFile.OutputBlob outputBlob = new MemoryOutputFile.OutputBlob(){

            @Override
            public boolean exists() {
                return MemoryFileSystem.this.blobs.containsKey(key);
            }

            @Override
            public void createBlob(Slice data) throws FileAlreadyExistsException {
                if (MemoryFileSystem.this.blobs.putIfAbsent(key, new MemoryBlob(data)) != null) {
                    throw new FileAlreadyExistsException(location);
                }
            }

            @Override
            public void overwriteBlob(Slice data) {
                MemoryFileSystem.this.blobs.put(key, new MemoryBlob(data));
            }
        };
        return new MemoryOutputFile(location, outputBlob);
    }

    @Override
    public void deleteFile(String location) throws IOException {
        if (this.blobs.remove(MemoryFileSystem.toBlobKey(location)) == null) {
            throw new NoSuchFileException(location);
        }
    }

    @Override
    public void deleteDirectory(String location) throws IOException {
        String prefix = MemoryFileSystem.toBlobPrefix(location);
        this.blobs.keySet().removeIf(path -> path.startsWith(prefix));
    }

    @Override
    public void renameFile(String source, String target) throws IOException {
        String sourceKey = MemoryFileSystem.toBlobKey(source);
        String targetKey = MemoryFileSystem.toBlobKey(target);
        MemoryBlob sourceData = (MemoryBlob)this.blobs.get(sourceKey);
        if (sourceData == null) {
            throw new IOException("File rename from %s to %s failed: Source does not exist".formatted(source, target));
        }
        if (this.blobs.putIfAbsent(targetKey, sourceData) != null) {
            throw new IOException("File rename from %s to %s failed: Target already exists".formatted(source, target));
        }
        this.blobs.remove(sourceKey, sourceData);
    }

    @Override
    public FileIterator listFiles(String location) throws IOException {
        String prefix = MemoryFileSystem.toBlobPrefix(location);
        final Iterator iterator = this.blobs.entrySet().stream().filter(entry -> ((String)entry.getKey()).startsWith(prefix)).map(entry -> new FileEntry("memory:///" + (String)entry.getKey(), ((MemoryBlob)entry.getValue()).data().length(), ((MemoryBlob)entry.getValue()).lastModified(), Optional.empty())).iterator();
        return new FileIterator(){

            @Override
            public boolean hasNext() {
                return iterator.hasNext();
            }

            @Override
            public FileEntry next() {
                return (FileEntry)iterator.next();
            }
        };
    }

    private static String toBlobKey(String location) {
        ParsedLocation parsedLocation = MemoryFileSystem.parseMemoryLocation(location);
        parsedLocation.verifyValidFileLocation();
        return parsedLocation.path();
    }

    private static String toBlobPrefix(String location) {
        Object directoryPath = MemoryFileSystem.parseMemoryLocation(location).path();
        if (!((String)directoryPath).isEmpty() && !((String)directoryPath).endsWith("/")) {
            directoryPath = (String)directoryPath + "/";
        }
        return directoryPath;
    }

    private static ParsedLocation parseMemoryLocation(String location) {
        ParsedLocation parsedLocation = ParsedLocation.parseLocation(location);
        Preconditions.checkArgument((boolean)"memory".equals(parsedLocation.scheme()), (String)"Only 'memory' scheme is supported: %s", (Object)location);
        Preconditions.checkArgument((boolean)parsedLocation.userInfo().isEmpty(), (String)"Memory location cannot contain user info: %s", (Object)location);
        Preconditions.checkArgument((boolean)parsedLocation.host().isEmpty(), (String)"Memory location cannot contain a host: %s", (Object)location);
        return parsedLocation;
    }
}

