/*
 * Decompiled with CFR 0.152.
 */
package io.trino.filesystem.local;

import com.google.common.base.Preconditions;
import com.google.common.collect.ImmutableSet;
import io.trino.filesystem.FileIterator;
import io.trino.filesystem.Location;
import io.trino.filesystem.TrinoFileSystem;
import io.trino.filesystem.TrinoInputFile;
import io.trino.filesystem.TrinoOutputFile;
import io.trino.filesystem.local.LocalFileIterator;
import io.trino.filesystem.local.LocalInputFile;
import io.trino.filesystem.local.LocalOutputFile;
import io.trino.filesystem.local.LocalUtils;
import java.io.IOException;
import java.nio.file.CopyOption;
import java.nio.file.FileVisitResult;
import java.nio.file.FileVisitor;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.NoSuchFileException;
import java.nio.file.Path;
import java.nio.file.SimpleFileVisitor;
import java.nio.file.attribute.BasicFileAttributes;
import java.nio.file.attribute.FileAttribute;
import java.util.Optional;
import java.util.Set;
import java.util.UUID;
import java.util.stream.Stream;

public class LocalFileSystem
implements TrinoFileSystem {
    private final Path rootPath;

    public LocalFileSystem(Path rootPath) {
        this.rootPath = rootPath;
        Preconditions.checkArgument((boolean)Files.isDirectory(rootPath, new LinkOption[0]), (Object)"root is not a directory");
    }

    @Override
    public TrinoInputFile newInputFile(Location location) {
        return new LocalInputFile(location, this.toFilePath(location));
    }

    @Override
    public TrinoInputFile newInputFile(Location location, long length) {
        return new LocalInputFile(location, this.toFilePath(location), length);
    }

    @Override
    public TrinoOutputFile newOutputFile(Location location) {
        return new LocalOutputFile(location, this.toFilePath(location));
    }

    @Override
    public void deleteFile(Location location) throws IOException {
        Path filePath = this.toFilePath(location);
        try {
            Files.delete(filePath);
        }
        catch (NoSuchFileException noSuchFileException) {
        }
        catch (IOException e) {
            throw LocalUtils.handleException(location, e);
        }
    }

    @Override
    public void deleteDirectory(Location location) throws IOException {
        Path directoryPath = this.toDirectoryPath(location);
        if (!Files.exists(directoryPath, new LinkOption[0])) {
            return;
        }
        if (!Files.isDirectory(directoryPath, new LinkOption[0])) {
            throw new IOException("Location is not a directory: " + location);
        }
        try {
            Files.walkFileTree(directoryPath, (FileVisitor<? super Path>)new SimpleFileVisitor<Path>(){

                @Override
                public FileVisitResult visitFile(Path file, BasicFileAttributes attrs) throws IOException {
                    Files.delete(file);
                    return FileVisitResult.CONTINUE;
                }

                @Override
                public FileVisitResult postVisitDirectory(Path directory, IOException exception) throws IOException {
                    if (exception != null) {
                        throw exception;
                    }
                    if (!directory.equals(LocalFileSystem.this.rootPath)) {
                        Files.delete(directory);
                    }
                    return FileVisitResult.CONTINUE;
                }
            });
        }
        catch (IOException e) {
            throw LocalUtils.handleException(location, e);
        }
    }

    @Override
    public void renameFile(Location source, Location target) throws IOException {
        Path sourcePath = this.toFilePath(source);
        Path targetPath = this.toFilePath(target);
        try {
            if (!Files.exists(sourcePath, new LinkOption[0])) {
                throw new IOException("Source does not exist: " + source);
            }
            if (!Files.isRegularFile(sourcePath, new LinkOption[0])) {
                throw new IOException("Source is not a file: " + source);
            }
            Files.createDirectories(targetPath.getParent(), new FileAttribute[0]);
            Files.move(sourcePath, targetPath, new CopyOption[0]);
        }
        catch (IOException e) {
            throw new IOException("File rename from %s to %s failed: %s".formatted(source, target, e.getMessage()), e);
        }
    }

    @Override
    public FileIterator listFiles(Location location) throws IOException {
        return new LocalFileIterator(location, this.rootPath, this.toDirectoryPath(location));
    }

    @Override
    public Optional<Boolean> directoryExists(Location location) {
        return Optional.of(Files.isDirectory(this.toDirectoryPath(location), new LinkOption[0]));
    }

    @Override
    public void createDirectory(Location location) throws IOException {
        LocalFileSystem.validateLocalLocation(location);
        try {
            Files.createDirectories(this.toDirectoryPath(location), new FileAttribute[0]);
        }
        catch (IOException e) {
            throw new IOException("Failed to create directory: " + location, e);
        }
    }

    @Override
    public void renameDirectory(Location source, Location target) throws IOException {
        Path sourcePath = this.toDirectoryPath(source);
        Path targetPath = this.toDirectoryPath(target);
        try {
            if (!Files.exists(sourcePath, new LinkOption[0])) {
                throw new IOException("Source does not exist: " + source);
            }
            if (!Files.isDirectory(sourcePath, new LinkOption[0])) {
                throw new IOException("Source is not a directory: " + source);
            }
            Files.createDirectories(targetPath.getParent(), new FileAttribute[0]);
            Files.move(sourcePath, targetPath, new CopyOption[0]);
        }
        catch (IOException e) {
            throw new IOException("Directory rename from %s to %s failed: %s".formatted(source, target, e.getMessage()), e);
        }
    }

    @Override
    public Set<Location> listDirectories(Location location) throws IOException {
        Path path = this.toDirectoryPath(location);
        if (Files.isRegularFile(path, new LinkOption[0])) {
            throw new IOException("Location is a file: " + location);
        }
        if (!Files.isDirectory(path, new LinkOption[0])) {
            return ImmutableSet.of();
        }
        try (Stream<Path> stream = Files.list(path);){
            Set set = (Set)stream.filter(file -> Files.isDirectory(file, LinkOption.NOFOLLOW_LINKS)).map(file -> file.getFileName() + "/").map(location::appendPath).collect(ImmutableSet.toImmutableSet());
            return set;
        }
    }

    @Override
    public Optional<Location> createTemporaryDirectory(Location targetPath, String temporaryPrefix, String relativePrefix) throws IOException {
        Location temporary;
        if (temporaryPrefix.startsWith("/")) {
            String prefix = temporaryPrefix;
            while (prefix.startsWith("/")) {
                prefix = prefix.substring(1);
            }
            temporary = Location.of("local:///").appendPath(prefix);
        } else {
            temporary = targetPath.appendPath(temporaryPrefix);
        }
        temporary = temporary.appendPath(UUID.randomUUID().toString());
        this.createDirectory(temporary);
        return Optional.of(temporary);
    }

    private Path toFilePath(Location location) {
        LocalFileSystem.validateLocalLocation(location);
        location.verifyValidFileLocation();
        Path localPath = this.toPath(location);
        Preconditions.checkArgument((!localPath.equals(this.rootPath) ? 1 : 0) != 0, (String)"Local file location must contain a path: %s", (Object)localPath);
        return localPath;
    }

    private Path toDirectoryPath(Location location) {
        LocalFileSystem.validateLocalLocation(location);
        return this.toPath(location);
    }

    private static void validateLocalLocation(Location location) {
        Preconditions.checkArgument((boolean)location.scheme().equals(Optional.of("local")), (String)"Only 'local' scheme is supported: %s", (Object)location);
        Preconditions.checkArgument((boolean)location.userInfo().isEmpty(), (String)"Local location cannot contain user info: %s", (Object)location);
        Preconditions.checkArgument((boolean)location.host().isEmpty(), (String)"Local location cannot contain a host: %s", (Object)location);
    }

    private Path toPath(Location location) {
        Path localPath = this.rootPath.resolve(location.path()).normalize();
        Preconditions.checkArgument((boolean)localPath.startsWith(this.rootPath), (String)"Location references data outside of the root: %s", (Object)location);
        return localPath;
    }
}

