/*
 * Decompiled with CFR 0.152.
 */
package io.trino.filesystem;

import com.google.common.collect.ImmutableList;
import com.google.common.collect.Ordering;
import com.google.common.io.ByteStreams;
import com.google.common.io.Closer;
import io.airlift.concurrent.Threads;
import io.airlift.slice.Slice;
import io.airlift.slice.Slices;
import io.airlift.units.Duration;
import io.trino.filesystem.FileEntry;
import io.trino.filesystem.FileIterator;
import io.trino.filesystem.Location;
import io.trino.filesystem.TrinoFileSystem;
import io.trino.filesystem.TrinoInput;
import io.trino.filesystem.TrinoInputFile;
import io.trino.filesystem.TrinoInputStream;
import io.trino.filesystem.TrinoOutputFile;
import io.trino.filesystem.UriLocation;
import io.trino.filesystem.encryption.EncryptionEnforcingFileSystem;
import io.trino.testing.assertions.Assert;
import java.io.Closeable;
import java.io.EOFException;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.UncheckedIOException;
import java.net.http.HttpClient;
import java.net.http.HttpRequest;
import java.net.http.HttpResponse;
import java.nio.charset.StandardCharsets;
import java.nio.file.FileAlreadyExistsException;
import java.time.Instant;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Comparator;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.UUID;
import java.util.concurrent.CyclicBarrier;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;
import java.util.concurrent.ThreadLocalRandom;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import org.assertj.core.api.AbstractBooleanAssert;
import org.assertj.core.api.AbstractByteArrayAssert;
import org.assertj.core.api.AbstractThrowableAssert;
import org.assertj.core.api.Assertions;
import org.assertj.core.api.Fail;
import org.junit.jupiter.api.Assumptions;
import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;
import org.junit.jupiter.api.TestInstance;
import org.junit.jupiter.api.parallel.Execution;
import org.junit.jupiter.api.parallel.ExecutionMode;

@TestInstance(value=TestInstance.Lifecycle.PER_CLASS)
@Execution(value=ExecutionMode.SAME_THREAD)
public abstract class AbstractTestTrinoFileSystem {
    protected static final String TEST_BLOB_CONTENT_PREFIX = "test blob content for ";
    private static final int MEGABYTE = 0x100000;

    protected abstract boolean isHierarchical();

    protected abstract TrinoFileSystem getFileSystem();

    protected abstract Location getRootLocation();

    protected abstract void verifyFileSystemIsEmpty();

    protected boolean useServerSideEncryptionWithCustomerKey() {
        return false;
    }

    protected boolean isCreateExclusive() {
        return true;
    }

    protected boolean supportsCreateExclusive() {
        return true;
    }

    protected boolean supportsRenameFile() {
        return true;
    }

    protected boolean supportsIncompleteWriteNoClobber() {
        return true;
    }

    protected boolean supportsPreSignedUri() {
        return false;
    }

    protected boolean normalizesListFilesResult() {
        return false;
    }

    protected boolean seekPastEndOfFileFails() {
        return true;
    }

    protected Location createLocation(String path) {
        if (path.isEmpty()) {
            return this.getRootLocation();
        }
        return this.getRootLocation().appendPath(path);
    }

    @BeforeEach
    void beforeEach() {
        this.verifyFileSystemIsEmpty();
    }

    @Test
    void testInputFileMetadata() throws IOException {
        ((AbstractThrowableAssert)Assertions.assertThatThrownBy(() -> this.getFileSystem().newInputFile(this.getRootLocation())).isInstanceOf(IllegalStateException.class)).hasMessageContaining(this.getRootLocation().toString());
        ((AbstractThrowableAssert)Assertions.assertThatThrownBy(() -> this.getFileSystem().newInputFile(Location.of((String)(String.valueOf(this.getRootLocation()) + "/")))).isInstanceOf(IllegalStateException.class)).hasMessageContaining(this.getRootLocation().toString() + "/");
        ((AbstractThrowableAssert)Assertions.assertThatThrownBy(() -> this.getFileSystem().newInputFile(this.createLocation("foo/"))).isInstanceOf(IllegalStateException.class)).hasMessageContaining(this.createLocation("foo/").toString());
        try (TempBlob tempBlob = this.randomBlobLocation("inputFileMetadata");){
            TrinoInputFile inputFile = this.getFileSystem().newInputFile(tempBlob.location());
            Assertions.assertThat((Object)inputFile.location()).isEqualTo((Object)tempBlob.location());
            Assertions.assertThat((boolean)inputFile.exists()).isFalse();
            ((AbstractThrowableAssert)Assertions.assertThatThrownBy(() -> ((TrinoInputFile)inputFile).length()).isInstanceOf(FileNotFoundException.class)).hasMessageContaining(tempBlob.location().toString());
            ((AbstractThrowableAssert)Assertions.assertThatThrownBy(() -> ((TrinoInputFile)inputFile).lastModified()).isInstanceOf(FileNotFoundException.class)).hasMessageContaining(tempBlob.location().toString());
            tempBlob.createOrOverwrite("123456");
            Assertions.assertThat((long)inputFile.length()).isEqualTo(6L);
            Instant lastModified = inputFile.lastModified();
            Assertions.assertThat((Instant)lastModified).isEqualTo((Object)tempBlob.inputFile().lastModified());
            tempBlob.close();
            Assertions.assertThat((boolean)inputFile.exists()).isFalse();
            Assertions.assertThat((long)inputFile.length()).isEqualTo(6L);
            Assertions.assertThat((Instant)inputFile.lastModified()).isEqualTo((Object)lastModified);
        }
    }

    @Test
    void testInputFileWithLengthMetadata() throws IOException {
        ((AbstractThrowableAssert)Assertions.assertThatThrownBy(() -> this.getFileSystem().newInputFile(this.getRootLocation(), 22L)).isInstanceOf(IllegalStateException.class)).hasMessageContaining(this.getRootLocation().toString());
        ((AbstractThrowableAssert)Assertions.assertThatThrownBy(() -> this.getFileSystem().newInputFile(Location.of((String)(String.valueOf(this.getRootLocation()) + "/")), 22L)).isInstanceOf(IllegalStateException.class)).hasMessageContaining(String.valueOf(this.getRootLocation()) + "/");
        ((AbstractThrowableAssert)Assertions.assertThatThrownBy(() -> this.getFileSystem().newInputFile(this.createLocation("foo/"), 22L)).isInstanceOf(IllegalStateException.class)).hasMessageContaining(this.createLocation("foo/").toString());
        try (TempBlob tempBlob = this.randomBlobLocation("inputFileWithLengthMetadata");){
            TrinoInputFile inputFile = this.getFileSystem().newInputFile(tempBlob.location(), 22L);
            Assertions.assertThat((boolean)inputFile.exists()).isFalse();
            Assertions.assertThat((long)inputFile.length()).isEqualTo(22L);
            ((AbstractThrowableAssert)Assertions.assertThatThrownBy(() -> ((TrinoInputFile)inputFile).lastModified()).isInstanceOf(FileNotFoundException.class)).hasMessageContaining(tempBlob.location().toString());
            Assertions.assertThat((long)inputFile.length()).isEqualTo(22L);
            tempBlob.createOrOverwrite("123456");
            Assertions.assertThat((long)inputFile.length()).isEqualTo(22L);
            Instant lastModified = inputFile.lastModified();
            Assertions.assertThat((Instant)lastModified).isEqualTo((Object)tempBlob.inputFile().lastModified());
            Assertions.assertThat((long)inputFile.length()).isEqualTo(22L);
            tempBlob.close();
            Assertions.assertThat((boolean)inputFile.exists()).isFalse();
            Assertions.assertThat((long)inputFile.length()).isEqualTo(22L);
            Assertions.assertThat((Instant)inputFile.lastModified()).isEqualTo((Object)lastModified);
        }
    }

    @Test
    void testInputFileWithLastModifiedMetadata() throws IOException {
        try (TempBlob tempBlob = this.randomBlobLocation("inputFileWithLastModifiedMetadata");){
            TrinoInputFile inputFile = this.getFileSystem().newInputFile(tempBlob.location(), 22L, Instant.ofEpochMilli(12345L));
            Assertions.assertThat((boolean)inputFile.exists()).isFalse();
            Assertions.assertThat((long)inputFile.length()).isEqualTo(22L);
            Assertions.assertThat((Instant)inputFile.lastModified()).isEqualTo((Object)Instant.ofEpochMilli(12345L));
            Assertions.assertThat((long)inputFile.length()).isEqualTo(22L);
            tempBlob.createOrOverwrite("123456");
            Assertions.assertThat((long)inputFile.length()).isEqualTo(22L);
            Assertions.assertThat((Instant)inputFile.lastModified()).isEqualTo((Object)Instant.ofEpochMilli(12345L));
            Assertions.assertThat((long)inputFile.length()).isEqualTo(22L);
            tempBlob.close();
            Assertions.assertThat((boolean)inputFile.exists()).isFalse();
            Assertions.assertThat((long)inputFile.length()).isEqualTo(22L);
            Assertions.assertThat((Instant)inputFile.lastModified()).isEqualTo((Object)Instant.ofEpochMilli(12345L));
        }
    }

    @Test
    public void testInputFile() throws IOException {
        try (TempBlob tempBlob = this.randomBlobLocation("inputStream");){
            Slice slice;
            byte[] bytes;
            TrinoInputFile inputFile = this.getFileSystem().newInputFile(tempBlob.location());
            ((AbstractThrowableAssert)Assertions.assertThatThrownBy(() -> {
                try (TrinoInputStream inputStream = inputFile.newStream();){
                    inputStream.readAllBytes();
                }
            }).isInstanceOf(FileNotFoundException.class)).hasMessageContaining(tempBlob.location().toString());
            ((AbstractThrowableAssert)Assertions.assertThatThrownBy(() -> {
                try (TrinoInput input = inputFile.newInput();){
                    input.readFully(0L, 10);
                }
            }).isInstanceOf(FileNotFoundException.class)).hasMessageContaining(tempBlob.location().toString());
            ((AbstractThrowableAssert)Assertions.assertThatThrownBy(() -> {
                try (TrinoInput input = inputFile.newInput();){
                    input.readTail(10);
                }
            }).isInstanceOf(FileNotFoundException.class)).hasMessageContaining(tempBlob.location().toString());
            try (OutputStream outputStream = tempBlob.outputFile().create();){
                byte[] bytes2 = new byte[4];
                Slice slice2 = Slices.wrappedBuffer((byte[])bytes2);
                for (int i = 0; i < 0x400000; ++i) {
                    slice2.setInt(0, i);
                    outputStream.write(bytes2);
                }
            }
            int fileSize = 0x1000000;
            Assertions.assertThat((boolean)inputFile.exists()).isTrue();
            Assertions.assertThat((long)inputFile.length()).isEqualTo((long)fileSize);
            try (TrinoInputStream inputStream = inputFile.newStream();){
                int intPosition;
                bytes = new byte[4];
                slice = Slices.wrappedBuffer((byte[])bytes);
                for (intPosition = 0; intPosition < 0x400000; ++intPosition) {
                    Assertions.assertThat((long)inputStream.getPosition()).isEqualTo((long)intPosition * 4L);
                    int size = inputStream.readNBytes(bytes, 0, bytes.length);
                    Assertions.assertThat((int)size).isEqualTo(4);
                    Assertions.assertThat((int)slice.getInt(0)).isEqualTo(intPosition);
                    Assertions.assertThat((long)inputStream.getPosition()).isEqualTo((long)(intPosition * 4 + size));
                }
                Assertions.assertThat((long)inputStream.getPosition()).isEqualTo((long)fileSize);
                Assertions.assertThat((int)inputStream.read()).isLessThan(0);
                Assertions.assertThat((int)inputStream.read(bytes)).isLessThan(0);
                if (this.seekPastEndOfFileFails()) {
                    Assertions.assertThat((long)inputStream.skip(10L)).isEqualTo(0L);
                } else {
                    Assertions.assertThat((long)inputStream.skip(10L)).isEqualTo(10L);
                }
                inputStream.seek(0x400000L);
                for (intPosition = 0x100000; intPosition < 0x400000; ++intPosition) {
                    slice.setInt(0, intPosition);
                    for (byte b : bytes) {
                        int value = inputStream.read();
                        Assertions.assertThat((int)value).isGreaterThanOrEqualTo(0);
                        Assertions.assertThat((byte)((byte)value)).isEqualTo(b);
                    }
                }
                Assertions.assertThat((long)inputStream.getPosition()).isEqualTo((long)fileSize);
                Assertions.assertThat((int)inputStream.read()).isLessThan(0);
                Assertions.assertThat((int)inputStream.read(bytes)).isLessThan(0);
                if (this.seekPastEndOfFileFails()) {
                    Assertions.assertThat((long)inputStream.skip(10L)).isEqualTo(0L);
                } else {
                    Assertions.assertThat((long)inputStream.skip(10L)).isEqualTo(10L);
                }
                for (int i = 0; i < 16; ++i) {
                    int expectedPosition = i * 0x100000;
                    inputStream.seek((long)expectedPosition);
                    Assertions.assertThat((long)inputStream.getPosition()).isEqualTo((long)expectedPosition);
                    int size = inputStream.readNBytes(bytes, 0, bytes.length);
                    Assertions.assertThat((int)size).isEqualTo(4);
                    Assertions.assertThat((int)slice.getInt(0)).isEqualTo(expectedPosition / 4);
                }
                inputStream.seek(0L);
                long expectedPosition = 0L;
                for (int i = 0; i < 15; ++i) {
                    long skipSize = inputStream.skip(0x100000L);
                    Assertions.assertThat((long)skipSize).isEqualTo(0x100000L);
                    Assertions.assertThat((long)inputStream.getPosition()).isEqualTo(expectedPosition += skipSize);
                    int size = inputStream.readNBytes(bytes, 0, bytes.length);
                    Assertions.assertThat((int)size).isEqualTo(4);
                    Assertions.assertThat((int)slice.getInt(0)).isEqualTo(expectedPosition / 4L);
                    expectedPosition += (long)size;
                }
                if (this.seekPastEndOfFileFails()) {
                    long skipSize = inputStream.skip(0x100000L);
                    Assertions.assertThat((long)skipSize).isEqualTo((long)fileSize - expectedPosition);
                    Assertions.assertThat((long)inputStream.getPosition()).isEqualTo((long)fileSize);
                }
                inputStream.seek(0L);
                expectedPosition = 0L;
                for (int i = 1; i <= 11; ++i) {
                    int size = Math.min(262144 * i, 0x200000);
                    inputStream.skipNBytes((long)size);
                    Assertions.assertThat((long)inputStream.getPosition()).isEqualTo(expectedPosition += (long)size);
                    size = inputStream.readNBytes(bytes, 0, bytes.length);
                    Assertions.assertThat((int)size).isEqualTo(4);
                    Assertions.assertThat((int)slice.getInt(0)).isEqualTo(expectedPosition / 4L);
                    expectedPosition += (long)size;
                }
                inputStream.skipNBytes((long)fileSize - expectedPosition);
                Assertions.assertThat((long)inputStream.getPosition()).isEqualTo((long)fileSize);
                if (this.seekPastEndOfFileFails()) {
                    inputStream.seek(expectedPosition);
                    Assertions.assertThat((long)(expectedPosition + 0x100000L)).isGreaterThan((long)fileSize);
                    Assertions.assertThatThrownBy(() -> inputStream.skipNBytes(0x100000L)).isInstanceOf(EOFException.class);
                }
                inputStream.seek((long)fileSize);
                if (this.seekPastEndOfFileFails()) {
                    Assertions.assertThatThrownBy(() -> inputStream.skipNBytes(1L)).isInstanceOf(EOFException.class);
                }
                inputStream.seek((long)fileSize);
                if (this.seekPastEndOfFileFails()) {
                    Assertions.assertThat((long)inputStream.skip(1L)).isEqualTo(0L);
                } else {
                    Assertions.assertThat((long)inputStream.skip(1L)).isEqualTo(1L);
                }
                long currentPosition = fileSize - 500;
                inputStream.seek(currentPosition);
                Assertions.assertThat((int)inputStream.read()).isGreaterThanOrEqualTo(0);
                ++currentPosition;
                if (this.seekPastEndOfFileFails()) {
                    ((AbstractThrowableAssert)Assertions.assertThatThrownBy(() -> inputStream.seek((long)(fileSize + 100))).isInstanceOf(IOException.class)).hasMessageContaining(tempBlob.location().toString());
                    Assertions.assertThat((long)inputStream.getPosition()).isEqualTo(currentPosition);
                    Assertions.assertThat((int)inputStream.read()).isGreaterThanOrEqualTo(0);
                    Assertions.assertThat((long)inputStream.getPosition()).isEqualTo(currentPosition + 1L);
                } else {
                    inputStream.seek((long)(fileSize + 100));
                    Assertions.assertThat((long)inputStream.getPosition()).isEqualTo((long)(fileSize + 100));
                    Assertions.assertThat((int)inputStream.read()).isEqualTo(-1);
                    Assertions.assertThat((byte[])inputStream.readNBytes(50)).isEmpty();
                    Assertions.assertThat((long)inputStream.getPosition()).isEqualTo((long)(fileSize + 100));
                }
                Assertions.assertThatThrownBy(() -> inputStream.read(new byte[1], -1, 0)).isInstanceOf(IndexOutOfBoundsException.class);
                Assertions.assertThatThrownBy(() -> inputStream.read(new byte[1], 0, -1)).isInstanceOf(IndexOutOfBoundsException.class);
                Assertions.assertThatThrownBy(() -> inputStream.read(new byte[1], 1, 3)).isInstanceOf(IndexOutOfBoundsException.class);
                inputStream.close();
                ((AbstractThrowableAssert)Assertions.assertThatThrownBy(() -> inputStream.available()).isInstanceOf(IOException.class)).hasMessageContaining(tempBlob.location().toString());
                ((AbstractThrowableAssert)Assertions.assertThatThrownBy(() -> inputStream.seek(0L)).isInstanceOf(IOException.class)).hasMessageContaining(tempBlob.location().toString());
                ((AbstractThrowableAssert)Assertions.assertThatThrownBy(() -> inputStream.read()).isInstanceOf(IOException.class)).hasMessageContaining(tempBlob.location().toString());
                ((AbstractThrowableAssert)Assertions.assertThatThrownBy(() -> inputStream.read(new byte[10])).isInstanceOf(IOException.class)).hasMessageContaining(tempBlob.location().toString());
                ((AbstractThrowableAssert)Assertions.assertThatThrownBy(() -> inputStream.read(new byte[10], 2, 3)).isInstanceOf(IOException.class)).hasMessageContaining(tempBlob.location().toString());
            }
            try (TrinoInput trinoInput = inputFile.newInput();){
                int i;
                int i2;
                bytes = new byte[40];
                slice = Slices.wrappedBuffer((byte[])bytes);
                trinoInput.readFully(0L, bytes, 0, bytes.length);
                for (i2 = 0; i2 < 10; ++i2) {
                    Assertions.assertThat((int)slice.getInt(i2 * 4)).isEqualTo(i2);
                }
                Assertions.assertThat((Comparable)trinoInput.readFully(0L, bytes.length)).isEqualTo((Object)Slices.wrappedBuffer((byte[])bytes));
                trinoInput.readFully(0L, bytes, 2, bytes.length - 2);
                for (i2 = 0; i2 < 9; ++i2) {
                    Assertions.assertThat((int)slice.getInt(2 + i2 * 4)).isEqualTo(i2);
                }
                trinoInput.readFully(0x100000L, bytes, 0, bytes.length);
                for (i2 = 0; i2 < 10; ++i2) {
                    Assertions.assertThat((int)slice.getInt(i2 * 4)).isEqualTo(i2 + 262144);
                }
                Assertions.assertThat((Comparable)trinoInput.readFully(0x100000L, bytes.length)).isEqualTo((Object)Slices.wrappedBuffer((byte[])bytes));
                ((AbstractThrowableAssert)Assertions.assertThatThrownBy(() -> trinoInput.readFully((long)(fileSize - bytes.length + 1), bytes, 0, bytes.length)).isInstanceOf(IOException.class)).hasMessageContaining(tempBlob.location().toString());
                trinoInput.readTail(bytes, 0, bytes.length);
                int totalPositions = 0x400000;
                for (i = 0; i < 10; ++i) {
                    Assertions.assertThat((int)slice.getInt(i * 4)).isEqualTo(totalPositions - 10 + i);
                }
                Assertions.assertThat((Comparable)trinoInput.readTail(bytes.length)).isEqualTo((Object)Slices.wrappedBuffer((byte[])bytes));
                trinoInput.readTail(bytes, 2, bytes.length - 2);
                for (i = 0; i < 9; ++i) {
                    Assertions.assertThat((int)slice.getInt(4 + i * 4)).isEqualTo(totalPositions - 9 + i);
                }
                trinoInput.close();
                ((AbstractThrowableAssert)Assertions.assertThatThrownBy(() -> trinoInput.readFully(0L, 10)).isInstanceOf(IOException.class)).hasMessageContaining(tempBlob.location().toString());
                ((AbstractThrowableAssert)Assertions.assertThatThrownBy(() -> trinoInput.readFully(0L, bytes, 0, 10)).isInstanceOf(IOException.class)).hasMessageContaining(tempBlob.location().toString());
                ((AbstractThrowableAssert)Assertions.assertThatThrownBy(() -> trinoInput.readTail(10)).isInstanceOf(IOException.class)).hasMessageContaining(tempBlob.location().toString());
                ((AbstractThrowableAssert)Assertions.assertThatThrownBy(() -> trinoInput.readTail(bytes, 0, 10)).isInstanceOf(IOException.class)).hasMessageContaining(tempBlob.location().toString());
            }
            tempBlob.outputFile().createOrOverwrite(new byte[0]);
            inputStream = tempBlob.inputFile().newStream();
            try {
                Assertions.assertThat((int)inputStream.read()).isLessThan(0);
            }
            finally {
                if (inputStream != null) {
                    inputStream.close();
                }
            }
        }
    }

    @Test
    void testOutputFile() throws IOException {
        ((AbstractThrowableAssert)Assertions.assertThatThrownBy(() -> this.getFileSystem().newOutputFile(this.getRootLocation())).isInstanceOf(IllegalStateException.class)).hasMessageContaining(this.getRootLocation().toString());
        ((AbstractThrowableAssert)Assertions.assertThatThrownBy(() -> this.getFileSystem().newOutputFile(Location.of((String)(String.valueOf(this.getRootLocation()) + "/")))).isInstanceOf(IllegalStateException.class)).hasMessageContaining(String.valueOf(this.getRootLocation()) + "/");
        ((AbstractThrowableAssert)Assertions.assertThatThrownBy(() -> this.getFileSystem().newOutputFile(this.createLocation("foo/"))).isInstanceOf(IllegalStateException.class)).hasMessageContaining(this.createLocation("foo/").toString());
        try (TempBlob tempBlob = this.randomBlobLocation("outputFile");){
            TrinoOutputFile outputFile = this.getFileSystem().newOutputFile(tempBlob.location());
            Assertions.assertThat((Object)outputFile.location()).isEqualTo((Object)tempBlob.location());
            Assertions.assertThat((boolean)tempBlob.exists()).isFalse();
            try (OutputStream outputStream = outputFile.create();){
                outputStream.write("initial".getBytes(StandardCharsets.UTF_8));
            }
            if (this.isCreateExclusive()) {
                ((AbstractThrowableAssert)Assertions.assertThatThrownBy(() -> outputFile.create().close()).isInstanceOf(FileAlreadyExistsException.class)).hasMessageContaining(tempBlob.location().toString());
                Assertions.assertThat((String)tempBlob.read()).isEqualTo("initial");
                if (this.supportsCreateExclusive()) {
                    ((AbstractThrowableAssert)Assertions.assertThatThrownBy(() -> outputFile.createExclusive(new byte[0])).isInstanceOf(FileAlreadyExistsException.class)).hasMessageContaining(tempBlob.location().toString());
                } else {
                    ((AbstractThrowableAssert)Assertions.assertThatThrownBy(() -> outputFile.createExclusive(new byte[0])).isInstanceOf(UnsupportedOperationException.class)).hasMessageStartingWith("createExclusive not supported");
                }
                Assertions.assertThat((String)tempBlob.read()).isEqualTo("initial");
            } else {
                outputStream = outputFile.create();
                try {
                    outputStream.write("replaced".getBytes(StandardCharsets.UTF_8));
                }
                finally {
                    if (outputStream != null) {
                        outputStream.close();
                    }
                }
                Assertions.assertThat((String)tempBlob.read()).isEqualTo("replaced");
                if (this.supportsCreateExclusive()) {
                    ((AbstractThrowableAssert)Assertions.assertThatThrownBy(() -> outputFile.createExclusive(new byte[0])).isInstanceOf(FileAlreadyExistsException.class)).hasMessageContaining(tempBlob.location().toString());
                } else {
                    ((AbstractThrowableAssert)Assertions.assertThatThrownBy(() -> outputFile.createExclusive(new byte[0])).isInstanceOf(UnsupportedOperationException.class)).hasMessageStartingWith("createExclusive not supported");
                }
            }
            outputFile.createOrOverwrite("overwrite".getBytes(StandardCharsets.UTF_8));
            Assertions.assertThat((String)tempBlob.read()).isEqualTo("overwrite");
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Test
    void testCreateExclusiveIsAtomic() throws Exception {
        if (!this.supportsCreateExclusive()) {
            return;
        }
        int timeoutSeconds = 20;
        ExecutorService executor = Executors.newCachedThreadPool(Threads.daemonThreadsNamed((String)"testCreateExclusiveIsAtomic-%s"));
        AtomicBoolean finishing = new AtomicBoolean(false);
        try (TempBlob tempBlob = this.randomBlobLocation("outputFile");){
            TrinoFileSystem fileSystem = this.getFileSystem();
            byte[] content = "a".repeat(0x100000).getBytes(StandardCharsets.US_ASCII);
            fileSystem.deleteFile(tempBlob.location());
            CyclicBarrier barrier = new CyclicBarrier(2);
            Future<Object> write = executor.submit(() -> {
                barrier.await(timeoutSeconds, TimeUnit.SECONDS);
                fileSystem.newOutputFile(tempBlob.location()).createExclusive(content);
                return null;
            });
            Future<Slice> read = executor.submit(() -> {
                TrinoInputFile inputFile = fileSystem.newInputFile(tempBlob.location());
                ((AbstractBooleanAssert)Assertions.assertThat((boolean)inputFile.exists()).as("inputFile.exists()", new Object[0])).isFalse();
                barrier.await(timeoutSeconds, TimeUnit.SECONDS);
                while (!finishing.get()) {
                    Slice slice;
                    block9: {
                        TrinoInput input = inputFile.newInput();
                        try {
                            slice = input.readFully(0L, content.length);
                            if (input == null) break block9;
                        }
                        catch (Throwable throwable) {
                            try {
                                if (input != null) {
                                    try {
                                        input.close();
                                    }
                                    catch (Throwable throwable2) {
                                        throwable.addSuppressed(throwable2);
                                    }
                                }
                                throw throwable;
                            }
                            catch (FileNotFoundException fileNotFoundException) {}
                        }
                        input.close();
                    }
                    return slice;
                }
                throw new RuntimeException("File not created");
            });
            ((AbstractByteArrayAssert)Assertions.assertThat((byte[])read.get(timeoutSeconds, TimeUnit.SECONDS).getBytes()).as("read content", new Object[0])).isEqualTo((Object)content);
            write.get(timeoutSeconds, TimeUnit.SECONDS);
        }
        finally {
            finishing.set(true);
            executor.shutdownNow();
            ((AbstractBooleanAssert)Assertions.assertThat((boolean)executor.awaitTermination(timeoutSeconds, TimeUnit.SECONDS)).as("executor terminated", new Object[0])).isTrue();
        }
    }

    @Test
    void testOutputStreamByteAtATime() throws IOException {
        try (TempBlob tempBlob = this.randomBlobLocation("inputStream");){
            int i;
            try (OutputStream outputStream = tempBlob.outputFile().create();){
                for (i = 0; i < 0x100000; ++i) {
                    outputStream.write(i);
                    if (i % 1024 != 0) continue;
                    outputStream.flush();
                }
                outputStream.close();
                ((AbstractThrowableAssert)Assertions.assertThatThrownBy(() -> outputStream.write(42)).isInstanceOf(IOException.class)).hasMessageContaining(tempBlob.location().toString());
                ((AbstractThrowableAssert)Assertions.assertThatThrownBy(() -> outputStream.write(new byte[10])).isInstanceOf(IOException.class)).hasMessageContaining(tempBlob.location().toString());
                ((AbstractThrowableAssert)Assertions.assertThatThrownBy(() -> outputStream.write(new byte[10], 1, 3)).isInstanceOf(IOException.class)).hasMessageContaining(tempBlob.location().toString());
                ((AbstractThrowableAssert)Assertions.assertThatThrownBy(outputStream::flush).isInstanceOf(IOException.class)).hasMessageContaining(tempBlob.location().toString());
            }
            try (TrinoInputStream inputStream = tempBlob.inputFile().newStream();){
                for (i = 0; i < 0x100000; ++i) {
                    int value = inputStream.read();
                    Assertions.assertThat((int)value).isGreaterThanOrEqualTo(0);
                    Assertions.assertThat((byte)((byte)value)).isEqualTo((byte)i);
                }
            }
        }
    }

    @Test
    void testOutputStreamLargeWrites() throws IOException {
        try (TempBlob tempBlob = this.randomBlobLocation("inputStream");){
            int i;
            try (OutputStream outputStream = tempBlob.outputFile().create();){
                for (i = 0; i < 8; ++i) {
                    byte[] bytes = new byte[524288];
                    Arrays.fill(bytes, (byte)i);
                    outputStream.write(bytes);
                }
            }
            try (TrinoInputStream inputStream = tempBlob.inputFile().newStream();){
                for (i = 0; i < 8; ++i) {
                    byte[] expected = new byte[524288];
                    Arrays.fill(expected, (byte)i);
                    byte[] actual = inputStream.readNBytes(expected.length);
                    Assertions.assertThat((int)actual.length).isEqualTo(expected.length);
                    Assertions.assertThat((byte[])actual).isEqualTo((Object)expected);
                }
            }
        }
    }

    @Test
    public void testPaths() throws IOException {
        if (this.isHierarchical()) {
            this.testPathHierarchical();
        } else {
            this.testPathBlob();
        }
    }

    protected void testPathHierarchical() throws IOException {
        ((AbstractThrowableAssert)Assertions.assertThatThrownBy(() -> this.getFileSystem().newOutputFile(this.createLocation("../file")).createOrOverwrite("test".getBytes(StandardCharsets.UTF_8))).isInstanceOfAny(new Class[]{IOException.class, IllegalArgumentException.class})).hasMessageContaining(this.createLocation("../file").toString());
        try (TempBlob absolute = new TempBlob(this.createLocation("b"));
             TempBlob alias = new TempBlob(this.createLocation("a/../b"));){
            absolute.createOrOverwrite(TEST_BLOB_CONTENT_PREFIX + absolute.location().toString());
            Assertions.assertThat((boolean)alias.exists()).isTrue();
            Assertions.assertThat((boolean)absolute.exists()).isTrue();
            Assertions.assertThat((String)alias.read()).isEqualTo(TEST_BLOB_CONTENT_PREFIX + absolute.location().toString());
            Assertions.assertThat(this.listPath("")).containsExactly((Object[])new Location[]{absolute.location()});
            this.getFileSystem().deleteFile(alias.location());
            Assertions.assertThat((boolean)alias.exists()).isFalse();
            Assertions.assertThat((boolean)absolute.exists()).isFalse();
        }
    }

    protected void testPathBlob() throws IOException {
        try (TempBlob tempBlob = new TempBlob(this.createLocation("test/.././/file"));){
            TrinoInputFile inputFile = this.getFileSystem().newInputFile(tempBlob.location());
            Assertions.assertThat((Object)inputFile.location()).isEqualTo((Object)tempBlob.location());
            Assertions.assertThat((boolean)inputFile.exists()).isFalse();
            tempBlob.createOrOverwrite(TEST_BLOB_CONTENT_PREFIX + tempBlob.location().toString());
            Assertions.assertThat((long)inputFile.length()).isEqualTo((long)(TEST_BLOB_CONTENT_PREFIX.length() + tempBlob.location().toString().length()));
            Assertions.assertThat((String)tempBlob.read()).isEqualTo(TEST_BLOB_CONTENT_PREFIX + tempBlob.location().toString());
            if (!this.normalizesListFilesResult()) {
                Assertions.assertThat(this.listPath("test/..")).containsExactly((Object[])new Location[]{tempBlob.location()});
            }
            if (this.supportsRenameFile()) {
                this.getFileSystem().renameFile(tempBlob.location(), this.createLocation("file"));
                Assertions.assertThat((boolean)inputFile.exists()).isFalse();
                Assertions.assertThat((String)this.readLocation(this.createLocation("file"))).isEqualTo(TEST_BLOB_CONTENT_PREFIX + tempBlob.location().toString());
                this.getFileSystem().renameFile(this.createLocation("file"), tempBlob.location());
                Assertions.assertThat((boolean)inputFile.exists()).isTrue();
                Assertions.assertThat((String)tempBlob.read()).isEqualTo(TEST_BLOB_CONTENT_PREFIX + tempBlob.location().toString());
            }
            this.getFileSystem().deleteFile(tempBlob.location());
            Assertions.assertThat((boolean)inputFile.exists()).isFalse();
        }
    }

    @Test
    void testDeleteFile() throws IOException {
        ((AbstractThrowableAssert)Assertions.assertThatThrownBy(() -> this.getFileSystem().deleteFile(this.getRootLocation())).isInstanceOf(IllegalStateException.class)).hasMessageContaining(this.getRootLocation().toString());
        ((AbstractThrowableAssert)Assertions.assertThatThrownBy(() -> this.getFileSystem().deleteFile(Location.of((String)(String.valueOf(this.getRootLocation()) + "/")))).isInstanceOf(IllegalStateException.class)).hasMessageContaining(String.valueOf(this.getRootLocation()) + "/");
        ((AbstractThrowableAssert)Assertions.assertThatThrownBy(() -> this.getFileSystem().deleteFile(this.createLocation("foo/"))).isInstanceOf(IllegalStateException.class)).hasMessageContaining(this.createLocation("foo/").toString());
        try (TempBlob tempBlob = this.randomBlobLocation("delete");){
            this.getFileSystem().deleteFile(tempBlob.location());
            tempBlob.createOrOverwrite("delete me");
            this.getFileSystem().deleteFile(tempBlob.location());
            Assertions.assertThat((boolean)tempBlob.exists()).isFalse();
        }
    }

    @Test
    void testDeleteFiles() throws IOException {
        try (Closer closer = Closer.create();){
            Set<Location> locations = this.createTestDirectoryStructure(closer, this.isHierarchical());
            this.getFileSystem().deleteFiles(locations);
            for (Location location : locations) {
                Assertions.assertThat((boolean)this.getFileSystem().newInputFile(location).exists()).isFalse();
            }
        }
    }

    @Test
    public void testDeleteDirectory() throws IOException {
        this.testDeleteDirectory(this.isHierarchical());
    }

    protected void testDeleteDirectory(boolean hierarchicalNamingConstraints) throws IOException {
        this.verifyFileSystemIsEmpty();
        try (Closer closer = Closer.create();){
            Set<Location> locations = this.createTestDirectoryStructure(closer, hierarchicalNamingConstraints);
            Assertions.assertThatThrownBy(this::verifyFileSystemIsEmpty).isInstanceOf(Throwable.class);
            this.getFileSystem().deleteDirectory(this.createLocation("unknown"));
            for (Location location : locations) {
                Assertions.assertThat((boolean)this.getFileSystem().newInputFile(location).exists()).isTrue();
            }
            if (this.isHierarchical()) {
                ((AbstractThrowableAssert)Assertions.assertThatThrownBy(() -> this.getFileSystem().deleteDirectory(this.createLocation("level0-file0"))).isInstanceOf(IOException.class)).hasMessageContaining(this.createLocation("level0-file0").toString());
            }
            this.getFileSystem().deleteDirectory(this.createLocation("level0"));
            Location deletedLocationPrefix = this.createLocation("level0/");
            for (Location location : Ordering.usingToString().sortedCopy(locations)) {
                ((AbstractBooleanAssert)Assertions.assertThat((boolean)this.getFileSystem().newInputFile(location).exists()).as("%s exists", new Object[]{location})).isEqualTo(!location.toString().startsWith(deletedLocationPrefix.toString()));
            }
            this.getFileSystem().deleteDirectory(this.getRootLocation());
            for (Location location : locations) {
                Assertions.assertThat((boolean)this.getFileSystem().newInputFile(location).exists()).isFalse();
            }
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Test
    protected void testRenameFile() throws IOException {
        TempBlob targetBlob;
        TempBlob sourceBlob3;
        block58: {
            if (!this.supportsRenameFile()) {
                try (TempBlob sourceBlob2 = this.randomBlobLocation("renameSource");
                     TempBlob targetBlob2 = this.randomBlobLocation("renameTarget");){
                    sourceBlob2.createOrOverwrite("data");
                    ((AbstractThrowableAssert)Assertions.assertThatThrownBy(() -> this.getFileSystem().renameFile(sourceBlob2.location(), targetBlob2.location())).isInstanceOf(IOException.class)).hasMessageContaining("does not support renames");
                }
                return;
            }
            ((AbstractThrowableAssert)Assertions.assertThatThrownBy(() -> this.getFileSystem().renameFile(this.getRootLocation(), this.createLocation("file"))).isInstanceOf(IllegalStateException.class)).hasMessageContaining(this.getRootLocation().toString());
            ((AbstractThrowableAssert)Assertions.assertThatThrownBy(() -> this.getFileSystem().renameFile(this.createLocation("file"), this.getRootLocation())).isInstanceOf(IllegalStateException.class)).hasMessageContaining(this.getRootLocation().toString());
            ((AbstractThrowableAssert)Assertions.assertThatThrownBy(() -> this.getFileSystem().renameFile(Location.of((String)(String.valueOf(this.getRootLocation()) + "/")), this.createLocation("file"))).isInstanceOf(IllegalStateException.class)).hasMessageContaining(String.valueOf(this.getRootLocation()) + "/");
            ((AbstractThrowableAssert)Assertions.assertThatThrownBy(() -> this.getFileSystem().renameFile(this.createLocation("file"), Location.of((String)(String.valueOf(this.getRootLocation()) + "/")))).isInstanceOf(IllegalStateException.class)).hasMessageContaining(String.valueOf(this.getRootLocation()) + "/");
            ((AbstractThrowableAssert)Assertions.assertThatThrownBy(() -> this.getFileSystem().renameFile(this.createLocation("foo/"), this.createLocation("file"))).isInstanceOf(IllegalStateException.class)).hasMessageContaining(this.createLocation("foo/").toString());
            ((AbstractThrowableAssert)Assertions.assertThatThrownBy(() -> this.getFileSystem().renameFile(this.createLocation("file"), this.createLocation("foo/"))).isInstanceOf(IllegalStateException.class)).hasMessageContaining(this.createLocation("foo/").toString());
            try {
                sourceBlob3 = this.randomBlobLocation("renameSource");
                try {
                    targetBlob = this.randomBlobLocation("renameTarget");
                    try {
                        ((AbstractThrowableAssert)Assertions.assertThatThrownBy(() -> this.getFileSystem().renameFile(sourceBlob3.location(), targetBlob.location())).isInstanceOf(IOException.class)).hasMessageContaining(sourceBlob3.location().toString()).hasMessageContaining(targetBlob.location().toString());
                        this.getFileSystem().createDirectory(targetBlob.location().parentDirectory());
                        sourceBlob3.createOrOverwrite("data");
                        this.getFileSystem().renameFile(sourceBlob3.location(), targetBlob.location());
                        Assertions.assertThat((boolean)sourceBlob3.exists()).isFalse();
                        Assertions.assertThat((boolean)targetBlob.exists()).isTrue();
                        Assertions.assertThat((String)targetBlob.read()).isEqualTo("data");
                        sourceBlob3.createOrOverwrite("new data");
                        ((AbstractThrowableAssert)Assertions.assertThatThrownBy(() -> this.getFileSystem().renameFile(sourceBlob3.location(), targetBlob.location())).isInstanceOf(IOException.class)).hasMessageContaining(sourceBlob3.location().toString()).hasMessageContaining(targetBlob.location().toString());
                        Assertions.assertThat((boolean)sourceBlob3.exists()).isTrue();
                        Assertions.assertThat((boolean)targetBlob.exists()).isTrue();
                        Assertions.assertThat((String)sourceBlob3.read()).isEqualTo("new data");
                        Assertions.assertThat((String)targetBlob.read()).isEqualTo("data");
                        if (!this.isHierarchical()) break block58;
                        try (Closer closer = Closer.create();){
                            this.createBlob(closer, "a/b");
                            Assertions.assertThatThrownBy(() -> this.getFileSystem().renameFile(this.createLocation("a"), this.createLocation("b"))).isInstanceOf(IOException.class);
                        }
                    }
                    finally {
                        if (targetBlob != null) {
                            targetBlob.close();
                        }
                    }
                }
                finally {
                    if (sourceBlob3 != null) {
                        sourceBlob3.close();
                    }
                }
            }
            finally {
                try {
                    this.getFileSystem().deleteFile(this.createLocation("renameTarget"));
                }
                catch (IOException sourceBlob3) {}
            }
        }
        try {
            sourceBlob3 = this.randomBlobLocation("renameSource");
            try {
                targetBlob = this.randomBlobLocation("renameTarget%25special");
                try {
                    sourceBlob3.createOrOverwrite("data");
                    this.getFileSystem().createDirectory(targetBlob.location().parentDirectory());
                    this.getFileSystem().renameFile(sourceBlob3.location(), targetBlob.location());
                    Assertions.assertThat((boolean)sourceBlob3.exists()).isFalse();
                    Assertions.assertThat((boolean)targetBlob.exists()).isTrue();
                    Assertions.assertThat((String)targetBlob.read()).isEqualTo("data");
                }
                finally {
                    if (targetBlob != null) {
                        targetBlob.close();
                    }
                }
            }
            finally {
                if (sourceBlob3 != null) {
                    sourceBlob3.close();
                }
            }
        }
        finally {
            try {
                this.getFileSystem().deleteFile(this.createLocation("renameTarget%25special"));
            }
            catch (IOException iOException) {}
        }
    }

    @Test
    public void testListFiles() throws IOException {
        this.testListFiles(this.isHierarchical());
    }

    @Test
    public void testPreSignedUris() throws IOException {
        try (Closer closer = Closer.create();){
            Location location = this.createBlob(closer, "pre_signed");
            if (!this.supportsPreSignedUri()) {
                Assertions.assertThatThrownBy(() -> this.getFileSystem().preSignedUri(location, new Duration(1.0, TimeUnit.SECONDS))).isInstanceOf(UnsupportedOperationException.class);
                Assumptions.abort((String)"Generating pre-signed URI is not supported");
            }
            Optional directLocation = this.getFileSystem().preSignedUri(location, new Duration(3.0, TimeUnit.SECONDS));
            Assertions.assertThat((Optional)directLocation).isPresent();
            Assertions.assertThat((String)AbstractTestTrinoFileSystem.retrieveUri((UriLocation)directLocation.get())).isEqualTo(TEST_BLOB_CONTENT_PREFIX + String.valueOf(location));
            Assertions.assertThat((String)AbstractTestTrinoFileSystem.retrieveUri((UriLocation)directLocation.get())).isEqualTo(TEST_BLOB_CONTENT_PREFIX + String.valueOf(location));
            Assert.assertEventually((Duration)new Duration(5.0, TimeUnit.SECONDS), (Duration)new Duration(1.0, TimeUnit.SECONDS), () -> Assertions.assertThatThrownBy(() -> AbstractTestTrinoFileSystem.retrieveUri((UriLocation)directLocation.get())).isInstanceOf(IOException.class));
        }
    }

    private static String retrieveUri(UriLocation uriLocation) throws IOException {
        try (HttpClient client = HttpClient.newHttpClient();){
            HttpRequest request = AbstractTestTrinoFileSystem.addHeaders(HttpRequest.newBuilder(), uriLocation.headers()).uri(uriLocation.uri()).GET().build();
            try {
                HttpResponse<String> response = client.send(request, HttpResponse.BodyHandlers.ofString());
                if (response.statusCode() != 200) {
                    throw new IOException("Failed to retrieve, got response code: %d, body: %s".formatted(response.statusCode(), response.body()));
                }
                String string = response.body();
                return string;
            }
            catch (InterruptedException e) {
                Thread.currentThread().interrupt();
                throw new RuntimeException(e);
            }
        }
    }

    private static HttpRequest.Builder addHeaders(HttpRequest.Builder builder, Map<String, List<String>> headers) {
        headers.forEach((headerName, headerValues) -> headerValues.forEach(headerValue -> builder.header((String)headerName, (String)headerValue)));
        return builder;
    }

    protected void testListFiles(boolean hierarchicalNamingConstraints) throws IOException {
        try (Closer closer = Closer.create();){
            Set<Location> locations = this.createTestDirectoryStructure(closer, hierarchicalNamingConstraints);
            Assertions.assertThat(this.listPath("")).containsExactlyInAnyOrderElementsOf(locations);
            Assertions.assertThat(this.listPath("level0")).containsExactlyInAnyOrderElementsOf(locations.stream().filter(location -> location.toString().startsWith(this.createLocation("level0/").toString())).toList());
            Assertions.assertThat(this.listPath("level0/")).containsExactlyInAnyOrderElementsOf(locations.stream().filter(location -> location.toString().startsWith(this.createLocation("level0/").toString())).toList());
            Assertions.assertThat(this.listPath("level0/level1/")).containsExactlyInAnyOrderElementsOf(locations.stream().filter(location -> location.toString().startsWith(this.createLocation("level0/level1/").toString())).toList());
            Assertions.assertThat(this.listPath("level0/level1")).containsExactlyInAnyOrderElementsOf(locations.stream().filter(location -> location.toString().startsWith(this.createLocation("level0/level1/").toString())).toList());
            Assertions.assertThat(this.listPath("level0/level1/level2/")).containsExactlyInAnyOrderElementsOf(locations.stream().filter(location -> location.toString().startsWith(this.createLocation("level0/level1/level2/").toString())).toList());
            Assertions.assertThat(this.listPath("level0/level1/level2")).containsExactlyInAnyOrderElementsOf(locations.stream().filter(location -> location.toString().startsWith(this.createLocation("level0/level1/level2/").toString())).toList());
            Assertions.assertThat(this.listPath("level0/level1/level2/level3")).isEmpty();
            Assertions.assertThat(this.listPath("level0/level1/level2/level3/")).isEmpty();
            Assertions.assertThat(this.listPath("unknown/")).isEmpty();
            if (this.isHierarchical()) {
                ((AbstractThrowableAssert)Assertions.assertThatThrownBy(() -> this.listPath("level0-file0")).isInstanceOf(IOException.class)).hasMessageContaining(this.createLocation("level0-file0").toString());
            } else {
                Assertions.assertThat(this.listPath("level0-file0")).isEmpty();
            }
            if (!hierarchicalNamingConstraints && !this.normalizesListFilesResult()) {
                Assertions.assertThat(this.listPath("/")).isEmpty();
            }
        }
    }

    @Test
    public void testDirectoryExists() throws IOException {
        try (Closer closer = Closer.create();){
            String directoryName = "testDirectoryExistsDir";
            String fileName = "file.csv";
            Assertions.assertThat(this.listPath("")).isEmpty();
            Assertions.assertThat((Optional)this.getFileSystem().directoryExists(this.getRootLocation())).contains((Object)true);
            if (this.isHierarchical()) {
                Assertions.assertThat((Optional)this.getFileSystem().directoryExists(this.createLocation(directoryName))).contains((Object)false);
                this.createBlob(closer, this.createLocation(directoryName).appendPath(fileName).path());
                Assertions.assertThat((Optional)this.getFileSystem().directoryExists(this.createLocation(directoryName))).contains((Object)true);
                Assertions.assertThat((Optional)this.getFileSystem().directoryExists(this.createLocation(UUID.randomUUID().toString()))).contains((Object)false);
                Assertions.assertThat((Optional)this.getFileSystem().directoryExists(this.createLocation(directoryName).appendPath(fileName))).contains((Object)false);
            } else {
                Assertions.assertThat((Optional)this.getFileSystem().directoryExists(this.createLocation(directoryName))).isEmpty();
                this.createBlob(closer, this.createLocation(directoryName).appendPath(fileName).path());
                Assertions.assertThat((Optional)this.getFileSystem().directoryExists(this.createLocation(directoryName))).contains((Object)true);
                Assertions.assertThat((Optional)this.getFileSystem().directoryExists(this.createLocation(UUID.randomUUID().toString()))).isEmpty();
                Assertions.assertThat((Optional)this.getFileSystem().directoryExists(this.createLocation(directoryName).appendPath(fileName))).isEmpty();
            }
        }
    }

    @Test
    public void testFileWithTrailingWhitespace() throws IOException {
        block26: {
            try (Closer closer = Closer.create();){
                Location location = this.createBlob(closer, "dir/whitespace ");
                Assertions.assertThat(this.listPath("dir")).isEqualTo(List.of(location));
                TrinoInputFile inputFile = this.getFileSystem().newInputFile(location);
                ((AbstractBooleanAssert)Assertions.assertThat((boolean)inputFile.exists()).as("exists", new Object[0])).isTrue();
                try (TrinoInputStream inputStream = inputFile.newStream();){
                    byte[] bytes = ByteStreams.toByteArray((InputStream)inputStream);
                    Assertions.assertThat((byte[])bytes).isEqualTo((Object)(TEST_BLOB_CONTENT_PREFIX + String.valueOf(location)).getBytes(StandardCharsets.UTF_8));
                }
                byte[] newContents = "bar bar baz new content".getBytes(StandardCharsets.UTF_8);
                this.getFileSystem().newOutputFile(location).createOrOverwrite(newContents);
                TrinoInputFile newInputFile = this.getFileSystem().newInputFile(location);
                try (TrinoInputStream inputStream = newInputFile.newStream();){
                    byte[] bytes = ByteStreams.toByteArray((InputStream)inputStream);
                    Assertions.assertThat((byte[])bytes).isEqualTo((Object)newContents);
                }
                this.getFileSystem().deleteFile(location);
                ((AbstractBooleanAssert)Assertions.assertThat((boolean)newInputFile.exists()).as("exists after delete", new Object[0])).isFalse();
                if (!this.supportsRenameFile()) break block26;
                Location source = this.createBlob(closer, "dir/another trailing whitespace ");
                Location target = this.getRootLocation().appendPath("dir/after rename still whitespace ");
                this.getFileSystem().renameFile(source, target);
                ((AbstractBooleanAssert)Assertions.assertThat((boolean)this.getFileSystem().newInputFile(source).exists()).as("source exists after rename", new Object[0])).isFalse();
                ((AbstractBooleanAssert)Assertions.assertThat((boolean)this.getFileSystem().newInputFile(target).exists()).as("target exists after rename", new Object[0])).isTrue();
                try (TrinoInputStream inputStream = this.getFileSystem().newInputFile(target).newStream();){
                    byte[] bytes = ByteStreams.toByteArray((InputStream)inputStream);
                    Assertions.assertThat((byte[])bytes).isEqualTo((Object)(TEST_BLOB_CONTENT_PREFIX + String.valueOf(source)).getBytes(StandardCharsets.UTF_8));
                }
                this.getFileSystem().deleteFile(target);
                ((AbstractBooleanAssert)Assertions.assertThat((boolean)this.getFileSystem().newInputFile(target).exists()).as("target exists after delete", new Object[0])).isFalse();
            }
        }
    }

    @Test
    public void testListLexicographicalOrder() throws IOException {
        try (Closer closer = Closer.create();){
            List<TempBlob> blobs = this.randomBlobs(closer);
            List<Location> sortedLocations = blobs.stream().map(TempBlob::location).sorted(Comparator.comparing(Location::fileName)).toList();
            Assertions.assertThat(this.listPath("")).isEqualTo(sortedLocations);
        }
    }

    @Test
    public void testCreateDirectory() throws IOException {
        try (Closer closer = Closer.create();){
            this.getFileSystem().createDirectory(this.createLocation("level0/level1/level2"));
            Optional expectedExists = this.isHierarchical() ? Optional.of(true) : Optional.empty();
            Assertions.assertThat((Optional)this.getFileSystem().directoryExists(this.createLocation("level0/level1/level2"))).isEqualTo(expectedExists);
            Assertions.assertThat((Optional)this.getFileSystem().directoryExists(this.createLocation("level0/level1"))).isEqualTo(expectedExists);
            Assertions.assertThat((Optional)this.getFileSystem().directoryExists(this.createLocation("level0"))).isEqualTo(expectedExists);
            Location blob = this.createBlob(closer, "level0/level1/level2-file");
            if (this.isHierarchical()) {
                ((AbstractThrowableAssert)Assertions.assertThatThrownBy(() -> this.getFileSystem().createDirectory(blob)).isInstanceOf(IOException.class)).hasMessageContaining(blob.toString());
            } else {
                this.getFileSystem().createDirectory(blob);
            }
            Assertions.assertThat((String)this.readLocation(blob)).isEqualTo(TEST_BLOB_CONTENT_PREFIX + String.valueOf(blob));
            this.getFileSystem().createDirectory(this.createLocation("level0"));
            this.getFileSystem().createDirectory(this.createLocation("level0/level1"));
            this.getFileSystem().createDirectory(this.createLocation("level0/level1/level2"));
        }
    }

    @Test
    public void testRenameDirectory() throws IOException {
        if (!this.isHierarchical()) {
            this.getFileSystem().createDirectory(this.createLocation("abc"));
            ((AbstractThrowableAssert)Assertions.assertThatThrownBy(() -> this.getFileSystem().renameDirectory(this.createLocation("source"), this.createLocation("target"))).isInstanceOf(IOException.class)).hasMessageContaining("does not support directory renames");
            return;
        }
        ((AbstractThrowableAssert)Assertions.assertThatThrownBy(() -> this.getFileSystem().renameDirectory(this.getRootLocation(), this.createLocation("dir"))).isInstanceOf(IOException.class)).hasMessageContaining(this.getRootLocation().toString());
        ((AbstractThrowableAssert)Assertions.assertThatThrownBy(() -> this.getFileSystem().renameDirectory(this.createLocation("dir"), this.getRootLocation())).isInstanceOf(IOException.class)).hasMessageContaining(this.getRootLocation().toString());
        try (Closer closer = Closer.create();){
            this.getFileSystem().createDirectory(this.createLocation("level0/level1/level2"));
            Location blob = this.createBlob(closer, "level0/level1/level2-file");
            Assertions.assertThat((Optional)this.getFileSystem().directoryExists(this.createLocation("level0/level1/level2"))).contains((Object)true);
            Assertions.assertThat((Optional)this.getFileSystem().directoryExists(this.createLocation("level0/level1"))).contains((Object)true);
            Assertions.assertThat((Optional)this.getFileSystem().directoryExists(this.createLocation("level0"))).contains((Object)true);
            this.getFileSystem().renameDirectory(this.createLocation("level0/level1"), this.createLocation("level0/renamed"));
            Assertions.assertThat((Optional)this.getFileSystem().directoryExists(this.createLocation("level0/level1"))).contains((Object)false);
            Assertions.assertThat((Optional)this.getFileSystem().directoryExists(this.createLocation("level0/level1/level2"))).contains((Object)false);
            Assertions.assertThat((Optional)this.getFileSystem().directoryExists(this.createLocation("level0/renamed"))).contains((Object)true);
            Assertions.assertThat((Optional)this.getFileSystem().directoryExists(this.createLocation("level0/renamed/level2"))).contains((Object)true);
            Assertions.assertThat((boolean)this.getFileSystem().newInputFile(blob).exists()).isFalse();
            Location renamedBlob = this.createLocation("level0/renamed/level2-file");
            Assertions.assertThat((String)this.readLocation(renamedBlob)).isEqualTo(TEST_BLOB_CONTENT_PREFIX + String.valueOf(blob));
            Location blob2 = this.createBlob(closer, "abc/xyz-file");
            Assertions.assertThat((Optional)this.getFileSystem().directoryExists(this.createLocation("abc"))).contains((Object)true);
            ((AbstractThrowableAssert)Assertions.assertThatThrownBy(() -> this.getFileSystem().renameDirectory(this.createLocation("abc"), this.createLocation("level0"))).isInstanceOf(IOException.class)).hasMessageContaining(this.createLocation("abc").toString()).hasMessageContaining(this.createLocation("level0").toString());
            Assertions.assertThat((boolean)this.getFileSystem().newInputFile(blob2).exists()).isTrue();
            Assertions.assertThat((boolean)this.getFileSystem().newInputFile(renamedBlob).exists()).isTrue();
        }
    }

    @Test
    public void testListDirectories() throws IOException {
        this.testListDirectories(this.isHierarchical());
    }

    protected void testListDirectories(boolean hierarchicalNamingConstraints) throws IOException {
        try (Closer closer = Closer.create();){
            this.createTestDirectoryStructure(closer, hierarchicalNamingConstraints);
            this.createBlob(closer, "level0/level1/level2/level3-file0");
            this.createBlob(closer, "level0/level1x/level2x-file0");
            this.createBlob(closer, "other/file");
            Assertions.assertThat(this.listDirectories("")).containsOnly((Object[])new Location[]{this.createLocation("level0/"), this.createLocation("other/")});
            Assertions.assertThat(this.listDirectories("level0")).containsOnly((Object[])new Location[]{this.createLocation("level0/level1/"), this.createLocation("level0/level1x/")});
            Assertions.assertThat(this.listDirectories("level0/")).containsOnly((Object[])new Location[]{this.createLocation("level0/level1/"), this.createLocation("level0/level1x/")});
            Assertions.assertThat(this.listDirectories("level0/level1")).containsOnly((Object[])new Location[]{this.createLocation("level0/level1/level2/")});
            Assertions.assertThat(this.listDirectories("level0/level1/")).containsOnly((Object[])new Location[]{this.createLocation("level0/level1/level2/")});
            Assertions.assertThat(this.listDirectories("level0/level1/level2/level3")).isEmpty();
            Assertions.assertThat(this.listDirectories("level0/level1/level2/level3/")).isEmpty();
            Assertions.assertThat(this.listDirectories("unknown")).isEmpty();
            Assertions.assertThat(this.listDirectories("unknown/")).isEmpty();
            if (this.isHierarchical()) {
                ((AbstractThrowableAssert)Assertions.assertThatThrownBy(() -> this.listDirectories("level0-file0")).isInstanceOf(IOException.class)).hasMessageContaining(this.createLocation("level0-file0").toString());
            } else {
                Assertions.assertThat(this.listDirectories("level0-file0")).isEmpty();
            }
            if (!hierarchicalNamingConstraints && !this.normalizesListFilesResult()) {
                Assertions.assertThat(this.listDirectories("/")).isEmpty();
            }
        }
    }

    private Set<Location> listDirectories(String path) throws IOException {
        return this.getFileSystem().listDirectories(this.createListingLocation(path));
    }

    private List<Location> listPath(String path) throws IOException {
        ArrayList<Location> locations = new ArrayList<Location>();
        FileIterator fileIterator = this.getFileSystem().listFiles(this.createListingLocation(path));
        while (fileIterator.hasNext()) {
            FileEntry fileEntry = fileIterator.next();
            Location location = fileEntry.location();
            Assertions.assertThat((long)fileEntry.length()).isEqualTo((long)(TEST_BLOB_CONTENT_PREFIX.length() + location.toString().length()));
            locations.add(location);
        }
        return locations;
    }

    private Location createListingLocation(String path) {
        if (path.equals("/")) {
            return this.createLocation("").appendSuffix("/");
        }
        return this.createLocation(path);
    }

    @Test
    void testFileDoesNotExistUntilClosed() throws Exception {
        if (!this.supportsIncompleteWriteNoClobber()) {
            Assumptions.abort((String)"skipped");
        }
        Location location = this.getRootLocation().appendPath("testFileDoesNotExistUntilClosed-%s".formatted(UUID.randomUUID()));
        this.getFileSystem().deleteFile(location);
        try (OutputStream out = this.getFileSystem().newOutputFile(location).create();){
            Assertions.assertThat((boolean)this.fileExistsInListing(location)).isFalse();
            Assertions.assertThat((boolean)this.fileExists(location)).isFalse();
            out.write("test".getBytes(StandardCharsets.UTF_8));
            Assertions.assertThat((boolean)this.fileExistsInListing(location)).isFalse();
            Assertions.assertThat((boolean)this.fileExists(location)).isFalse();
        }
        Assertions.assertThat((boolean)this.fileExistsInListing(location)).isTrue();
        Assertions.assertThat((boolean)this.fileExists(location)).isTrue();
        this.getFileSystem().deleteFile(location);
    }

    @Test
    public void testLargeFileDoesNotExistUntilClosed() throws IOException {
        if (!this.supportsIncompleteWriteNoClobber()) {
            Assumptions.abort((String)"skipped");
        }
        Location location = this.getRootLocation().appendPath("testLargeFileDoesNotExistUntilClosed-%s".formatted(UUID.randomUUID()));
        this.getFileSystem().deleteFile(location);
        try (OutputStream outputStream = this.getFileSystem().newOutputFile(location).create();){
            byte[] bytes = AbstractTestTrinoFileSystem.getBytes();
            int target = 0x1100000;
            int count = 0;
            while (count < target) {
                outputStream.write(bytes);
                if ((count += bytes.length) + bytes.length < target) continue;
                Assertions.assertThat((boolean)this.fileExistsInListing(location)).isFalse();
                Assertions.assertThat((boolean)this.fileExists(location)).isFalse();
            }
        }
        Assertions.assertThat((boolean)this.fileExistsInListing(location)).isTrue();
        Assertions.assertThat((boolean)this.fileExists(location)).isTrue();
        this.getFileSystem().deleteFile(location);
    }

    @Test
    void testServerSideEncryptionWithCustomerKey() throws IOException {
        if (!this.useServerSideEncryptionWithCustomerKey()) {
            Assumptions.abort((String)"Test is specific to SSE-C");
        }
        Location location = this.getRootLocation().appendPath("encrypted");
        byte[] data = "this is encrypted data".getBytes(StandardCharsets.UTF_8);
        this.getFileSystem().newOutputFile(location).createOrOverwrite(data);
        TrinoFileSystem trinoFileSystem = this.getFileSystem();
        if (!(trinoFileSystem instanceof EncryptionEnforcingFileSystem)) {
            Fail.fail((String)"Expected file system to enforce server side encryption");
            return;
        }
        EncryptionEnforcingFileSystem encryptionEnforcingFileSystem = (EncryptionEnforcingFileSystem)trinoFileSystem;
        Assertions.assertThatThrownBy(() -> encryptionEnforcingFileSystem.getDelegate().newInputFile(location).newStream().readAllBytes()).isInstanceOf(IOException.class);
        Assertions.assertThat((byte[])this.getFileSystem().newInputFile(location).newStream().readAllBytes()).isEqualTo((Object)data);
        this.getFileSystem().deleteFile(location);
    }

    private static byte[] getBytes() {
        byte[] bytes = new byte[8192];
        for (int i = 0; i < bytes.length; ++i) {
            bytes[i] = (byte)i;
        }
        Assertions.assertThat((int)(0x100000 % bytes.length)).isEqualTo(0);
        return bytes;
    }

    private boolean fileExistsInListing(Location location) throws IOException {
        FileIterator fileIterator = this.getFileSystem().listFiles(this.getRootLocation());
        while (fileIterator.hasNext()) {
            FileEntry fileEntry = fileIterator.next();
            if (!fileEntry.location().equals((Object)location)) continue;
            return true;
        }
        return false;
    }

    private boolean fileExists(Location location) throws IOException {
        return this.getFileSystem().newInputFile(location).exists();
    }

    private String readLocation(Location path) {
        String string;
        block8: {
            TrinoInputStream inputStream = this.getFileSystem().newInputFile(path).newStream();
            try {
                string = new String(inputStream.readAllBytes(), StandardCharsets.UTF_8);
                if (inputStream == null) break block8;
            }
            catch (Throwable throwable) {
                try {
                    if (inputStream != null) {
                        try {
                            inputStream.close();
                        }
                        catch (Throwable throwable2) {
                            throwable.addSuppressed(throwable2);
                        }
                    }
                    throw throwable;
                }
                catch (IOException e) {
                    throw new UncheckedIOException(e);
                }
            }
            inputStream.close();
        }
        return string;
    }

    protected Location createBlob(Closer closer, String path) {
        Location location = this.createLocation(path);
        ((TempBlob)closer.register((Closeable)new TempBlob(location))).createOrOverwrite(TEST_BLOB_CONTENT_PREFIX + location.toString());
        return location;
    }

    protected TempBlob randomBlobLocation(String nameHint) {
        TempBlob tempBlob = new TempBlob(this.createLocation("%s/%s".formatted(nameHint, UUID.randomUUID())));
        Assertions.assertThat((boolean)tempBlob.exists()).isFalse();
        return tempBlob;
    }

    private List<TempBlob> randomBlobs(Closer closer) {
        char[] chars = new char[]{'a', 'b', 'c', 'd', 'A', 'B', 'C', 'D'};
        ImmutableList.Builder names = ImmutableList.builder();
        for (int i = 0; i < 100; ++i) {
            StringBuilder name = new StringBuilder();
            for (int j = 0; j < 10; ++j) {
                name.append(chars[ThreadLocalRandom.current().nextInt(chars.length)]);
            }
            TempBlob tempBlob = new TempBlob(this.createLocation(name.toString()));
            Assertions.assertThat((boolean)tempBlob.exists()).isFalse();
            tempBlob.createOrOverwrite(TEST_BLOB_CONTENT_PREFIX + tempBlob.location().toString());
            closer.register((Closeable)tempBlob);
            names.add((Object)tempBlob);
        }
        return names.build();
    }

    private Set<Location> createTestDirectoryStructure(Closer closer, boolean hierarchicalNamingConstraints) {
        HashSet<Location> locations = new HashSet<Location>();
        if (!hierarchicalNamingConstraints) {
            locations.add(this.createBlob(closer, "level0"));
        }
        locations.add(this.createBlob(closer, "level0-file0"));
        locations.add(this.createBlob(closer, "level0-file1"));
        locations.add(this.createBlob(closer, "level0-file2"));
        if (!hierarchicalNamingConstraints) {
            locations.add(this.createBlob(closer, "level0/level1"));
        }
        locations.add(this.createBlob(closer, "level0/level1-file0"));
        locations.add(this.createBlob(closer, "level0/level1-file1"));
        locations.add(this.createBlob(closer, "level0/level1-file2"));
        if (!hierarchicalNamingConstraints) {
            locations.add(this.createBlob(closer, "level0/level1/level2"));
        }
        locations.add(this.createBlob(closer, "level0/level1/level2-file0"));
        locations.add(this.createBlob(closer, "level0/level1/level2-file1"));
        locations.add(this.createBlob(closer, "level0/level1/level2-file2"));
        return locations;
    }

    protected class TempBlob
    implements Closeable {
        private final Location location;
        private final TrinoFileSystem fileSystem;

        public TempBlob(Location location) {
            this.location = Objects.requireNonNull(location, "location is null");
            this.fileSystem = AbstractTestTrinoFileSystem.this.getFileSystem();
        }

        public Location location() {
            return this.location;
        }

        public boolean exists() {
            try {
                return this.inputFile().exists();
            }
            catch (IOException e) {
                throw new UncheckedIOException(e);
            }
        }

        public TrinoInputFile inputFile() {
            return this.fileSystem.newInputFile(this.location);
        }

        public TrinoOutputFile outputFile() {
            return this.fileSystem.newOutputFile(this.location);
        }

        public void createOrOverwrite(String data) {
            try {
                this.outputFile().createOrOverwrite(data.getBytes(StandardCharsets.UTF_8));
            }
            catch (IOException e) {
                throw new UncheckedIOException(e);
            }
            Assertions.assertThat((boolean)this.exists()).isTrue();
        }

        public String read() {
            return AbstractTestTrinoFileSystem.this.readLocation(this.location);
        }

        @Override
        public void close() {
            try {
                this.fileSystem.deleteFile(this.location);
            }
            catch (IOException iOException) {
                // empty catch block
            }
        }
    }
}

