/*
 * Decompiled with CFR 0.152.
 */
package io.trino.hdfs;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.MoreObjects;
import com.google.common.base.Strings;
import com.google.common.base.Throwables;
import com.google.common.collect.ImmutableSet;
import io.airlift.log.Logger;
import io.trino.hdfs.FileSystemFinalizerService;
import io.trino.hdfs.TrinoFileSystemCacheStats;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.net.URI;
import java.util.Collection;
import java.util.EnumSet;
import java.util.Locale;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.atomic.AtomicLong;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.BlockLocation;
import org.apache.hadoop.fs.CreateFlag;
import org.apache.hadoop.fs.FSDataInputStream;
import org.apache.hadoop.fs.FSDataOutputStream;
import org.apache.hadoop.fs.FileSystem;
import org.apache.hadoop.fs.FileSystemCache;
import org.apache.hadoop.fs.FilterFileSystem;
import org.apache.hadoop.fs.LocatedFileStatus;
import org.apache.hadoop.fs.Options;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.fs.RemoteIterator;
import org.apache.hadoop.fs.permission.FsPermission;
import org.apache.hadoop.security.UserGroupInformation;
import org.apache.hadoop.security.UserGroupInformationShim;
import org.apache.hadoop.util.Progressable;
import org.apache.hadoop.util.ReflectionUtils;
import org.gaul.modernizer_maven_annotations.SuppressModernizer;

public class TrinoFileSystemCache
implements FileSystemCache {
    private static final Logger log = Logger.get(TrinoFileSystemCache.class);
    public static final String CACHE_KEY = "fs.cache.credentials";
    static final TrinoFileSystemCache INSTANCE = new TrinoFileSystemCache();
    private final AtomicLong unique = new AtomicLong();
    private final TrinoFileSystemCacheStats stats;
    private final Map<FileSystemKey, FileSystemHolder> cache = new ConcurrentHashMap<FileSystemKey, FileSystemHolder>();
    private final AtomicLong cacheSize = new AtomicLong();

    @VisibleForTesting
    TrinoFileSystemCache() {
        this.stats = new TrinoFileSystemCacheStats(this.cache::size);
    }

    public FileSystem get(URI uri, Configuration conf) throws IOException {
        this.stats.newGetCall();
        return this.getInternal(uri, conf, 0L);
    }

    public FileSystem getUnique(URI uri, Configuration conf) throws IOException {
        this.stats.newGetUniqueCall();
        return this.getInternal(uri, conf, this.unique.incrementAndGet());
    }

    @VisibleForTesting
    int getCacheSize() {
        return this.cache.size();
    }

    TrinoFileSystemCacheStats getStats() {
        return this.stats;
    }

    private FileSystem getInternal(URI uri, Configuration conf, long unique) throws IOException {
        FileSystemHolder fileSystemHolder;
        UserGroupInformation userGroupInformation = UserGroupInformation.getCurrentUser();
        FileSystemKey key = TrinoFileSystemCache.createFileSystemKey(uri, userGroupInformation, unique);
        Set<?> privateCredentials = TrinoFileSystemCache.getPrivateCredentials(userGroupInformation);
        int maxSize = conf.getInt("fs.cache.max-size", 1000);
        try {
            fileSystemHolder = this.cache.compute(key, (k, currentFileSystemHolder) -> {
                if (currentFileSystemHolder == null) {
                    if (this.cacheSize.getAndUpdate(currentSize -> Math.min(currentSize + 1L, (long)maxSize)) >= (long)maxSize) {
                        throw new RuntimeException(new IOException(String.format("FileSystem max cache size has been reached: %s", maxSize)));
                    }
                    return new FileSystemHolder(conf, privateCredentials);
                }
                if (currentFileSystemHolder.credentialsChanged(uri, conf, privateCredentials)) {
                    return new FileSystemHolder(conf, privateCredentials);
                }
                return currentFileSystemHolder;
            });
            fileSystemHolder.createFileSystemOnce(uri, conf);
        }
        catch (IOException | RuntimeException e) {
            this.stats.newGetCallFailed();
            Throwables.throwIfInstanceOf((Throwable)e, IOException.class);
            Throwables.throwIfInstanceOf((Throwable)e.getCause(), IOException.class);
            throw e;
        }
        return fileSystemHolder.getFileSystem();
    }

    private static FileSystem createFileSystem(URI uri, Configuration conf) throws IOException {
        Class clazz = FileSystem.getFileSystemClass((String)uri.getScheme(), (Configuration)conf);
        if (clazz == null) {
            throw new IOException("No FileSystem for scheme: " + uri.getScheme());
        }
        FileSystem original = (FileSystem)ReflectionUtils.newInstance((Class)clazz, (Configuration)conf);
        original.initialize(uri, conf);
        FileSystemWrapper wrapper = new FileSystemWrapper(original);
        FileSystemFinalizerService.getInstance().addFinalizer((Object)wrapper, () -> {
            try {
                TrinoFileSystemCache.closeFileSystem(original);
            }
            catch (IOException e) {
                log.error((Throwable)e, "Error occurred when finalizing file system");
            }
        });
        return wrapper;
    }

    public void remove(FileSystem fileSystem) {
        this.stats.newRemoveCall();
        this.cache.forEach((key, fileSystemHolder) -> {
            if (fileSystem.equals(fileSystemHolder.getFileSystem())) {
                this.cache.compute((FileSystemKey)key, (k, currentFileSystemHolder) -> {
                    if (currentFileSystemHolder != null && fileSystem.equals(currentFileSystemHolder.getFileSystem())) {
                        this.cacheSize.decrementAndGet();
                        return null;
                    }
                    return currentFileSystemHolder;
                });
            }
        });
    }

    public void closeAll() throws IOException {
        try {
            this.cache.forEach((key, fileSystemHolder) -> {
                try {
                    this.cache.compute((FileSystemKey)key, (k, currentFileSystemHolder) -> {
                        if (currentFileSystemHolder != null) {
                            this.cacheSize.decrementAndGet();
                        }
                        return null;
                    });
                    FileSystem fs = fileSystemHolder.getFileSystem();
                    if (fs != null) {
                        TrinoFileSystemCache.closeFileSystem(fs);
                    }
                }
                catch (IOException e) {
                    throw new RuntimeException(e);
                }
            });
        }
        catch (RuntimeException e) {
            Throwables.throwIfInstanceOf((Throwable)e.getCause(), IOException.class);
            throw e;
        }
    }

    @SuppressModernizer
    private static void closeFileSystem(FileSystem fileSystem) throws IOException {
        fileSystem.close();
    }

    private static FileSystemKey createFileSystemKey(URI uri, UserGroupInformation userGroupInformation, long unique) {
        String realUser;
        String scheme = Strings.nullToEmpty((String)uri.getScheme()).toLowerCase(Locale.ENGLISH);
        String authority = Strings.nullToEmpty((String)uri.getAuthority()).toLowerCase(Locale.ENGLISH);
        UserGroupInformation.AuthenticationMethod authenticationMethod = userGroupInformation.getAuthenticationMethod();
        return new FileSystemKey(scheme, authority, unique, realUser, switch (authenticationMethod) {
            case UserGroupInformation.AuthenticationMethod.SIMPLE, UserGroupInformation.AuthenticationMethod.KERBEROS -> {
                realUser = userGroupInformation.getUserName();
                yield null;
            }
            case UserGroupInformation.AuthenticationMethod.PROXY -> {
                realUser = userGroupInformation.getRealUser().getUserName();
                yield userGroupInformation.getUserName();
            }
            default -> throw new IllegalArgumentException("Unsupported authentication method: " + authenticationMethod);
        });
    }

    private static Set<?> getPrivateCredentials(UserGroupInformation userGroupInformation) {
        UserGroupInformation.AuthenticationMethod authenticationMethod = userGroupInformation.getAuthenticationMethod();
        return switch (authenticationMethod) {
            case UserGroupInformation.AuthenticationMethod.SIMPLE -> ImmutableSet.of();
            case UserGroupInformation.AuthenticationMethod.KERBEROS -> ImmutableSet.copyOf(UserGroupInformationShim.getSubject((UserGroupInformation)userGroupInformation).getPrivateCredentials());
            case UserGroupInformation.AuthenticationMethod.PROXY -> TrinoFileSystemCache.getPrivateCredentials(userGroupInformation.getRealUser());
            default -> throw new IllegalArgumentException("Unsupported authentication method: " + authenticationMethod);
        };
    }

    private static boolean isHdfs(URI uri) {
        String scheme = uri.getScheme();
        return "hdfs".equals(scheme) || "viewfs".equals(scheme);
    }

    private record FileSystemKey(String scheme, String authority, long unique, String realUser, String proxyUser) {
        private FileSystemKey {
            Objects.requireNonNull(scheme, "scheme is null");
            Objects.requireNonNull(authority, "authority is null");
            Objects.requireNonNull(realUser, "realUser");
        }
    }

    private static class FileSystemHolder {
        private final Set<?> privateCredentials;
        private final String cacheCredentials;
        private volatile FileSystem fileSystem;

        public FileSystemHolder(Configuration conf, Set<?> privateCredentials) {
            this.privateCredentials = ImmutableSet.copyOf((Collection)Objects.requireNonNull(privateCredentials, "privateCredentials is null"));
            this.cacheCredentials = conf.get(TrinoFileSystemCache.CACHE_KEY, "");
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        public void createFileSystemOnce(URI uri, Configuration conf) throws IOException {
            if (this.fileSystem == null) {
                FileSystemHolder fileSystemHolder = this;
                synchronized (fileSystemHolder) {
                    if (this.fileSystem == null) {
                        this.fileSystem = TrinoFileSystemCache.createFileSystem(uri, conf);
                    }
                }
            }
        }

        public boolean credentialsChanged(URI newUri, Configuration newConf, Set<?> newPrivateCredentials) {
            return TrinoFileSystemCache.isHdfs(newUri) && !this.privateCredentials.equals(newPrivateCredentials) || !this.cacheCredentials.equals(newConf.get(TrinoFileSystemCache.CACHE_KEY, ""));
        }

        public FileSystem getFileSystem() {
            return this.fileSystem;
        }

        public String toString() {
            return MoreObjects.toStringHelper((Object)this).add("fileSystem", (Object)this.fileSystem).add("privateCredentials", this.privateCredentials).add("cacheCredentials", (Object)this.cacheCredentials).toString();
        }
    }

    private static class FileSystemWrapper
    extends FilterFileSystem {
        public FileSystemWrapper(FileSystem fs) {
            super(fs);
        }

        public FSDataInputStream open(Path f, int bufferSize) throws IOException {
            return new InputStreamWrapper(this.getRawFileSystem().open(f, bufferSize), this);
        }

        public String getScheme() {
            return this.getRawFileSystem().getScheme();
        }

        public FSDataOutputStream append(Path f, int bufferSize, Progressable progress) throws IOException {
            return new OutputStreamWrapper(this.getRawFileSystem().append(f, bufferSize, progress), this);
        }

        public FSDataOutputStream create(Path f, FsPermission permission, boolean overwrite, int bufferSize, short replication, long blockSize, Progressable progress) throws IOException {
            return new OutputStreamWrapper(this.getRawFileSystem().create(f, permission, overwrite, bufferSize, replication, blockSize, progress), this);
        }

        public FSDataOutputStream create(Path f, FsPermission permission, EnumSet<CreateFlag> flags, int bufferSize, short replication, long blockSize, Progressable progress, Options.ChecksumOpt checksumOpt) throws IOException {
            return new OutputStreamWrapper(this.getRawFileSystem().create(f, permission, flags, bufferSize, replication, blockSize, progress, checksumOpt), this);
        }

        public FSDataOutputStream createNonRecursive(Path f, FsPermission permission, EnumSet<CreateFlag> flags, int bufferSize, short replication, long blockSize, Progressable progress) throws IOException {
            return new OutputStreamWrapper(this.getRawFileSystem().createNonRecursive(f, permission, flags, bufferSize, replication, blockSize, progress), this);
        }

        public BlockLocation[] getFileBlockLocations(Path p, long start, long len) throws IOException {
            return this.fs.getFileBlockLocations(p, start, len);
        }

        public RemoteIterator<LocatedFileStatus> listFiles(Path path, boolean recursive) throws IOException {
            return new RemoteIteratorWrapper((RemoteIterator<LocatedFileStatus>)this.fs.listFiles(path, recursive), this);
        }
    }

    private static class RemoteIteratorWrapper
    implements RemoteIterator<LocatedFileStatus> {
        private final RemoteIterator<LocatedFileStatus> delegate;
        private final FileSystemWrapper owningFileSystemWrapper;

        public RemoteIteratorWrapper(RemoteIterator<LocatedFileStatus> delegate, FileSystemWrapper owningFileSystemWrapper) {
            this.delegate = delegate;
            this.owningFileSystemWrapper = Objects.requireNonNull(owningFileSystemWrapper, "owningFileSystemWrapper is null");
        }

        public boolean hasNext() throws IOException {
            return this.delegate.hasNext();
        }

        public LocatedFileStatus next() throws IOException {
            return (LocatedFileStatus)this.delegate.next();
        }
    }

    private static class InputStreamWrapper
    extends FSDataInputStream {
        private final FileSystemWrapper owningFileSystemWrapper;

        public InputStreamWrapper(FSDataInputStream inputStream, FileSystemWrapper owningFileSystemWrapper) {
            super((InputStream)inputStream);
            this.owningFileSystemWrapper = Objects.requireNonNull(owningFileSystemWrapper, "owningFileSystemWrapper is null");
        }

        public InputStream getWrappedStream() {
            return ((FSDataInputStream)super.getWrappedStream()).getWrappedStream();
        }
    }

    private static class OutputStreamWrapper
    extends FSDataOutputStream {
        private final FileSystemWrapper owningFileSystemWrapper;

        public OutputStreamWrapper(FSDataOutputStream delegate, FileSystemWrapper owningFileSystemWrapper) {
            super((OutputStream)delegate, null, delegate.getPos());
            this.owningFileSystemWrapper = Objects.requireNonNull(owningFileSystemWrapper, "owningFileSystemWrapper is null");
        }

        public OutputStream getWrappedStream() {
            return ((FSDataOutputStream)super.getWrappedStream()).getWrappedStream();
        }
    }
}

