/*
 * Decompiled with CFR 0.152.
 */
package io.trino.hive.formats.rcfile;

import com.google.common.base.Preconditions;
import com.google.common.io.Closer;
import io.airlift.slice.DynamicSliceOutput;
import io.airlift.slice.SizeOf;
import io.airlift.slice.Slice;
import io.airlift.slice.SliceOutput;
import io.airlift.slice.Slices;
import io.airlift.units.DataSize;
import io.trino.filesystem.TrinoInputFile;
import io.trino.hive.formats.DataOutputStream;
import io.trino.hive.formats.FileCorruptionException;
import io.trino.hive.formats.ReadWriteUtils;
import io.trino.hive.formats.compression.Codec;
import io.trino.hive.formats.compression.CompressionKind;
import io.trino.hive.formats.compression.MemoryCompressedSliceOutput;
import io.trino.hive.formats.encodings.ColumnEncoding;
import io.trino.hive.formats.encodings.ColumnEncodingFactory;
import io.trino.hive.formats.encodings.EncodeOutput;
import io.trino.hive.formats.rcfile.PageSplitterUtil;
import io.trino.hive.formats.rcfile.RcFileReader;
import io.trino.hive.formats.rcfile.RcFileWriteValidation;
import io.trino.spi.Page;
import io.trino.spi.block.Block;
import io.trino.spi.type.Type;
import java.io.Closeable;
import java.io.IOException;
import java.io.OutputStream;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.concurrent.ThreadLocalRandom;
import java.util.function.Consumer;
import javax.annotation.Nullable;

public class RcFileWriter
implements Closeable {
    private static final int INSTANCE_SIZE = SizeOf.instanceSize(RcFileWriter.class);
    private static final Slice RCFILE_MAGIC = Slices.utf8Slice((String)"RCF");
    private static final int CURRENT_VERSION = 1;
    private static final String COLUMN_COUNT_METADATA_KEY = "hive.io.rcfile.column.number";
    private static final DataSize DEFAULT_TARGET_MIN_ROW_GROUP_SIZE = DataSize.of((long)4L, (DataSize.Unit)DataSize.Unit.MEGABYTE);
    private static final DataSize DEFAULT_TARGET_MAX_ROW_GROUP_SIZE = DataSize.of((long)8L, (DataSize.Unit)DataSize.Unit.MEGABYTE);
    private static final DataSize MIN_BUFFER_SIZE = DataSize.of((long)4L, (DataSize.Unit)DataSize.Unit.KILOBYTE);
    private static final DataSize MAX_BUFFER_SIZE = DataSize.of((long)1L, (DataSize.Unit)DataSize.Unit.MEGABYTE);
    static final String PRESTO_RCFILE_WRITER_VERSION_METADATA_KEY = "presto.writer.version";
    static final String PRESTO_RCFILE_WRITER_VERSION;
    private final DataOutputStream output;
    private final List<Type> types;
    private final ColumnEncodingFactory encoding;
    private final long syncFirst = ThreadLocalRandom.current().nextLong();
    private final long syncSecond = ThreadLocalRandom.current().nextLong();
    private MemoryCompressedSliceOutput keySectionOutput;
    private final ColumnEncoder[] columnEncoders;
    private final int targetMinRowGroupSize;
    private final int targetMaxRowGroupSize;
    private int bufferedSize;
    private int bufferedRows;
    private long totalRowCount;
    @Nullable
    private final RcFileWriteValidation.RcFileWriteValidationBuilder validationBuilder;

    public RcFileWriter(OutputStream rawOutput, List<Type> types, ColumnEncodingFactory encoding, Optional<CompressionKind> compressionKind, Map<String, String> metadata, boolean validate) throws IOException {
        this(rawOutput, types, encoding, compressionKind, metadata, DEFAULT_TARGET_MIN_ROW_GROUP_SIZE, DEFAULT_TARGET_MAX_ROW_GROUP_SIZE, validate);
    }

    public RcFileWriter(OutputStream rawOutput, List<Type> types, ColumnEncodingFactory encoding, Optional<CompressionKind> compressionKind, Map<String, String> metadata, DataSize targetMinRowGroupSize, DataSize targetMaxRowGroupSize, boolean validate) throws IOException {
        Objects.requireNonNull(rawOutput, "rawOutput is null");
        Objects.requireNonNull(types, "types is null");
        Preconditions.checkArgument((!types.isEmpty() ? 1 : 0) != 0, (Object)"types is empty");
        Objects.requireNonNull(encoding, "encoding is null");
        Objects.requireNonNull(compressionKind, "compressionKind is null");
        Preconditions.checkArgument((!compressionKind.equals(Optional.of(CompressionKind.LZOP)) ? 1 : 0) != 0, (Object)"LZOP cannot be use with RCFile.  LZO compression can be used, but LZ4 is preferred.");
        Objects.requireNonNull(metadata, "metadata is null");
        Preconditions.checkArgument((!metadata.containsKey(PRESTO_RCFILE_WRITER_VERSION_METADATA_KEY) ? 1 : 0) != 0, (String)"Cannot set property %s", (Object)PRESTO_RCFILE_WRITER_VERSION_METADATA_KEY);
        Preconditions.checkArgument((!metadata.containsKey(COLUMN_COUNT_METADATA_KEY) ? 1 : 0) != 0, (String)"Cannot set property %s", (Object)COLUMN_COUNT_METADATA_KEY);
        Objects.requireNonNull(targetMinRowGroupSize, "targetMinRowGroupSize is null");
        Objects.requireNonNull(targetMaxRowGroupSize, "targetMaxRowGroupSize is null");
        Preconditions.checkArgument((targetMinRowGroupSize.compareTo(targetMaxRowGroupSize) <= 0 ? 1 : 0) != 0, (Object)"targetMinRowGroupSize must be less than or equal to targetMaxRowGroupSize");
        this.validationBuilder = validate ? new RcFileWriteValidation.RcFileWriteValidationBuilder(types) : null;
        this.output = new DataOutputStream(rawOutput);
        this.types = types;
        this.encoding = encoding;
        this.output.write(RCFILE_MAGIC);
        this.output.writeByte(1);
        this.recordValidation(validation -> validation.setVersion((byte)1));
        this.output.writeBoolean(compressionKind.isPresent());
        if (compressionKind.isPresent()) {
            ReadWriteUtils.writeLengthPrefixedString(this.output, Slices.utf8Slice((String)compressionKind.get().getHadoopClassName()));
        }
        this.recordValidation(validation -> validation.setCodecClassName(compressionKind.map(CompressionKind::getHadoopClassName)));
        this.output.writeInt(Integer.reverseBytes(metadata.size() + 2));
        this.writeMetadataProperty(COLUMN_COUNT_METADATA_KEY, Integer.toString(types.size()));
        this.writeMetadataProperty(PRESTO_RCFILE_WRITER_VERSION_METADATA_KEY, PRESTO_RCFILE_WRITER_VERSION);
        for (Map.Entry<String, String> entry : metadata.entrySet()) {
            this.writeMetadataProperty(entry.getKey(), entry.getValue());
        }
        this.output.writeLong(this.syncFirst);
        this.recordValidation(validation -> validation.setSyncFirst(this.syncFirst));
        this.output.writeLong(this.syncSecond);
        this.recordValidation(validation -> validation.setSyncSecond(this.syncSecond));
        Optional<Codec> codec = compressionKind.map(CompressionKind::createCodec);
        this.keySectionOutput = RcFileWriter.createMemoryCompressedSliceOutput(codec);
        this.keySectionOutput.close();
        this.columnEncoders = new ColumnEncoder[types.size()];
        for (int columnIndex = 0; columnIndex < types.size(); ++columnIndex) {
            Type type = types.get(columnIndex);
            ColumnEncoding columnEncoding = encoding.getEncoding(type);
            this.columnEncoders[columnIndex] = new ColumnEncoder(columnEncoding, codec);
        }
        this.targetMinRowGroupSize = StrictMath.toIntExact(targetMinRowGroupSize.toBytes());
        this.targetMaxRowGroupSize = StrictMath.toIntExact(targetMaxRowGroupSize.toBytes());
    }

    private void writeMetadataProperty(String key, String value) throws IOException {
        ReadWriteUtils.writeLengthPrefixedString(this.output, Slices.utf8Slice((String)key));
        ReadWriteUtils.writeLengthPrefixedString(this.output, Slices.utf8Slice((String)value));
        this.recordValidation(validation -> validation.addMetadataProperty(key, value));
    }

    @Override
    public void close() throws IOException {
        try (Closer closer = Closer.create();){
            closer.register((Closeable)this.output);
            closer.register(this.keySectionOutput::destroy);
            for (ColumnEncoder columnEncoder : this.columnEncoders) {
                closer.register(columnEncoder::destroy);
            }
            this.writeRowGroup();
        }
    }

    private void recordValidation(Consumer<RcFileWriteValidation.RcFileWriteValidationBuilder> task) {
        if (this.validationBuilder != null) {
            task.accept(this.validationBuilder);
        }
    }

    public void validate(TrinoInputFile inputFile) throws FileCorruptionException {
        Preconditions.checkState((this.validationBuilder != null ? 1 : 0) != 0, (Object)"validation is not enabled");
        RcFileReader.validateFile(this.validationBuilder.build(), inputFile, this.encoding, this.types);
    }

    public long getRetainedSizeInBytes() {
        long retainedSize = INSTANCE_SIZE;
        retainedSize += this.output.getRetainedSize();
        retainedSize += this.keySectionOutput.getRetainedSize();
        for (ColumnEncoder columnEncoder : this.columnEncoders) {
            retainedSize += columnEncoder.getRetainedSizeInBytes();
        }
        return retainedSize;
    }

    public void write(Page page) throws IOException {
        if (page.getPositionCount() == 0) {
            return;
        }
        List<Page> pages = PageSplitterUtil.splitPage(page, this.targetMaxRowGroupSize);
        for (Page splitPage : pages) {
            this.bufferPage(splitPage);
        }
    }

    private void bufferPage(Page page) throws IOException {
        this.bufferedRows += page.getPositionCount();
        this.bufferedSize = 0;
        for (int i = 0; i < page.getChannelCount(); ++i) {
            Block block = page.getBlock(i);
            this.columnEncoders[i].writeBlock(block);
            this.bufferedSize += this.columnEncoders[i].getBufferedSize();
        }
        this.recordValidation(validation -> validation.addPage(page));
        if (this.bufferedSize >= this.targetMinRowGroupSize) {
            this.writeRowGroup();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void writeRowGroup() throws IOException {
        if (this.bufferedRows == 0) {
            return;
        }
        if (this.totalRowCount != 0L) {
            this.output.writeInt(-1);
            this.output.writeLong(this.syncFirst);
            this.output.writeLong(this.syncSecond);
        }
        for (ColumnEncoder columnEncoder : this.columnEncoders) {
            columnEncoder.closeColumn();
        }
        int valueLength = 0;
        this.keySectionOutput = this.keySectionOutput.createRecycledCompressedSliceOutput();
        try {
            ReadWriteUtils.writeVInt(this.keySectionOutput, this.bufferedRows);
            this.recordValidation(validation -> validation.addRowGroup(this.bufferedRows));
            for (ColumnEncoder columnEncoder : this.columnEncoders) {
                valueLength += columnEncoder.getCompressedSize();
                ReadWriteUtils.writeVInt(this.keySectionOutput, columnEncoder.getCompressedSize());
                ReadWriteUtils.writeVInt(this.keySectionOutput, columnEncoder.getUncompressedSize());
                Slice lengthData = columnEncoder.getLengthData();
                ReadWriteUtils.writeVInt(this.keySectionOutput, lengthData.length());
                this.keySectionOutput.writeBytes(lengthData);
            }
        }
        finally {
            this.keySectionOutput.close();
        }
        this.output.writeInt(Integer.reverseBytes(this.keySectionOutput.size() + valueLength));
        this.output.writeInt(Integer.reverseBytes(this.keySectionOutput.size()));
        this.output.writeInt(Integer.reverseBytes(this.keySectionOutput.getCompressedSize()));
        for (Slice slice : this.keySectionOutput.getCompressedSlices()) {
            this.output.write(slice);
        }
        for (ColumnEncoder columnEncoder : this.columnEncoders) {
            List<Slice> slices = columnEncoder.getCompressedData();
            for (Slice slice : slices) {
                this.output.write(slice);
            }
            columnEncoder.reset();
        }
        this.totalRowCount += (long)this.bufferedRows;
        this.bufferedSize = 0;
        this.bufferedRows = 0;
    }

    private static MemoryCompressedSliceOutput createMemoryCompressedSliceOutput(Optional<Codec> codec) throws IOException {
        if (codec.isPresent()) {
            return codec.get().createMemoryCompressedSliceOutput((int)MIN_BUFFER_SIZE.toBytes(), (int)MAX_BUFFER_SIZE.toBytes());
        }
        return MemoryCompressedSliceOutput.createUncompressedMemorySliceOutput((int)MIN_BUFFER_SIZE.toBytes(), (int)MAX_BUFFER_SIZE.toBytes());
    }

    static {
        String version = RcFileWriter.class.getPackage().getImplementationVersion();
        PRESTO_RCFILE_WRITER_VERSION = version == null ? "UNKNOWN" : version;
    }

    private static class ColumnEncoder {
        private static final int INSTANCE_SIZE = SizeOf.instanceSize(ColumnEncoder.class) + SizeOf.instanceSize(ColumnEncodeOutput.class);
        private final ColumnEncoding columnEncoding;
        private ColumnEncodeOutput encodeOutput;
        private final SliceOutput lengthOutput = new DynamicSliceOutput(512);
        private MemoryCompressedSliceOutput output;
        private boolean columnClosed;

        public ColumnEncoder(ColumnEncoding columnEncoding, Optional<Codec> codec) throws IOException {
            this.columnEncoding = columnEncoding;
            this.output = RcFileWriter.createMemoryCompressedSliceOutput(codec);
            this.encodeOutput = new ColumnEncodeOutput(this.lengthOutput, this.output);
        }

        private void writeBlock(Block block) throws IOException {
            Preconditions.checkArgument((!this.columnClosed ? 1 : 0) != 0, (Object)"Column is closed");
            this.columnEncoding.encodeColumn(block, this.output, this.encodeOutput);
        }

        public void closeColumn() throws IOException {
            Preconditions.checkArgument((!this.columnClosed ? 1 : 0) != 0, (Object)"Column is not open");
            this.encodeOutput.flush();
            this.output.close();
            this.columnClosed = true;
        }

        public int getBufferedSize() {
            return this.lengthOutput.size() + this.output.size();
        }

        public Slice getLengthData() {
            Preconditions.checkArgument((boolean)this.columnClosed, (Object)"Column is open");
            return this.lengthOutput.slice();
        }

        public int getUncompressedSize() {
            Preconditions.checkArgument((boolean)this.columnClosed, (Object)"Column is open");
            return this.output.size();
        }

        public int getCompressedSize() {
            Preconditions.checkArgument((boolean)this.columnClosed, (Object)"Column is open");
            return this.output.getCompressedSize();
        }

        public List<Slice> getCompressedData() {
            Preconditions.checkArgument((boolean)this.columnClosed, (Object)"Column is open");
            return this.output.getCompressedSlices();
        }

        public void reset() throws IOException {
            Preconditions.checkArgument((boolean)this.columnClosed, (Object)"Column is open");
            this.lengthOutput.reset();
            this.output = this.output.createRecycledCompressedSliceOutput();
            this.encodeOutput = new ColumnEncodeOutput(this.lengthOutput, this.output);
            this.columnClosed = false;
        }

        public void destroy() throws IOException {
            this.output.destroy();
        }

        public long getRetainedSizeInBytes() {
            return (long)INSTANCE_SIZE + this.lengthOutput.getRetainedSize() + this.output.getRetainedSize();
        }

        private static class ColumnEncodeOutput
        implements EncodeOutput {
            private final SliceOutput lengthOutput;
            private final SliceOutput valueOutput;
            private int previousOffset;
            private int previousLength;
            private int runLength;

            public ColumnEncodeOutput(SliceOutput lengthOutput, SliceOutput valueOutput) {
                this.lengthOutput = lengthOutput;
                this.valueOutput = valueOutput;
                this.previousOffset = valueOutput.size();
                this.previousLength = -1;
            }

            @Override
            public void closeEntry() {
                int offset = this.valueOutput.size();
                int length = offset - this.previousOffset;
                this.previousOffset = offset;
                if (length == this.previousLength) {
                    ++this.runLength;
                } else {
                    if (this.runLength > 0) {
                        int value = ~this.runLength;
                        ReadWriteUtils.writeVInt(this.lengthOutput, value);
                    }
                    ReadWriteUtils.writeVInt(this.lengthOutput, length);
                    this.previousLength = length;
                    this.runLength = 0;
                }
            }

            private void flush() {
                if (this.runLength > 0) {
                    int value = ~this.runLength;
                    ReadWriteUtils.writeVInt(this.lengthOutput, value);
                }
                this.previousLength = -1;
                this.runLength = 0;
            }
        }
    }
}

