/*
 * Decompiled with CFR 0.152.
 */
package io.trino.hive.formats;

import com.google.common.base.Preconditions;
import io.airlift.slice.Slice;
import io.airlift.slice.SliceInput;
import io.airlift.slice.SliceOutput;
import io.airlift.slice.SliceUtf8;
import io.airlift.slice.Slices;
import io.trino.filesystem.TrinoInput;
import io.trino.filesystem.TrinoInputFile;
import io.trino.hive.formats.DataOutputStream;
import io.trino.hive.formats.TrinoDataInputStream;
import io.trino.spi.type.CharType;
import io.trino.spi.type.Type;
import io.trino.spi.type.VarcharType;
import java.io.IOException;
import java.util.Objects;

public final class ReadWriteUtils {
    private static final int SYNC_SEQUENCE_LENGTH = 20;

    private ReadWriteUtils() {
    }

    public static int decodeVIntSize(Slice slice, int offset) {
        return ReadWriteUtils.decodeVIntSize(slice.getByte(offset));
    }

    public static int decodeVIntSize(byte value) {
        if (value >= -112) {
            return 1;
        }
        if (value < -120) {
            return -119 - value;
        }
        return -111 - value;
    }

    public static boolean isNegativeVInt(Slice slice, int offset) {
        return ReadWriteUtils.isNegativeVInt(slice.getByte(offset));
    }

    public static boolean isNegativeVInt(byte value) {
        return value < -120 || value >= -112 && value < 0;
    }

    public static long readVInt(TrinoDataInputStream in) throws IOException {
        byte firstByte = in.readByte();
        int length = ReadWriteUtils.decodeVIntSize(firstByte);
        if (length == 1) {
            return firstByte;
        }
        long value = 0L;
        for (int i = 1; i < length; ++i) {
            value <<= 8;
            value |= (long)(in.readByte() & 0xFF);
        }
        return ReadWriteUtils.isNegativeVInt(firstByte) ? value ^ 0xFFFFFFFFFFFFFFFFL : value;
    }

    public static long readVInt(SliceInput in) {
        byte firstByte = in.readByte();
        int length = ReadWriteUtils.decodeVIntSize(firstByte);
        if (length == 1) {
            return firstByte;
        }
        long value = 0L;
        for (int i = 1; i < length; ++i) {
            value <<= 8;
            value |= (long)(in.readByte() & 0xFF);
        }
        return ReadWriteUtils.isNegativeVInt(firstByte) ? value ^ 0xFFFFFFFFFFFFFFFFL : value;
    }

    public static long readVInt(Slice slice, int start) {
        byte firstByte = slice.getByte(start);
        int length = ReadWriteUtils.decodeVIntSize(firstByte);
        if (length == 1) {
            return firstByte;
        }
        return ReadWriteUtils.readVIntInternal(slice, start, length);
    }

    public static long readVInt(Slice slice, int start, int length) {
        if (length == 1) {
            return slice.getByte(start);
        }
        return ReadWriteUtils.readVIntInternal(slice, start, length);
    }

    private static long readVIntInternal(Slice slice, int start, int length) {
        long value = 0L;
        for (int i = 1; i < length; ++i) {
            value <<= 8;
            value |= (long)(slice.getByte(start + i) & 0xFF);
        }
        return ReadWriteUtils.isNegativeVInt(slice.getByte(start)) ? value ^ 0xFFFFFFFFFFFFFFFFL : value;
    }

    public static long findFirstSyncPosition(TrinoInputFile inputFile, long offset, long length, long syncFirst, long syncSecond) throws IOException {
        Objects.requireNonNull(inputFile, "inputFile is null");
        Preconditions.checkArgument((offset >= 0L ? 1 : 0) != 0, (Object)"offset is negative");
        Preconditions.checkArgument((length >= 1L ? 1 : 0) != 0, (Object)"length must be at least 1");
        Preconditions.checkArgument((offset + length <= inputFile.length() ? 1 : 0) != 0, (Object)"offset plus length is greater than data size");
        Slice sync = Slices.allocate((int)20);
        sync.setInt(0, -1);
        sync.setLong(4, syncFirst);
        sync.setLong(12, syncSecond);
        byte[] buffer = new byte[Math.toIntExact(Math.min(0x400000L, length + 19L))];
        Slice bufferSlice = Slices.wrappedBuffer((byte[])buffer);
        try (TrinoInput input = inputFile.newInput();){
            for (long position = 0L; position < length; position += (long)(bufferSlice.length() - 19)) {
                int bufferSize = Math.toIntExact(Math.min((long)buffer.length, length + 19L - position));
                bufferSize = Math.toIntExact(Math.min((long)bufferSize, inputFile.length() - offset - position));
                input.readFully(offset + position, buffer, 0, bufferSize);
                int index = bufferSlice.indexOf(sync);
                if (index < 0) continue;
                if (position + (long)index < length) {
                    long startOfSyncSequence;
                    long l = startOfSyncSequence = offset + position + (long)index;
                    return l;
                }
                long l = -1L;
                return l;
            }
        }
        return -1L;
    }

    public static void writeLengthPrefixedString(DataOutputStream out, Slice slice) throws IOException {
        ReadWriteUtils.writeVInt(out, slice.length());
        out.write(slice);
    }

    public static void writeVInt(DataOutputStream out, int value) throws IOException {
        if (value >= -112 && value <= 127) {
            out.writeByte(value);
            return;
        }
        int length = -112;
        if (value < 0) {
            value ^= 0xFFFFFFFF;
            length = -120;
        }
        int tmp = value;
        while (tmp != 0) {
            tmp >>= 8;
            --length;
        }
        out.writeByte(length);
        for (int idx = length = length < -120 ? -(length + 120) : -(length + 112); idx != 0; --idx) {
            int shiftBits = (idx - 1) * 8;
            out.writeByte(value >> shiftBits & 0xFF);
        }
    }

    public static int computeVIntLength(int value) {
        if (value >= -112 && value <= 127) {
            return 1;
        }
        if (value < 0) {
            value ^= 0xFFFFFFFF;
        }
        return (31 - Integer.numberOfLeadingZeros(value)) / 8 + 2;
    }

    public static void writeVInt(SliceOutput out, int value) {
        if (value >= -112 && value <= 127) {
            out.writeByte(value);
            return;
        }
        int length = -112;
        if (value < 0) {
            value ^= 0xFFFFFFFF;
            length = -120;
        }
        int tmp = value;
        while (tmp != 0) {
            tmp >>= 8;
            --length;
        }
        out.writeByte(length);
        for (int idx = length = length < -120 ? -(length + 120) : -(length + 112); idx != 0; --idx) {
            int shiftBits = (idx - 1) * 8;
            out.writeByte(value >> shiftBits & 0xFF);
        }
    }

    public static void writeVLong(SliceOutput out, long value) {
        if (value >= -112L && value <= 127L) {
            out.writeByte((int)((byte)value));
            return;
        }
        int length = -112;
        if (value < 0L) {
            value ^= 0xFFFFFFFFFFFFFFFFL;
            length = -120;
        }
        long tmp = value;
        while (tmp != 0L) {
            tmp >>= 8;
            --length;
        }
        out.writeByte(length);
        for (int idx = length = length < -120 ? -(length + 120) : -(length + 112); idx != 0; --idx) {
            int shiftBits = (idx - 1) * 8;
            long mask = 255L << shiftBits;
            out.writeByte((int)((byte)((value & mask) >> shiftBits)));
        }
    }

    public static int calculateTruncationLength(Type type, Slice slice, int offset, int length) {
        Objects.requireNonNull(type, "type is null");
        if (type instanceof VarcharType) {
            VarcharType varcharType = (VarcharType)type;
            if (varcharType.isUnbounded()) {
                return length;
            }
            return ReadWriteUtils.calculateTruncationLength(varcharType.getBoundedLength(), slice, offset, length);
        }
        if (type instanceof CharType) {
            int truncationLength;
            CharType charType = (CharType)type;
            for (truncationLength = ReadWriteUtils.calculateTruncationLength(charType.getLength(), slice, offset, length); truncationLength > 0 && slice.getByte(offset + truncationLength - 1) == 32; --truncationLength) {
            }
            return truncationLength;
        }
        return length;
    }

    private static int calculateTruncationLength(int maxCharacterCount, Slice slice, int offset, int length) {
        Objects.requireNonNull(slice, "slice is null");
        if (maxCharacterCount < 0) {
            throw new IllegalArgumentException("Max length must be greater or equal than zero");
        }
        if (length <= maxCharacterCount) {
            return length;
        }
        int indexEnd = SliceUtf8.offsetOfCodePoint((Slice)slice, (int)offset, (int)maxCharacterCount);
        if (indexEnd < 0) {
            return length;
        }
        return Math.min(indexEnd - offset, length);
    }
}

