/*
 * Decompiled with CFR 0.152.
 */
package io.trino.plugin.hive;

import com.google.common.base.Joiner;
import com.google.common.base.Preconditions;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.Iterables;
import com.google.common.primitives.Shorts;
import com.google.common.primitives.SignedBytes;
import io.airlift.slice.Slice;
import io.trino.plugin.hive.HiveColumnHandle;
import io.trino.spi.predicate.Domain;
import io.trino.spi.predicate.Range;
import io.trino.spi.predicate.TupleDomain;
import io.trino.spi.type.BigintType;
import io.trino.spi.type.BooleanType;
import io.trino.spi.type.DateType;
import io.trino.spi.type.DecimalType;
import io.trino.spi.type.Decimals;
import io.trino.spi.type.IntegerType;
import io.trino.spi.type.SmallintType;
import io.trino.spi.type.TinyintType;
import io.trino.spi.type.Type;
import io.trino.spi.type.TypeManager;
import io.trino.spi.type.VarcharType;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;
import org.joda.time.Chronology;
import org.joda.time.chrono.ISOChronology;
import org.joda.time.format.DateTimeFormatter;
import org.joda.time.format.ISODateTimeFormat;

public class IonSqlQueryBuilder {
    private static final DateTimeFormatter FORMATTER = ISODateTimeFormat.date().withChronology((Chronology)ISOChronology.getInstanceUTC());
    private static final String DATA_SOURCE = "S3Object s";
    private final TypeManager typeManager;

    public IonSqlQueryBuilder(TypeManager typeManager) {
        this.typeManager = Objects.requireNonNull(typeManager, "typeManager is null");
    }

    public String buildSql(List<HiveColumnHandle> columns, TupleDomain<HiveColumnHandle> tupleDomain) {
        columns.forEach(column -> Preconditions.checkArgument((boolean)column.isBaseColumn(), (String)"%s is not a base column", (Object)column));
        tupleDomain.getDomains().ifPresent(domains -> domains.keySet().forEach(column -> Preconditions.checkArgument((boolean)column.isBaseColumn(), (String)"%s is not a base column", (Object)column)));
        StringBuilder sql = new StringBuilder("SELECT ");
        if (columns.isEmpty()) {
            sql.append("' '");
        } else {
            String columnNames = columns.stream().map(column -> String.format("s._%d", column.getBaseHiveColumnIndex() + 1)).collect(Collectors.joining(", "));
            sql.append(columnNames);
        }
        sql.append(" FROM ");
        sql.append(DATA_SOURCE);
        List<String> clauses = this.toConjuncts(columns, tupleDomain);
        if (!clauses.isEmpty()) {
            sql.append(" WHERE ").append(Joiner.on((String)" AND ").join(clauses));
        }
        return sql.toString();
    }

    private List<String> toConjuncts(List<HiveColumnHandle> columns, TupleDomain<HiveColumnHandle> tupleDomain) {
        ImmutableList.Builder builder = ImmutableList.builder();
        for (HiveColumnHandle column : columns) {
            Domain domain;
            Type type = column.getHiveType().getType(this.typeManager);
            if (!tupleDomain.getDomains().isPresent() || !IonSqlQueryBuilder.isSupported(type) || (domain = (Domain)((Map)tupleDomain.getDomains().get()).get(column)) == null) continue;
            builder.add((Object)this.toPredicate(domain, type, column.getBaseHiveColumnIndex()));
        }
        return builder.build();
    }

    private static boolean isSupported(Type type) {
        Type validType = Objects.requireNonNull(type, "type is null");
        return validType.equals(BigintType.BIGINT) || validType.equals(TinyintType.TINYINT) || validType.equals(SmallintType.SMALLINT) || validType.equals(IntegerType.INTEGER) || validType instanceof DecimalType || validType.equals(BooleanType.BOOLEAN) || validType.equals(DateType.DATE) || validType instanceof VarcharType;
    }

    private String toPredicate(Domain domain, Type type, int position) {
        Preconditions.checkArgument((boolean)domain.getType().isOrderable(), (Object)"Domain type must be orderable");
        if (domain.getValues().isNone()) {
            if (domain.isNullAllowed()) {
                return String.format("s._%d", position + 1) + " = '' ";
            }
            return "FALSE";
        }
        if (domain.getValues().isAll()) {
            if (domain.isNullAllowed()) {
                return "TRUE";
            }
            return String.format("s._%d", position + 1) + " <> '' ";
        }
        ArrayList<Object> disjuncts = new ArrayList<Object>();
        ArrayList<Object> singleValues = new ArrayList<Object>();
        for (Range range : domain.getValues().getRanges().getOrderedRanges()) {
            Preconditions.checkState((!range.isAll() ? 1 : 0) != 0);
            if (range.isSingleValue()) {
                singleValues.add(range.getLow().getValue());
                continue;
            }
            ArrayList<String> arrayList = new ArrayList<String>();
            if (!range.getLow().isLowerUnbounded()) {
                switch (range.getLow().getBound()) {
                    case ABOVE: {
                        arrayList.add(this.toPredicate(">", range.getLow().getValue(), type, position));
                        break;
                    }
                    case EXACTLY: {
                        arrayList.add(this.toPredicate(">=", range.getLow().getValue(), type, position));
                        break;
                    }
                    case BELOW: {
                        throw new IllegalArgumentException("Low marker should never use BELOW bound");
                    }
                    default: {
                        throw new AssertionError((Object)("Unhandled bound: " + range.getLow().getBound()));
                    }
                }
            }
            if (!range.getHigh().isUpperUnbounded()) {
                switch (range.getHigh().getBound()) {
                    case ABOVE: {
                        throw new IllegalArgumentException("High marker should never use ABOVE bound");
                    }
                    case EXACTLY: {
                        arrayList.add(this.toPredicate("<=", range.getHigh().getValue(), type, position));
                        break;
                    }
                    case BELOW: {
                        arrayList.add(this.toPredicate("<", range.getHigh().getValue(), type, position));
                        break;
                    }
                    default: {
                        throw new AssertionError((Object)("Unhandled bound: " + range.getHigh().getBound()));
                    }
                }
            }
            Preconditions.checkState((!arrayList.isEmpty() ? 1 : 0) != 0);
            disjuncts.add("(" + Joiner.on((String)" AND ").join(arrayList) + ")");
        }
        if (singleValues.size() == 1) {
            disjuncts.add(this.toPredicate("=", Iterables.getOnlyElement(singleValues), type, position));
        } else if (singleValues.size() > 1) {
            ArrayList<String> values = new ArrayList<String>();
            for (Object e : singleValues) {
                IonSqlQueryBuilder.checkType(type);
                values.add(IonSqlQueryBuilder.valueToQuery(type, e));
            }
            disjuncts.add(this.createColumn(type, position) + " IN (" + Joiner.on((String)",").join(values) + ")");
        }
        Preconditions.checkState((!disjuncts.isEmpty() ? 1 : 0) != 0);
        if (domain.isNullAllowed()) {
            disjuncts.add(String.format("s._%d", position + 1) + " = '' ");
        }
        return "(" + Joiner.on((String)" OR ").join(disjuncts) + ")";
    }

    private String toPredicate(String operator, Object value, Type type, int position) {
        IonSqlQueryBuilder.checkType(type);
        return String.format("%s %s %s", this.createColumn(type, position), operator, IonSqlQueryBuilder.valueToQuery(type, value));
    }

    private static void checkType(Type type) {
        Preconditions.checkArgument((boolean)IonSqlQueryBuilder.isSupported(type), (String)"Type not supported: %s", (Object)type);
    }

    private static String valueToQuery(Type type, Object value) {
        if (type.equals(BigintType.BIGINT)) {
            return String.valueOf((Long)value);
        }
        if (type.equals(IntegerType.INTEGER)) {
            return String.valueOf(Math.toIntExact((Long)value));
        }
        if (type.equals(SmallintType.SMALLINT)) {
            return String.valueOf(Shorts.checkedCast((long)((Long)value)));
        }
        if (type.equals(TinyintType.TINYINT)) {
            return String.valueOf(SignedBytes.checkedCast((long)((Long)value)));
        }
        if (type.equals(BooleanType.BOOLEAN)) {
            return String.valueOf((Boolean)value);
        }
        if (type.equals(DateType.DATE)) {
            return "`" + FORMATTER.print(TimeUnit.DAYS.toMillis((Long)value)) + "`";
        }
        if (type.equals(VarcharType.VARCHAR)) {
            return "'" + ((Slice)value).toStringUtf8() + "'";
        }
        if (type instanceof DecimalType) {
            if (Decimals.isLongDecimal((Type)type)) {
                return Decimals.toString((Slice)((Slice)value), (int)((DecimalType)type).getScale());
            }
            return Decimals.toString((long)((Long)value), (int)((DecimalType)type).getScale());
        }
        return "'" + ((Slice)value).toStringUtf8() + "'";
    }

    private String createColumn(Type type, int position) {
        String column = String.format("s._%d", position + 1);
        if (type.equals(BigintType.BIGINT) || type.equals(IntegerType.INTEGER) || type.equals(SmallintType.SMALLINT) || type.equals(TinyintType.TINYINT)) {
            return this.formatPredicate(column, "INT");
        }
        if (type.equals(BooleanType.BOOLEAN)) {
            return this.formatPredicate(column, "BOOL");
        }
        if (type.equals(DateType.DATE)) {
            return this.formatPredicate(column, "TIMESTAMP");
        }
        if (type instanceof DecimalType) {
            DecimalType decimalType = (DecimalType)type;
            return this.formatPredicate(column, String.format("DECIMAL(%s,%s)", decimalType.getPrecision(), decimalType.getScale()));
        }
        return column;
    }

    private String formatPredicate(String column, String type) {
        return String.format("case %s when '' then null else CAST(%s AS %s) end", column, column, type);
    }
}

