/*
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package io.trino.plugin.hive.metastore.cache;

import com.google.inject.Binder;
import com.google.inject.Module;
import com.google.inject.Provides;
import io.trino.plugin.base.CatalogName;
import io.trino.plugin.hive.metastore.HiveMetastore;
import io.trino.plugin.hive.metastore.HiveMetastoreDecorator;
import io.trino.spi.NodeManager;

import javax.inject.Singleton;

import java.util.Optional;
import java.util.concurrent.Executor;

import static com.google.inject.multibindings.OptionalBinder.newOptionalBinder;
import static io.airlift.concurrent.Threads.daemonThreadsNamed;
import static io.airlift.configuration.ConfigBinder.configBinder;
import static io.trino.plugin.hive.metastore.cache.CachingHiveMetastore.cachingHiveMetastore;
import static java.util.concurrent.Executors.newCachedThreadPool;
import static org.weakref.jmx.guice.ExportBinder.newExporter;

public class CachingHiveMetastoreModule
        implements Module
{
    @Override
    public void configure(Binder binder)
    {
        configBinder(binder).bindConfig(CachingHiveMetastoreConfig.class);
        newOptionalBinder(binder, HiveMetastoreDecorator.class);
        newExporter(binder).export(HiveMetastore.class)
                .as(generator -> generator.generatedNameOf(CachingHiveMetastore.class));
    }

    @Provides
    @Singleton
    public HiveMetastore createCachingHiveMetastore(
            NodeManager nodeManager,
            @ForCachingHiveMetastore HiveMetastore delegate,
            CachingHiveMetastoreConfig config,
            CatalogName catalogName,
            Optional<HiveMetastoreDecorator> hiveMetastoreDecorator)
    {
        HiveMetastore decoratedDelegate = hiveMetastoreDecorator
                .map(decorator -> decorator.decorate(delegate))
                .orElse(delegate);

        if (!nodeManager.getCurrentNode().isCoordinator()) {
            // Disable caching on workers, because there currently is no way to invalidate such a cache.
            // Note: while we could skip CachingHiveMetastoreModule altogether on workers, we retain it so that catalog
            // configuration can remain identical for all nodes, making cluster configuration easier.
            return decoratedDelegate;
        }

        Executor executor = new ReentrantBoundedExecutor(
                newCachedThreadPool(daemonThreadsNamed("hive-metastore-" + catalogName + "-%s")),
                config.getMaxMetastoreRefreshThreads());

        return cachingHiveMetastore(decoratedDelegate, executor, config);
    }
}
