/*
 * Decompiled with CFR 0.152.
 */
package io.trino.plugin.hive.util;

import com.fasterxml.jackson.annotation.JsonCreator;
import com.fasterxml.jackson.annotation.JsonProperty;
import com.google.common.annotations.VisibleForTesting;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.ImmutableSet;
import com.google.common.collect.Lists;
import io.trino.plugin.hive.HiveBucketHandle;
import io.trino.plugin.hive.HiveBucketProperty;
import io.trino.plugin.hive.HiveColumnHandle;
import io.trino.plugin.hive.HiveErrorCode;
import io.trino.plugin.hive.HiveSessionProperties;
import io.trino.plugin.hive.HiveTableHandle;
import io.trino.plugin.hive.HiveTimestampPrecision;
import io.trino.plugin.hive.HiveType;
import io.trino.plugin.hive.metastore.Column;
import io.trino.plugin.hive.metastore.SortingColumn;
import io.trino.plugin.hive.metastore.Table;
import io.trino.plugin.hive.util.HiveBucketingV1;
import io.trino.plugin.hive.util.HiveBucketingV2;
import io.trino.plugin.hive.util.HiveUtil;
import io.trino.spi.ErrorCodeSupplier;
import io.trino.spi.Page;
import io.trino.spi.StandardErrorCode;
import io.trino.spi.TrinoException;
import io.trino.spi.connector.ColumnHandle;
import io.trino.spi.connector.ConnectorSession;
import io.trino.spi.predicate.Domain;
import io.trino.spi.predicate.NullableValue;
import io.trino.spi.predicate.TupleDomain;
import io.trino.spi.predicate.ValueSet;
import io.trino.spi.type.TypeManager;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.function.Function;
import java.util.stream.Collectors;
import org.apache.hadoop.hive.serde2.objectinspector.PrimitiveObjectInspector;
import org.apache.hadoop.hive.serde2.typeinfo.ListTypeInfo;
import org.apache.hadoop.hive.serde2.typeinfo.MapTypeInfo;
import org.apache.hadoop.hive.serde2.typeinfo.PrimitiveTypeInfo;
import org.apache.hadoop.hive.serde2.typeinfo.TypeInfo;

public final class HiveBucketing {
    private static final long BUCKETS_EXPLORATION_LIMIT_FACTOR = 4L;
    private static final long BUCKETS_EXPLORATION_GUARANTEED_LIMIT = 1000L;
    private static final Set<HiveType> SUPPORTED_TYPES_FOR_BUCKET_FILTER = ImmutableSet.of((Object)HiveType.HIVE_BYTE, (Object)HiveType.HIVE_SHORT, (Object)HiveType.HIVE_INT, (Object)HiveType.HIVE_LONG, (Object)HiveType.HIVE_BOOLEAN, (Object)HiveType.HIVE_STRING, (Object[])new HiveType[0]);

    private HiveBucketing() {
    }

    public static int getHiveBucket(BucketingVersion bucketingVersion, int bucketCount, List<TypeInfo> types, Page page, int position) {
        return HiveBucketing.getBucketNumber(bucketingVersion.getBucketHashCode(types, page, position), bucketCount);
    }

    public static int getHiveBucket(BucketingVersion bucketingVersion, int bucketCount, List<TypeInfo> types, Object[] values) {
        return HiveBucketing.getBucketNumber(bucketingVersion.getBucketHashCode(types, values), bucketCount);
    }

    @VisibleForTesting
    static Optional<Set<Integer>> getHiveBuckets(BucketingVersion bucketingVersion, int bucketCount, List<TypeInfo> types, List<List<NullableValue>> values) {
        long explorationCount;
        try {
            explorationCount = values.stream().mapToLong(List::size).reduce(1L, Math::multiplyExact);
        }
        catch (ArithmeticException e) {
            return Optional.empty();
        }
        long explorationLimit = Math.max((long)bucketCount * 4L, 1000L);
        if (explorationCount > explorationLimit) {
            return Optional.empty();
        }
        HashSet<Integer> buckets = new HashSet<Integer>();
        for (List combination : Lists.cartesianProduct(values)) {
            buckets.add(HiveBucketing.getBucketNumber(bucketingVersion.getBucketHashCode(types, combination.stream().map(NullableValue::getValue).toArray()), bucketCount));
            if (buckets.size() < bucketCount) continue;
            return Optional.empty();
        }
        return Optional.of(ImmutableSet.copyOf(buckets));
    }

    @VisibleForTesting
    static int getBucketNumber(int hashCode, int bucketCount) {
        return (hashCode & Integer.MAX_VALUE) % bucketCount;
    }

    public static Optional<HiveBucketHandle> getHiveBucketHandle(ConnectorSession session, Table table, TypeManager typeManager) {
        if (table.getParameters().containsKey("spark.sql.sources.provider")) {
            return Optional.empty();
        }
        Optional<HiveBucketProperty> hiveBucketProperty = table.getStorage().getBucketProperty();
        if (hiveBucketProperty.isEmpty()) {
            return Optional.empty();
        }
        HiveTimestampPrecision timestampPrecision = HiveSessionProperties.getTimestampPrecision(session);
        Map map = HiveUtil.getRegularColumnHandles(table, typeManager, timestampPrecision).stream().collect(Collectors.toMap(HiveColumnHandle::getName, Function.identity()));
        ImmutableList.Builder bucketColumns = ImmutableList.builder();
        for (String bucketColumnName : hiveBucketProperty.get().getBucketedBy()) {
            HiveColumnHandle bucketColumnHandle = (HiveColumnHandle)map.get(bucketColumnName);
            if (bucketColumnHandle == null) {
                throw new TrinoException((ErrorCodeSupplier)HiveErrorCode.HIVE_INVALID_METADATA, String.format("Table '%s.%s' is bucketed on non-existent column '%s'", table.getDatabaseName(), table.getTableName(), bucketColumnName));
            }
            bucketColumns.add((Object)bucketColumnHandle);
        }
        BucketingVersion bucketingVersion = hiveBucketProperty.get().getBucketingVersion();
        int bucketCount = hiveBucketProperty.get().getBucketCount();
        List<SortingColumn> sortedBy = hiveBucketProperty.get().getSortedBy();
        return Optional.of(new HiveBucketHandle((List<HiveColumnHandle>)bucketColumns.build(), bucketingVersion, bucketCount, bucketCount, sortedBy));
    }

    public static Optional<HiveBucketFilter> getHiveBucketFilter(HiveTableHandle hiveTable, TupleDomain<ColumnHandle> effectivePredicate) {
        List dataColumns;
        if (hiveTable.getBucketHandle().isEmpty()) {
            return Optional.empty();
        }
        HiveBucketProperty hiveBucketProperty = hiveTable.getBucketHandle().get().toTableBucketProperty();
        if (HiveBucketing.bucketedOnTimestamp(hiveBucketProperty, dataColumns = (List)hiveTable.getDataColumns().stream().map(HiveColumnHandle::toMetastoreColumn).collect(ImmutableList.toImmutableList()), hiveTable.getTableName())) {
            return Optional.empty();
        }
        Optional bindings = TupleDomain.extractDiscreteValues(effectivePredicate);
        if (bindings.isEmpty()) {
            return Optional.empty();
        }
        Optional<Set<Integer>> buckets = HiveBucketing.getHiveBuckets(hiveBucketProperty, dataColumns, (Map)bindings.get());
        if (buckets.isPresent()) {
            return Optional.of(new HiveBucketFilter(buckets.get()));
        }
        Optional domain = effectivePredicate.getDomains().flatMap(domains -> domains.entrySet().stream().filter(entry -> ((HiveColumnHandle)entry.getKey()).getName().equals("$bucket")).findFirst().map(Map.Entry::getValue));
        if (domain.isEmpty()) {
            return Optional.empty();
        }
        ValueSet values = ((Domain)domain.get()).getValues();
        ImmutableSet.Builder builder = ImmutableSet.builder();
        int bucketCount = hiveBucketProperty.getBucketCount();
        for (int i = 0; i < bucketCount; ++i) {
            if (!values.containsValue((Object)i)) continue;
            builder.add((Object)i);
        }
        return Optional.of(new HiveBucketFilter((Set<Integer>)builder.build()));
    }

    private static Optional<Set<Integer>> getHiveBuckets(HiveBucketProperty hiveBucketProperty, List<Column> dataColumns, Map<ColumnHandle, List<NullableValue>> bindings) {
        if (bindings.isEmpty()) {
            return Optional.empty();
        }
        List<String> bucketColumns = hiveBucketProperty.getBucketedBy();
        HashMap<String, HiveType> hiveTypes = new HashMap<String, HiveType>();
        for (Column column : dataColumns) {
            hiveTypes.put(column.getName(), column.getType());
        }
        for (String string : bucketColumns) {
            if (SUPPORTED_TYPES_FOR_BUCKET_FILTER.contains(hiveTypes.get(string))) continue;
            return Optional.empty();
        }
        HashMap<String, List<NullableValue>> bucketBindings = new HashMap<String, List<NullableValue>>();
        for (Map.Entry<ColumnHandle, List<NullableValue>> entry : bindings.entrySet()) {
            HiveColumnHandle columnHandle = (HiveColumnHandle)entry.getKey();
            if (!bucketColumns.contains(columnHandle.getName())) continue;
            bucketBindings.put(columnHandle.getName(), entry.getValue());
        }
        if (bucketBindings.size() != bucketColumns.size()) {
            return Optional.empty();
        }
        List list = (List)bucketColumns.stream().map(bucketBindings::get).collect(ImmutableList.toImmutableList());
        List typeInfos = (List)bucketColumns.stream().map(name -> ((HiveType)hiveTypes.get(name)).getTypeInfo()).collect(ImmutableList.toImmutableList());
        return HiveBucketing.getHiveBuckets(hiveBucketProperty.getBucketingVersion(), hiveBucketProperty.getBucketCount(), typeInfos, list);
    }

    public static BucketingVersion getBucketingVersion(Map<String, String> tableProperties) {
        String bucketingVersion;
        switch (bucketingVersion = tableProperties.getOrDefault("bucketing_version", "1")) {
            case "1": {
                return BucketingVersion.BUCKETING_V1;
            }
            case "2": {
                return BucketingVersion.BUCKETING_V2;
            }
        }
        throw new TrinoException((ErrorCodeSupplier)StandardErrorCode.NOT_SUPPORTED, String.format("Unsupported bucketing version: '%s'", bucketingVersion));
    }

    public static boolean bucketedOnTimestamp(HiveBucketProperty bucketProperty, Table table) {
        return HiveBucketing.bucketedOnTimestamp(bucketProperty, table.getDataColumns(), table.getTableName());
    }

    public static boolean bucketedOnTimestamp(HiveBucketProperty bucketProperty, List<Column> dataColumns, String tableName) {
        return bucketProperty.getBucketedBy().stream().map(columnName -> dataColumns.stream().filter(column -> column.getName().equals(columnName)).findFirst().orElseThrow(() -> new IllegalArgumentException(String.format("Cannot find column '%s' in %s", columnName, tableName)))).map(Column::getType).map(HiveType::getTypeInfo).anyMatch(HiveBucketing::bucketedOnTimestamp);
    }

    private static boolean bucketedOnTimestamp(TypeInfo type) {
        switch (type.getCategory()) {
            case PRIMITIVE: {
                return ((PrimitiveTypeInfo)type).getPrimitiveCategory() == PrimitiveObjectInspector.PrimitiveCategory.TIMESTAMP;
            }
            case LIST: {
                return HiveBucketing.bucketedOnTimestamp(((ListTypeInfo)type).getListElementTypeInfo());
            }
            case MAP: {
                MapTypeInfo mapTypeInfo = (MapTypeInfo)type;
                return HiveBucketing.bucketedOnTimestamp(mapTypeInfo.getMapKeyTypeInfo()) || HiveBucketing.bucketedOnTimestamp(mapTypeInfo.getMapValueTypeInfo());
            }
        }
        throw new UnsupportedOperationException("Computation of Hive bucket hashCode is not supported for Hive category: " + type.getCategory());
    }

    public static class HiveBucketFilter {
        private final Set<Integer> bucketsToKeep;

        @JsonCreator
        public HiveBucketFilter(@JsonProperty(value="bucketsToKeep") Set<Integer> bucketsToKeep) {
            this.bucketsToKeep = bucketsToKeep;
        }

        @JsonProperty
        public Set<Integer> getBucketsToKeep() {
            return this.bucketsToKeep;
        }

        public boolean equals(Object obj) {
            if (this == obj) {
                return true;
            }
            if (obj == null || this.getClass() != obj.getClass()) {
                return false;
            }
            HiveBucketFilter other = (HiveBucketFilter)obj;
            return Objects.equals(this.bucketsToKeep, other.bucketsToKeep);
        }

        public int hashCode() {
            return Objects.hash(this.bucketsToKeep);
        }
    }

    public static enum BucketingVersion {
        BUCKETING_V1(1){

            @Override
            int getBucketHashCode(List<TypeInfo> types, Object[] values) {
                return HiveBucketingV1.getBucketHashCode(types, values);
            }

            @Override
            int getBucketHashCode(List<TypeInfo> types, Page page, int position) {
                return HiveBucketingV1.getBucketHashCode(types, page, position);
            }
        }
        ,
        BUCKETING_V2(2){

            @Override
            int getBucketHashCode(List<TypeInfo> types, Object[] values) {
                return HiveBucketingV2.getBucketHashCode(types, values);
            }

            @Override
            int getBucketHashCode(List<TypeInfo> types, Page page, int position) {
                return HiveBucketingV2.getBucketHashCode(types, page, position);
            }
        };

        private final int version;

        private BucketingVersion(int version) {
            this.version = version;
        }

        public int getVersion() {
            return this.version;
        }

        abstract int getBucketHashCode(List<TypeInfo> var1, Object[] var2);

        abstract int getBucketHashCode(List<TypeInfo> var1, Page var2, int var3);
    }
}

