/*
 * Decompiled with CFR 0.152.
 */
package io.trino.plugin.hive.orc;

import com.google.common.base.Preconditions;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.ImmutableMap;
import com.google.common.io.Resources;
import io.trino.filesystem.TrinoFileSystemFactory;
import io.trino.filesystem.hdfs.HdfsFileSystemFactory;
import io.trino.hadoop.ConfigurationInstantiator;
import io.trino.plugin.hive.AcidInfo;
import io.trino.plugin.hive.FileFormatDataSourceStats;
import io.trino.plugin.hive.HiveColumnHandle;
import io.trino.plugin.hive.HiveConfig;
import io.trino.plugin.hive.HivePageSourceFactory;
import io.trino.plugin.hive.HiveStorageFormat;
import io.trino.plugin.hive.HiveTestUtils;
import io.trino.plugin.hive.HiveType;
import io.trino.plugin.hive.ReaderPageSource;
import io.trino.plugin.hive.acid.AcidTransaction;
import io.trino.plugin.hive.orc.OrcPageSourceFactory;
import io.trino.plugin.hive.orc.OrcReaderConfig;
import io.trino.spi.Page;
import io.trino.spi.connector.ConnectorPageSource;
import io.trino.spi.predicate.Domain;
import io.trino.spi.predicate.TupleDomain;
import io.trino.spi.type.BigintType;
import io.trino.spi.type.IntegerType;
import io.trino.spi.type.Type;
import io.trino.spi.type.VarcharType;
import io.trino.tpch.Nation;
import io.trino.tpch.NationColumn;
import io.trino.tpch.NationGenerator;
import io.trino.tpch.TpchColumnType;
import java.io.File;
import java.net.URISyntaxException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.OptionalInt;
import java.util.OptionalLong;
import java.util.Properties;
import java.util.Set;
import java.util.function.LongPredicate;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.hive.ql.io.AcidUtils;
import org.apache.hadoop.mapred.JobConf;
import org.assertj.core.api.Assertions;
import org.testng.Assert;
import org.testng.annotations.Test;

public class TestOrcPageSourceFactory {
    private static final Map<NationColumn, Integer> ALL_COLUMNS = ImmutableMap.of((Object)NationColumn.NATION_KEY, (Object)0, (Object)NationColumn.NAME, (Object)1, (Object)NationColumn.REGION_KEY, (Object)2, (Object)NationColumn.COMMENT, (Object)3);
    private static final HivePageSourceFactory PAGE_SOURCE_FACTORY = new OrcPageSourceFactory(new OrcReaderConfig(), (TrinoFileSystemFactory)new HdfsFileSystemFactory(HiveTestUtils.HDFS_ENVIRONMENT), new FileFormatDataSourceStats(), new HiveConfig());

    @Test
    public void testFullFileRead() {
        TestOrcPageSourceFactory.assertRead((Map<NationColumn, Integer>)ImmutableMap.of((Object)NationColumn.NATION_KEY, (Object)0, (Object)NationColumn.NAME, (Object)1, (Object)NationColumn.REGION_KEY, (Object)2, (Object)NationColumn.COMMENT, (Object)3), OptionalLong.empty(), Optional.empty(), nationKey -> false);
    }

    @Test
    public void testSingleColumnRead() {
        TestOrcPageSourceFactory.assertRead((Map<NationColumn, Integer>)ImmutableMap.of((Object)NationColumn.REGION_KEY, (Object)ALL_COLUMNS.get(NationColumn.REGION_KEY)), OptionalLong.empty(), Optional.empty(), nationKey -> false);
    }

    @Test
    public void testFullFileSkipped() {
        TestOrcPageSourceFactory.assertRead(ALL_COLUMNS, OptionalLong.of(100L), Optional.empty(), nationKey -> false);
    }

    @Test
    public void testSomeStripesAndRowGroupRead() {
        TestOrcPageSourceFactory.assertRead(ALL_COLUMNS, OptionalLong.of(5L), Optional.empty(), nationKey -> false);
    }

    @Test
    public void testDeletedRows() {
        Path partitionLocation = new Path(this.getClass().getClassLoader().getResource("nation_delete_deltas") + "/");
        Optional acidInfo = AcidInfo.builder((Path)partitionLocation).addDeleteDelta(new Path(partitionLocation, AcidUtils.deleteDeltaSubdir((long)3L, (long)3L, (int)0))).addDeleteDelta(new Path(partitionLocation, AcidUtils.deleteDeltaSubdir((long)4L, (long)4L, (int)0))).build();
        TestOrcPageSourceFactory.assertRead(ALL_COLUMNS, OptionalLong.empty(), acidInfo, nationKey -> nationKey == 5L || nationKey == 19L);
    }

    @Test
    public void testReadWithAcidVersionValidationHive3() throws Exception {
        File tableFile = new File(Resources.getResource((String)"acid_version_validation/acid_version_hive_3/00000_0").toURI());
        String tablePath = tableFile.getParent();
        Optional acidInfo = AcidInfo.builder((Path)new Path(tablePath)).setOrcAcidVersionValidated(false).build();
        List<Nation> result = TestOrcPageSourceFactory.readFile(Map.of(), OptionalLong.empty(), acidInfo, tableFile.getPath(), 625L);
        Assert.assertEquals((int)result.size(), (int)1);
    }

    @Test
    public void testReadWithAcidVersionValidationNoVersionInMetadata() throws Exception {
        File tableFile = new File(Resources.getResource((String)"acid_version_validation/no_orc_acid_version_in_metadata/00000_0").toURI());
        String tablePath = tableFile.getParent();
        Optional acidInfo = AcidInfo.builder((Path)new Path(tablePath)).setOrcAcidVersionValidated(false).build();
        Assertions.assertThatThrownBy(() -> TestOrcPageSourceFactory.readFile(Map.of(), OptionalLong.empty(), acidInfo, tableFile.getPath(), 730L)).hasMessageMatching("Hive transactional tables are supported since Hive 3.0. Expected `hive.acid.version` in ORC metadata in .*/acid_version_validation/no_orc_acid_version_in_metadata/00000_0 to be >=2 but was <empty>. If you have upgraded from an older version of Hive, make sure a major compaction has been run at least once after the upgrade.");
    }

    @Test
    public void testFullFileReadOriginalFilesTable() throws Exception {
        File tableFile = new File(Resources.getResource((String)"fullacidNationTableWithOriginalFiles/000000_0").toURI());
        String tablePath = tableFile.getParent();
        AcidInfo acidInfo = AcidInfo.builder((Path)new Path(tablePath)).addDeleteDelta(new Path(tablePath, AcidUtils.deleteDeltaSubdir((long)10000001L, (long)10000001L, (int)0))).addOriginalFile(new Path(tablePath, "000000_0"), 1780L, 0).setOrcAcidVersionValidated(true).buildWithRequiredOriginalFiles(0);
        List<Nation> expected = TestOrcPageSourceFactory.expectedResult(OptionalLong.empty(), nationKey -> nationKey == 24L, 1);
        List<Nation> result = TestOrcPageSourceFactory.readFile(ALL_COLUMNS, OptionalLong.empty(), Optional.of(acidInfo), tablePath + "/000000_0", 1780L);
        Assert.assertEquals((int)result.size(), (int)expected.size());
        int deletedRowKey = 24;
        String deletedRowNameColumn = "UNITED STATES";
        Assert.assertFalse((boolean)result.stream().anyMatch(acidNationRow -> acidNationRow.getName().equals(deletedRowNameColumn) && acidNationRow.getNationKey() == (long)deletedRowKey), (String)"Deleted row shouldn't be present in the result");
    }

    private static void assertRead(Map<NationColumn, Integer> columns, OptionalLong nationKeyPredicate, Optional<AcidInfo> acidInfo, LongPredicate deletedRows) {
        List<Nation> actual = TestOrcPageSourceFactory.readFile(columns, nationKeyPredicate, acidInfo);
        List<Nation> expected = TestOrcPageSourceFactory.expectedResult(nationKeyPredicate, deletedRows, 1000);
        TestOrcPageSourceFactory.assertEqualsByColumns(columns.keySet(), actual, expected);
    }

    private static List<Nation> expectedResult(OptionalLong nationKeyPredicate, LongPredicate deletedRows, int replicationFactor) {
        ArrayList<Nation> expected = new ArrayList<Nation>();
        for (Nation nation : ImmutableList.copyOf((Iterator)new NationGenerator().iterator())) {
            if (nationKeyPredicate.isPresent() && nationKeyPredicate.getAsLong() != nation.getNationKey() || deletedRows.test(nation.getNationKey())) continue;
            expected.addAll(Collections.nCopies(replicationFactor, nation));
        }
        return expected;
    }

    private static List<Nation> readFile(Map<NationColumn, Integer> columns, OptionalLong nationKeyPredicate, Optional<AcidInfo> acidInfo) {
        try {
            File testFile = new File(Resources.getResource((String)"nationFile25kRowsSortedOnNationKey/bucket_00000").toURI());
            return TestOrcPageSourceFactory.readFile(columns, nationKeyPredicate, acidInfo, testFile.toURI().getPath(), testFile.length());
        }
        catch (URISyntaxException e) {
            throw new RuntimeException(e);
        }
    }

    private static List<Nation> readFile(Map<NationColumn, Integer> columns, OptionalLong nationKeyPredicate, Optional<AcidInfo> acidInfo, String filePath, long fileSize) {
        TupleDomain tupleDomain = TupleDomain.all();
        if (nationKeyPredicate.isPresent()) {
            tupleDomain = TupleDomain.withColumnDomains((Map)ImmutableMap.of((Object)TestOrcPageSourceFactory.toHiveColumnHandle(NationColumn.NATION_KEY, 0), (Object)Domain.singleValue((Type)IntegerType.INTEGER, (Object)nationKeyPredicate.getAsLong())));
        }
        List columnHandles = (List)columns.entrySet().stream().map(entry -> TestOrcPageSourceFactory.toHiveColumnHandle((NationColumn)entry.getKey(), (Integer)entry.getValue())).collect(ImmutableList.toImmutableList());
        List columnNames = (List)columnHandles.stream().map(HiveColumnHandle::getName).collect(ImmutableList.toImmutableList());
        Optional pageSourceWithProjections = PAGE_SOURCE_FACTORY.createPageSource((Configuration)new JobConf(ConfigurationInstantiator.newEmptyConfiguration()), HiveTestUtils.SESSION, new Path(filePath), 0L, fileSize, fileSize, TestOrcPageSourceFactory.createSchema(), columnHandles, tupleDomain, acidInfo, OptionalInt.empty(), false, AcidTransaction.NO_ACID_TRANSACTION);
        Preconditions.checkArgument((boolean)pageSourceWithProjections.isPresent());
        Preconditions.checkArgument((boolean)((ReaderPageSource)pageSourceWithProjections.get()).getReaderColumns().isEmpty(), (Object)"projected columns not expected here");
        ConnectorPageSource pageSource = ((ReaderPageSource)pageSourceWithProjections.get()).get();
        int nationKeyColumn = columnNames.indexOf("n_nationkey");
        int nameColumn = columnNames.indexOf("n_name");
        int regionKeyColumn = columnNames.indexOf("n_regionkey");
        int commentColumn = columnNames.indexOf("n_comment");
        ImmutableList.Builder rows = ImmutableList.builder();
        while (!pageSource.isFinished()) {
            Page page = pageSource.getNextPage();
            if (page == null) continue;
            page = page.getLoadedPage();
            for (int position = 0; position < page.getPositionCount(); ++position) {
                long nationKey = -42L;
                if (nationKeyColumn >= 0) {
                    nationKey = BigintType.BIGINT.getLong(page.getBlock(nationKeyColumn), position);
                }
                String name = "<not read>";
                if (nameColumn >= 0) {
                    name = VarcharType.VARCHAR.getSlice(page.getBlock(nameColumn), position).toStringUtf8();
                }
                long regionKey = -42L;
                if (regionKeyColumn >= 0) {
                    regionKey = BigintType.BIGINT.getLong(page.getBlock(regionKeyColumn), position);
                }
                String comment = "<not read>";
                if (commentColumn >= 0) {
                    comment = VarcharType.VARCHAR.getSlice(page.getBlock(commentColumn), position).toStringUtf8();
                }
                rows.add((Object)new Nation((long)position, nationKey, name, regionKey, comment));
            }
        }
        return rows.build();
    }

    private static HiveColumnHandle toHiveColumnHandle(NationColumn nationColumn, int hiveColumnIndex) {
        BigintType trinoType = switch (nationColumn.getType().getBase()) {
            case TpchColumnType.Base.IDENTIFIER -> BigintType.BIGINT;
            case TpchColumnType.Base.VARCHAR -> VarcharType.VARCHAR;
            default -> throw new IllegalStateException("Unexpected value: " + nationColumn.getType().getBase());
        };
        return HiveColumnHandle.createBaseColumn((String)nationColumn.getColumnName(), (int)hiveColumnIndex, (HiveType)HiveType.toHiveType((Type)trinoType), (Type)trinoType, (HiveColumnHandle.ColumnType)HiveColumnHandle.ColumnType.REGULAR, Optional.empty());
    }

    private static Properties createSchema() {
        Properties schema = new Properties();
        schema.setProperty("serialization.lib", HiveStorageFormat.ORC.getSerde());
        schema.setProperty("file.inputformat", HiveStorageFormat.ORC.getInputFormat());
        schema.setProperty("transactional", "true");
        return schema;
    }

    private static void assertEqualsByColumns(Set<NationColumn> columns, List<Nation> actualRows, List<Nation> expectedRows) {
        Assert.assertEquals((int)actualRows.size(), (int)expectedRows.size(), (String)"row count");
        for (int i = 0; i < actualRows.size(); ++i) {
            Nation actual = actualRows.get(i);
            Nation expected = expectedRows.get(i);
            Assert.assertEquals((long)actual.getNationKey(), (long)(columns.contains(NationColumn.NATION_KEY) ? expected.getNationKey() : -42L));
            Assert.assertEquals((String)actual.getName(), (String)(columns.contains(NationColumn.NAME) ? expected.getName() : "<not read>"));
            Assert.assertEquals((long)actual.getRegionKey(), (long)(columns.contains(NationColumn.REGION_KEY) ? expected.getRegionKey() : -42L));
            Assert.assertEquals((String)actual.getComment(), (String)(columns.contains(NationColumn.COMMENT) ? expected.getComment() : "<not read>"));
        }
    }
}

