/*
 * Decompiled with CFR 0.152.
 */
package io.trino.plugin.hive.s3;

import com.amazonaws.AmazonWebServiceClient;
import com.amazonaws.ClientConfiguration;
import com.amazonaws.Protocol;
import com.amazonaws.auth.AWSCredentials;
import com.amazonaws.auth.AWSCredentialsProvider;
import com.amazonaws.auth.AWSSessionCredentials;
import com.amazonaws.auth.AWSStaticCredentialsProvider;
import com.amazonaws.auth.DefaultAWSCredentialsProviderChain;
import com.amazonaws.auth.STSAssumeRoleSessionCredentialsProvider;
import com.amazonaws.services.s3.AmazonS3;
import com.amazonaws.services.s3.AmazonS3Client;
import com.amazonaws.services.s3.AmazonS3EncryptionClient;
import com.amazonaws.services.s3.S3ClientOptions;
import com.amazonaws.services.s3.model.AmazonS3Exception;
import com.amazonaws.services.s3.model.CannedAccessControlList;
import com.amazonaws.services.s3.model.EncryptionMaterials;
import com.amazonaws.services.s3.model.EncryptionMaterialsProvider;
import com.amazonaws.services.s3.model.GetObjectMetadataRequest;
import com.amazonaws.services.s3.model.ListObjectsV2Request;
import com.amazonaws.services.s3.model.ListObjectsV2Result;
import com.amazonaws.services.s3.model.ObjectMetadata;
import com.amazonaws.services.s3.model.S3ObjectSummary;
import com.amazonaws.services.s3.model.StorageClass;
import com.amazonaws.services.s3.model.UploadPartRequest;
import com.amazonaws.services.securitytoken.AWSSecurityTokenService;
import com.amazonaws.services.securitytoken.AWSSecurityTokenServiceClient;
import com.google.common.base.Preconditions;
import com.google.common.base.VerifyException;
import com.google.common.collect.ImmutableList;
import com.google.common.io.ByteStreams;
import com.google.common.io.MoreFiles;
import com.google.common.io.RecursiveDeleteOption;
import com.google.common.util.concurrent.ListenableFuture;
import io.trino.hadoop.ConfigurationInstantiator;
import io.trino.memory.context.AggregatedMemoryContext;
import io.trino.memory.context.MemoryReservationHandler;
import io.trino.plugin.hive.s3.HiveS3Config;
import io.trino.plugin.hive.s3.MockAmazonS3;
import io.trino.plugin.hive.s3.TrinoS3ConfigurationInitializer;
import io.trino.plugin.hive.s3.TrinoS3FileSystem;
import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.SequenceInputStream;
import java.lang.reflect.Field;
import java.net.URI;
import java.net.URISyntaxException;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Paths;
import java.nio.file.attribute.FileAttribute;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Date;
import java.util.List;
import java.util.Map;
import javax.crypto.SecretKey;
import javax.crypto.spec.SecretKeySpec;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.FSDataInputStream;
import org.apache.hadoop.fs.FSDataOutputStream;
import org.apache.hadoop.fs.FileStatus;
import org.apache.hadoop.fs.LocatedFileStatus;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.fs.RemoteIterator;
import org.assertj.core.api.AbstractThrowableAssert;
import org.assertj.core.api.Assertions;
import org.testng.Assert;
import org.testng.SkipException;
import org.testng.annotations.Test;

public class TestTrinoS3FileSystem {
    private static final int HTTP_RANGE_NOT_SATISFIABLE = 416;
    private static final String S3_DIRECTORY_OBJECT_CONTENT_TYPE = "application/x-directory; charset=UTF-8";

    @Test
    public void testEmbeddedCredentials() throws Exception {
        Configuration config = ConfigurationInstantiator.newEmptyConfiguration();
        try (TrinoS3FileSystem fs = new TrinoS3FileSystem();){
            AWSCredentials credentials = TestTrinoS3FileSystem.getStaticCredentials(config, fs, "s3n://testAccess:testSecret@test-bucket/");
            Assert.assertEquals((String)credentials.getAWSAccessKeyId(), (String)"testAccess");
            Assert.assertEquals((String)credentials.getAWSSecretKey(), (String)"testSecret");
            Assertions.assertThat((Object)credentials).isNotInstanceOf(AWSSessionCredentials.class);
        }
    }

    @Test
    public void testStaticCredentials() throws Exception {
        AWSCredentials credentials;
        Configuration config = ConfigurationInstantiator.newEmptyConfiguration();
        config.set("trino.s3.access-key", "test_access_key");
        config.set("trino.s3.secret-key", "test_secret_key");
        try (TrinoS3FileSystem fs = new TrinoS3FileSystem();){
            credentials = TestTrinoS3FileSystem.getStaticCredentials(config, fs, "s3n://test-bucket/");
            Assert.assertEquals((String)credentials.getAWSAccessKeyId(), (String)"test_access_key");
            Assert.assertEquals((String)credentials.getAWSSecretKey(), (String)"test_secret_key");
            Assertions.assertThat((Object)credentials).isNotInstanceOf(AWSSessionCredentials.class);
        }
        config.set("trino.s3.session-token", "test_token");
        fs = new TrinoS3FileSystem();
        try {
            credentials = TestTrinoS3FileSystem.getStaticCredentials(config, fs, "s3n://test-bucket/");
            Assert.assertEquals((String)credentials.getAWSAccessKeyId(), (String)"test_access_key");
            Assert.assertEquals((String)credentials.getAWSSecretKey(), (String)"test_secret_key");
            Assertions.assertThat((Object)credentials).isInstanceOfSatisfying(AWSSessionCredentials.class, sessionCredentials -> Assert.assertEquals((String)sessionCredentials.getSessionToken(), (String)"test_token"));
        }
        finally {
            fs.close();
        }
    }

    private static AWSCredentials getStaticCredentials(Configuration config, TrinoS3FileSystem fileSystem, String uri) throws IOException, URISyntaxException {
        fileSystem.initialize(new URI(uri), config);
        AWSCredentialsProvider awsCredentialsProvider = TestTrinoS3FileSystem.getAwsCredentialsProvider(fileSystem);
        io.airlift.testing.Assertions.assertInstanceOf((Object)awsCredentialsProvider, AWSStaticCredentialsProvider.class);
        return awsCredentialsProvider.getCredentials();
    }

    @Test(expectedExceptions={VerifyException.class}, expectedExceptionsMessageRegExp="Invalid configuration: either endpoint can be set or S3 client can be pinned to the current region")
    public void testEndpointWithPinToCurrentRegionConfiguration() throws Exception {
        Configuration config = ConfigurationInstantiator.newEmptyConfiguration();
        config.set("trino.s3.endpoint", "test.example.endpoint.com");
        config.set("trino.s3.pin-client-to-current-region", "true");
        try (TrinoS3FileSystem fs = new TrinoS3FileSystem();){
            fs.initialize(new URI("s3a://test-bucket/"), config);
        }
    }

    @Test
    public void testEndpointWithExplicitRegionConfiguration() throws Exception {
        Configuration config = ConfigurationInstantiator.newEmptyConfiguration();
        config.set("trino.s3.endpoint", "test.example.endpoint.com");
        try (TrinoS3FileSystem fs = new TrinoS3FileSystem();){
            fs.initialize(new URI("s3a://test-bucket/"), config);
            Assertions.assertThat((String)((AmazonS3Client)fs.getS3Client()).getSignerRegionOverride()).isNull();
        }
        config.set("trino.s3.endpoint", "test.example.endpoint.com");
        config.set("trino.s3.region", "region1");
        fs = new TrinoS3FileSystem();
        try {
            fs.initialize(new URI("s3a://test-bucket/"), config);
            Assertions.assertThat((String)((AmazonS3Client)fs.getS3Client()).getSignerRegionOverride()).isEqualTo("region1");
        }
        finally {
            fs.close();
        }
        config.set("trino.s3.region", "region1");
        fs = new TrinoS3FileSystem();
        try {
            fs.initialize(new URI("s3a://test-bucket/"), config);
            Assertions.assertThat((String)((AmazonS3Client)fs.getS3Client()).getSignerRegionOverride()).isEqualTo("region1");
        }
        finally {
            fs.close();
        }
    }

    @Test
    public void testAssumeRoleDefaultCredentials() throws Exception {
        Configuration config = ConfigurationInstantiator.newEmptyConfiguration();
        config.set("trino.s3.iam-role", "test_role");
        try (TrinoS3FileSystem fs = new TrinoS3FileSystem();){
            fs.initialize(new URI("s3n://test-bucket/"), config);
            AWSCredentialsProvider tokenService = TestTrinoS3FileSystem.getStsCredentialsProvider(fs, "test_role");
            io.airlift.testing.Assertions.assertInstanceOf((Object)tokenService, DefaultAWSCredentialsProviderChain.class);
        }
    }

    @Test
    public void testAssumeRoleStaticCredentials() throws Exception {
        Configuration config = ConfigurationInstantiator.newEmptyConfiguration();
        config.set("trino.s3.access-key", "test_access_key");
        config.set("trino.s3.secret-key", "test_secret_key");
        config.set("trino.s3.iam-role", "test_role");
        try (TrinoS3FileSystem fs = new TrinoS3FileSystem();){
            fs.initialize(new URI("s3n://test-bucket/"), config);
            AWSCredentialsProvider tokenService = TestTrinoS3FileSystem.getStsCredentialsProvider(fs, "test_role");
            io.airlift.testing.Assertions.assertInstanceOf((Object)tokenService, AWSStaticCredentialsProvider.class);
            AWSCredentials credentials = tokenService.getCredentials();
            Assert.assertEquals((String)credentials.getAWSAccessKeyId(), (String)"test_access_key");
            Assert.assertEquals((String)credentials.getAWSSecretKey(), (String)"test_secret_key");
        }
    }

    private static AWSCredentialsProvider getStsCredentialsProvider(TrinoS3FileSystem fs, String expectedRole) {
        AWSCredentialsProvider awsCredentialsProvider = TestTrinoS3FileSystem.getAwsCredentialsProvider(fs);
        io.airlift.testing.Assertions.assertInstanceOf((Object)awsCredentialsProvider, STSAssumeRoleSessionCredentialsProvider.class);
        Assert.assertEquals((String)TestTrinoS3FileSystem.getFieldValue(awsCredentialsProvider, "roleArn", String.class), (String)expectedRole);
        AWSSecurityTokenService tokenService = TestTrinoS3FileSystem.getFieldValue(awsCredentialsProvider, "securityTokenService", AWSSecurityTokenService.class);
        io.airlift.testing.Assertions.assertInstanceOf((Object)tokenService, AWSSecurityTokenServiceClient.class);
        return TestTrinoS3FileSystem.getFieldValue(tokenService, "awsCredentialsProvider", AWSCredentialsProvider.class);
    }

    @Test
    public void testAssumeRoleCredentialsWithExternalId() throws Exception {
        Configuration config = ConfigurationInstantiator.newEmptyConfiguration();
        config.set("trino.s3.iam-role", "role");
        config.set("trino.s3.external-id", "externalId");
        try (TrinoS3FileSystem fs = new TrinoS3FileSystem();){
            fs.initialize(new URI("s3n://test-bucket/"), config);
            AWSCredentialsProvider awsCredentialsProvider = TestTrinoS3FileSystem.getAwsCredentialsProvider(fs);
            io.airlift.testing.Assertions.assertInstanceOf((Object)awsCredentialsProvider, STSAssumeRoleSessionCredentialsProvider.class);
            Assert.assertEquals((String)TestTrinoS3FileSystem.getFieldValue(awsCredentialsProvider, "roleArn", String.class), (String)"role");
            Assert.assertEquals((String)TestTrinoS3FileSystem.getFieldValue(awsCredentialsProvider, "roleExternalId", String.class), (String)"externalId");
        }
    }

    @Test
    public void testDefaultCredentials() throws Exception {
        Configuration config = ConfigurationInstantiator.newEmptyConfiguration();
        try (TrinoS3FileSystem fs = new TrinoS3FileSystem();){
            fs.initialize(new URI("s3n://test-bucket/"), config);
            io.airlift.testing.Assertions.assertInstanceOf((Object)TestTrinoS3FileSystem.getAwsCredentialsProvider(fs), DefaultAWSCredentialsProviderChain.class);
        }
    }

    @Test
    public void testPathStyleAccess() throws Exception {
        Configuration config = ConfigurationInstantiator.newEmptyConfiguration();
        config.setBoolean("trino.s3.path-style-access", true);
        try (TrinoS3FileSystem fs = new TrinoS3FileSystem();){
            fs.initialize(new URI("s3n://test-bucket/"), config);
            S3ClientOptions clientOptions = TestTrinoS3FileSystem.getFieldValue(fs.getS3Client(), AmazonS3Client.class, "clientOptions", S3ClientOptions.class);
            Assert.assertTrue((boolean)clientOptions.isPathStyleAccess());
        }
    }

    @Test
    public void testUnderscoreBucket() throws Exception {
        Configuration config = ConfigurationInstantiator.newEmptyConfiguration();
        config.setBoolean("trino.s3.path-style-access", true);
        try (TrinoS3FileSystem fs = new TrinoS3FileSystem();){
            MockAmazonS3 s3 = new MockAmazonS3();
            String expectedBucketName = "test-bucket_underscore";
            URI uri = new URI("s3n://" + expectedBucketName + "/");
            Assert.assertEquals((String)fs.getBucketName(uri), (String)expectedBucketName);
            fs.initialize(uri, config);
            fs.setS3Client((AmazonS3)s3);
            fs.getS3ObjectMetadata(new Path("/test/path"));
            Assert.assertEquals((String)expectedBucketName, (String)s3.getGetObjectMetadataRequest().getBucketName());
        }
    }

    @Test
    public void testReadRetryCounters() throws Exception {
        try (TrinoS3FileSystem fs = new TrinoS3FileSystem();){
            int maxRetries = 2;
            MockAmazonS3 s3 = new MockAmazonS3();
            s3.setGetObjectHttpErrorCode(500);
            Configuration configuration = ConfigurationInstantiator.newEmptyConfiguration();
            configuration.set("trino.s3.max-backoff-time", "1ms");
            configuration.set("trino.s3.max-retry-time", "5s");
            configuration.setInt("trino.s3.max-client-retries", maxRetries);
            fs.initialize(new URI("s3n://test-bucket/"), configuration);
            fs.setS3Client((AmazonS3)s3);
            try (FSDataInputStream inputStream = fs.open(new Path("s3n://test-bucket/test"));){
                inputStream.read();
            }
            catch (Throwable expected) {
                io.airlift.testing.Assertions.assertInstanceOf((Object)expected, AmazonS3Exception.class);
                Assert.assertEquals((int)((AmazonS3Exception)expected).getStatusCode(), (int)500);
                Assert.assertEquals((long)TrinoS3FileSystem.getFileSystemStats().getReadRetries().getTotalCount(), (long)maxRetries);
                Assert.assertEquals((long)TrinoS3FileSystem.getFileSystemStats().getGetObjectRetries().getTotalCount(), (long)(((long)maxRetries + 1L) * (long)maxRetries));
            }
        }
    }

    @Test
    public void testGetMetadataRetryCounter() {
        int maxRetries = 2;
        try (TrinoS3FileSystem fs = new TrinoS3FileSystem();){
            MockAmazonS3 s3 = new MockAmazonS3();
            s3.setGetObjectMetadataHttpCode(500);
            Configuration configuration = ConfigurationInstantiator.newEmptyConfiguration();
            configuration.set("trino.s3.max-backoff-time", "1ms");
            configuration.set("trino.s3.max-retry-time", "5s");
            configuration.setInt("trino.s3.max-client-retries", maxRetries);
            fs.initialize(new URI("s3n://test-bucket/"), configuration);
            fs.setS3Client((AmazonS3)s3);
            fs.getS3ObjectMetadata(new Path("s3n://test-bucket/test"));
        }
        catch (Throwable expected) {
            io.airlift.testing.Assertions.assertInstanceOf((Object)expected, AmazonS3Exception.class);
            Assert.assertEquals((int)((AmazonS3Exception)expected).getStatusCode(), (int)500);
            Assert.assertEquals((long)TrinoS3FileSystem.getFileSystemStats().getGetMetadataRetries().getTotalCount(), (long)maxRetries);
        }
    }

    @Test
    public void testReadNotFound() throws Exception {
        try (TrinoS3FileSystem fs = new TrinoS3FileSystem();){
            MockAmazonS3 s3 = new MockAmazonS3();
            s3.setGetObjectHttpErrorCode(404);
            fs.initialize(new URI("s3n://test-bucket/"), ConfigurationInstantiator.newEmptyConfiguration());
            fs.setS3Client((AmazonS3)s3);
            try (FSDataInputStream inputStream = fs.open(new Path("s3n://test-bucket/test"));){
                ((AbstractThrowableAssert)Assertions.assertThatThrownBy(() -> inputStream.read()).isInstanceOf(IOException.class)).hasMessageContaining("Failing getObject call with 404");
            }
        }
    }

    @Test
    public void testReadForbidden() throws Exception {
        try (TrinoS3FileSystem fs = new TrinoS3FileSystem();){
            MockAmazonS3 s3 = new MockAmazonS3();
            s3.setGetObjectHttpErrorCode(403);
            fs.initialize(new URI("s3n://test-bucket/"), ConfigurationInstantiator.newEmptyConfiguration());
            fs.setS3Client((AmazonS3)s3);
            try (FSDataInputStream inputStream = fs.open(new Path("s3n://test-bucket/test"));){
                ((AbstractThrowableAssert)Assertions.assertThatThrownBy(() -> inputStream.read()).isInstanceOf(IOException.class)).hasMessageContaining("Failing getObject call with 403");
            }
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Test
    public void testCreateWithNonexistentStagingDirectory() throws Exception {
        java.nio.file.Path stagingParent = Files.createTempDirectory("test", new FileAttribute[0]);
        java.nio.file.Path staging = Paths.get(stagingParent.toString(), "staging");
        try (TrinoS3FileSystem fs = new TrinoS3FileSystem();){
            MockAmazonS3 s3 = new MockAmazonS3();
            Configuration conf = ConfigurationInstantiator.newEmptyConfiguration();
            conf.set("trino.s3.staging-directory", staging.toString());
            conf.set("trino.s3.streaming.enabled", "false");
            fs.initialize(new URI("s3n://test-bucket/"), conf);
            fs.setS3Client((AmazonS3)s3);
            FSDataOutputStream stream = fs.create(new Path("s3n://test-bucket/test"));
            stream.close();
            Assert.assertTrue((boolean)Files.exists(staging, new LinkOption[0]));
        }
        catch (Throwable throwable) {
            MoreFiles.deleteRecursively((java.nio.file.Path)stagingParent, (RecursiveDeleteOption[])new RecursiveDeleteOption[]{RecursiveDeleteOption.ALLOW_INSECURE});
            throw throwable;
        }
        MoreFiles.deleteRecursively((java.nio.file.Path)stagingParent, (RecursiveDeleteOption[])new RecursiveDeleteOption[]{RecursiveDeleteOption.ALLOW_INSECURE});
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Test
    public void testCreateWithStagingDirectoryFile() throws Exception {
        java.nio.file.Path staging = Files.createTempFile("staging", null, new FileAttribute[0]);
        try (TrinoS3FileSystem fs = new TrinoS3FileSystem();){
            MockAmazonS3 s3 = new MockAmazonS3();
            Configuration conf = ConfigurationInstantiator.newEmptyConfiguration();
            conf.set("trino.s3.staging-directory", staging.toString());
            conf.set("trino.s3.streaming.enabled", "false");
            fs.initialize(new URI("s3n://test-bucket/"), conf);
            fs.setS3Client((AmazonS3)s3);
            ((AbstractThrowableAssert)Assertions.assertThatThrownBy(() -> fs.create(new Path("s3n://test-bucket/test"))).isInstanceOf(IOException.class)).hasMessageStartingWith("Configured staging path is not a directory:");
        }
        finally {
            Files.deleteIfExists(staging);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Test
    public void testCreateWithStagingDirectorySymlink() throws Exception {
        java.nio.file.Path staging = Files.createTempDirectory("staging", new FileAttribute[0]);
        java.nio.file.Path link = Paths.get(staging + ".symlink", new String[0]);
        try {
            try {
                Files.createSymbolicLink(link, staging, new FileAttribute[0]);
            }
            catch (UnsupportedOperationException e) {
                throw new SkipException("Filesystem does not support symlinks", (Throwable)e);
            }
            try (TrinoS3FileSystem fs = new TrinoS3FileSystem();){
                MockAmazonS3 s3 = new MockAmazonS3();
                Configuration conf = ConfigurationInstantiator.newEmptyConfiguration();
                conf.set("trino.s3.staging-directory", link.toString());
                fs.initialize(new URI("s3n://test-bucket/"), conf);
                fs.setS3Client((AmazonS3)s3);
                FSDataOutputStream stream = fs.create(new Path("s3n://test-bucket/test"));
                stream.close();
                Assert.assertTrue((boolean)Files.exists(link, new LinkOption[0]));
            }
        }
        catch (Throwable throwable) {
            MoreFiles.deleteRecursively((java.nio.file.Path)link, (RecursiveDeleteOption[])new RecursiveDeleteOption[]{RecursiveDeleteOption.ALLOW_INSECURE});
            MoreFiles.deleteRecursively((java.nio.file.Path)staging, (RecursiveDeleteOption[])new RecursiveDeleteOption[]{RecursiveDeleteOption.ALLOW_INSECURE});
            throw throwable;
        }
        MoreFiles.deleteRecursively((java.nio.file.Path)link, (RecursiveDeleteOption[])new RecursiveDeleteOption[]{RecursiveDeleteOption.ALLOW_INSECURE});
        MoreFiles.deleteRecursively((java.nio.file.Path)staging, (RecursiveDeleteOption[])new RecursiveDeleteOption[]{RecursiveDeleteOption.ALLOW_INSECURE});
    }

    @Test
    public void testReadRequestRangeNotSatisfiable() throws Exception {
        try (TrinoS3FileSystem fs = new TrinoS3FileSystem();){
            MockAmazonS3 s3 = new MockAmazonS3();
            s3.setGetObjectHttpErrorCode(416);
            fs.initialize(new URI("s3n://test-bucket/"), ConfigurationInstantiator.newEmptyConfiguration());
            fs.setS3Client((AmazonS3)s3);
            try (FSDataInputStream inputStream = fs.open(new Path("s3n://test-bucket/test"));){
                Assert.assertEquals((int)inputStream.read(), (int)-1);
            }
        }
    }

    @Test
    public void testGetMetadataForbidden() throws Exception {
        try (TrinoS3FileSystem fs = new TrinoS3FileSystem();){
            MockAmazonS3 s3 = new MockAmazonS3();
            s3.setGetObjectMetadataHttpCode(403);
            fs.initialize(new URI("s3n://test-bucket/"), ConfigurationInstantiator.newEmptyConfiguration());
            fs.setS3Client((AmazonS3)s3);
            ((AbstractThrowableAssert)Assertions.assertThatThrownBy(() -> fs.getS3ObjectMetadata(new Path("s3n://test-bucket/test"))).isInstanceOf(IOException.class)).hasMessageContaining("Failing getObjectMetadata call with 403");
        }
    }

    @Test
    public void testGetMetadataNotFound() throws Exception {
        try (TrinoS3FileSystem fs = new TrinoS3FileSystem();){
            MockAmazonS3 s3 = new MockAmazonS3();
            s3.setGetObjectMetadataHttpCode(404);
            fs.initialize(new URI("s3n://test-bucket/"), ConfigurationInstantiator.newEmptyConfiguration());
            fs.setS3Client((AmazonS3)s3);
            Assert.assertNull((Object)fs.getS3ObjectMetadata(new Path("s3n://test-bucket/test")));
        }
    }

    @Test
    public void testEncryptionMaterialsProvider() throws Exception {
        Configuration config = ConfigurationInstantiator.newEmptyConfiguration();
        config.set("trino.s3.encryption-materials-provider", TestEncryptionMaterialsProvider.class.getName());
        try (TrinoS3FileSystem fs = new TrinoS3FileSystem();){
            fs.initialize(new URI("s3n://test-bucket/"), config);
            io.airlift.testing.Assertions.assertInstanceOf((Object)fs.getS3Client(), AmazonS3EncryptionClient.class);
        }
    }

    @Test
    public void testKMSEncryptionMaterialsProvider() throws Exception {
        Configuration config = ConfigurationInstantiator.newEmptyConfiguration();
        config.set("trino.s3.kms-key-id", "test-key-id");
        try (TrinoS3FileSystem fs = new TrinoS3FileSystem();){
            fs.initialize(new URI("s3n://test-bucket/"), config);
            io.airlift.testing.Assertions.assertInstanceOf((Object)fs.getS3Client(), AmazonS3EncryptionClient.class);
        }
    }

    @Test(expectedExceptions={TrinoS3FileSystem.UnrecoverableS3OperationException.class}, expectedExceptionsMessageRegExp=".*\\Q (Path: /tmp/test/path)\\E")
    public void testUnrecoverableS3ExceptionMessage() throws Exception {
        throw new TrinoS3FileSystem.UnrecoverableS3OperationException(new Path("/tmp/test/path"), (Throwable)new IOException("test io exception"));
    }

    @Test
    public void testCustomCredentialsProvider() throws Exception {
        Configuration config = ConfigurationInstantiator.newEmptyConfiguration();
        config.set("trino.s3.credentials-provider", TestCredentialsProvider.class.getName());
        try (TrinoS3FileSystem fs = new TrinoS3FileSystem();){
            fs.initialize(new URI("s3n://test-bucket/"), config);
            io.airlift.testing.Assertions.assertInstanceOf((Object)TestTrinoS3FileSystem.getAwsCredentialsProvider(fs), TestCredentialsProvider.class);
        }
    }

    @Test(expectedExceptions={RuntimeException.class}, expectedExceptionsMessageRegExp="Error creating an instance of .*")
    public void testCustomCredentialsClassCannotBeFound() throws Exception {
        Configuration config = ConfigurationInstantiator.newEmptyConfiguration();
        config.set("trino.s3.credentials-provider", "com.example.DoesNotExist");
        try (TrinoS3FileSystem fs = new TrinoS3FileSystem();){
            fs.initialize(new URI("s3n://test-bucket/"), config);
        }
    }

    @Test
    public void testUserAgentPrefix() throws Exception {
        String userAgentPrefix = "agent_prefix";
        Configuration config = ConfigurationInstantiator.newEmptyConfiguration();
        config.set("trino.s3.user-agent-prefix", userAgentPrefix);
        try (TrinoS3FileSystem fs = new TrinoS3FileSystem();){
            fs.initialize(new URI("s3n://test-bucket/"), config);
            ClientConfiguration clientConfig = TestTrinoS3FileSystem.getFieldValue(fs.getS3Client(), AmazonWebServiceClient.class, "clientConfiguration", ClientConfiguration.class);
            Assert.assertEquals((String)clientConfig.getUserAgentSuffix(), (String)"Trino");
            Assert.assertEquals((String)clientConfig.getUserAgentPrefix(), (String)userAgentPrefix);
        }
    }

    @Test
    public void testDefaultS3ClientConfiguration() throws Exception {
        HiveS3Config defaults = new HiveS3Config();
        try (TrinoS3FileSystem fs = new TrinoS3FileSystem();){
            fs.initialize(new URI("s3n://test-bucket/"), ConfigurationInstantiator.newEmptyConfiguration());
            ClientConfiguration config = TestTrinoS3FileSystem.getFieldValue(fs.getS3Client(), AmazonWebServiceClient.class, "clientConfiguration", ClientConfiguration.class);
            Assert.assertEquals((int)config.getMaxErrorRetry(), (int)defaults.getS3MaxErrorRetries());
            Assert.assertEquals((long)config.getConnectionTimeout(), (long)defaults.getS3ConnectTimeout().toMillis());
            Assert.assertEquals((long)config.getSocketTimeout(), (long)defaults.getS3SocketTimeout().toMillis());
            Assert.assertEquals((int)config.getMaxConnections(), (int)defaults.getS3MaxConnections());
            Assert.assertEquals((String)config.getUserAgentSuffix(), (String)"Trino");
            Assert.assertEquals((String)config.getUserAgentPrefix(), (String)"");
        }
    }

    @Test
    public void testSkipGlacierObjectsEnabled() throws Exception {
        TestTrinoS3FileSystem.assertSkipGlacierObjects(true);
        TestTrinoS3FileSystem.assertSkipGlacierObjects(false);
    }

    @Test
    public void testProxyDefaultsS3ClientConfiguration() throws Exception {
        HiveS3Config hiveS3Config = new HiveS3Config();
        TrinoS3ConfigurationInitializer configurationInitializer = new TrinoS3ConfigurationInitializer(hiveS3Config);
        Configuration trinoFsConfiguration = ConfigurationInstantiator.newEmptyConfiguration();
        configurationInitializer.initializeConfiguration(trinoFsConfiguration);
        try (TrinoS3FileSystem fs = new TrinoS3FileSystem();){
            fs.initialize(new URI("s3n://test-bucket/"), trinoFsConfiguration);
            ClientConfiguration config = TestTrinoS3FileSystem.getFieldValue(fs.getS3Client(), AmazonWebServiceClient.class, "clientConfiguration", ClientConfiguration.class);
            Assert.assertNull((Object)config.getProxyHost());
            Assert.assertEquals((int)config.getProxyPort(), (int)-1);
            Assert.assertEquals((Object)config.getProxyProtocol(), (Object)Protocol.HTTP);
            Assert.assertEquals((String)config.getNonProxyHosts(), (String)System.getProperty("http.nonProxyHosts"));
            Assert.assertNull((Object)config.getProxyUsername());
            Assert.assertNull((Object)config.getProxyPassword());
            Assert.assertFalse((boolean)config.isPreemptiveBasicProxyAuth());
        }
    }

    @Test
    public void testOnNoHostProxyDefaultsS3ClientConfiguration() throws Exception {
        HiveS3Config hiveS3Config = new HiveS3Config();
        hiveS3Config.setS3ProxyHost(null);
        hiveS3Config.setS3ProxyPort(40000);
        hiveS3Config.setS3ProxyProtocol("https");
        hiveS3Config.setS3NonProxyHosts((List)ImmutableList.of((Object)"firsthost.com", (Object)"secondhost.com"));
        hiveS3Config.setS3ProxyUsername("dummy_username");
        hiveS3Config.setS3ProxyPassword("dummy_password");
        hiveS3Config.setS3PreemptiveBasicProxyAuth(true);
        TrinoS3ConfigurationInitializer configurationInitializer = new TrinoS3ConfigurationInitializer(hiveS3Config);
        Configuration trinoFsConfiguration = ConfigurationInstantiator.newEmptyConfiguration();
        configurationInitializer.initializeConfiguration(trinoFsConfiguration);
        try (TrinoS3FileSystem fs = new TrinoS3FileSystem();){
            fs.initialize(new URI("s3n://test-bucket/"), trinoFsConfiguration);
            ClientConfiguration config = TestTrinoS3FileSystem.getFieldValue(fs.getS3Client(), AmazonWebServiceClient.class, "clientConfiguration", ClientConfiguration.class);
            Assert.assertNull((Object)config.getProxyHost());
            Assert.assertEquals((int)config.getProxyPort(), (int)-1);
            Assert.assertEquals((Object)config.getProxyProtocol(), (Object)Protocol.HTTP);
            Assert.assertEquals((String)config.getNonProxyHosts(), (String)System.getProperty("http.nonProxyHosts"));
            Assert.assertNull((Object)config.getProxyUsername());
            Assert.assertNull((Object)config.getProxyPassword());
            Assert.assertFalse((boolean)config.isPreemptiveBasicProxyAuth());
        }
    }

    @Test
    public void testExplicitProxyS3ClientConfiguration() throws Exception {
        HiveS3Config hiveS3Config = new HiveS3Config();
        hiveS3Config.setS3ProxyHost("dummy.com");
        hiveS3Config.setS3ProxyPort(40000);
        hiveS3Config.setS3ProxyProtocol("https");
        hiveS3Config.setS3NonProxyHosts((List)ImmutableList.of((Object)"firsthost.com", (Object)"secondhost.com"));
        hiveS3Config.setS3ProxyUsername("dummy_username");
        hiveS3Config.setS3ProxyPassword("dummy_password");
        hiveS3Config.setS3PreemptiveBasicProxyAuth(true);
        TrinoS3ConfigurationInitializer configurationInitializer = new TrinoS3ConfigurationInitializer(hiveS3Config);
        Configuration trinoFsConfiguration = ConfigurationInstantiator.newEmptyConfiguration();
        configurationInitializer.initializeConfiguration(trinoFsConfiguration);
        try (TrinoS3FileSystem fs = new TrinoS3FileSystem();){
            fs.initialize(new URI("s3n://test-bucket/"), trinoFsConfiguration);
            ClientConfiguration config = TestTrinoS3FileSystem.getFieldValue(fs.getS3Client(), AmazonWebServiceClient.class, "clientConfiguration", ClientConfiguration.class);
            Assert.assertEquals((String)config.getProxyHost(), (String)"dummy.com");
            Assert.assertEquals((int)config.getProxyPort(), (int)40000);
            Assert.assertEquals((Object)config.getProxyProtocol(), (Object)Protocol.HTTPS);
            Assert.assertEquals((String)config.getNonProxyHosts(), (String)"firsthost.com|secondhost.com");
            Assert.assertEquals((String)config.getProxyUsername(), (String)"dummy_username");
            Assert.assertEquals((String)config.getProxyPassword(), (String)"dummy_password");
            Assert.assertTrue((boolean)config.isPreemptiveBasicProxyAuth());
        }
    }

    private static void assertSkipGlacierObjects(boolean skipGlacierObjects) throws Exception {
        Configuration config = ConfigurationInstantiator.newEmptyConfiguration();
        config.set("trino.s3.skip-glacier-objects", String.valueOf(skipGlacierObjects));
        try (TrinoS3FileSystem fs = new TrinoS3FileSystem();){
            MockAmazonS3 s3 = new MockAmazonS3();
            s3.setHasGlacierObjects(true);
            fs.initialize(new URI("s3n://test-bucket/"), config);
            fs.setS3Client((AmazonS3)s3);
            FileStatus[] statuses = fs.listStatus(new Path("s3n://test-bucket/test"));
            Assert.assertEquals((int)statuses.length, (int)(skipGlacierObjects ? 2 : 4));
        }
    }

    @Test
    public void testSkipHadoopFolderMarkerObjectsEnabled() throws Exception {
        Configuration config = ConfigurationInstantiator.newEmptyConfiguration();
        try (TrinoS3FileSystem fs = new TrinoS3FileSystem();){
            MockAmazonS3 s3 = new MockAmazonS3();
            s3.setHasHadoopFolderMarkerObjects(true);
            fs.initialize(new URI("s3n://test-bucket/"), config);
            fs.setS3Client((AmazonS3)s3);
            FileStatus[] statuses = fs.listStatus(new Path("s3n://test-bucket/test"));
            Assert.assertEquals((int)statuses.length, (int)2);
        }
    }

    public static AWSCredentialsProvider getAwsCredentialsProvider(TrinoS3FileSystem fs) {
        return TestTrinoS3FileSystem.getFieldValue(fs.getS3Client(), "awsCredentialsProvider", AWSCredentialsProvider.class);
    }

    private static <T> T getFieldValue(Object instance, String name, Class<T> type) {
        return TestTrinoS3FileSystem.getFieldValue(instance, instance.getClass(), name, type);
    }

    private static <T> T getFieldValue(Object instance, Class<?> clazz, String name, Class<T> type) {
        try {
            Field field = clazz.getDeclaredField(name);
            Preconditions.checkArgument((field.getType() == type ? 1 : 0) != 0, (String)"expected %s but found %s", type, field.getType());
            field.setAccessible(true);
            return (T)field.get(instance);
        }
        catch (ReflectiveOperationException e) {
            throw new RuntimeException(e);
        }
    }

    @Test
    public void testDefaultAcl() throws Exception {
        Configuration config = ConfigurationInstantiator.newEmptyConfiguration();
        try (TrinoS3FileSystem fs = new TrinoS3FileSystem();){
            MockAmazonS3 s3 = new MockAmazonS3();
            String expectedBucketName = "test-bucket";
            fs.initialize(new URI("s3n://" + expectedBucketName + "/"), config);
            fs.setS3Client((AmazonS3)s3);
            FSDataOutputStream stream = fs.create(new Path("s3n://test-bucket/test"));
            if (stream != null) {
                stream.close();
            }
            Assert.assertEquals((Object)CannedAccessControlList.Private, (Object)s3.getAcl());
        }
    }

    @Test
    public void testFullBucketOwnerControlAcl() throws Exception {
        Configuration config = ConfigurationInstantiator.newEmptyConfiguration();
        config.set("trino.s3.upload-acl-type", "BUCKET_OWNER_FULL_CONTROL");
        try (TrinoS3FileSystem fs = new TrinoS3FileSystem();){
            MockAmazonS3 s3 = new MockAmazonS3();
            String expectedBucketName = "test-bucket";
            fs.initialize(new URI("s3n://" + expectedBucketName + "/"), config);
            fs.setS3Client((AmazonS3)s3);
            FSDataOutputStream stream = fs.create(new Path("s3n://test-bucket/test"));
            if (stream != null) {
                stream.close();
            }
            Assert.assertEquals((Object)CannedAccessControlList.BucketOwnerFullControl, (Object)s3.getAcl());
        }
    }

    @Test
    public void testStreamingUpload() throws Exception {
        Configuration config = ConfigurationInstantiator.newEmptyConfiguration();
        config.set("trino.s3.streaming.enabled", "true");
        try (TrinoS3FileSystem fs = new TrinoS3FileSystem();){
            MockAmazonS3 s3 = new MockAmazonS3();
            String expectedBucketName = "test-bucket";
            config.set("trino.s3.streaming.part-size", "128");
            fs.initialize(new URI("s3n://" + expectedBucketName + "/"), config);
            fs.setS3Client((AmazonS3)s3);
            String objectKey = "test";
            try (FSDataOutputStream stream = fs.create(new Path("s3n://test-bucket/" + objectKey));){
                stream.write(97);
                stream.write("foo".repeat(21).getBytes(StandardCharsets.US_ASCII));
                stream.write("bar".repeat(44).getBytes(StandardCharsets.US_ASCII));
                stream.write("orange".repeat(25).getBytes(StandardCharsets.US_ASCII), 6, 132);
            }
            List<UploadPartRequest> parts = s3.getUploadParts();
            Assertions.assertThat(parts).size().isEqualTo(3);
            InputStream concatInputStream = parts.stream().map(UploadPartRequest::getInputStream).reduce(new ByteArrayInputStream(new byte[0]), SequenceInputStream::new);
            String data = new String(ByteStreams.toByteArray((InputStream)concatInputStream), StandardCharsets.US_ASCII);
            Assert.assertEquals((String)data, (String)("a" + "foo".repeat(21) + "bar".repeat(44) + "orange".repeat(22)));
        }
    }

    @Test
    public void testEmptyDirectory() throws Exception {
        try (TrinoS3FileSystem fs = new TrinoS3FileSystem();){
            MockAmazonS3 s3 = new MockAmazonS3(){

                @Override
                public ObjectMetadata getObjectMetadata(GetObjectMetadataRequest getObjectMetadataRequest) {
                    if (getObjectMetadataRequest.getKey().equals("empty-dir/")) {
                        ObjectMetadata objectMetadata = new ObjectMetadata();
                        objectMetadata.setContentType(TestTrinoS3FileSystem.S3_DIRECTORY_OBJECT_CONTENT_TYPE);
                        return objectMetadata;
                    }
                    return super.getObjectMetadata(getObjectMetadataRequest);
                }
            };
            fs.initialize(new URI("s3n://test-bucket/"), ConfigurationInstantiator.newEmptyConfiguration());
            fs.setS3Client((AmazonS3)s3);
            FileStatus fileStatus = fs.getFileStatus(new Path("s3n://test-bucket/empty-dir/"));
            Assert.assertTrue((boolean)fileStatus.isDirectory());
            fileStatus = fs.getFileStatus(new Path("s3n://test-bucket/empty-dir"));
            Assert.assertTrue((boolean)fileStatus.isDirectory());
        }
    }

    @Test
    public void testListPrefixModes() throws Exception {
        final S3ObjectSummary rootObject = new S3ObjectSummary();
        rootObject.setStorageClass(StorageClass.Standard.toString());
        rootObject.setKey("standard-object-at-root.txt");
        rootObject.setLastModified(new Date());
        final S3ObjectSummary childObject = new S3ObjectSummary();
        childObject.setStorageClass(StorageClass.Standard.toString());
        childObject.setKey("prefix/child-object.txt");
        childObject.setLastModified(new Date());
        try (TrinoS3FileSystem fs = new TrinoS3FileSystem();){
            MockAmazonS3 s3 = new MockAmazonS3(){

                @Override
                public ListObjectsV2Result listObjectsV2(ListObjectsV2Request listObjectsV2Request) {
                    ListObjectsV2Result listing = new ListObjectsV2Result();
                    if ("/".equals(listObjectsV2Request.getDelimiter())) {
                        listing.getCommonPrefixes().add("prefix");
                        listing.getObjectSummaries().add(rootObject);
                        return listing;
                    }
                    listing.getObjectSummaries().addAll(Arrays.asList(childObject, rootObject));
                    return listing;
                }
            };
            Path rootPath = new Path("s3n://test-bucket/");
            fs.initialize(rootPath.toUri(), ConfigurationInstantiator.newEmptyConfiguration());
            fs.setS3Client((AmazonS3)s3);
            List<LocatedFileStatus> shallowAll = TestTrinoS3FileSystem.remoteIteratorToList((RemoteIterator<LocatedFileStatus>)fs.listLocatedStatus(rootPath));
            Assert.assertEquals((int)shallowAll.size(), (int)2);
            Assert.assertTrue((boolean)shallowAll.get(0).isDirectory());
            Assert.assertFalse((boolean)shallowAll.get(1).isDirectory());
            Assert.assertEquals((Object)shallowAll.get(0).getPath(), (Object)new Path(rootPath, "prefix"));
            Assert.assertEquals((Object)shallowAll.get(1).getPath(), (Object)new Path(rootPath, rootObject.getKey()));
            List<LocatedFileStatus> shallowFiles = TestTrinoS3FileSystem.remoteIteratorToList((RemoteIterator<LocatedFileStatus>)fs.listFiles(rootPath, false));
            Assert.assertEquals((int)shallowFiles.size(), (int)1);
            Assert.assertFalse((boolean)shallowFiles.get(0).isDirectory());
            Assert.assertEquals((Object)shallowFiles.get(0).getPath(), (Object)new Path(rootPath, rootObject.getKey()));
            List<LocatedFileStatus> recursiveFiles = TestTrinoS3FileSystem.remoteIteratorToList((RemoteIterator<LocatedFileStatus>)fs.listFiles(rootPath, true));
            Assert.assertEquals((int)recursiveFiles.size(), (int)2);
            Assert.assertFalse((boolean)recursiveFiles.get(0).isDirectory());
            Assert.assertFalse((boolean)recursiveFiles.get(1).isDirectory());
            Assert.assertEquals((Object)recursiveFiles.get(0).getPath(), (Object)new Path(rootPath, childObject.getKey()));
            Assert.assertEquals((Object)recursiveFiles.get(1).getPath(), (Object)new Path(rootPath, rootObject.getKey()));
        }
    }

    @Test
    public void testThatTrinoS3FileSystemReportsConsumedMemory() throws IOException {
        TestMemoryReservationHandler memoryReservationHandler = new TestMemoryReservationHandler();
        AggregatedMemoryContext memoryContext = AggregatedMemoryContext.newRootAggregatedMemoryContext((MemoryReservationHandler)memoryReservationHandler, (long)1024000000L);
        try (TrinoS3FileSystem fs = new TrinoS3FileSystem();){
            MockAmazonS3 s3 = new MockAmazonS3();
            Path rootPath = new Path("s3n://test-bucket/");
            fs.initialize(rootPath.toUri(), ConfigurationInstantiator.newEmptyConfiguration());
            fs.setS3Client((AmazonS3)s3);
            OutputStream outputStream = fs.create(new Path("s3n://test-bucket/test1"), memoryContext);
            outputStream.write(new byte[]{1, 2, 3, 4, 5, 6}, 0, 6);
            outputStream.close();
        }
        Assertions.assertThat((long)memoryReservationHandler.getReserved()).isEqualTo(0L);
        Assertions.assertThat((long)memoryReservationHandler.getMaxReserved()).isGreaterThan(0L);
    }

    private static List<LocatedFileStatus> remoteIteratorToList(RemoteIterator<LocatedFileStatus> statuses) throws IOException {
        ArrayList<LocatedFileStatus> result = new ArrayList<LocatedFileStatus>();
        while (statuses.hasNext()) {
            result.add((LocatedFileStatus)statuses.next());
        }
        return result;
    }

    private static class TestEncryptionMaterialsProvider
    implements EncryptionMaterialsProvider {
        private final EncryptionMaterials encryptionMaterials = new EncryptionMaterials((SecretKey)new SecretKeySpec(new byte[]{1, 2, 3}, "AES"));

        public void refresh() {
        }

        public EncryptionMaterials getEncryptionMaterials(Map<String, String> materialsDescription) {
            return this.encryptionMaterials;
        }

        public EncryptionMaterials getEncryptionMaterials() {
            return this.encryptionMaterials;
        }
    }

    private static class TestCredentialsProvider
    implements AWSCredentialsProvider {
        public TestCredentialsProvider(URI uri, Configuration conf) {
        }

        public AWSCredentials getCredentials() {
            return null;
        }

        public void refresh() {
        }
    }

    private static class TestMemoryReservationHandler
    implements MemoryReservationHandler {
        private long reserved;
        private long maxReserved;

        private TestMemoryReservationHandler() {
        }

        public ListenableFuture<Void> reserveMemory(String allocationTag, long delta) {
            this.reserved += delta;
            if (delta > this.maxReserved) {
                this.maxReserved = delta;
            }
            return null;
        }

        public boolean tryReserveMemory(String allocationTag, long delta) {
            this.reserved += delta;
            if (delta > this.maxReserved) {
                this.maxReserved = delta;
            }
            return true;
        }

        public long getReserved() {
            return this.reserved;
        }

        public long getMaxReserved() {
            return this.maxReserved;
        }
    }
}

