/*
 * Decompiled with CFR 0.152.
 */
package io.trino.plugin.hive.util;

import com.google.common.base.CharMatcher;
import com.google.common.base.Joiner;
import com.google.common.base.MoreObjects;
import com.google.common.base.Preconditions;
import com.google.common.base.Splitter;
import com.google.common.base.Strings;
import com.google.common.base.VerifyException;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.ImmutableMap;
import com.google.common.collect.ImmutableSet;
import com.google.common.collect.Iterables;
import com.google.common.collect.Lists;
import com.google.common.collect.Sets;
import io.airlift.compress.lzo.LzoCodec;
import io.airlift.compress.lzo.LzopCodec;
import io.airlift.slice.Slice;
import io.airlift.slice.SliceUtf8;
import io.airlift.slice.Slices;
import io.trino.hadoop.TextLineLengthLimitExceededException;
import io.trino.hdfs.ConfigurationUtils;
import io.trino.hive.formats.compression.CompressionKind;
import io.trino.orc.OrcWriterOptions;
import io.trino.plugin.hive.HiveColumnHandle;
import io.trino.plugin.hive.HiveErrorCode;
import io.trino.plugin.hive.HivePartitionKey;
import io.trino.plugin.hive.HiveStorageFormat;
import io.trino.plugin.hive.HiveTimestampPrecision;
import io.trino.plugin.hive.HiveType;
import io.trino.plugin.hive.avro.TrinoAvroSerDe;
import io.trino.plugin.hive.metastore.Column;
import io.trino.plugin.hive.metastore.SortingColumn;
import io.trino.plugin.hive.metastore.Table;
import io.trino.plugin.hive.type.StructTypeInfo;
import io.trino.plugin.hive.util.FooterAwareRecordReader;
import io.trino.plugin.hive.util.HiveBucketing;
import io.trino.spi.ErrorCodeSupplier;
import io.trino.spi.StandardErrorCode;
import io.trino.spi.TrinoException;
import io.trino.spi.connector.ColumnMetadata;
import io.trino.spi.predicate.NullableValue;
import io.trino.spi.type.ArrayType;
import io.trino.spi.type.BigintType;
import io.trino.spi.type.BooleanType;
import io.trino.spi.type.CharType;
import io.trino.spi.type.Chars;
import io.trino.spi.type.DateTimeEncoding;
import io.trino.spi.type.DateType;
import io.trino.spi.type.DecimalType;
import io.trino.spi.type.DoubleType;
import io.trino.spi.type.Int128;
import io.trino.spi.type.IntegerType;
import io.trino.spi.type.MapType;
import io.trino.spi.type.RealType;
import io.trino.spi.type.RowType;
import io.trino.spi.type.SmallintType;
import io.trino.spi.type.TimestampType;
import io.trino.spi.type.TimestampWithTimeZoneType;
import io.trino.spi.type.TinyintType;
import io.trino.spi.type.Type;
import io.trino.spi.type.TypeManager;
import io.trino.spi.type.VarbinaryType;
import io.trino.spi.type.VarcharType;
import java.io.IOException;
import java.lang.reflect.Field;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.math.BigDecimal;
import java.math.BigInteger;
import java.math.RoundingMode;
import java.nio.charset.StandardCharsets;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HexFormat;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.OptionalInt;
import java.util.Properties;
import java.util.Set;
import java.util.function.Function;
import javax.annotation.Nullable;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.FileSystem;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.hive.ql.io.parquet.MapredParquetInputFormat;
import org.apache.hadoop.hive.serde2.AbstractSerDe;
import org.apache.hadoop.hive.serde2.Deserializer;
import org.apache.hadoop.hive.serde2.SerDeException;
import org.apache.hadoop.hive.serde2.objectinspector.ObjectInspector;
import org.apache.hadoop.hive.serde2.objectinspector.StructObjectInspector;
import org.apache.hadoop.io.compress.CompressionCodec;
import org.apache.hadoop.io.compress.CompressionCodecFactory;
import org.apache.hadoop.mapred.FileSplit;
import org.apache.hadoop.mapred.InputFormat;
import org.apache.hadoop.mapred.InputSplit;
import org.apache.hadoop.mapred.JobConf;
import org.apache.hadoop.mapred.RecordReader;
import org.apache.hadoop.mapred.Reporter;
import org.apache.hadoop.mapred.TextInputFormat;
import org.apache.hadoop.util.ReflectionUtils;
import org.joda.time.DateTimeZone;
import org.joda.time.Days;
import org.joda.time.LocalDateTime;
import org.joda.time.LocalTime;
import org.joda.time.ReadablePartial;
import org.joda.time.format.DateTimeFormat;
import org.joda.time.format.DateTimeFormatter;
import org.joda.time.format.DateTimeFormatterBuilder;
import org.joda.time.format.DateTimeParser;
import org.joda.time.format.DateTimePrinter;

public final class HiveUtil {
    public static final String SPARK_TABLE_PROVIDER_KEY = "spark.sql.sources.provider";
    public static final String DELTA_LAKE_PROVIDER = "delta";
    private static final String SPARK_TABLE_BUCKET_NUMBER_KEY = "spark.sql.sources.schema.numBuckets";
    public static final String ICEBERG_TABLE_TYPE_NAME = "table_type";
    public static final String ICEBERG_TABLE_TYPE_VALUE = "iceberg";
    private static final String HUDI_PARQUET_INPUT_FORMAT = "org.apache.hudi.hadoop.HoodieParquetInputFormat";
    private static final String HUDI_PARQUET_REALTIME_INPUT_FORMAT = "org.apache.hudi.hadoop.realtime.HoodieParquetRealtimeInputFormat";
    private static final String HUDI_INPUT_FORMAT = "com.uber.hoodie.hadoop.HoodieInputFormat";
    private static final String HUDI_REALTIME_INPUT_FORMAT = "com.uber.hoodie.hadoop.realtime.HoodieRealtimeInputFormat";
    private static final HexFormat HEX_UPPER_FORMAT = HexFormat.of().withUpperCase();
    private static final LocalDateTime EPOCH_DAY = new LocalDateTime(1970, 1, 1, 0, 0);
    private static final DateTimeFormatter HIVE_DATE_PARSER;
    private static final DateTimeFormatter HIVE_TIMESTAMP_PARSER;
    private static final Field COMPRESSION_CODECS_FIELD;
    private static final String BIG_DECIMAL_POSTFIX = "BD";
    private static final Splitter COLUMN_NAMES_SPLITTER;
    private static final CharMatcher PATH_CHAR_TO_ESCAPE;

    private HiveUtil() {
    }

    public static RecordReader<?, ?> createRecordReader(Configuration configuration, Path path, long start, long length, Properties schema, List<HiveColumnHandle> columns) {
        List readColumns = (List)columns.stream().filter(column -> column.getColumnType() == HiveColumnHandle.ColumnType.REGULAR).collect(ImmutableList.toImmutableList());
        readColumns.forEach(readColumn -> Preconditions.checkArgument((boolean)readColumn.isBaseColumn(), (String)"column %s is not a base column", (Object)readColumn.getName()));
        List readHiveColumnIndexes = (List)readColumns.stream().map(HiveColumnHandle::getBaseHiveColumnIndex).collect(ImmutableList.toImmutableList());
        configuration = ConfigurationUtils.copy((Configuration)configuration);
        HiveUtil.setReadColumns(configuration, readHiveColumnIndexes);
        InputFormat<?, ?> inputFormat = HiveUtil.getInputFormat(configuration, schema, true);
        JobConf jobConf = ConfigurationUtils.toJobConf((Configuration)configuration);
        FileSplit fileSplit = new FileSplit(path, start, length, (String[])null);
        schema.stringPropertyNames().stream().filter(name -> name.startsWith("serialization.")).forEach(name -> jobConf.set(name, schema.getProperty((String)name)));
        HiveUtil.configureCompressionCodecs(jobConf);
        try {
            int footerCount;
            FooterAwareRecordReader recordReader = inputFormat.getRecordReader((InputSplit)fileSplit, jobConf, Reporter.NULL);
            int headerCount = HiveUtil.getHeaderCount(schema);
            if (start == 0L && headerCount > 0) {
                HiveUtil.skipHeader(recordReader, headerCount);
            }
            if ((footerCount = HiveUtil.getFooterCount(schema)) > 0) {
                recordReader = new FooterAwareRecordReader(recordReader, footerCount, jobConf);
            }
            return recordReader;
        }
        catch (IOException e) {
            if (e instanceof TextLineLengthLimitExceededException) {
                throw new TrinoException((ErrorCodeSupplier)HiveErrorCode.HIVE_BAD_DATA, "Line too long in text file: " + path, (Throwable)e);
            }
            throw new TrinoException((ErrorCodeSupplier)HiveErrorCode.HIVE_CANNOT_OPEN_SPLIT, String.format("Error opening Hive split %s (offset=%s, length=%s) using %s: %s", path, start, length, HiveUtil.getInputFormatName(schema), MoreObjects.firstNonNull((Object)e.getMessage(), (Object)e.getClass().getName())), (Throwable)e);
        }
    }

    private static <K, V> void skipHeader(RecordReader<K, V> reader, int headerCount) throws IOException {
        Object key = reader.createKey();
        Object value = reader.createValue();
        while (headerCount > 0) {
            if (!reader.next(key, value)) {
                return;
            }
            --headerCount;
        }
    }

    private static void setReadColumns(Configuration configuration, List<Integer> readHiveColumnIndexes) {
        configuration.set("hive.io.file.readcolumn.ids", Joiner.on((char)',').join(readHiveColumnIndexes));
        configuration.setBoolean("hive.io.file.read.all.columns", false);
    }

    private static void configureCompressionCodecs(JobConf jobConf) {
        ArrayList codecs = Lists.newArrayList((Iterable)Splitter.on((String)",").trimResults().omitEmptyStrings().split((CharSequence)jobConf.get("io.compression.codecs", "")));
        if (!codecs.contains(LzoCodec.class.getName())) {
            codecs.add(0, LzoCodec.class.getName());
        }
        if (!codecs.contains(LzopCodec.class.getName())) {
            codecs.add(0, LzopCodec.class.getName());
        }
        jobConf.set("io.compression.codecs", String.join((CharSequence)",", codecs));
    }

    public static Optional<CompressionCodec> getCompressionCodec(TextInputFormat inputFormat, Path file) {
        CompressionCodecFactory compressionCodecFactory;
        try {
            compressionCodecFactory = (CompressionCodecFactory)COMPRESSION_CODECS_FIELD.get(inputFormat);
        }
        catch (IllegalAccessException e) {
            throw new TrinoException((ErrorCodeSupplier)StandardErrorCode.GENERIC_INTERNAL_ERROR, "Failed to find compressionCodec for inputFormat: " + inputFormat.getClass().getName(), (Throwable)e);
        }
        if (compressionCodecFactory == null) {
            return Optional.empty();
        }
        return Optional.ofNullable(compressionCodecFactory.getCodec(file));
    }

    public static InputFormat<?, ?> getInputFormat(Configuration configuration, Properties schema, boolean symlinkTarget) {
        String inputFormatName = HiveUtil.getInputFormatName(schema);
        try {
            JobConf jobConf = ConfigurationUtils.toJobConf((Configuration)configuration);
            HiveUtil.configureCompressionCodecs(jobConf);
            Class<InputFormat<?, ?>> inputFormatClass = HiveUtil.getInputFormatClass(jobConf, inputFormatName);
            if (symlinkTarget && inputFormatClass.getName().equals("org.apache.hadoop.hive.ql.io.SymlinkTextInputFormat")) {
                String serde = HiveUtil.getDeserializerClassName(schema);
                if (serde.equals(HiveStorageFormat.TEXTFILE.getSerde())) {
                    inputFormatClass = HiveUtil.getInputFormatClass(jobConf, HiveStorageFormat.TEXTFILE.getInputFormat());
                    return (InputFormat)ReflectionUtils.newInstance(inputFormatClass, (Configuration)jobConf);
                }
                for (HiveStorageFormat format : HiveStorageFormat.values()) {
                    if (!serde.equals(format.getSerde())) continue;
                    inputFormatClass = HiveUtil.getInputFormatClass(jobConf, format.getInputFormat());
                    return (InputFormat)ReflectionUtils.newInstance(inputFormatClass, (Configuration)jobConf);
                }
                throw new TrinoException((ErrorCodeSupplier)HiveErrorCode.HIVE_UNSUPPORTED_FORMAT, "Unknown SerDe for SymlinkTextInputFormat: " + serde);
            }
            return (InputFormat)ReflectionUtils.newInstance(inputFormatClass, (Configuration)jobConf);
        }
        catch (ClassNotFoundException | RuntimeException e) {
            throw new TrinoException((ErrorCodeSupplier)HiveErrorCode.HIVE_UNSUPPORTED_FORMAT, "Unable to create input format " + inputFormatName, (Throwable)e);
        }
    }

    private static Class<? extends InputFormat<?, ?>> getInputFormatClass(JobConf conf, String inputFormatName) throws ClassNotFoundException {
        if ("parquet.hive.DeprecatedParquetInputFormat".equals(inputFormatName) || "parquet.hive.MapredParquetInputFormat".equals(inputFormatName)) {
            return MapredParquetInputFormat.class;
        }
        Class clazz = conf.getClassByName(inputFormatName);
        return clazz.asSubclass(InputFormat.class);
    }

    public static String getInputFormatName(Properties schema) {
        String name = schema.getProperty("file.inputformat");
        HiveUtil.checkCondition(name != null, HiveErrorCode.HIVE_INVALID_METADATA, "Table or partition is missing Hive input format property: %s", "file.inputformat");
        return name;
    }

    private static long parseHiveDate(String value) {
        LocalDateTime date = HIVE_DATE_PARSER.parseLocalDateTime(value);
        if (!date.toLocalTime().equals((Object)LocalTime.MIDNIGHT)) {
            throw new IllegalArgumentException(String.format("The value should be a whole round date: '%s'", value));
        }
        return Days.daysBetween((ReadablePartial)EPOCH_DAY, (ReadablePartial)date).getDays();
    }

    public static long parseHiveTimestamp(String value) {
        return HIVE_TIMESTAMP_PARSER.parseMillis(value) * 1000L;
    }

    public static boolean isSplittable(InputFormat<?, ?> inputFormat, FileSystem fileSystem, Path path) {
        switch (inputFormat.getClass().getSimpleName()) {
            case "OrcInputFormat": 
            case "MapredParquetInputFormat": 
            case "AvroContainerInputFormat": 
            case "RCFileInputFormat": 
            case "SequenceFileInputFormat": {
                return true;
            }
            case "TextInputFormat": {
                return CompressionKind.forFile((String)path.getName()).isEmpty();
            }
        }
        Method method = null;
        for (Class<?> clazz = inputFormat.getClass(); clazz != null; clazz = clazz.getSuperclass()) {
            try {
                method = clazz.getDeclaredMethod("isSplitable", FileSystem.class, Path.class);
                break;
            }
            catch (NoSuchMethodException noSuchMethodException) {
                continue;
            }
        }
        if (method == null) {
            return false;
        }
        try {
            method.setAccessible(true);
            return (Boolean)method.invoke(inputFormat, fileSystem, path);
        }
        catch (IllegalAccessException | InvocationTargetException e) {
            throw new RuntimeException(e);
        }
    }

    public static StructObjectInspector getTableObjectInspector(Deserializer deserializer) {
        try {
            ObjectInspector inspector = deserializer.getObjectInspector();
            Preconditions.checkArgument((inspector.getCategory() == ObjectInspector.Category.STRUCT ? 1 : 0) != 0, (String)"expected STRUCT: %s", (Object)inspector.getCategory());
            return (StructObjectInspector)inspector;
        }
        catch (SerDeException e) {
            throw new RuntimeException(e);
        }
    }

    public static String getDeserializerClassName(Properties schema) {
        String name = schema.getProperty("serialization.lib");
        HiveUtil.checkCondition(name != null, HiveErrorCode.HIVE_INVALID_METADATA, "Table or partition is missing Hive deserializer property: %s", "serialization.lib");
        return name;
    }

    public static Deserializer getDeserializer(Configuration configuration, Properties schema) {
        String name = HiveUtil.getDeserializerClassName(schema);
        if (name.equals("org.apache.hadoop.hive.serde2.lazy.LazySimpleSerDe") && schema.containsKey("colelction.delim") && !schema.containsKey("collection.delim")) {
            schema.setProperty("collection.delim", schema.getProperty("colelction.delim"));
        }
        Deserializer deserializer = HiveUtil.createDeserializer(HiveUtil.getDeserializerClass(name));
        HiveUtil.initializeDeserializer(configuration, deserializer, schema);
        return deserializer;
    }

    private static Class<? extends Deserializer> getDeserializerClass(String name) {
        if ("org.apache.hadoop.hive.serde2.avro.AvroSerDe".equals(name)) {
            return TrinoAvroSerDe.class;
        }
        try {
            return Class.forName(name).asSubclass(Deserializer.class);
        }
        catch (ClassNotFoundException e) {
            throw new TrinoException((ErrorCodeSupplier)HiveErrorCode.HIVE_SERDE_NOT_FOUND, "deserializer does not exist: " + name);
        }
        catch (ClassCastException e) {
            throw new RuntimeException("invalid deserializer class: " + name);
        }
    }

    private static Deserializer createDeserializer(Class<? extends Deserializer> clazz) {
        try {
            return clazz.getConstructor(new Class[0]).newInstance(new Object[0]);
        }
        catch (ReflectiveOperationException e) {
            throw new RuntimeException("error creating deserializer: " + clazz.getName(), e);
        }
    }

    private static void initializeDeserializer(Configuration configuration, Deserializer deserializer, Properties schema) {
        try {
            configuration = ConfigurationUtils.copy((Configuration)configuration);
            deserializer.initialize(configuration, schema);
            HiveUtil.validate(deserializer);
        }
        catch (RuntimeException | SerDeException e) {
            throw new RuntimeException("error initializing deserializer: " + deserializer.getClass().getName(), e);
        }
    }

    private static void validate(Deserializer deserializer) {
        if (deserializer instanceof AbstractSerDe && !((AbstractSerDe)deserializer).getConfigurationErrors().isEmpty()) {
            throw new RuntimeException("There are configuration errors: " + ((AbstractSerDe)deserializer).getConfigurationErrors());
        }
    }

    private static boolean isHiveNull(byte[] bytes) {
        return bytes.length == 2 && bytes[0] == 92 && bytes[1] == 78;
    }

    public static void verifyPartitionTypeSupported(String partitionName, Type type) {
        if (!HiveUtil.isValidPartitionType(type)) {
            throw new TrinoException((ErrorCodeSupplier)StandardErrorCode.NOT_SUPPORTED, String.format("Unsupported type [%s] for partition: %s", type, partitionName));
        }
    }

    private static boolean isValidPartitionType(Type type) {
        return type instanceof DecimalType || BooleanType.BOOLEAN.equals((Object)type) || TinyintType.TINYINT.equals((Object)type) || SmallintType.SMALLINT.equals((Object)type) || IntegerType.INTEGER.equals((Object)type) || BigintType.BIGINT.equals((Object)type) || RealType.REAL.equals((Object)type) || DoubleType.DOUBLE.equals((Object)type) || DateType.DATE.equals((Object)type) || TimestampType.TIMESTAMP_MILLIS.equals((Object)type) || type instanceof VarcharType || type instanceof CharType;
    }

    public static NullableValue parsePartitionValue(String partitionName, String value, Type type) {
        HiveUtil.verifyPartitionTypeSupported(partitionName, type);
        boolean isNull = "__HIVE_DEFAULT_PARTITION__".equals(value);
        if (type instanceof DecimalType) {
            DecimalType decimalType = (DecimalType)type;
            if (isNull) {
                return NullableValue.asNull((Type)decimalType);
            }
            if (decimalType.isShort()) {
                if (value.isEmpty()) {
                    return NullableValue.of((Type)decimalType, (Object)0L);
                }
                return NullableValue.of((Type)decimalType, (Object)HiveUtil.shortDecimalPartitionKey(value, decimalType, partitionName));
            }
            if (value.isEmpty()) {
                return NullableValue.of((Type)decimalType, (Object)Int128.ZERO);
            }
            return NullableValue.of((Type)decimalType, (Object)HiveUtil.longDecimalPartitionKey(value, decimalType, partitionName));
        }
        if (BooleanType.BOOLEAN.equals((Object)type)) {
            if (isNull) {
                return NullableValue.asNull((Type)BooleanType.BOOLEAN);
            }
            if (value.isEmpty()) {
                return NullableValue.of((Type)BooleanType.BOOLEAN, (Object)false);
            }
            return NullableValue.of((Type)BooleanType.BOOLEAN, (Object)HiveUtil.booleanPartitionKey(value, partitionName));
        }
        if (TinyintType.TINYINT.equals((Object)type)) {
            if (isNull) {
                return NullableValue.asNull((Type)TinyintType.TINYINT);
            }
            if (value.isEmpty()) {
                return NullableValue.of((Type)TinyintType.TINYINT, (Object)0L);
            }
            return NullableValue.of((Type)TinyintType.TINYINT, (Object)HiveUtil.tinyintPartitionKey(value, partitionName));
        }
        if (SmallintType.SMALLINT.equals((Object)type)) {
            if (isNull) {
                return NullableValue.asNull((Type)SmallintType.SMALLINT);
            }
            if (value.isEmpty()) {
                return NullableValue.of((Type)SmallintType.SMALLINT, (Object)0L);
            }
            return NullableValue.of((Type)SmallintType.SMALLINT, (Object)HiveUtil.smallintPartitionKey(value, partitionName));
        }
        if (IntegerType.INTEGER.equals((Object)type)) {
            if (isNull) {
                return NullableValue.asNull((Type)IntegerType.INTEGER);
            }
            if (value.isEmpty()) {
                return NullableValue.of((Type)IntegerType.INTEGER, (Object)0L);
            }
            return NullableValue.of((Type)IntegerType.INTEGER, (Object)HiveUtil.integerPartitionKey(value, partitionName));
        }
        if (BigintType.BIGINT.equals((Object)type)) {
            if (isNull) {
                return NullableValue.asNull((Type)BigintType.BIGINT);
            }
            if (value.isEmpty()) {
                return NullableValue.of((Type)BigintType.BIGINT, (Object)0L);
            }
            return NullableValue.of((Type)BigintType.BIGINT, (Object)HiveUtil.bigintPartitionKey(value, partitionName));
        }
        if (DateType.DATE.equals((Object)type)) {
            if (isNull) {
                return NullableValue.asNull((Type)DateType.DATE);
            }
            return NullableValue.of((Type)DateType.DATE, (Object)HiveUtil.datePartitionKey(value, partitionName));
        }
        if (TimestampType.TIMESTAMP_MILLIS.equals((Object)type)) {
            if (isNull) {
                return NullableValue.asNull((Type)TimestampType.TIMESTAMP_MILLIS);
            }
            return NullableValue.of((Type)TimestampType.TIMESTAMP_MILLIS, (Object)HiveUtil.timestampPartitionKey(value, partitionName));
        }
        if (RealType.REAL.equals((Object)type)) {
            if (isNull) {
                return NullableValue.asNull((Type)RealType.REAL);
            }
            if (value.isEmpty()) {
                return NullableValue.of((Type)RealType.REAL, (Object)Float.floatToRawIntBits(0.0f));
            }
            return NullableValue.of((Type)RealType.REAL, (Object)HiveUtil.floatPartitionKey(value, partitionName));
        }
        if (DoubleType.DOUBLE.equals((Object)type)) {
            if (isNull) {
                return NullableValue.asNull((Type)DoubleType.DOUBLE);
            }
            if (value.isEmpty()) {
                return NullableValue.of((Type)DoubleType.DOUBLE, (Object)0.0);
            }
            return NullableValue.of((Type)DoubleType.DOUBLE, (Object)HiveUtil.doublePartitionKey(value, partitionName));
        }
        if (type instanceof VarcharType) {
            if (isNull) {
                return NullableValue.asNull((Type)type);
            }
            return NullableValue.of((Type)type, (Object)HiveUtil.varcharPartitionKey(value, partitionName, type));
        }
        if (type instanceof CharType) {
            if (isNull) {
                return NullableValue.asNull((Type)type);
            }
            return NullableValue.of((Type)type, (Object)HiveUtil.charPartitionKey(value, partitionName, type));
        }
        if (type instanceof VarbinaryType) {
            if (isNull) {
                return NullableValue.asNull((Type)type);
            }
            return NullableValue.of((Type)type, (Object)Slices.utf8Slice((String)value));
        }
        throw new VerifyException(String.format("Unhandled type [%s] for partition: %s", type, partitionName));
    }

    public static boolean isStructuralType(Type type) {
        return type instanceof ArrayType || type instanceof MapType || type instanceof RowType;
    }

    private static boolean booleanPartitionKey(String value, String name) {
        if (value.equalsIgnoreCase("true")) {
            return true;
        }
        if (value.equalsIgnoreCase("false")) {
            return false;
        }
        throw new TrinoException((ErrorCodeSupplier)HiveErrorCode.HIVE_INVALID_PARTITION_VALUE, String.format("Invalid partition value '%s' for BOOLEAN partition key: %s", value, name));
    }

    private static long bigintPartitionKey(String value, String name) {
        try {
            return Long.parseLong(value);
        }
        catch (NumberFormatException e) {
            throw new TrinoException((ErrorCodeSupplier)HiveErrorCode.HIVE_INVALID_PARTITION_VALUE, String.format("Invalid partition value '%s' for BIGINT partition key: %s", value, name));
        }
    }

    private static long integerPartitionKey(String value, String name) {
        try {
            return Integer.parseInt(value);
        }
        catch (NumberFormatException e) {
            throw new TrinoException((ErrorCodeSupplier)HiveErrorCode.HIVE_INVALID_PARTITION_VALUE, String.format("Invalid partition value '%s' for INTEGER partition key: %s", value, name));
        }
    }

    private static long smallintPartitionKey(String value, String name) {
        try {
            return Short.parseShort(value);
        }
        catch (NumberFormatException e) {
            throw new TrinoException((ErrorCodeSupplier)HiveErrorCode.HIVE_INVALID_PARTITION_VALUE, String.format("Invalid partition value '%s' for SMALLINT partition key: %s", value, name));
        }
    }

    private static long tinyintPartitionKey(String value, String name) {
        try {
            return Byte.parseByte(value);
        }
        catch (NumberFormatException e) {
            throw new TrinoException((ErrorCodeSupplier)HiveErrorCode.HIVE_INVALID_PARTITION_VALUE, String.format("Invalid partition value '%s' for TINYINT partition key: %s", value, name));
        }
    }

    private static long floatPartitionKey(String value, String name) {
        try {
            return Float.floatToRawIntBits(Float.parseFloat(value));
        }
        catch (NumberFormatException e) {
            throw new TrinoException((ErrorCodeSupplier)HiveErrorCode.HIVE_INVALID_PARTITION_VALUE, String.format("Invalid partition value '%s' for FLOAT partition key: %s", value, name));
        }
    }

    private static double doublePartitionKey(String value, String name) {
        try {
            return Double.parseDouble(value);
        }
        catch (NumberFormatException e) {
            throw new TrinoException((ErrorCodeSupplier)HiveErrorCode.HIVE_INVALID_PARTITION_VALUE, String.format("Invalid partition value '%s' for DOUBLE partition key: %s", value, name));
        }
    }

    private static long datePartitionKey(String value, String name) {
        try {
            return HiveUtil.parseHiveDate(value);
        }
        catch (IllegalArgumentException e) {
            throw new TrinoException((ErrorCodeSupplier)HiveErrorCode.HIVE_INVALID_PARTITION_VALUE, String.format("Invalid partition value '%s' for DATE partition key: %s", value, name));
        }
    }

    private static long timestampPartitionKey(String value, String name) {
        try {
            return HiveUtil.parseHiveTimestamp(value);
        }
        catch (IllegalArgumentException e) {
            throw new TrinoException((ErrorCodeSupplier)HiveErrorCode.HIVE_INVALID_PARTITION_VALUE, String.format("Invalid partition value '%s' for TIMESTAMP partition key: %s", value, name));
        }
    }

    private static long shortDecimalPartitionKey(String value, DecimalType type, String name) {
        return HiveUtil.decimalPartitionKey(value, type, name).unscaledValue().longValue();
    }

    private static Int128 longDecimalPartitionKey(String value, DecimalType type, String name) {
        return Int128.valueOf((BigInteger)HiveUtil.decimalPartitionKey(value, type, name).unscaledValue());
    }

    private static BigDecimal decimalPartitionKey(String value, DecimalType type, String name) {
        try {
            if (value.endsWith(BIG_DECIMAL_POSTFIX)) {
                value = value.substring(0, value.length() - BIG_DECIMAL_POSTFIX.length());
            }
            BigDecimal decimal = new BigDecimal(value);
            if ((decimal = decimal.setScale(type.getScale(), RoundingMode.UNNECESSARY)).precision() > type.getPrecision()) {
                throw new TrinoException((ErrorCodeSupplier)HiveErrorCode.HIVE_INVALID_PARTITION_VALUE, String.format("Invalid partition value '%s' for %s partition key: %s", value, type, name));
            }
            return decimal;
        }
        catch (NumberFormatException e) {
            throw new TrinoException((ErrorCodeSupplier)HiveErrorCode.HIVE_INVALID_PARTITION_VALUE, String.format("Invalid partition value '%s' for %s partition key: %s", value, type, name));
        }
    }

    private static Slice varcharPartitionKey(String value, String name, Type columnType) {
        Slice partitionKey = Slices.utf8Slice((String)value);
        VarcharType varcharType = (VarcharType)columnType;
        if (!varcharType.isUnbounded() && SliceUtf8.countCodePoints((Slice)partitionKey) > varcharType.getBoundedLength()) {
            throw new TrinoException((ErrorCodeSupplier)HiveErrorCode.HIVE_INVALID_PARTITION_VALUE, String.format("Invalid partition value '%s' for %s partition key: %s", value, columnType, name));
        }
        return partitionKey;
    }

    private static Slice charPartitionKey(String value, String name, Type columnType) {
        Slice partitionKey = Chars.trimTrailingSpaces((Slice)Slices.utf8Slice((String)value));
        CharType charType = (CharType)columnType;
        if (SliceUtf8.countCodePoints((Slice)partitionKey) > charType.getLength()) {
            throw new TrinoException((ErrorCodeSupplier)HiveErrorCode.HIVE_INVALID_PARTITION_VALUE, String.format("Invalid partition value '%s' for %s partition key: %s", value, columnType, name));
        }
        return partitionKey;
    }

    public static List<HiveColumnHandle> hiveColumnHandles(Table table, TypeManager typeManager, HiveTimestampPrecision timestampPrecision) {
        ImmutableList.Builder columns = ImmutableList.builder();
        columns.addAll(HiveUtil.getRegularColumnHandles(table, typeManager, timestampPrecision));
        columns.addAll(HiveUtil.getPartitionKeyColumnHandles(table, typeManager));
        columns.add((Object)HiveColumnHandle.pathColumnHandle());
        if (table.getStorage().getBucketProperty().isPresent() && HiveBucketing.isSupportedBucketing(table)) {
            columns.add((Object)HiveColumnHandle.bucketColumnHandle());
        }
        columns.add((Object)HiveColumnHandle.fileSizeColumnHandle());
        columns.add((Object)HiveColumnHandle.fileModifiedTimeColumnHandle());
        if (!table.getPartitionColumns().isEmpty()) {
            columns.add((Object)HiveColumnHandle.partitionColumnHandle());
        }
        return columns.build();
    }

    public static List<HiveColumnHandle> getRegularColumnHandles(Table table, TypeManager typeManager, HiveTimestampPrecision timestampPrecision) {
        ImmutableList.Builder columns = ImmutableList.builder();
        int hiveColumnIndex = 0;
        for (Column field : table.getDataColumns()) {
            HiveType hiveType = field.getType();
            if (hiveType.isSupportedType(table.getStorage().getStorageFormat())) {
                columns.add((Object)HiveColumnHandle.createBaseColumn(field.getName(), hiveColumnIndex, hiveType, hiveType.getType(typeManager, timestampPrecision), HiveColumnHandle.ColumnType.REGULAR, field.getComment()));
            }
            ++hiveColumnIndex;
        }
        return columns.build();
    }

    public static List<HiveColumnHandle> getPartitionKeyColumnHandles(Table table, TypeManager typeManager) {
        ImmutableList.Builder columns = ImmutableList.builder();
        List<Column> partitionKeys = table.getPartitionColumns();
        for (Column field : partitionKeys) {
            HiveType hiveType = field.getType();
            if (!hiveType.isSupportedType(table.getStorage().getStorageFormat())) {
                throw new TrinoException((ErrorCodeSupplier)StandardErrorCode.NOT_SUPPORTED, String.format("Unsupported Hive type %s found in partition keys of table %s.%s", hiveType, table.getDatabaseName(), table.getTableName()));
            }
            columns.add((Object)HiveColumnHandle.createBaseColumn(field.getName(), -1, hiveType, hiveType.getType(typeManager), HiveColumnHandle.ColumnType.PARTITION_KEY, field.getComment()));
        }
        return columns.build();
    }

    public static void checkCondition(boolean condition, ErrorCodeSupplier errorCode, String formatString, Object ... args) {
        if (!condition) {
            throw new TrinoException(errorCode, String.format(formatString, args));
        }
    }

    @Nullable
    public static String columnExtraInfo(boolean partitionKey) {
        return partitionKey ? "partition key" : null;
    }

    public static List<String> toPartitionValues(String partitionName) {
        ImmutableList.Builder resultBuilder = ImmutableList.builder();
        int start = 0;
        while (true) {
            int end;
            if (start < partitionName.length() && partitionName.charAt(start) != '=') {
                ++start;
                continue;
            }
            for (end = ++start; end < partitionName.length() && partitionName.charAt(end) != '/'; ++end) {
            }
            if (start > partitionName.length()) break;
            resultBuilder.add((Object)HiveUtil.unescapePathName(partitionName.substring(start, end)));
            start = end + 1;
        }
        return resultBuilder.build();
    }

    public static NullableValue getPrefilledColumnValue(HiveColumnHandle columnHandle, HivePartitionKey partitionKey, Path path, OptionalInt bucketNumber, long fileSize, long fileModifiedTime, String partitionName) {
        String columnValue;
        if (partitionKey != null) {
            columnValue = partitionKey.getValue();
        } else if (HiveColumnHandle.isPathColumnHandle(columnHandle)) {
            columnValue = path.toString();
        } else if (HiveColumnHandle.isBucketColumnHandle(columnHandle)) {
            columnValue = String.valueOf(bucketNumber.getAsInt());
        } else if (HiveColumnHandle.isFileSizeColumnHandle(columnHandle)) {
            columnValue = String.valueOf(fileSize);
        } else if (HiveColumnHandle.isFileModifiedTimeColumnHandle(columnHandle)) {
            columnValue = HIVE_TIMESTAMP_PARSER.print(fileModifiedTime);
        } else if (HiveColumnHandle.isPartitionColumnHandle(columnHandle)) {
            columnValue = partitionName;
        } else {
            throw new TrinoException((ErrorCodeSupplier)StandardErrorCode.NOT_SUPPORTED, "unsupported hidden column: " + columnHandle);
        }
        byte[] bytes = columnValue.getBytes(StandardCharsets.UTF_8);
        String name = columnHandle.getName();
        Type type = columnHandle.getType();
        if (HiveUtil.isHiveNull(bytes)) {
            return NullableValue.asNull((Type)type);
        }
        if (type.equals(BooleanType.BOOLEAN)) {
            return NullableValue.of((Type)type, (Object)HiveUtil.booleanPartitionKey(columnValue, name));
        }
        if (type.equals(BigintType.BIGINT)) {
            return NullableValue.of((Type)type, (Object)HiveUtil.bigintPartitionKey(columnValue, name));
        }
        if (type.equals(IntegerType.INTEGER)) {
            return NullableValue.of((Type)type, (Object)HiveUtil.integerPartitionKey(columnValue, name));
        }
        if (type.equals(SmallintType.SMALLINT)) {
            return NullableValue.of((Type)type, (Object)HiveUtil.smallintPartitionKey(columnValue, name));
        }
        if (type.equals(TinyintType.TINYINT)) {
            return NullableValue.of((Type)type, (Object)HiveUtil.tinyintPartitionKey(columnValue, name));
        }
        if (type.equals(RealType.REAL)) {
            return NullableValue.of((Type)type, (Object)HiveUtil.floatPartitionKey(columnValue, name));
        }
        if (type.equals(DoubleType.DOUBLE)) {
            return NullableValue.of((Type)type, (Object)HiveUtil.doublePartitionKey(columnValue, name));
        }
        if (type instanceof VarcharType) {
            return NullableValue.of((Type)type, (Object)HiveUtil.varcharPartitionKey(columnValue, name, type));
        }
        if (type instanceof CharType) {
            return NullableValue.of((Type)type, (Object)HiveUtil.charPartitionKey(columnValue, name, type));
        }
        if (type.equals(DateType.DATE)) {
            return NullableValue.of((Type)type, (Object)HiveUtil.datePartitionKey(columnValue, name));
        }
        if (type.equals(TimestampType.TIMESTAMP_MILLIS)) {
            return NullableValue.of((Type)type, (Object)HiveUtil.timestampPartitionKey(columnValue, name));
        }
        if (type.equals(TimestampWithTimeZoneType.TIMESTAMP_TZ_MILLIS)) {
            return NullableValue.of((Type)type, (Object)DateTimeEncoding.packDateTimeWithZone((long)Math.floorDiv(HiveUtil.timestampPartitionKey(columnValue, name), 1000), (String)DateTimeZone.getDefault().getID()));
        }
        if (type instanceof DecimalType) {
            DecimalType decimalType = (DecimalType)type;
            if (decimalType.isShort()) {
                return NullableValue.of((Type)type, (Object)HiveUtil.shortDecimalPartitionKey(columnValue, decimalType, name));
            }
            return NullableValue.of((Type)type, (Object)HiveUtil.longDecimalPartitionKey(columnValue, decimalType, name));
        }
        if (type.equals(VarbinaryType.VARBINARY)) {
            return NullableValue.of((Type)type, (Object)Slices.utf8Slice((String)columnValue));
        }
        throw new TrinoException((ErrorCodeSupplier)StandardErrorCode.NOT_SUPPORTED, String.format("Unsupported column type %s for prefilled column: %s", type.getDisplayName(), name));
    }

    public static List<HiveType> extractStructFieldTypes(HiveType hiveType) {
        return (List)((StructTypeInfo)hiveType.getTypeInfo()).getAllStructFieldTypeInfos().stream().map(typeInfo -> HiveType.valueOf(typeInfo.getTypeName())).collect(ImmutableList.toImmutableList());
    }

    public static int getHeaderCount(Properties schema) {
        return HiveUtil.getPositiveIntegerValue(schema, "skip.header.line.count", "0");
    }

    public static int getFooterCount(Properties schema) {
        return HiveUtil.getPositiveIntegerValue(schema, "skip.footer.line.count", "0");
    }

    private static int getPositiveIntegerValue(Properties schema, String key, String defaultValue) {
        String value = schema.getProperty(key, defaultValue);
        try {
            int intValue = Integer.parseInt(value);
            if (intValue < 0) {
                throw new TrinoException((ErrorCodeSupplier)HiveErrorCode.HIVE_INVALID_METADATA, String.format("Invalid value for %s property: %s", key, value));
            }
            return intValue;
        }
        catch (NumberFormatException e) {
            throw new TrinoException((ErrorCodeSupplier)HiveErrorCode.HIVE_INVALID_METADATA, String.format("Invalid value for %s property: %s", key, value));
        }
    }

    public static List<String> getColumnNames(Properties schema) {
        return COLUMN_NAMES_SPLITTER.splitToList((CharSequence)schema.getProperty("columns", ""));
    }

    public static List<HiveType> getColumnTypes(Properties schema) {
        return HiveType.toHiveTypes(schema.getProperty("columns.types", ""));
    }

    public static OrcWriterOptions getOrcWriterOptions(Properties schema, OrcWriterOptions orcWriterOptions) {
        if (schema.containsKey("orc.bloom.filter.columns")) {
            if (!schema.containsKey("orc.bloom.filter.fpp")) {
                throw new TrinoException((ErrorCodeSupplier)HiveErrorCode.HIVE_INVALID_METADATA, "FPP for bloom filter is missing");
            }
            try {
                double fpp = Double.parseDouble(schema.getProperty("orc.bloom.filter.fpp"));
                return orcWriterOptions.withBloomFilterColumns((Set)ImmutableSet.copyOf((Collection)COLUMN_NAMES_SPLITTER.splitToList((CharSequence)schema.getProperty("orc.bloom.filter.columns")))).withBloomFilterFpp(fpp);
            }
            catch (NumberFormatException e) {
                throw new TrinoException((ErrorCodeSupplier)HiveErrorCode.HIVE_UNSUPPORTED_FORMAT, String.format("Invalid value for %s property: %s", "orc_bloom_filter_fpp", schema.getProperty("orc.bloom.filter.fpp")));
            }
        }
        return orcWriterOptions;
    }

    public static SortingColumn sortingColumnFromString(String name) {
        SortingColumn.Order order = SortingColumn.Order.ASCENDING;
        String lower = name.toUpperCase(Locale.ENGLISH);
        if (lower.endsWith(" ASC")) {
            name = name.substring(0, name.length() - 4).trim();
        } else if (lower.endsWith(" DESC")) {
            name = name.substring(0, name.length() - 5).trim();
            order = SortingColumn.Order.DESCENDING;
        }
        return new SortingColumn(name, order);
    }

    public static String sortingColumnToString(SortingColumn column) {
        return column.getColumnName() + (column.getOrder() == SortingColumn.Order.DESCENDING ? " DESC" : "");
    }

    public static boolean isHiveSystemSchema(String schemaName) {
        if ("information_schema".equals(schemaName)) {
            return true;
        }
        return "sys".equals(schemaName);
    }

    public static boolean isDeltaLakeTable(Table table) {
        return HiveUtil.isDeltaLakeTable(table.getParameters());
    }

    public static boolean isDeltaLakeTable(Map<String, String> tableParameters) {
        return tableParameters.containsKey(SPARK_TABLE_PROVIDER_KEY) && tableParameters.get(SPARK_TABLE_PROVIDER_KEY).toLowerCase(Locale.ENGLISH).equals(DELTA_LAKE_PROVIDER);
    }

    public static boolean isIcebergTable(Table table) {
        return HiveUtil.isIcebergTable(table.getParameters());
    }

    public static boolean isIcebergTable(Map<String, String> tableParameters) {
        return ICEBERG_TABLE_TYPE_VALUE.equalsIgnoreCase(tableParameters.get(ICEBERG_TABLE_TYPE_NAME));
    }

    public static boolean isHudiTable(Table table) {
        Objects.requireNonNull(table, "table is null");
        String inputFormat = table.getStorage().getStorageFormat().getInputFormatNullable();
        return HUDI_PARQUET_INPUT_FORMAT.equals(inputFormat) || HUDI_PARQUET_REALTIME_INPUT_FORMAT.equals(inputFormat) || HUDI_INPUT_FORMAT.equals(inputFormat) || HUDI_REALTIME_INPUT_FORMAT.equals(inputFormat);
    }

    public static boolean isSparkBucketedTable(Table table) {
        return table.getParameters().containsKey(SPARK_TABLE_PROVIDER_KEY) && table.getParameters().containsKey(SPARK_TABLE_BUCKET_NUMBER_KEY);
    }

    public static Function<HiveColumnHandle, ColumnMetadata> columnMetadataGetter(Table table) {
        ImmutableList.Builder columnNames = ImmutableList.builder();
        table.getPartitionColumns().stream().map(Column::getName).forEach(arg_0 -> ((ImmutableList.Builder)columnNames).add(arg_0));
        table.getDataColumns().stream().map(Column::getName).forEach(arg_0 -> ((ImmutableList.Builder)columnNames).add(arg_0));
        ImmutableList allColumnNames = columnNames.build();
        if (allColumnNames.size() > Sets.newHashSet((Iterable)allColumnNames).size()) {
            throw new TrinoException((ErrorCodeSupplier)HiveErrorCode.HIVE_INVALID_METADATA, String.format("Hive metadata for table %s is invalid: Table descriptor contains duplicate columns", table.getTableName()));
        }
        List<Column> tableColumns = table.getDataColumns();
        ImmutableMap.Builder builder = ImmutableMap.builder();
        for (Column field : Iterables.concat(tableColumns, table.getPartitionColumns())) {
            if (field.getComment().isPresent() && !field.getComment().get().equals("from deserializer")) {
                builder.put((Object)field.getName(), field.getComment());
                continue;
            }
            builder.put((Object)field.getName(), Optional.empty());
        }
        ImmutableMap columnComment = builder.buildOrThrow();
        return arg_0 -> HiveUtil.lambda$columnMetadataGetter$5((Map)columnComment, arg_0);
    }

    public static String unescapePathName(String path) {
        int escapedAtIndex = path.indexOf(37);
        if (escapedAtIndex < 0 || escapedAtIndex + 2 >= path.length()) {
            return path;
        }
        StringBuilder sb = new StringBuilder();
        int fromIndex = 0;
        while (escapedAtIndex >= 0 && escapedAtIndex + 2 < path.length()) {
            if (escapedAtIndex > fromIndex) {
                sb.append(path, fromIndex, escapedAtIndex);
            }
            try {
                int code = HexFormat.fromHexDigits(path, escapedAtIndex + 1, escapedAtIndex + 3);
                sb.append((char)code);
                fromIndex = escapedAtIndex + 3;
            }
            catch (NumberFormatException e) {
                sb.append('%');
                fromIndex = escapedAtIndex + 1;
            }
            escapedAtIndex = path.indexOf(37, fromIndex);
        }
        if (fromIndex < path.length()) {
            sb.append(path, fromIndex, path.length());
        }
        return sb.toString();
    }

    public static String escapePathName(String path) {
        if (Strings.isNullOrEmpty((String)path)) {
            return "__HIVE_DEFAULT_PARTITION__";
        }
        int escapeAtIndex = PATH_CHAR_TO_ESCAPE.indexIn((CharSequence)path);
        if (escapeAtIndex < 0) {
            return path;
        }
        StringBuilder sb = new StringBuilder();
        int fromIndex = 0;
        while (escapeAtIndex >= 0 && escapeAtIndex < path.length()) {
            if (escapeAtIndex > fromIndex) {
                sb.append(path, fromIndex, escapeAtIndex);
            }
            char c = path.charAt(escapeAtIndex);
            sb.append('%').append(HEX_UPPER_FORMAT.toHighHexDigit(c)).append(HEX_UPPER_FORMAT.toLowHexDigit(c));
            fromIndex = escapeAtIndex + 1;
            if (fromIndex < path.length()) {
                escapeAtIndex = PATH_CHAR_TO_ESCAPE.indexIn((CharSequence)path, fromIndex);
                continue;
            }
            escapeAtIndex = -1;
        }
        if (fromIndex < path.length()) {
            sb.append(path, fromIndex, path.length());
        }
        return sb.toString();
    }

    public static String makePartName(List<String> columns, List<String> values) {
        StringBuilder name = new StringBuilder();
        for (int i = 0; i < columns.size(); ++i) {
            if (i > 0) {
                name.append('/');
            }
            name.append(HiveUtil.escapePathName(columns.get(i).toLowerCase(Locale.ENGLISH)));
            name.append('=');
            name.append(HiveUtil.escapePathName(values.get(i)));
        }
        return name.toString();
    }

    private static /* synthetic */ ColumnMetadata lambda$columnMetadataGetter$5(Map columnComment, HiveColumnHandle handle) {
        return ColumnMetadata.builder().setName(handle.getName()).setType(handle.getType()).setComment(handle.isHidden() ? Optional.empty() : (Optional)columnComment.get(handle.getName())).setExtraInfo(Optional.ofNullable(HiveUtil.columnExtraInfo(handle.isPartitionKey()))).setHidden(handle.isHidden()).build();
    }

    static {
        COLUMN_NAMES_SPLITTER = Splitter.on((char)',').trimResults().omitEmptyStrings();
        PATH_CHAR_TO_ESCAPE = CharMatcher.inRange((char)'\u0000', (char)'\u001f').or(CharMatcher.anyOf((CharSequence)"\"#%'*/:=?\\\u007f{[]^")).precomputed();
        DateTimeParser[] timestampWithoutTimeZoneParser = new DateTimeParser[]{DateTimeFormat.forPattern((String)"yyyy-M-d").getParser(), DateTimeFormat.forPattern((String)"yyyy-M-d H:m").getParser(), DateTimeFormat.forPattern((String)"yyyy-M-d H:m:s").getParser(), DateTimeFormat.forPattern((String)"yyyy-M-d H:m:s.SSS").getParser(), DateTimeFormat.forPattern((String)"yyyy-M-d H:m:s.SSSSSSS").getParser(), DateTimeFormat.forPattern((String)"yyyy-M-d H:m:s.SSSSSSSSS").getParser()};
        DateTimePrinter timestampWithoutTimeZonePrinter = DateTimeFormat.forPattern((String)"yyyy-MM-dd HH:mm:ss.SSSSSSSSS").getPrinter();
        HIVE_TIMESTAMP_PARSER = new DateTimeFormatterBuilder().append(timestampWithoutTimeZonePrinter, timestampWithoutTimeZoneParser).toFormatter().withZoneUTC();
        HIVE_DATE_PARSER = new DateTimeFormatterBuilder().append(timestampWithoutTimeZonePrinter, timestampWithoutTimeZoneParser).toFormatter().withZoneUTC();
        try {
            COMPRESSION_CODECS_FIELD = TextInputFormat.class.getDeclaredField("compressionCodecs");
            COMPRESSION_CODECS_FIELD.setAccessible(true);
        }
        catch (ReflectiveOperationException e) {
            throw new AssertionError((Object)e);
        }
    }
}

