/*
 * Decompiled with CFR 0.152.
 */
package io.trino.parquet;

import com.google.common.base.Preconditions;
import com.google.common.io.ByteStreams;
import io.airlift.compress.Decompressor;
import io.airlift.compress.lz4.Lz4Decompressor;
import io.airlift.compress.lzo.LzoDecompressor;
import io.airlift.compress.snappy.SnappyDecompressor;
import io.airlift.compress.zstd.ZstdDecompressor;
import io.airlift.slice.Slice;
import io.airlift.slice.Slices;
import io.trino.parquet.ParquetCorruptionException;
import io.trino.parquet.ParquetDataSourceId;
import java.io.IOException;
import java.io.InputStream;
import java.util.Objects;
import java.util.zip.GZIPInputStream;
import org.apache.parquet.format.CompressionCodec;

public final class ParquetCompressionUtils {
    private static final int GZIP_BUFFER_SIZE = 8192;

    private ParquetCompressionUtils() {
    }

    public static Slice decompress(ParquetDataSourceId dataSourceId, CompressionCodec codec, Slice input, int uncompressedSize) throws IOException {
        Objects.requireNonNull(input, "input is null");
        if (input.length() == 0) {
            return Slices.EMPTY_SLICE;
        }
        return switch (codec) {
            default -> throw new IncompatibleClassChangeError();
            case CompressionCodec.UNCOMPRESSED -> input;
            case CompressionCodec.GZIP -> ParquetCompressionUtils.decompressGzip(input, uncompressedSize);
            case CompressionCodec.SNAPPY -> ParquetCompressionUtils.decompressSnappy(input, uncompressedSize);
            case CompressionCodec.LZO -> ParquetCompressionUtils.decompressLZO(input, uncompressedSize);
            case CompressionCodec.LZ4 -> ParquetCompressionUtils.decompressLz4(input, uncompressedSize);
            case CompressionCodec.ZSTD -> ParquetCompressionUtils.decompressZstd(input, uncompressedSize);
            case CompressionCodec.BROTLI, CompressionCodec.LZ4_RAW -> throw new ParquetCorruptionException(dataSourceId, "Codec not supported in Parquet: %s", codec);
        };
    }

    private static Slice decompressSnappy(Slice input, int uncompressedSize) {
        byte[] buffer = new byte[uncompressedSize + 8];
        int actualUncompressedSize = ParquetCompressionUtils.decompress((Decompressor)new SnappyDecompressor(), input, 0, input.length(), buffer, 0);
        if (actualUncompressedSize != uncompressedSize) {
            throw new IllegalArgumentException(String.format("Invalid uncompressedSize for SNAPPY input. Expected %s, actual: %s", uncompressedSize, actualUncompressedSize));
        }
        return Slices.wrappedBuffer((byte[])buffer, (int)0, (int)uncompressedSize);
    }

    private static Slice decompressZstd(Slice input, int uncompressedSize) {
        byte[] buffer = new byte[uncompressedSize];
        ParquetCompressionUtils.decompress((Decompressor)new ZstdDecompressor(), input, 0, input.length(), buffer, 0);
        return Slices.wrappedBuffer((byte[])buffer);
    }

    private static Slice decompressGzip(Slice input, int uncompressedSize) throws IOException {
        if (uncompressedSize == 0) {
            return Slices.EMPTY_SLICE;
        }
        try (GZIPInputStream gzipInputStream = new GZIPInputStream((InputStream)input.getInput(), Math.min(8192, input.length()));){
            byte[] buffer = new byte[uncompressedSize];
            int bytesRead = ByteStreams.read((InputStream)gzipInputStream, (byte[])buffer, (int)0, (int)buffer.length);
            if (bytesRead != uncompressedSize) {
                throw new IllegalArgumentException(String.format("Invalid uncompressedSize for GZIP input. Expected %s, actual: %s", uncompressedSize, bytesRead));
            }
            Preconditions.checkArgument((gzipInputStream.read() == -1 ? 1 : 0) != 0, (String)"Invalid uncompressedSize for GZIP input. Actual size exceeds %s bytes", (int)uncompressedSize);
            Slice slice = Slices.wrappedBuffer((byte[])buffer, (int)0, (int)bytesRead);
            return slice;
        }
    }

    private static Slice decompressLz4(Slice input, int uncompressedSize) {
        return ParquetCompressionUtils.decompressFramed((Decompressor)new Lz4Decompressor(), input, uncompressedSize);
    }

    private static Slice decompressLZO(Slice input, int uncompressedSize) {
        return ParquetCompressionUtils.decompressFramed((Decompressor)new LzoDecompressor(), input, uncompressedSize);
    }

    private static Slice decompressFramed(Decompressor decompressor, Slice input, int uncompressedSize) {
        long totalDecompressedCount = 0L;
        byte[] output = new byte[uncompressedSize + 8];
        int outputOffset = 0;
        int inputOffset = 0;
        int cumulativeUncompressedBlockLength = 0;
        while (totalDecompressedCount < (long)uncompressedSize) {
            if (totalDecompressedCount == (long)cumulativeUncompressedBlockLength) {
                cumulativeUncompressedBlockLength += Integer.reverseBytes(input.getInt(inputOffset));
                inputOffset += 4;
            }
            int compressedChunkLength = Integer.reverseBytes(input.getInt(inputOffset));
            int decompressionSize = ParquetCompressionUtils.decompress(decompressor, input, inputOffset += 4, compressedChunkLength, output, outputOffset);
            totalDecompressedCount += (long)decompressionSize;
            outputOffset += decompressionSize;
            inputOffset += compressedChunkLength;
        }
        Preconditions.checkArgument((outputOffset == uncompressedSize ? 1 : 0) != 0);
        return Slices.wrappedBuffer((byte[])output, (int)0, (int)uncompressedSize);
    }

    private static int decompress(Decompressor decompressor, Slice input, int inputOffset, int inputLength, byte[] output, int outputOffset) {
        byte[] byteArray = input.byteArray();
        int byteArrayOffset = inputOffset + input.byteArrayOffset();
        return decompressor.decompress(byteArray, byteArrayOffset, inputLength, output, outputOffset, output.length - outputOffset);
    }
}

