/*
 * Decompiled with CFR 0.152.
 */
package io.trino.plugin.hive.projection;

import com.google.common.collect.ImmutableList;
import com.google.common.collect.ImmutableSet;
import io.airlift.slice.Slices;
import io.trino.plugin.hive.projection.InvalidProjectionException;
import io.trino.plugin.hive.projection.PartitionProjectionProperties;
import io.trino.plugin.hive.projection.Projection;
import io.trino.spi.TrinoException;
import io.trino.spi.predicate.Domain;
import io.trino.spi.type.DateType;
import io.trino.spi.type.TimestampType;
import io.trino.spi.type.Type;
import io.trino.spi.type.VarcharType;
import java.text.DateFormat;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.time.Instant;
import java.time.ZoneId;
import java.time.temporal.ChronoField;
import java.time.temporal.ChronoUnit;
import java.util.Date;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.TimeZone;
import java.util.concurrent.TimeUnit;
import java.util.function.Supplier;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

final class DateProjection
implements Projection {
    private static final ZoneId UTC_TIME_ZONE_ID = ZoneId.of("UTC");
    private static final Set<ChronoUnit> DATE_PROJECTION_INTERVAL_UNITS = ImmutableSet.of((Object)ChronoUnit.DAYS, (Object)ChronoUnit.HOURS, (Object)ChronoUnit.MINUTES, (Object)ChronoUnit.SECONDS);
    private static final Pattern DATE_RANGE_BOUND_EXPRESSION_PATTERN = Pattern.compile("^\\s*NOW\\s*(([+-])\\s*([0-9]+)\\s*(DAY|HOUR|MINUTE|SECOND)S?\\s*)?$");
    private final String columnName;
    private final DateFormat dateFormat;
    private final Supplier<Instant> leftBound;
    private final Supplier<Instant> rightBound;
    private final int interval;
    private final ChronoUnit intervalUnit;

    public DateProjection(String columnName, Type columnType, Map<String, Object> columnProperties) {
        TimestampType timestampType;
        if (!(columnType instanceof VarcharType || columnType instanceof DateType || columnType instanceof TimestampType && (timestampType = (TimestampType)columnType).isShort())) {
            throw new InvalidProjectionException(columnName, columnType);
        }
        this.columnName = Objects.requireNonNull(columnName, "columnName is null");
        String dateFormatPattern = PartitionProjectionProperties.getProjectionPropertyRequiredValue(columnName, columnProperties, "partition_projection_format", String::valueOf);
        List range = (List)PartitionProjectionProperties.getProjectionPropertyRequiredValue(columnName, columnProperties, "partition_projection_range", value -> (ImmutableList)((List)value).stream().map(String.class::cast).collect(ImmutableList.toImmutableList()));
        if (range.size() != 2) {
            throw DateProjection.invalidRangeProperty(columnName, dateFormatPattern, Optional.empty());
        }
        SimpleDateFormat dateFormat = new SimpleDateFormat(dateFormatPattern);
        dateFormat.setLenient(false);
        dateFormat.setTimeZone(TimeZone.getTimeZone(UTC_TIME_ZONE_ID));
        this.dateFormat = Objects.requireNonNull(dateFormat, "dateFormatPattern is null");
        this.leftBound = DateProjection.parseDateRangerBound(columnName, (String)range.get(0), dateFormat);
        this.rightBound = DateProjection.parseDateRangerBound(columnName, (String)range.get(1), dateFormat);
        if (!this.leftBound.get().isBefore(this.rightBound.get())) {
            throw DateProjection.invalidRangeProperty(columnName, dateFormatPattern, Optional.empty());
        }
        this.interval = PartitionProjectionProperties.getProjectionPropertyValue(columnProperties, "partition_projection_interval", Integer.class::cast).orElse(1);
        this.intervalUnit = PartitionProjectionProperties.getProjectionPropertyValue(columnProperties, "partition_projection_interval_unit", ChronoUnit.class::cast).orElseGet(() -> DateProjection.resolveDefaultChronoUnit(columnName, dateFormatPattern));
        if (!DATE_PROJECTION_INTERVAL_UNITS.contains(this.intervalUnit)) {
            throw new InvalidProjectionException(columnName, String.format("Property: '%s' value '%s' is invalid. Available options: %s", "partition_projection_interval_unit", this.intervalUnit, DATE_PROJECTION_INTERVAL_UNITS));
        }
    }

    @Override
    public List<String> getProjectedValues(Optional<Domain> partitionValueFilter) {
        ImmutableList.Builder builder = ImmutableList.builder();
        Instant leftBound = this.adjustBoundToDateFormat(this.leftBound.get());
        Instant rightBound = this.adjustBoundToDateFormat(this.rightBound.get());
        Instant currentValue = leftBound;
        while (!currentValue.isAfter(rightBound)) {
            String currentValueFormatted = this.formatValue(currentValue);
            if (this.isValueInDomain(partitionValueFilter, currentValue, currentValueFormatted)) {
                builder.add((Object)currentValueFormatted);
            }
            currentValue = currentValue.atZone(UTC_TIME_ZONE_ID).plus(this.interval, this.intervalUnit).toInstant();
        }
        return builder.build();
    }

    private Instant adjustBoundToDateFormat(Instant value) {
        String formatted = this.formatValue(value.with(ChronoField.MILLI_OF_SECOND, 0L));
        try {
            return this.dateFormat.parse(formatted).toInstant();
        }
        catch (ParseException e) {
            throw new InvalidProjectionException(formatted, e.getMessage());
        }
    }

    private String formatValue(Instant current) {
        return this.dateFormat.format(new Date(current.toEpochMilli()));
    }

    private boolean isValueInDomain(Optional<Domain> valueDomain, Instant value, String formattedValue) {
        if (valueDomain.isEmpty() || valueDomain.get().isAll()) {
            return true;
        }
        Domain domain = valueDomain.get();
        Type type = domain.getType();
        if (type instanceof VarcharType) {
            return domain.contains(Domain.singleValue((Type)type, (Object)Slices.utf8Slice((String)formattedValue)));
        }
        if (type instanceof DateType) {
            return domain.contains(Domain.singleValue((Type)type, (Object)TimeUnit.MILLISECONDS.toDays(value.toEpochMilli())));
        }
        if (type instanceof TimestampType && ((TimestampType)type).isShort()) {
            return domain.contains(Domain.singleValue((Type)type, (Object)TimeUnit.MILLISECONDS.toMicros(value.toEpochMilli())));
        }
        throw new InvalidProjectionException(this.columnName, type);
    }

    private static ChronoUnit resolveDefaultChronoUnit(String columnName, String dateFormatPattern) {
        String datePatternWithoutText = dateFormatPattern.replaceAll("'.*?'", "");
        if (datePatternWithoutText.contains("S") || datePatternWithoutText.contains("s") || datePatternWithoutText.contains("m") || datePatternWithoutText.contains("H")) {
            throw new InvalidProjectionException(columnName, String.format("Property: '%s' needs to be set when provided '%s' is less that single-day precision. Interval defaults to 1 day or 1 month, respectively. Otherwise, interval is required", "partition_projection_interval_unit", "partition_projection_format"));
        }
        if (datePatternWithoutText.contains("d")) {
            return ChronoUnit.DAYS;
        }
        return ChronoUnit.MONTHS;
    }

    private static Supplier<Instant> parseDateRangerBound(String columnName, String value, SimpleDateFormat dateFormat) {
        Instant dateBound;
        Matcher matcher = DATE_RANGE_BOUND_EXPRESSION_PATTERN.matcher(value);
        if (matcher.matches()) {
            String operator = matcher.group(2);
            String multiplierString = matcher.group(3);
            String unitString = matcher.group(4);
            if (Objects.nonNull(operator) && Objects.nonNull(multiplierString) && Objects.nonNull(unitString)) {
                unitString = unitString.toUpperCase(Locale.ENGLISH);
                return new DateExpressionBound(Integer.parseInt(multiplierString), ChronoUnit.valueOf(unitString + "S"), operator.charAt(0) == '+');
            }
            if (value.trim().equals("NOW")) {
                Instant now = Instant.now();
                return () -> now;
            }
            throw DateProjection.invalidRangeProperty(columnName, dateFormat.toPattern(), Optional.of("Invalid expression"));
        }
        try {
            dateBound = dateFormat.parse(value).toInstant();
        }
        catch (ParseException e) {
            throw DateProjection.invalidRangeProperty(columnName, dateFormat.toPattern(), Optional.of(e.getMessage()));
        }
        return () -> dateBound;
    }

    private static TrinoException invalidRangeProperty(String columnName, String dateFormatPattern, Optional<String> errorDetail) {
        throw new InvalidProjectionException(columnName, String.format("Property: '%s' needs to be a list of 2 valid dates formatted as '%s' or '%s' that are sequential%s", "partition_projection_range", dateFormatPattern, DATE_RANGE_BOUND_EXPRESSION_PATTERN.pattern(), errorDetail.map(error -> ": " + error).orElse("")));
    }

    private record DateExpressionBound(int multiplier, ChronoUnit unit, boolean increment) implements Supplier<Instant>
    {
        @Override
        public Instant get() {
            return Instant.now().plus(this.increment ? (long)this.multiplier : (long)(-this.multiplier), this.unit);
        }
    }
}

