/*
 * Decompiled with CFR 0.152.
 */
package io.trino.plugin.hive.metastore.file;

import com.google.common.base.Preconditions;
import com.google.common.base.Splitter;
import com.google.common.base.Throwables;
import com.google.common.base.Verify;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.ImmutableMap;
import com.google.common.collect.ImmutableSet;
import com.google.common.collect.Lists;
import com.google.common.collect.Sets;
import com.google.common.hash.Hashing;
import com.google.errorprone.annotations.ThreadSafe;
import io.airlift.json.JsonCodec;
import io.trino.filesystem.FileIterator;
import io.trino.filesystem.Location;
import io.trino.filesystem.TrinoFileSystem;
import io.trino.filesystem.TrinoFileSystemFactory;
import io.trino.filesystem.TrinoInputStream;
import io.trino.filesystem.TrinoOutputFile;
import io.trino.metastore.Database;
import io.trino.metastore.HiveBasicStatistics;
import io.trino.metastore.HiveColumnStatistics;
import io.trino.metastore.HiveMetastore;
import io.trino.metastore.HivePrincipal;
import io.trino.metastore.HivePrivilegeInfo;
import io.trino.metastore.HiveType;
import io.trino.metastore.Partition;
import io.trino.metastore.PartitionStatistics;
import io.trino.metastore.PartitionWithStatistics;
import io.trino.metastore.Partitions;
import io.trino.metastore.PrincipalPrivileges;
import io.trino.metastore.SchemaAlreadyExistsException;
import io.trino.metastore.StatisticsUpdateMode;
import io.trino.metastore.Table;
import io.trino.metastore.TableAlreadyExistsException;
import io.trino.metastore.TableInfo;
import io.trino.plugin.hive.HiveErrorCode;
import io.trino.plugin.hive.HivePartitionManager;
import io.trino.plugin.hive.NodeVersion;
import io.trino.plugin.hive.PartitionNotFoundException;
import io.trino.plugin.hive.TableType;
import io.trino.plugin.hive.ViewReaderUtil;
import io.trino.plugin.hive.metastore.MetastoreUtil;
import io.trino.plugin.hive.metastore.file.Column;
import io.trino.plugin.hive.metastore.file.ColumnStatistics;
import io.trino.plugin.hive.metastore.file.DatabaseMetadata;
import io.trino.plugin.hive.metastore.file.FileHiveMetastoreConfig;
import io.trino.plugin.hive.metastore.file.PartitionMetadata;
import io.trino.plugin.hive.metastore.file.PermissionMetadata;
import io.trino.plugin.hive.metastore.file.TableMetadata;
import io.trino.plugin.hive.util.HiveUtil;
import io.trino.spi.ErrorCodeSupplier;
import io.trino.spi.StandardErrorCode;
import io.trino.spi.TrinoException;
import io.trino.spi.connector.ColumnNotFoundException;
import io.trino.spi.connector.SchemaNotFoundException;
import io.trino.spi.connector.SchemaTableName;
import io.trino.spi.connector.TableNotFoundException;
import io.trino.spi.function.LanguageFunction;
import io.trino.spi.predicate.TupleDomain;
import io.trino.spi.security.ConnectorIdentity;
import io.trino.spi.security.PrincipalType;
import io.trino.spi.security.RoleGrant;
import io.trino.spi.security.TrinoPrincipal;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.util.ArrayList;
import java.util.Collection;
import java.util.EnumSet;
import java.util.HashMap;
import java.util.HashSet;
import java.util.LinkedHashMap;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.OptionalLong;
import java.util.Set;
import java.util.function.Function;
import java.util.function.Predicate;
import java.util.stream.Collectors;

@ThreadSafe
public class FileHiveMetastore
implements HiveMetastore {
    private static final String PUBLIC_ROLE_NAME = "public";
    private static final String ADMIN_ROLE_NAME = "admin";
    private static final String TRINO_SCHEMA_FILE_NAME_SUFFIX = ".trinoSchema";
    private static final String TRINO_PERMISSIONS_DIRECTORY_NAME = ".trinoPermissions";
    private static final String TRINO_FUNCTIONS_DIRECTORY_NAME = ".trinoFunction";
    private static final String ROLES_FILE_NAME = ".roles";
    private static final String ROLE_GRANTS_FILE_NAME = ".roleGrants";
    private static final Set<String> ADMIN_USERS = ImmutableSet.of((Object)"admin", (Object)"hive", (Object)"hdfs");
    private static final int MAX_NAME_LENGTH = 128;
    private final String currentVersion;
    private final FileHiveMetastoreConfig.VersionCompatibility versionCompatibility;
    private final TrinoFileSystem fileSystem;
    private final Location catalogDirectory;
    private final boolean disableLocationChecks;
    private final boolean hideDeltaLakeTables;
    private final JsonCodec<DatabaseMetadata> databaseCodec = JsonCodec.jsonCodec(DatabaseMetadata.class);
    private final JsonCodec<TableMetadata> tableCodec = JsonCodec.jsonCodec(TableMetadata.class);
    private final JsonCodec<PartitionMetadata> partitionCodec = JsonCodec.jsonCodec(PartitionMetadata.class);
    private final JsonCodec<List<PermissionMetadata>> permissionsCodec = JsonCodec.listJsonCodec(PermissionMetadata.class);
    private final JsonCodec<LanguageFunction> functionCodec = JsonCodec.jsonCodec(LanguageFunction.class);
    private final JsonCodec<List<String>> rolesCodec = JsonCodec.listJsonCodec(String.class);
    private final JsonCodec<List<RoleGrant>> roleGrantsCodec = JsonCodec.listJsonCodec(RoleGrant.class);

    public FileHiveMetastore(NodeVersion nodeVersion, TrinoFileSystemFactory fileSystemFactory, boolean hideDeltaLakeTables, FileHiveMetastoreConfig config) {
        this.currentVersion = nodeVersion.toString();
        this.versionCompatibility = Objects.requireNonNull(config.getVersionCompatibility(), "config.getVersionCompatibility() is null");
        this.fileSystem = fileSystemFactory.create(ConnectorIdentity.ofUser((String)config.getMetastoreUser()));
        this.catalogDirectory = Location.of((String)Objects.requireNonNull(config.getCatalogDirectory(), "catalogDirectory is null"));
        this.disableLocationChecks = config.isDisableLocationChecks();
        this.hideDeltaLakeTables = hideDeltaLakeTables;
    }

    public synchronized void createDatabase(Database database) {
        Objects.requireNonNull(database, "database is null");
        database = new Database(database.getDatabaseName().toLowerCase(Locale.ENGLISH), database.getLocation(), database.getOwnerName(), database.getOwnerType(), database.getComment(), database.getParameters());
        FileHiveMetastore.verifyDatabaseNameLength(database.getDatabaseName());
        Optional<Database> existingDatabase = this.getDatabase(database.getDatabaseName());
        if (existingDatabase.isPresent()) {
            String expectedQueryId = (String)database.getParameters().get("trino_query_id");
            if (expectedQueryId != null && expectedQueryId.equals(existingDatabase.get().getParameters().get("trino_query_id"))) {
                return;
            }
            throw new SchemaAlreadyExistsException(database.getDatabaseName());
        }
        Location databaseMetadataDirectory = this.getDatabaseMetadataDirectory(database.getDatabaseName());
        this.writeSchemaFile(SchemaType.DATABASE, databaseMetadataDirectory, this.databaseCodec, new DatabaseMetadata(this.currentVersion, database), false);
        try {
            this.fileSystem.createDirectory(databaseMetadataDirectory);
        }
        catch (IOException e) {
            throw new TrinoException((ErrorCodeSupplier)HiveErrorCode.HIVE_METASTORE_ERROR, "Could not write database", (Throwable)e);
        }
    }

    public synchronized void dropDatabase(String databaseName, boolean deleteData) {
        Objects.requireNonNull(databaseName, "databaseName is null");
        databaseName = databaseName.toLowerCase(Locale.ENGLISH);
        this.getRequiredDatabase(databaseName);
        if (!this.getTables(databaseName).isEmpty()) {
            throw new TrinoException((ErrorCodeSupplier)HiveErrorCode.HIVE_METASTORE_ERROR, "Database " + databaseName + " is not empty");
        }
        if (deleteData) {
            this.deleteDirectoryAndSchema(SchemaType.DATABASE, this.getDatabaseMetadataDirectory(databaseName));
        } else {
            this.deleteSchemaFile(SchemaType.DATABASE, this.getDatabaseMetadataDirectory(databaseName));
        }
    }

    public synchronized void renameDatabase(String databaseName, String newDatabaseName) {
        Objects.requireNonNull(databaseName, "databaseName is null");
        Objects.requireNonNull(newDatabaseName, "newDatabaseName is null");
        FileHiveMetastore.verifyDatabaseNameLength(newDatabaseName);
        this.getRequiredDatabase(databaseName);
        this.verifyDatabaseNotExists(newDatabaseName);
        Location oldDatabaseMetadataDirectory = this.getDatabaseMetadataDirectory(databaseName);
        Location newDatabaseMetadataDirectory = this.getDatabaseMetadataDirectory(newDatabaseName);
        try {
            this.renameSchemaFile(SchemaType.DATABASE, oldDatabaseMetadataDirectory, newDatabaseMetadataDirectory);
            this.fileSystem.renameDirectory(oldDatabaseMetadataDirectory, newDatabaseMetadataDirectory);
        }
        catch (IOException e) {
            throw new TrinoException((ErrorCodeSupplier)HiveErrorCode.HIVE_METASTORE_ERROR, (Throwable)e);
        }
    }

    public synchronized void setDatabaseOwner(String databaseName, HivePrincipal principal) {
        Database database = this.getRequiredDatabase(databaseName);
        Location databaseMetadataDirectory = this.getDatabaseMetadataDirectory(database.getDatabaseName());
        Database newDatabase = Database.builder((Database)database).setOwnerName(Optional.of(principal.getName())).setOwnerType(Optional.of(principal.getType())).build();
        this.writeSchemaFile(SchemaType.DATABASE, databaseMetadataDirectory, this.databaseCodec, new DatabaseMetadata(this.currentVersion, newDatabase), true);
    }

    public synchronized Optional<Database> getDatabase(String databaseName) {
        Objects.requireNonNull(databaseName, "databaseName is null");
        String normalizedName = databaseName.toLowerCase(Locale.ENGLISH);
        Location databaseMetadataDirectory = this.getDatabaseMetadataDirectory(normalizedName);
        return this.readSchemaFile(SchemaType.DATABASE, databaseMetadataDirectory, this.databaseCodec).map(databaseMetadata -> {
            this.checkVersion(databaseMetadata.getWriterVersion());
            return databaseMetadata.toDatabase(normalizedName, databaseMetadataDirectory.toString());
        });
    }

    private Database getRequiredDatabase(String databaseName) {
        return this.getDatabase(databaseName).orElseThrow(() -> new SchemaNotFoundException(databaseName));
    }

    private static void verifyDatabaseNameLength(String databaseName) {
        if (databaseName.length() > 128) {
            throw new TrinoException((ErrorCodeSupplier)StandardErrorCode.NOT_SUPPORTED, String.format("Schema name must be shorter than or equal to '%s' characters but got '%s'", 128, databaseName.length()));
        }
    }

    private static void verifyTableNameLength(String tableName) {
        if (tableName.length() > 128) {
            throw new TrinoException((ErrorCodeSupplier)StandardErrorCode.NOT_SUPPORTED, String.format("Table name must be shorter than or equal to '%s' characters but got '%s'", 128, tableName.length()));
        }
    }

    private void verifyDatabaseNotExists(String databaseName) {
        if (this.getDatabase(databaseName).isPresent()) {
            throw new SchemaAlreadyExistsException(databaseName);
        }
    }

    public synchronized List<String> getAllDatabases() {
        try {
            String prefix = this.catalogDirectory.toString();
            HashSet<String> databases = new HashSet<String>();
            FileIterator iterator = this.fileSystem.listFiles(this.catalogDirectory);
            while (iterator.hasNext()) {
                int length;
                Location location = iterator.next().location();
                String child = location.toString().substring(prefix.length());
                if (child.startsWith("/")) {
                    child = child.substring(1);
                }
                if ((length = child.length() - TRINO_SCHEMA_FILE_NAME_SUFFIX.length()) <= 1 || child.contains("/") || !child.startsWith(".") || !child.endsWith(TRINO_SCHEMA_FILE_NAME_SUFFIX)) continue;
                databases.add(child.substring(1, length));
            }
            return ImmutableList.copyOf(databases);
        }
        catch (IOException e) {
            throw new TrinoException((ErrorCodeSupplier)HiveErrorCode.HIVE_METASTORE_ERROR, (Throwable)e);
        }
    }

    public synchronized void createTable(Table table, PrincipalPrivileges principalPrivileges) {
        FileHiveMetastore.verifyTableNameLength(table.getTableName());
        this.verifyDatabaseExists(table.getDatabaseName());
        Optional<Table> existingTable = this.getTable(table.getDatabaseName(), table.getTableName());
        if (existingTable.isPresent()) {
            String expectedQueryId = (String)table.getParameters().get("trino_query_id");
            if (expectedQueryId != null && expectedQueryId.equals(existingTable.get().getParameters().get("trino_query_id"))) {
                return;
            }
            throw new TableAlreadyExistsException(new SchemaTableName(table.getDatabaseName(), table.getTableName()));
        }
        Location tableMetadataDirectory = this.getTableMetadataDirectory(table);
        if (ViewReaderUtil.isSomeKindOfAView(table)) {
            Preconditions.checkArgument((boolean)table.getStorage().getLocation().isEmpty(), (Object)"Storage location for view must be empty");
        } else if (table.getTableType().equals(TableType.MANAGED_TABLE.name())) {
            if (!this.disableLocationChecks && !table.getStorage().getLocation().contains(tableMetadataDirectory.toString())) {
                throw new TrinoException((ErrorCodeSupplier)HiveErrorCode.HIVE_METASTORE_ERROR, "Table directory must be " + String.valueOf(tableMetadataDirectory));
            }
        } else if (table.getTableType().equals(TableType.EXTERNAL_TABLE.name())) {
            if (!this.disableLocationChecks) {
                try {
                    Location externalLocation = Location.of((String)table.getStorage().getLocation());
                    if (!this.fileSystem.directoryExists(externalLocation).orElse(true).booleanValue()) {
                        throw new TrinoException((ErrorCodeSupplier)HiveErrorCode.HIVE_METASTORE_ERROR, "External table location does not exist");
                    }
                }
                catch (IOException e) {
                    throw new TrinoException((ErrorCodeSupplier)HiveErrorCode.HIVE_METASTORE_ERROR, "Could not validate external location", (Throwable)e);
                }
            }
        } else if (!table.getTableType().equals(TableType.MATERIALIZED_VIEW.name())) {
            throw new TrinoException((ErrorCodeSupplier)StandardErrorCode.NOT_SUPPORTED, "Table type not supported: " + table.getTableType());
        }
        this.writeSchemaFile(SchemaType.TABLE, tableMetadataDirectory, this.tableCodec, new TableMetadata(this.currentVersion, table), false);
        for (Map.Entry entry : principalPrivileges.getUserPrivileges().asMap().entrySet()) {
            this.setTablePrivileges(new HivePrincipal(PrincipalType.USER, (String)entry.getKey()), table.getDatabaseName(), table.getTableName(), (Collection)entry.getValue());
        }
        for (Map.Entry entry : principalPrivileges.getRolePrivileges().asMap().entrySet()) {
            this.setTablePrivileges(new HivePrincipal(PrincipalType.ROLE, (String)entry.getKey()), table.getDatabaseName(), table.getTableName(), (Collection)entry.getValue());
        }
    }

    public synchronized Optional<Table> getTable(String databaseName, String tableName) {
        Objects.requireNonNull(databaseName, "databaseName is null");
        Objects.requireNonNull(tableName, "tableName is null");
        Location tableMetadataDirectory = this.getTableMetadataDirectory(databaseName, tableName);
        return this.readSchemaFile(SchemaType.TABLE, tableMetadataDirectory, this.tableCodec).map(tableMetadata -> {
            this.checkVersion(tableMetadata.getWriterVersion());
            return tableMetadata.toTable(databaseName, tableName, tableMetadataDirectory.toString());
        });
    }

    public synchronized void setTableOwner(String databaseName, String tableName, HivePrincipal principal) {
        if (principal.getType() != PrincipalType.USER) {
            throw new TrinoException((ErrorCodeSupplier)StandardErrorCode.NOT_SUPPORTED, "Setting table owner type as a role is not supported");
        }
        Table table = this.getRequiredTable(databaseName, tableName);
        Location tableMetadataDirectory = this.getTableMetadataDirectory(table);
        Table newTable = Table.builder((Table)table).setOwner(Optional.of(principal.getName())).build();
        this.writeSchemaFile(SchemaType.TABLE, tableMetadataDirectory, this.tableCodec, new TableMetadata(this.currentVersion, newTable), true);
    }

    public synchronized Map<String, HiveColumnStatistics> getTableColumnStatistics(String databaseName, String tableName, Set<String> columnNames) {
        Preconditions.checkArgument((!columnNames.isEmpty() ? 1 : 0) != 0, (Object)"columnNames is empty");
        Location tableMetadataDirectory = this.getTableMetadataDirectory(databaseName, tableName);
        TableMetadata tableMetadata = this.readSchemaFile(SchemaType.TABLE, tableMetadataDirectory, this.tableCodec).orElseThrow(() -> new TableNotFoundException(new SchemaTableName(databaseName, tableName)));
        this.checkVersion(tableMetadata.getWriterVersion());
        return FileHiveMetastore.toHiveColumnStats(columnNames, tableMetadata.getParameters(), tableMetadata.getColumnStatistics());
    }

    public synchronized Map<String, Map<String, HiveColumnStatistics>> getPartitionColumnStatistics(String databaseName, String tableName, Set<String> partitionNames, Set<String> columnNames) {
        Preconditions.checkArgument((!columnNames.isEmpty() ? 1 : 0) != 0, (Object)"columnNames is empty");
        ImmutableMap.Builder result = ImmutableMap.builder();
        for (String partitionName : partitionNames) {
            result.put((Object)partitionName, this.getPartitionStatisticsInternal(databaseName, tableName, partitionName, columnNames));
        }
        return result.buildOrThrow();
    }

    private synchronized Map<String, HiveColumnStatistics> getPartitionStatisticsInternal(String databaseName, String tableName, String partitionName, Set<String> columnNames) {
        Location partitionDirectory = this.getPartitionMetadataDirectory(databaseName, tableName, partitionName);
        PartitionMetadata partitionMetadata = this.readSchemaFile(SchemaType.PARTITION, partitionDirectory, this.partitionCodec).orElseThrow(() -> new PartitionNotFoundException(new SchemaTableName(databaseName, tableName), HivePartitionManager.extractPartitionValues(partitionName)));
        return FileHiveMetastore.toHiveColumnStats(columnNames, partitionMetadata.getParameters(), partitionMetadata.getColumnStatistics());
    }

    private Table getRequiredTable(String databaseName, String tableName) {
        return this.getTable(databaseName, tableName).orElseThrow(() -> new TableNotFoundException(new SchemaTableName(databaseName, tableName)));
    }

    private void verifyDatabaseExists(String databaseName) {
        if (this.getDatabase(databaseName).isEmpty()) {
            throw new SchemaNotFoundException(databaseName);
        }
    }

    private void verifyTableNotExists(String newDatabaseName, String newTableName) {
        if (this.getTable(newDatabaseName, newTableName).isPresent()) {
            throw new TableAlreadyExistsException(new SchemaTableName(newDatabaseName, newTableName));
        }
    }

    public synchronized void updateTableStatistics(String databaseName, String tableName, OptionalLong acidWriteId, StatisticsUpdateMode mode, PartitionStatistics statisticsUpdate) {
        Location tableMetadataDirectory = this.getTableMetadataDirectory(databaseName, tableName);
        TableMetadata tableMetadata = this.readSchemaFile(SchemaType.TABLE, tableMetadataDirectory, this.tableCodec).orElseThrow(() -> new TableNotFoundException(new SchemaTableName(databaseName, tableName)));
        this.checkVersion(tableMetadata.getWriterVersion());
        PartitionStatistics originalStatistics = FileHiveMetastore.toHivePartitionStatistics(tableMetadata.getParameters(), tableMetadata.getColumnStatistics());
        PartitionStatistics updatedStatistics = mode.updatePartitionStatistics(originalStatistics, statisticsUpdate);
        TableMetadata updatedMetadata = tableMetadata.withParameters(this.currentVersion, MetastoreUtil.updateStatisticsParameters(tableMetadata.getParameters(), updatedStatistics.basicStatistics())).withColumnStatistics(this.currentVersion, FileHiveMetastore.fromHiveColumnStats(updatedStatistics.columnStatistics()));
        this.writeSchemaFile(SchemaType.TABLE, tableMetadataDirectory, this.tableCodec, updatedMetadata, true);
    }

    public synchronized void updatePartitionStatistics(Table table, StatisticsUpdateMode mode, Map<String, PartitionStatistics> partitionUpdates) {
        partitionUpdates.forEach((partitionName, partitionUpdate) -> {
            Location partitionDirectory = this.getPartitionMetadataDirectory(table, (String)partitionName);
            PartitionMetadata partitionMetadata = this.readSchemaFile(SchemaType.PARTITION, partitionDirectory, this.partitionCodec).orElseThrow(() -> new PartitionNotFoundException(table.getSchemaTableName(), HivePartitionManager.extractPartitionValues(partitionName)));
            PartitionStatistics originalStatistics = FileHiveMetastore.toHivePartitionStatistics(partitionMetadata.getParameters(), partitionMetadata.getColumnStatistics());
            PartitionStatistics updatedStatistics = mode.updatePartitionStatistics(originalStatistics, partitionUpdate);
            PartitionMetadata updatedMetadata = partitionMetadata.withParameters(MetastoreUtil.updateStatisticsParameters(partitionMetadata.getParameters(), updatedStatistics.basicStatistics())).withColumnStatistics(FileHiveMetastore.fromHiveColumnStats(updatedStatistics.columnStatistics()));
            this.writeSchemaFile(SchemaType.PARTITION, partitionDirectory, this.partitionCodec, updatedMetadata, true);
        });
    }

    public synchronized List<TableInfo> getTables(String databaseName) {
        return this.doListAllTables(databaseName, tableMetadata -> true);
    }

    public synchronized List<String> getTableNamesWithParameters(String databaseName, String parameterKey, ImmutableSet<String> parameterValues) {
        Objects.requireNonNull(parameterKey, "parameterKey is null");
        return (List)this.doListAllTables(databaseName, table -> parameterValues.contains((Object)table.getParameters().get(parameterKey))).stream().map(tableInfo -> tableInfo.tableName().getTableName()).collect(ImmutableList.toImmutableList());
    }

    private synchronized List<TableInfo> doListAllTables(String databaseName, Predicate<TableMetadata> tableMetadataPredicate) {
        Objects.requireNonNull(databaseName, "databaseName is null");
        Optional<Database> database = this.getDatabase(databaseName);
        if (database.isEmpty()) {
            return ImmutableList.of();
        }
        Location metadataDirectory = this.getDatabaseMetadataDirectory(databaseName);
        try {
            Object prefix = metadataDirectory.toString();
            if (!((String)prefix).endsWith("/")) {
                prefix = (String)prefix + "/";
            }
            HashSet tables = new HashSet();
            for (Location subdirectory : this.fileSystem.listDirectories(metadataDirectory)) {
                String locationString = subdirectory.toString();
                Verify.verify((locationString.startsWith((String)prefix) && locationString.endsWith("/") ? 1 : 0) != 0, (String)"Unexpected subdirectory %s when listing %s", (Object)subdirectory, (Object)metadataDirectory);
                String tableName = locationString.substring(((String)prefix).length(), locationString.length() - 1);
                Location schemaFileLocation = subdirectory.appendPath(TRINO_SCHEMA_FILE_NAME_SUFFIX);
                this.readFile("table schema", schemaFileLocation, this.tableCodec).ifPresent(tableMetadata -> {
                    this.checkVersion(tableMetadata.getWriterVersion());
                    if (this.hideDeltaLakeTables && "delta".equals(tableMetadata.getParameters().get("spark.sql.sources.provider")) || !tableMetadataPredicate.test((TableMetadata)tableMetadata)) {
                        return;
                    }
                    tables.add(new TableInfo(new SchemaTableName(databaseName, tableName), TableInfo.ExtendedRelationType.fromTableTypeAndComment((String)tableMetadata.getTableType(), (String)tableMetadata.getParameters().get("comment"))));
                });
            }
            return ImmutableList.copyOf(tables);
        }
        catch (IOException e) {
            throw new TrinoException((ErrorCodeSupplier)HiveErrorCode.HIVE_METASTORE_ERROR, (Throwable)e);
        }
    }

    public synchronized void dropTable(String databaseName, String tableName, boolean deleteData) {
        Objects.requireNonNull(databaseName, "databaseName is null");
        Objects.requireNonNull(tableName, "tableName is null");
        Table table = this.getRequiredTable(databaseName, tableName);
        Location tableMetadataDirectory = this.getTableMetadataDirectory(databaseName, tableName);
        if (deleteData) {
            this.deleteDirectoryAndSchema(SchemaType.TABLE, tableMetadataDirectory);
        } else {
            this.deleteSchemaFile(SchemaType.TABLE, tableMetadataDirectory);
            this.deleteTablePrivileges(table);
        }
    }

    public synchronized void replaceTable(String databaseName, String tableName, Table newTable, PrincipalPrivileges principalPrivileges) {
        Table table = this.getRequiredTable(databaseName, tableName);
        if (!table.getDatabaseName().equals(databaseName) || !table.getTableName().equals(tableName)) {
            throw new TrinoException((ErrorCodeSupplier)HiveErrorCode.HIVE_METASTORE_ERROR, "Replacement table must have same name");
        }
        if (HiveUtil.isIcebergTable(table) && !Objects.equals(table.getParameters().get("metadata_location"), newTable.getParameters().get("previous_metadata_location"))) {
            throw new TrinoException((ErrorCodeSupplier)HiveErrorCode.HIVE_CONCURRENT_MODIFICATION_DETECTED, "Cannot update Iceberg table: supplied previous location does not match current location");
        }
        Location tableMetadataDirectory = this.getTableMetadataDirectory(table);
        this.writeSchemaFile(SchemaType.TABLE, tableMetadataDirectory, this.tableCodec, new TableMetadata(this.currentVersion, newTable), true);
        this.deleteTablePrivileges(table);
        for (Map.Entry entry : principalPrivileges.getUserPrivileges().asMap().entrySet()) {
            this.setTablePrivileges(new HivePrincipal(PrincipalType.USER, (String)entry.getKey()), table.getDatabaseName(), table.getTableName(), (Collection)entry.getValue());
        }
        for (Map.Entry entry : principalPrivileges.getRolePrivileges().asMap().entrySet()) {
            this.setTablePrivileges(new HivePrincipal(PrincipalType.ROLE, (String)entry.getKey()), table.getDatabaseName(), table.getTableName(), (Collection)entry.getValue());
        }
    }

    public synchronized void renameTable(String databaseName, String tableName, String newDatabaseName, String newTableName) {
        Objects.requireNonNull(databaseName, "databaseName is null");
        Objects.requireNonNull(tableName, "tableName is null");
        Objects.requireNonNull(newDatabaseName, "newDatabaseName is null");
        Objects.requireNonNull(newTableName, "newTableName is null");
        Table table = this.getRequiredTable(databaseName, tableName);
        this.getRequiredDatabase(newDatabaseName);
        FileHiveMetastore.verifyTableNameLength(newTableName);
        this.verifyTableNotExists(newDatabaseName, newTableName);
        Location oldPath = this.getTableMetadataDirectory(databaseName, tableName);
        Location newPath = this.getTableMetadataDirectory(newDatabaseName, newTableName);
        try {
            if (HiveUtil.isIcebergTable(table)) {
                this.fileSystem.createDirectory(newPath);
                this.fileSystem.renameFile(FileHiveMetastore.getSchemaFile(SchemaType.TABLE, oldPath), FileHiveMetastore.getSchemaFile(SchemaType.TABLE, newPath));
            } else {
                this.fileSystem.renameDirectory(oldPath, newPath);
            }
        }
        catch (IOException e) {
            throw new TrinoException((ErrorCodeSupplier)HiveErrorCode.HIVE_METASTORE_ERROR, (Throwable)e);
        }
    }

    public synchronized void commentTable(String databaseName, String tableName, Optional<String> comment) {
        this.alterTable(databaseName, tableName, oldTable -> {
            Map<String, String> parameters = oldTable.getParameters().entrySet().stream().filter(entry -> !((String)entry.getKey()).equals("comment")).collect(Collectors.toMap(Map.Entry::getKey, Map.Entry::getValue));
            comment.ifPresent(value -> parameters.put("comment", (String)value));
            return oldTable.withParameters(this.currentVersion, parameters);
        });
    }

    public synchronized void commentColumn(String databaseName, String tableName, String columnName, Optional<String> comment) {
        this.alterTable(databaseName, tableName, table -> table.withDataColumns(this.currentVersion, FileHiveMetastore.updateColumnComment(table.getDataColumns(), columnName, comment)).withPartitionColumns(this.currentVersion, FileHiveMetastore.updateColumnComment(table.getPartitionColumns(), columnName, comment)));
    }

    private static List<Column> updateColumnComment(List<Column> originalColumns, String columnName, Optional<String> comment) {
        return Lists.transform(originalColumns, column -> column.getName().equals(columnName) ? new Column(column.getName(), column.getType(), comment, column.getProperties()) : column);
    }

    public synchronized void addColumn(String databaseName, String tableName, String columnName, HiveType columnType, String columnComment) {
        this.alterTable(databaseName, tableName, oldTable -> {
            if (oldTable.getColumn(columnName).isPresent()) {
                throw new TrinoException((ErrorCodeSupplier)StandardErrorCode.ALREADY_EXISTS, "Column already exists: " + columnName);
            }
            return oldTable.withDataColumns(this.currentVersion, (List<Column>)ImmutableList.builder().addAll(oldTable.getDataColumns()).add((Object)new Column(columnName, columnType, Optional.ofNullable(columnComment), (Map<String, String>)ImmutableMap.of())).build());
        });
    }

    public synchronized void renameColumn(String databaseName, String tableName, String oldColumnName, String newColumnName) {
        this.alterTable(databaseName, tableName, oldTable -> {
            if (oldTable.getColumn(newColumnName).isPresent()) {
                throw new TrinoException((ErrorCodeSupplier)StandardErrorCode.ALREADY_EXISTS, "Column already exists: " + newColumnName);
            }
            if (oldTable.getColumn(oldColumnName).isEmpty()) {
                SchemaTableName name = new SchemaTableName(databaseName, tableName);
                throw new ColumnNotFoundException(name, oldColumnName);
            }
            for (Column column : oldTable.getPartitionColumns()) {
                if (!column.getName().equals(oldColumnName)) continue;
                throw new TrinoException((ErrorCodeSupplier)StandardErrorCode.NOT_SUPPORTED, "Renaming partition columns is not supported");
            }
            ImmutableList.Builder newDataColumns = ImmutableList.builder();
            for (Column fieldSchema : oldTable.getDataColumns()) {
                if (fieldSchema.getName().equals(oldColumnName)) {
                    newDataColumns.add((Object)new Column(newColumnName, fieldSchema.getType(), fieldSchema.getComment(), fieldSchema.getProperties()));
                    continue;
                }
                newDataColumns.add((Object)fieldSchema);
            }
            return oldTable.withDataColumns(this.currentVersion, (List<Column>)newDataColumns.build());
        });
    }

    public synchronized void dropColumn(String databaseName, String tableName, String columnName) {
        this.alterTable(databaseName, tableName, oldTable -> {
            MetastoreUtil.verifyCanDropColumn(this, databaseName, tableName, columnName);
            if (oldTable.getColumn(columnName).isEmpty()) {
                SchemaTableName name = new SchemaTableName(databaseName, tableName);
                throw new ColumnNotFoundException(name, columnName);
            }
            ImmutableList.Builder newDataColumns = ImmutableList.builder();
            for (Column fieldSchema : oldTable.getDataColumns()) {
                if (fieldSchema.getName().equals(columnName)) continue;
                newDataColumns.add((Object)fieldSchema);
            }
            return oldTable.withDataColumns(this.currentVersion, (List<Column>)newDataColumns.build());
        });
    }

    private void alterTable(String databaseName, String tableName, Function<TableMetadata, TableMetadata> alterFunction) {
        Objects.requireNonNull(databaseName, "databaseName is null");
        Objects.requireNonNull(tableName, "tableName is null");
        Location tableMetadataDirectory = this.getTableMetadataDirectory(databaseName, tableName);
        TableMetadata oldTableSchema = this.readSchemaFile(SchemaType.TABLE, tableMetadataDirectory, this.tableCodec).orElseThrow(() -> new TableNotFoundException(new SchemaTableName(databaseName, tableName)));
        this.checkVersion(oldTableSchema.getWriterVersion());
        TableMetadata newTableSchema = alterFunction.apply(oldTableSchema);
        if (oldTableSchema == newTableSchema) {
            return;
        }
        this.writeSchemaFile(SchemaType.TABLE, tableMetadataDirectory, this.tableCodec, newTableSchema, true);
    }

    /*
     * Enabled force condition propagation
     * Lifted jumps to return sites
     */
    public synchronized void addPartitions(String databaseName, String tableName, List<PartitionWithStatistics> partitions) {
        Objects.requireNonNull(databaseName, "databaseName is null");
        Objects.requireNonNull(tableName, "tableName is null");
        Objects.requireNonNull(partitions, "partitions is null");
        Table table = this.getRequiredTable(databaseName, tableName);
        TableType tableType = TableType.valueOf(table.getTableType());
        Preconditions.checkArgument((boolean)EnumSet.of(TableType.MANAGED_TABLE, TableType.EXTERNAL_TABLE).contains((Object)tableType), (String)"Invalid table type: %s", (Object)((Object)tableType));
        try {
            LinkedHashMap<Location, byte[]> schemaFiles = new LinkedHashMap<Location, byte[]>();
            for (PartitionWithStatistics partitionWithStatistics : partitions) {
                Partition partition = partitionWithStatistics.getPartition();
                this.verifiedPartition(table, partition);
                Location partitionMetadataDirectory = this.getPartitionMetadataDirectory(table, partition.getValues());
                Location schemaPath = FileHiveMetastore.getSchemaFile(SchemaType.PARTITION, partitionMetadataDirectory);
                if (this.fileSystem.directoryExists(schemaPath).orElse(false).booleanValue()) {
                    throw new TrinoException((ErrorCodeSupplier)HiveErrorCode.HIVE_METASTORE_ERROR, "Partition already exists");
                }
                byte[] schemaJson = this.partitionCodec.toJsonBytes((Object)new PartitionMetadata(table, partitionWithStatistics));
                schemaFiles.put(schemaPath, schemaJson);
            }
            LinkedHashSet<Location> createdFiles = new LinkedHashSet<Location>();
            try {
                for (Map.Entry entry : schemaFiles.entrySet()) {
                    try {
                        OutputStream outputStream = this.fileSystem.newOutputFile((Location)entry.getKey()).create();
                        try {
                            createdFiles.add((Location)entry.getKey());
                            outputStream.write((byte[])entry.getValue());
                        }
                        finally {
                            if (outputStream == null) continue;
                            outputStream.close();
                        }
                    }
                    catch (IOException e) {
                        throw new TrinoException((ErrorCodeSupplier)HiveErrorCode.HIVE_METASTORE_ERROR, "Could not write partition schema", (Throwable)e);
                        return;
                    }
                }
            }
            catch (Throwable throwable) {
                try {
                    this.fileSystem.deleteFiles(createdFiles);
                    throw throwable;
                }
                catch (IOException iOException) {
                    if (throwable.equals(iOException)) throw throwable;
                    throwable.addSuppressed(iOException);
                }
                throw throwable;
            }
        }
        catch (IOException e) {
            throw new TrinoException((ErrorCodeSupplier)HiveErrorCode.HIVE_METASTORE_ERROR, (Throwable)e);
        }
    }

    private void verifiedPartition(Table table, Partition partition) {
        block8: {
            Location partitionMetadataDirectory = this.getPartitionMetadataDirectory(table, partition.getValues());
            if (table.getTableType().equals(TableType.MANAGED_TABLE.name())) {
                if (!partitionMetadataDirectory.equals((Object)Location.of((String)partition.getStorage().getLocation()))) {
                    throw new TrinoException((ErrorCodeSupplier)HiveErrorCode.HIVE_METASTORE_ERROR, "Partition directory must be " + String.valueOf(partitionMetadataDirectory));
                }
            } else {
                if (table.getTableType().equals(TableType.EXTERNAL_TABLE.name())) {
                    try {
                        Location externalLocation = Location.of((String)partition.getStorage().getLocation());
                        if (!this.fileSystem.directoryExists(externalLocation).orElse(true).booleanValue()) {
                            throw new TrinoException((ErrorCodeSupplier)HiveErrorCode.HIVE_METASTORE_ERROR, "External partition location does not exist");
                        }
                        if (externalLocation.toString().startsWith(this.catalogDirectory.toString())) {
                            throw new TrinoException((ErrorCodeSupplier)HiveErrorCode.HIVE_METASTORE_ERROR, "External partition location cannot be inside the system metadata directory");
                        }
                        break block8;
                    }
                    catch (IOException e) {
                        throw new TrinoException((ErrorCodeSupplier)HiveErrorCode.HIVE_METASTORE_ERROR, "Could not validate external partition location", (Throwable)e);
                    }
                }
                throw new TrinoException((ErrorCodeSupplier)StandardErrorCode.NOT_SUPPORTED, "Partitions cannot be added to " + table.getTableType());
            }
        }
    }

    public synchronized void dropPartition(String databaseName, String tableName, List<String> partitionValues, boolean deleteData) {
        Objects.requireNonNull(databaseName, "databaseName is null");
        Objects.requireNonNull(tableName, "tableName is null");
        Objects.requireNonNull(partitionValues, "partitionValues is null");
        Optional<Table> tableReference = this.getTable(databaseName, tableName);
        if (tableReference.isEmpty()) {
            return;
        }
        Table table = tableReference.get();
        Location partitionMetadataDirectory = this.getPartitionMetadataDirectory(table, partitionValues);
        if (deleteData) {
            this.deleteDirectoryAndSchema(SchemaType.PARTITION, partitionMetadataDirectory);
        } else {
            this.deleteSchemaFile(SchemaType.PARTITION, partitionMetadataDirectory);
        }
    }

    public synchronized void alterPartition(String databaseName, String tableName, PartitionWithStatistics partitionWithStatistics) {
        Table table = this.getRequiredTable(databaseName, tableName);
        Partition partition = partitionWithStatistics.getPartition();
        this.verifiedPartition(table, partition);
        Location partitionMetadataDirectory = this.getPartitionMetadataDirectory(table, partition.getValues());
        this.writeSchemaFile(SchemaType.PARTITION, partitionMetadataDirectory, this.partitionCodec, new PartitionMetadata(table, partitionWithStatistics), true);
    }

    public synchronized void createRole(String role, String grantor) {
        HashSet<String> roles = new HashSet<String>(this.listRoles());
        roles.add(role);
        this.writeFile("roles", this.getRolesFile(), this.rolesCodec, ImmutableList.copyOf(roles), true);
    }

    public synchronized void dropRole(String role) {
        HashSet<String> roles = new HashSet<String>(this.listRoles());
        roles.remove(role);
        this.writeFile("roles", this.getRolesFile(), this.rolesCodec, ImmutableList.copyOf(roles), true);
        Set<RoleGrant> grants = this.listRoleGrantsSanitized();
        this.writeRoleGrantsFile(grants);
    }

    public synchronized Set<String> listRoles() {
        HashSet<String> roles = new HashSet<String>();
        roles.add(ADMIN_ROLE_NAME);
        this.readFile("roles", this.getRolesFile(), this.rolesCodec).ifPresent(roles::addAll);
        return ImmutableSet.copyOf(roles);
    }

    public synchronized void grantRoles(Set<String> roles, Set<HivePrincipal> grantees, boolean adminOption, HivePrincipal grantor) {
        Set<String> existingRoles = this.listRoles();
        Set<RoleGrant> existingGrants = this.listRoleGrantsSanitized();
        Set<RoleGrant> modifiedGrants = new HashSet<RoleGrant>(existingGrants);
        for (HivePrincipal grantee : grantees) {
            for (String role : roles) {
                Preconditions.checkArgument((boolean)existingRoles.contains(role), (String)"Role does not exist: %s", (Object)role);
                if (grantee.getType() == PrincipalType.ROLE) {
                    Preconditions.checkArgument((boolean)existingRoles.contains(grantee.getName()), (String)"Role does not exist: %s", (Object)grantee.getName());
                }
                RoleGrant grantWithAdminOption = new RoleGrant(grantee.toTrinoPrincipal(), role, true);
                RoleGrant grantWithoutAdminOption = new RoleGrant(grantee.toTrinoPrincipal(), role, false);
                if (adminOption) {
                    modifiedGrants.remove(grantWithoutAdminOption);
                    modifiedGrants.add(grantWithAdminOption);
                    continue;
                }
                modifiedGrants.remove(grantWithAdminOption);
                modifiedGrants.add(grantWithoutAdminOption);
            }
        }
        if (!existingGrants.equals(modifiedGrants = FileHiveMetastore.removeDuplicatedEntries(modifiedGrants))) {
            this.writeRoleGrantsFile(modifiedGrants);
        }
    }

    public synchronized void revokeRoles(Set<String> roles, Set<HivePrincipal> grantees, boolean adminOption, HivePrincipal grantor) {
        Set<RoleGrant> existingGrants = this.listRoleGrantsSanitized();
        Set<RoleGrant> modifiedGrants = new HashSet<RoleGrant>(existingGrants);
        for (HivePrincipal grantee : grantees) {
            for (String role : roles) {
                RoleGrant grantWithAdminOption = new RoleGrant(grantee.toTrinoPrincipal(), role, true);
                RoleGrant grantWithoutAdminOption = new RoleGrant(grantee.toTrinoPrincipal(), role, false);
                if (!modifiedGrants.contains(grantWithAdminOption) && !modifiedGrants.contains(grantWithoutAdminOption)) continue;
                if (adminOption) {
                    modifiedGrants.remove(grantWithAdminOption);
                    modifiedGrants.add(grantWithoutAdminOption);
                    continue;
                }
                modifiedGrants.remove(grantWithAdminOption);
                modifiedGrants.remove(grantWithoutAdminOption);
            }
        }
        if (!existingGrants.equals(modifiedGrants = FileHiveMetastore.removeDuplicatedEntries(modifiedGrants))) {
            this.writeRoleGrantsFile(modifiedGrants);
        }
    }

    public synchronized Set<RoleGrant> listRoleGrants(HivePrincipal principal) {
        ImmutableSet.Builder result = ImmutableSet.builder();
        if (principal.getType() == PrincipalType.USER) {
            result.add((Object)new RoleGrant(principal.toTrinoPrincipal(), PUBLIC_ROLE_NAME, false));
            if (ADMIN_USERS.contains(principal.getName())) {
                result.add((Object)new RoleGrant(principal.toTrinoPrincipal(), ADMIN_ROLE_NAME, true));
            }
        }
        result.addAll((Iterable)this.listRoleGrantsSanitized().stream().filter(grant -> HivePrincipal.from((TrinoPrincipal)grant.getGrantee()).equals((Object)principal)).collect(Collectors.toSet()));
        return result.build();
    }

    private synchronized Set<RoleGrant> listRoleGrantsSanitized() {
        Set<RoleGrant> grants = this.readRoleGrantsFile();
        Set<String> existingRoles = this.listRoles();
        return FileHiveMetastore.removeDuplicatedEntries(FileHiveMetastore.removeNonExistingRoles(grants, existingRoles));
    }

    private static Set<RoleGrant> removeDuplicatedEntries(Set<RoleGrant> grants) {
        HashMap<RoleGrantee, RoleGrant> map = new HashMap<RoleGrantee, RoleGrant>();
        for (RoleGrant grant : grants) {
            RoleGrantee tuple = new RoleGrantee(grant.getRoleName(), HivePrincipal.from((TrinoPrincipal)grant.getGrantee()));
            map.merge(tuple, grant, (first, second) -> first.isGrantable() ? first : second);
        }
        return ImmutableSet.copyOf(map.values());
    }

    private static Set<RoleGrant> removeNonExistingRoles(Set<RoleGrant> grants, Set<String> existingRoles) {
        ImmutableSet.Builder result = ImmutableSet.builder();
        for (RoleGrant grant : grants) {
            HivePrincipal grantee;
            if (!existingRoles.contains(grant.getRoleName()) || (grantee = HivePrincipal.from((TrinoPrincipal)grant.getGrantee())).getType() == PrincipalType.ROLE && !existingRoles.contains(grantee.getName())) continue;
            result.add((Object)grant);
        }
        return result.build();
    }

    private Set<RoleGrant> readRoleGrantsFile() {
        return ImmutableSet.copyOf((Collection)this.readFile("roleGrants", this.getRoleGrantsFile(), this.roleGrantsCodec).orElse((List<RoleGrant>)ImmutableList.of()));
    }

    private void writeRoleGrantsFile(Set<RoleGrant> roleGrants) {
        this.writeFile("roleGrants", this.getRoleGrantsFile(), this.roleGrantsCodec, ImmutableList.copyOf(roleGrants), true);
    }

    private synchronized Optional<List<String>> getAllPartitionNames(String databaseName, String tableName) {
        Objects.requireNonNull(databaseName, "databaseName is null");
        Objects.requireNonNull(tableName, "tableName is null");
        Optional<Table> tableReference = this.getTable(databaseName, tableName);
        if (tableReference.isEmpty()) {
            return Optional.empty();
        }
        Table table = tableReference.get();
        Location tableMetadataDirectory = this.getTableMetadataDirectory(table);
        List<List<String>> partitions = this.listPartitions(tableMetadataDirectory, table.getPartitionColumns());
        List partitionNames = (List)partitions.stream().map(partitionValues -> MetastoreUtil.makePartitionName(table.getPartitionColumns(), (List<String>)ImmutableList.copyOf((Collection)partitionValues))).filter(partitionName -> this.isValidPartition(table, (String)partitionName)).collect(ImmutableList.toImmutableList());
        return Optional.of(partitionNames);
    }

    private boolean isValidPartition(Table table, String partitionName) {
        Location location = FileHiveMetastore.getSchemaFile(SchemaType.PARTITION, this.getPartitionMetadataDirectory(table, partitionName));
        try {
            return this.fileSystem.newInputFile(location).exists();
        }
        catch (IOException e) {
            return false;
        }
    }

    private List<List<String>> listPartitions(Location directory, List<io.trino.metastore.Column> partitionColumns) {
        if (partitionColumns.isEmpty()) {
            return ImmutableList.of();
        }
        try {
            ArrayList<List<String>> partitionValues = new ArrayList<List<String>>();
            FileIterator iterator = this.fileSystem.listFiles(directory);
            while (iterator.hasNext()) {
                List values;
                Location location = iterator.next().location();
                String path = location.toString().substring(directory.toString().length());
                if (path.startsWith("/")) {
                    path = path.substring(1);
                }
                if (!path.endsWith("/.trinoSchema") || (values = Partitions.toPartitionValues((String)(path = path.substring(0, path.length() - TRINO_SCHEMA_FILE_NAME_SUFFIX.length() - 1)))).size() != partitionColumns.size()) continue;
                partitionValues.add(values);
            }
            return partitionValues;
        }
        catch (IOException e) {
            throw new TrinoException((ErrorCodeSupplier)HiveErrorCode.HIVE_METASTORE_ERROR, "Error listing partition directories", (Throwable)e);
        }
    }

    public synchronized Optional<Partition> getPartition(Table table, List<String> partitionValues) {
        Objects.requireNonNull(table, "table is null");
        Objects.requireNonNull(partitionValues, "partitionValues is null");
        Location partitionDirectory = this.getPartitionMetadataDirectory(table, partitionValues);
        return this.readSchemaFile(SchemaType.PARTITION, partitionDirectory, this.partitionCodec).map(partitionMetadata -> partitionMetadata.toPartition(table.getDatabaseName(), table.getTableName(), partitionValues, partitionDirectory.toString()));
    }

    public Optional<List<String>> getPartitionNamesByFilter(String databaseName, String tableName, List<String> columnNames, TupleDomain<String> partitionKeysFilter) {
        return this.getAllPartitionNames(databaseName, tableName);
    }

    public synchronized Map<String, Optional<Partition>> getPartitionsByNames(Table table, List<String> partitionNames) {
        ImmutableMap.Builder builder = ImmutableMap.builder();
        for (String partitionName : partitionNames) {
            List partitionValues = Partitions.toPartitionValues((String)partitionName);
            builder.put((Object)partitionName, this.getPartition(table, partitionValues));
        }
        return builder.buildOrThrow();
    }

    public synchronized Set<HivePrivilegeInfo> listTablePrivileges(String databaseName, String tableName, Optional<String> tableOwner, Optional<HivePrincipal> principal) {
        Table table = this.getRequiredTable(databaseName, tableName);
        Location permissionsDirectory = this.getPermissionsDirectory(table);
        if (principal.isEmpty()) {
            ImmutableSet.Builder privileges = ImmutableSet.builder().addAll(this.readAllPermissions(permissionsDirectory));
            tableOwner.ifPresent(owner -> privileges.add((Object)new HivePrivilegeInfo(HivePrivilegeInfo.HivePrivilege.OWNERSHIP, true, new HivePrincipal(PrincipalType.USER, owner), new HivePrincipal(PrincipalType.USER, owner))));
            return privileges.build();
        }
        ImmutableSet.Builder result = ImmutableSet.builder();
        if (principal.get().getType() == PrincipalType.USER && ((String)table.getOwner().orElseThrow()).equals(principal.get().getName())) {
            result.add((Object)new HivePrivilegeInfo(HivePrivilegeInfo.HivePrivilege.OWNERSHIP, true, principal.get(), principal.get()));
        }
        result.addAll(this.readPermissionsFile(FileHiveMetastore.getPermissionsPath(permissionsDirectory, principal.get())));
        return result.build();
    }

    public synchronized void grantTablePrivileges(String databaseName, String tableName, String tableOwner, HivePrincipal grantee, HivePrincipal grantor, Set<HivePrivilegeInfo.HivePrivilege> privileges, boolean grantOption) {
        this.setTablePrivileges(grantee, databaseName, tableName, (Collection)privileges.stream().map(privilege -> new HivePrivilegeInfo(privilege, grantOption, grantor, grantee)).collect(ImmutableList.toImmutableList()));
    }

    public synchronized void revokeTablePrivileges(String databaseName, String tableName, String tableOwner, HivePrincipal grantee, HivePrincipal grantor, Set<HivePrivilegeInfo.HivePrivilege> privileges, boolean grantOption) {
        Set<HivePrivilegeInfo> currentPrivileges = this.listTablePrivileges(databaseName, tableName, Optional.of(tableOwner), Optional.of(grantee));
        Set privilegesToRemove = (Set)privileges.stream().map(p -> new HivePrivilegeInfo(p, grantOption, grantor, grantee)).collect(ImmutableSet.toImmutableSet());
        this.setTablePrivileges(grantee, databaseName, tableName, (Collection<HivePrivilegeInfo>)Sets.difference(currentPrivileges, (Set)privilegesToRemove));
    }

    public synchronized boolean functionExists(String databaseName, String functionName, String signatureToken) {
        Location directory = this.getFunctionsDirectory(databaseName);
        Location file = FileHiveMetastore.getFunctionFile(directory, functionName, signatureToken);
        try {
            return this.fileSystem.newInputFile(file).exists();
        }
        catch (IOException e) {
            throw new TrinoException((ErrorCodeSupplier)HiveErrorCode.HIVE_METASTORE_ERROR, (Throwable)e);
        }
    }

    public synchronized Collection<LanguageFunction> getAllFunctions(String databaseName) {
        return this.getFunctions(databaseName, Optional.empty());
    }

    public synchronized Collection<LanguageFunction> getFunctions(String databaseName, String functionName) {
        return this.getFunctions(databaseName, Optional.of(functionName));
    }

    private synchronized Collection<LanguageFunction> getFunctions(String databaseName, Optional<String> functionName) {
        ImmutableList.Builder functions = ImmutableList.builder();
        Location directory = this.getFunctionsDirectory(databaseName);
        try {
            FileIterator iterator = this.fileSystem.listFiles(directory);
            while (iterator.hasNext()) {
                Location location = iterator.next().location();
                List parts = Splitter.on((char)'=').splitToList((CharSequence)location.fileName());
                if (parts.size() != 2) continue;
                String name = Partitions.unescapePathName((String)((String)parts.getFirst()));
                if (functionName.isPresent() && !name.equals(functionName.get())) continue;
                this.readFile("function", location, this.functionCodec).ifPresent(arg_0 -> ((ImmutableList.Builder)functions).add(arg_0));
            }
        }
        catch (IOException e) {
            throw new TrinoException((ErrorCodeSupplier)HiveErrorCode.HIVE_METASTORE_ERROR, (Throwable)e);
        }
        return functions.build();
    }

    public synchronized void createFunction(String databaseName, String functionName, LanguageFunction function) {
        Location directory = this.getFunctionsDirectory(databaseName);
        Location file = FileHiveMetastore.getFunctionFile(directory, functionName, function.signatureToken());
        byte[] json = this.functionCodec.toJsonBytes((Object)function);
        try {
            if (this.fileSystem.newInputFile(file).exists()) {
                throw new TrinoException((ErrorCodeSupplier)StandardErrorCode.ALREADY_EXISTS, "Function already exists");
            }
            try (OutputStream outputStream = this.fileSystem.newOutputFile(file).create();){
                outputStream.write(json);
            }
        }
        catch (IOException e) {
            throw new TrinoException((ErrorCodeSupplier)HiveErrorCode.HIVE_METASTORE_ERROR, "Could not write function", (Throwable)e);
        }
    }

    public synchronized void replaceFunction(String databaseName, String functionName, LanguageFunction function) {
        Location directory = this.getFunctionsDirectory(databaseName);
        Location file = FileHiveMetastore.getFunctionFile(directory, functionName, function.signatureToken());
        byte[] json = this.functionCodec.toJsonBytes((Object)function);
        try {
            this.fileSystem.newOutputFile(file).createOrOverwrite(json);
        }
        catch (IOException e) {
            throw new TrinoException((ErrorCodeSupplier)HiveErrorCode.HIVE_METASTORE_ERROR, "Could not write function", (Throwable)e);
        }
    }

    public synchronized void dropFunction(String databaseName, String functionName, String signatureToken) {
        Location directory = this.getFunctionsDirectory(databaseName);
        Location file = FileHiveMetastore.getFunctionFile(directory, functionName, signatureToken);
        try {
            if (!this.fileSystem.newInputFile(file).exists()) {
                throw new TrinoException((ErrorCodeSupplier)StandardErrorCode.NOT_FOUND, "Function not found");
            }
            this.fileSystem.deleteFile(file);
        }
        catch (IOException e) {
            throw new TrinoException((ErrorCodeSupplier)HiveErrorCode.HIVE_METASTORE_ERROR, (Throwable)e);
        }
    }

    private synchronized void setTablePrivileges(HivePrincipal grantee, String databaseName, String tableName, Collection<HivePrivilegeInfo> privileges) {
        Objects.requireNonNull(grantee, "grantee is null");
        Objects.requireNonNull(databaseName, "databaseName is null");
        Objects.requireNonNull(tableName, "tableName is null");
        Objects.requireNonNull(privileges, "privileges is null");
        try {
            Table table = this.getRequiredTable(databaseName, tableName);
            Location permissionsDirectory = this.getPermissionsDirectory(table);
            this.fileSystem.createDirectory(permissionsDirectory);
            Location permissionFilePath = FileHiveMetastore.getPermissionsPath(permissionsDirectory, grantee);
            List permissions = privileges.stream().map(hivePrivilegeInfo -> new PermissionMetadata(hivePrivilegeInfo.getHivePrivilege(), hivePrivilegeInfo.isGrantOption(), grantee)).collect(Collectors.toList());
            this.writeFile("permissions", permissionFilePath, this.permissionsCodec, permissions, true);
        }
        catch (IOException e) {
            throw new TrinoException((ErrorCodeSupplier)HiveErrorCode.HIVE_METASTORE_ERROR, (Throwable)e);
        }
    }

    private synchronized void deleteTablePrivileges(Table table) {
        try {
            Location permissionsDirectory = this.getPermissionsDirectory(table);
            this.fileSystem.deleteDirectory(permissionsDirectory);
        }
        catch (IOException e) {
            throw new TrinoException((ErrorCodeSupplier)HiveErrorCode.HIVE_METASTORE_ERROR, "Could not delete table permissions", (Throwable)e);
        }
    }

    private Set<HivePrivilegeInfo> readPermissionsFile(Location permissionFilePath) {
        return (Set)this.readFile("permissions", permissionFilePath, this.permissionsCodec).orElse((List<PermissionMetadata>)ImmutableList.of()).stream().map(PermissionMetadata::toHivePrivilegeInfo).collect(ImmutableSet.toImmutableSet());
    }

    private Set<HivePrivilegeInfo> readAllPermissions(Location permissionsDirectory) {
        try {
            ImmutableSet.Builder permissions = ImmutableSet.builder();
            FileIterator iterator = this.fileSystem.listFiles(permissionsDirectory);
            while (iterator.hasNext()) {
                Location location = iterator.next().location();
                if (location.fileName().startsWith(".")) continue;
                permissions.addAll(this.readPermissionsFile(location));
            }
            return permissions.build();
        }
        catch (IOException e) {
            throw new TrinoException((ErrorCodeSupplier)HiveErrorCode.HIVE_METASTORE_ERROR, (Throwable)e);
        }
    }

    private void deleteDirectoryAndSchema(SchemaType type, Location metadataDirectory) {
        try {
            Location schemaPath = FileHiveMetastore.getSchemaFile(type, metadataDirectory);
            if (!this.fileSystem.newInputFile(schemaPath).exists()) {
                return;
            }
            this.deleteSchemaFile(type, metadataDirectory);
            this.fileSystem.deleteDirectory(metadataDirectory);
        }
        catch (IOException e) {
            throw new TrinoException((ErrorCodeSupplier)HiveErrorCode.HIVE_METASTORE_ERROR, (Throwable)e);
        }
    }

    private void checkVersion(Optional<String> writerVersion) {
        if (writerVersion.isPresent() && writerVersion.get().equals(this.currentVersion)) {
            return;
        }
        if (this.versionCompatibility == FileHiveMetastoreConfig.VersionCompatibility.UNSAFE_ASSUME_COMPATIBILITY) {
            return;
        }
        throw new RuntimeException(String.format("The metadata file was written with %s while current version is %s. File metastore provides no compatibility for metadata written with a different version. You can disable this check by setting '%s=%s' configuration property.", new Object[]{writerVersion.map(version -> "version " + version).orElse("unknown version"), this.currentVersion, "hive.metastore.version-compatibility", FileHiveMetastoreConfig.VersionCompatibility.UNSAFE_ASSUME_COMPATIBILITY}));
    }

    private <T> Optional<T> readSchemaFile(SchemaType type, Location metadataDirectory, JsonCodec<T> codec) {
        return this.readFile(String.valueOf((Object)type) + " schema", FileHiveMetastore.getSchemaFile(type, metadataDirectory), codec);
    }

    private <T> Optional<T> readFile(String type, Location file, JsonCodec<T> codec) {
        Optional<Object> optional;
        block9: {
            TrinoInputStream inputStream = this.fileSystem.newInputFile(file).newStream();
            try {
                optional = Optional.of(codec.fromJson((InputStream)inputStream));
                if (inputStream == null) break block9;
            }
            catch (Throwable throwable) {
                try {
                    if (inputStream != null) {
                        try {
                            inputStream.close();
                        }
                        catch (Throwable throwable2) {
                            throwable.addSuppressed(throwable2);
                        }
                    }
                    throw throwable;
                }
                catch (Exception e) {
                    if (Throwables.getCausalChain((Throwable)e).stream().anyMatch(FileNotFoundException.class::isInstance)) {
                        return Optional.empty();
                    }
                    throw new TrinoException((ErrorCodeSupplier)HiveErrorCode.HIVE_METASTORE_ERROR, "Could not read " + type, (Throwable)e);
                }
            }
            inputStream.close();
        }
        return optional;
    }

    private <T> void writeSchemaFile(SchemaType type, Location directory, JsonCodec<T> codec, T value, boolean overwrite) {
        this.writeFile(String.valueOf((Object)type) + " schema", FileHiveMetastore.getSchemaFile(type, directory), codec, value, overwrite);
    }

    private <T> void writeFile(String type, Location location, JsonCodec<T> codec, T value, boolean overwrite) {
        block12: {
            try {
                byte[] json = codec.toJsonBytes(value);
                TrinoOutputFile output = this.fileSystem.newOutputFile(location);
                if (overwrite) {
                    output.createOrOverwrite(json);
                    break block12;
                }
                if (this.fileSystem.newInputFile(location).exists()) {
                    throw new TrinoException((ErrorCodeSupplier)HiveErrorCode.HIVE_METASTORE_ERROR, type + " file already exists");
                }
                try {
                    output.createExclusive(json);
                }
                catch (UnsupportedOperationException unsupportedOperationException) {
                    try (OutputStream out = output.create();){
                        out.write(json);
                    }
                }
            }
            catch (Exception e) {
                throw new TrinoException((ErrorCodeSupplier)HiveErrorCode.HIVE_METASTORE_ERROR, "Could not write " + type, (Throwable)e);
            }
        }
    }

    private void renameSchemaFile(SchemaType type, Location oldMetadataDirectory, Location newMetadataDirectory) {
        try {
            this.fileSystem.renameFile(FileHiveMetastore.getSchemaFile(type, oldMetadataDirectory), FileHiveMetastore.getSchemaFile(type, newMetadataDirectory));
        }
        catch (IOException e) {
            throw new TrinoException((ErrorCodeSupplier)HiveErrorCode.HIVE_METASTORE_ERROR, "Could not rename " + String.valueOf((Object)type) + " schema", (Throwable)e);
        }
    }

    private void deleteSchemaFile(SchemaType type, Location metadataDirectory) {
        try {
            this.fileSystem.deleteFile(FileHiveMetastore.getSchemaFile(type, metadataDirectory));
        }
        catch (IOException e) {
            throw new TrinoException((ErrorCodeSupplier)HiveErrorCode.HIVE_METASTORE_ERROR, "Could not delete " + String.valueOf((Object)type) + " schema", (Throwable)e);
        }
    }

    private Location getDatabaseMetadataDirectory(String databaseName) {
        return this.catalogDirectory.appendPath(HiveUtil.escapeSchemaName(databaseName));
    }

    private Location getFunctionsDirectory(String databaseName) {
        return this.getDatabaseMetadataDirectory(databaseName).appendPath(TRINO_FUNCTIONS_DIRECTORY_NAME);
    }

    private Location getTableMetadataDirectory(Table table) {
        return this.getTableMetadataDirectory(table.getDatabaseName(), table.getTableName());
    }

    private Location getTableMetadataDirectory(String databaseName, String tableName) {
        return this.getDatabaseMetadataDirectory(databaseName).appendPath(HiveUtil.escapeTableName(tableName));
    }

    private Location getPartitionMetadataDirectory(Table table, List<String> values) {
        String partitionName = MetastoreUtil.makePartitionName(table.getPartitionColumns(), values);
        return this.getPartitionMetadataDirectory(table, partitionName);
    }

    private Location getPartitionMetadataDirectory(Table table, String partitionName) {
        return this.getPartitionMetadataDirectory(table.getDatabaseName(), table.getTableName(), partitionName);
    }

    private Location getPartitionMetadataDirectory(String databaseName, String tableName, String partitionName) {
        return this.getTableMetadataDirectory(databaseName, tableName).appendPath(partitionName);
    }

    private Location getPermissionsDirectory(Table table) {
        return this.getTableMetadataDirectory(table).appendPath(TRINO_PERMISSIONS_DIRECTORY_NAME);
    }

    private static Location getPermissionsPath(Location permissionsDirectory, HivePrincipal grantee) {
        String granteeType = grantee.getType().toString().toLowerCase(Locale.US);
        return permissionsDirectory.appendPath(granteeType + "_" + grantee.getName());
    }

    private Location getRolesFile() {
        return this.catalogDirectory.appendPath(ROLES_FILE_NAME);
    }

    private Location getRoleGrantsFile() {
        return this.catalogDirectory.appendPath(ROLE_GRANTS_FILE_NAME);
    }

    private static Location getSchemaFile(SchemaType type, Location metadataDirectory) {
        if (type == SchemaType.DATABASE) {
            Object path = metadataDirectory.toString();
            if (((String)path).endsWith("/")) {
                path = ((String)path).substring(0, ((String)path).length() - 1);
            }
            Preconditions.checkArgument((!((String)path).isEmpty() ? 1 : 0) != 0, (String)"Can't use root directory as database path: %s", (Object)metadataDirectory);
            int index = ((String)path).lastIndexOf(47);
            path = index >= 0 ? ((String)path).substring(0, index + 1) + "." + ((String)path).substring(index + 1) : "." + (String)path;
            return Location.of((String)path).appendSuffix(TRINO_SCHEMA_FILE_NAME_SUFFIX);
        }
        return metadataDirectory.appendPath(TRINO_SCHEMA_FILE_NAME_SUFFIX);
    }

    private static Location getFunctionFile(Location directory, String functionName, String signatureToken) {
        return directory.appendPath("%s=%s".formatted(Partitions.escapePathName((String)functionName), Hashing.sha256().hashUnencodedChars((CharSequence)signatureToken)));
    }

    private static PartitionStatistics toHivePartitionStatistics(Map<String, String> parameters, Map<String, ColumnStatistics> columnStatistics) {
        HiveBasicStatistics basicStatistics = MetastoreUtil.getHiveBasicStatistics(parameters);
        Map hiveColumnStatistics = (Map)columnStatistics.entrySet().stream().collect(ImmutableMap.toImmutableMap(Map.Entry::getKey, column -> ((ColumnStatistics)column.getValue()).toHiveColumnStatistics(basicStatistics)));
        return new PartitionStatistics(basicStatistics, hiveColumnStatistics);
    }

    private static Map<String, ColumnStatistics> fromHiveColumnStats(Map<String, HiveColumnStatistics> columnStatistics) {
        return (Map)columnStatistics.entrySet().stream().collect(ImmutableMap.toImmutableMap(Map.Entry::getKey, entry -> ColumnStatistics.fromHiveColumnStatistics((HiveColumnStatistics)entry.getValue())));
    }

    private static Map<String, HiveColumnStatistics> toHiveColumnStats(Set<String> columnNames, Map<String, String> partitionMetadata, Map<String, ColumnStatistics> columnStatistics) {
        HiveBasicStatistics basicStatistics = MetastoreUtil.getHiveBasicStatistics(partitionMetadata);
        return (Map)columnStatistics.entrySet().stream().filter(entry -> columnNames.contains(entry.getKey())).collect(ImmutableMap.toImmutableMap(Map.Entry::getKey, entry -> ((ColumnStatistics)entry.getValue()).toHiveColumnStatistics(basicStatistics)));
    }

    static enum SchemaType {
        DATABASE,
        TABLE,
        PARTITION;


        public String toString() {
            return this.name().toLowerCase(Locale.ENGLISH);
        }
    }

    private record RoleGrantee(String role, HivePrincipal grantee) {
        private RoleGrantee {
            Objects.requireNonNull(role, "role is null");
            Objects.requireNonNull(grantee, "grantee is null");
        }
    }
}

