/*
 * Decompiled with CFR 0.152.
 */
package io.trino.plugin.hive.metastore.thrift;

import com.google.common.collect.ImmutableList;
import io.airlift.log.Logger;
import io.airlift.units.Duration;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Objects;
import java.util.concurrent.Callable;
import java.util.concurrent.ThreadLocalRandom;
import java.util.concurrent.TimeUnit;

public class RetryDriver {
    private static final Logger log = Logger.get(RetryDriver.class);
    public static final int DEFAULT_MAX_ATTEMPTS = 10;
    public static final Duration DEFAULT_MIN_BACKOFF_DELAY = new Duration(1.0, TimeUnit.SECONDS);
    public static final Duration DEFAULT_MAX_BACKOFF_DELAY = new Duration(2.0, TimeUnit.SECONDS);
    public static final Duration DEFAULT_MAX_RETRY_TIME = new Duration(30.0, TimeUnit.SECONDS);
    public static final double DEFAULT_SCALE_FACTOR = 2.0;
    private final int maxAttempts;
    private final Duration minSleepTime;
    private final Duration maxSleepTime;
    private final double scaleFactor;
    private final Duration maxRetryTime;
    private final List<Class<? extends Exception>> stopOnExceptions;

    private RetryDriver(int maxAttempts, Duration minSleepTime, Duration maxSleepTime, double scaleFactor, Duration maxRetryTime, List<Class<? extends Exception>> stopOnExceptions) {
        this.maxAttempts = maxAttempts;
        this.minSleepTime = minSleepTime;
        this.maxSleepTime = maxSleepTime;
        this.scaleFactor = scaleFactor;
        this.maxRetryTime = maxRetryTime;
        this.stopOnExceptions = stopOnExceptions;
    }

    private RetryDriver() {
        this(10, DEFAULT_MIN_BACKOFF_DELAY, DEFAULT_MAX_BACKOFF_DELAY, 2.0, DEFAULT_MAX_RETRY_TIME, (List<Class<? extends Exception>>)ImmutableList.of());
    }

    public static RetryDriver retry() {
        return new RetryDriver();
    }

    public final RetryDriver maxAttempts(int maxAttempts) {
        return new RetryDriver(maxAttempts, this.minSleepTime, this.maxSleepTime, this.scaleFactor, this.maxRetryTime, this.stopOnExceptions);
    }

    public final RetryDriver exponentialBackoff(Duration minSleepTime, Duration maxSleepTime, Duration maxRetryTime, double scaleFactor) {
        return new RetryDriver(this.maxAttempts, minSleepTime, maxSleepTime, scaleFactor, maxRetryTime, this.stopOnExceptions);
    }

    @SafeVarargs
    public final RetryDriver stopOn(Class<? extends Exception> ... classes) {
        Objects.requireNonNull(classes, "classes is null");
        ImmutableList exceptions = ImmutableList.builder().addAll(this.stopOnExceptions).addAll(Arrays.asList(classes)).build();
        return new RetryDriver(this.maxAttempts, this.minSleepTime, this.maxSleepTime, this.scaleFactor, this.maxRetryTime, (List<Class<? extends Exception>>)exceptions);
    }

    public RetryDriver stopOnIllegalExceptions() {
        return this.stopOn(NullPointerException.class, IllegalStateException.class, IllegalArgumentException.class);
    }

    public <V> V run(String callableName, Callable<V> callable) throws Exception {
        Objects.requireNonNull(callableName, "callableName is null");
        Objects.requireNonNull(callable, "callable is null");
        ArrayList<Throwable> suppressedExceptions = new ArrayList<Throwable>();
        long startTime = System.nanoTime();
        int attempt = 0;
        while (true) {
            ++attempt;
            try {
                return callable.call();
            }
            catch (Exception e) {
                if (e instanceof InterruptedException || Thread.currentThread().isInterrupted()) {
                    RetryDriver.addSuppressed(e, suppressedExceptions);
                    throw e;
                }
                for (Class<? extends Exception> clazz : this.stopOnExceptions) {
                    if (!clazz.isInstance(e)) continue;
                    RetryDriver.addSuppressed(e, suppressedExceptions);
                    throw e;
                }
                if (attempt >= this.maxAttempts || Duration.nanosSince((long)startTime).compareTo(this.maxRetryTime) >= 0) {
                    RetryDriver.addSuppressed(e, suppressedExceptions);
                    throw e;
                }
                log.debug("Failed on executing %s with attempt %d, will retry. Exception: %s", new Object[]{callableName, attempt, e.getMessage()});
                suppressedExceptions.add(e);
                int delayInMs = (int)Math.min((double)this.minSleepTime.toMillis() * Math.pow(this.scaleFactor, attempt - 1), (double)this.maxSleepTime.toMillis());
                int jitter = ThreadLocalRandom.current().nextInt(Math.max(1, (int)((double)delayInMs * 0.1)));
                try {
                    TimeUnit.MILLISECONDS.sleep(delayInMs + jitter);
                }
                catch (InterruptedException ie) {
                    Thread.currentThread().interrupt();
                    RuntimeException exception = new RuntimeException(ie);
                    RetryDriver.addSuppressed(exception, suppressedExceptions);
                    throw exception;
                }
            }
        }
    }

    private static void addSuppressed(Exception exception, List<Throwable> suppressedExceptions) {
        for (Throwable suppressedException : suppressedExceptions) {
            if (exception == suppressedException) continue;
            exception.addSuppressed(suppressedException);
        }
    }
}

