/*
 * Decompiled with CFR 0.152.
 */
package alluxio.client.file;

import alluxio.AlluxioURI;
import alluxio.client.ReadType;
import alluxio.client.block.BlockStoreClient;
import alluxio.client.block.stream.BlockInStream;
import alluxio.client.block.stream.BlockWorkerClient;
import alluxio.client.file.FileInStream;
import alluxio.client.file.FileSystemContext;
import alluxio.client.file.FileSystemMasterClient;
import alluxio.client.file.URIStatus;
import alluxio.client.file.options.InStreamOptions;
import alluxio.conf.AlluxioConfiguration;
import alluxio.conf.PropertyKey;
import alluxio.exception.PreconditionMessage;
import alluxio.exception.status.AlluxioStatusException;
import alluxio.exception.status.OutOfRangeException;
import alluxio.grpc.CacheRequest;
import alluxio.grpc.FileSystemMasterCommonPOptions;
import alluxio.grpc.ListStatusPOptions;
import alluxio.grpc.ReadPType;
import alluxio.resource.CloseableResource;
import alluxio.retry.ExponentialTimeBoundedRetry;
import alluxio.retry.RetryPolicy;
import alluxio.util.CommonUtils;
import alluxio.util.FileSystemOptionsUtils;
import alluxio.wire.BlockInfo;
import alluxio.wire.BlockLocation;
import alluxio.wire.WorkerNetAddress;
import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import com.google.common.base.Supplier;
import com.google.common.io.Closer;
import io.grpc.StatusRuntimeException;
import io.netty.util.internal.OutOfDirectMemoryError;
import java.io.Closeable;
import java.io.IOException;
import java.nio.ByteBuffer;
import java.time.Duration;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import javax.annotation.concurrent.NotThreadSafe;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@NotThreadSafe
public class AlluxioFileInStream
extends FileInStream {
    private static final Logger LOG = LoggerFactory.getLogger(AlluxioFileInStream.class);
    private Supplier<RetryPolicy> mRetryPolicySupplier;
    private final URIStatus mStatus;
    private final InStreamOptions mOptions;
    private final BlockStoreClient mBlockStore;
    private final FileSystemContext mContext;
    private final boolean mPassiveCachingEnabled;
    private final long mStatusOutdatedTime;
    private final long mLength;
    private final long mBlockSize;
    private long mPosition;
    private BlockInStream mBlockInStream;
    private BlockInStream mCachedPositionedReadStream;
    private long mLastBlockIdCached;
    private Map<WorkerNetAddress, Long> mFailedWorkers = new HashMap<WorkerNetAddress, Long>();
    private Closer mCloser = Closer.create();

    protected AlluxioFileInStream(URIStatus status, InStreamOptions options, FileSystemContext context) {
        this.mContext = context;
        this.mCloser.register((Closeable)this.mContext.blockReinit());
        try {
            AlluxioConfiguration conf = this.mContext.getPathConf(new AlluxioURI(status.getPath()));
            this.mPassiveCachingEnabled = conf.getBoolean(PropertyKey.USER_FILE_PASSIVE_CACHE_ENABLED);
            Duration blockReadRetryMaxDuration = conf.getDuration(PropertyKey.USER_BLOCK_READ_RETRY_MAX_DURATION);
            Duration blockReadRetrySleepBase = conf.getDuration(PropertyKey.USER_BLOCK_READ_RETRY_SLEEP_MIN);
            Duration blockReadRetrySleepMax = conf.getDuration(PropertyKey.USER_BLOCK_READ_RETRY_SLEEP_MAX);
            this.mRetryPolicySupplier = () -> ExponentialTimeBoundedRetry.builder().withMaxDuration(blockReadRetryMaxDuration).withInitialSleep(blockReadRetrySleepBase).withMaxSleep(blockReadRetrySleepMax).withSkipInitialSleep().build();
            this.mStatus = status;
            this.mStatusOutdatedTime = System.currentTimeMillis() + conf.getMs(PropertyKey.USER_FILE_IN_STREAM_STATUS_EXPIRATION_TIME);
            this.mOptions = options;
            this.mBlockStore = BlockStoreClient.create(this.mContext);
            this.mLength = this.mStatus.getLength();
            this.mBlockSize = this.mStatus.getBlockSizeBytes();
            this.mPosition = 0L;
            this.mBlockInStream = null;
            this.mCachedPositionedReadStream = null;
            this.mLastBlockIdCached = 0L;
        }
        catch (Throwable t) {
            throw CommonUtils.closeAndRethrowRuntimeException((Closer)this.mCloser, (Throwable)t);
        }
    }

    @Override
    public int read() throws IOException {
        if (this.mPosition == this.mLength) {
            return -1;
        }
        RetryPolicy retry = (RetryPolicy)this.mRetryPolicySupplier.get();
        IOException lastException = null;
        while (retry.attempt()) {
            try {
                this.updateStream();
                int result = this.mBlockInStream.read();
                if (result != -1) {
                    ++this.mPosition;
                }
                if (this.mBlockInStream.remaining() == 0L) {
                    this.closeBlockInStream(this.mBlockInStream);
                }
                return result;
            }
            catch (IOException e) {
                lastException = e;
                if (this.mBlockInStream != null) {
                    this.handleRetryableException(this.mBlockInStream, e);
                    this.mBlockInStream = null;
                }
                if (!(e instanceof OutOfRangeException)) continue;
                this.refreshMetadataOnMismatchedLength((OutOfRangeException)e);
            }
        }
        throw lastException;
    }

    @Override
    public int read(ByteBuffer byteBuffer, int off, int len) throws IOException {
        Preconditions.checkArgument((off >= 0 && len >= 0 && len + off <= byteBuffer.capacity() ? 1 : 0) != 0, (String)PreconditionMessage.ERR_BUFFER_STATE.toString(), (Object)byteBuffer.capacity(), (Object)off, (Object)len);
        if (len == 0) {
            return 0;
        }
        if (this.mPosition == this.mLength) {
            return -1;
        }
        int bytesLeft = len;
        int currentOffset = off;
        RetryPolicy retry = (RetryPolicy)this.mRetryPolicySupplier.get();
        IOException lastException = null;
        while (bytesLeft > 0 && this.mPosition != this.mLength && retry.attempt()) {
            try {
                this.updateStream();
                int bytesRead = this.mBlockInStream.read(byteBuffer, currentOffset, bytesLeft);
                if (bytesRead > 0) {
                    bytesLeft -= bytesRead;
                    currentOffset += bytesRead;
                    this.mPosition += (long)bytesRead;
                }
                retry = (RetryPolicy)this.mRetryPolicySupplier.get();
                lastException = null;
                if (this.mBlockInStream.remaining() != 0L) continue;
                this.closeBlockInStream(this.mBlockInStream);
            }
            catch (IOException e) {
                lastException = e;
                if (this.mBlockInStream != null) {
                    this.handleRetryableException(this.mBlockInStream, e);
                    this.mBlockInStream = null;
                }
                if (!(e instanceof OutOfRangeException)) continue;
                this.refreshMetadataOnMismatchedLength((OutOfRangeException)e);
            }
        }
        if (lastException != null) {
            throw lastException;
        }
        return len - bytesLeft;
    }

    private void refreshMetadataOnMismatchedLength(OutOfRangeException e) {
        try {
            CloseableResource<FileSystemMasterClient> client = this.mContext.acquireMasterClientResource();
            Throwable throwable = null;
            try {
                try {
                    AlluxioURI path = new AlluxioURI(this.mStatus.getPath());
                    ListStatusPOptions refreshPathOptions = ListStatusPOptions.newBuilder().setCommonOptions(FileSystemMasterCommonPOptions.newBuilder().setSyncIntervalMs(0L).build()).setLoadMetadataOnly(true).build();
                    ListStatusPOptions mergedOptions = FileSystemOptionsUtils.listStatusDefaults(this.mContext.getPathConf(path)).toBuilder().mergeFrom(refreshPathOptions).build();
                    ((FileSystemMasterClient)client.get()).listStatus(path, mergedOptions);
                    LOG.info("Notified the master that {} should be sync-ed with UFS on the next access", (Object)this.mStatus.getPath());
                    throw new IllegalStateException(e.getMessage());
                }
                catch (Throwable throwable2) {
                    throwable = throwable2;
                    throw throwable2;
                }
            }
            catch (Throwable throwable3) {
                if (client != null) {
                    if (throwable != null) {
                        try {
                            client.close();
                        }
                        catch (Throwable throwable4) {
                            throwable.addSuppressed(throwable4);
                        }
                    } else {
                        client.close();
                    }
                }
                throw throwable3;
            }
        }
        catch (AlluxioStatusException x) {
            String msg = String.format("Failed to force a metadata sync on path %s. Please manually sync metadata by `bin/alluxio fs loadMetadata -f {path}` before you retry reading this file.", this.mStatus.getPath());
            throw new IllegalStateException(msg, (Throwable)e);
        }
    }

    @Override
    public long skip(long n) throws IOException {
        if (n <= 0L) {
            return 0L;
        }
        long toSkip = Math.min(n, this.mLength - this.mPosition);
        this.seek(this.mPosition + toSkip);
        return toSkip;
    }

    @Override
    public void close() throws IOException {
        this.closeBlockInStream(this.mBlockInStream);
        this.closeBlockInStream(this.mCachedPositionedReadStream);
        this.mCloser.close();
    }

    @Override
    public long remaining() {
        return this.mLength - this.mPosition;
    }

    @Override
    public int positionedRead(long pos, byte[] b, int off, int len) throws IOException {
        return this.positionedReadInternal(pos, b, off, len);
    }

    private int positionedReadInternal(long pos, byte[] b, int off, int len) throws IOException {
        if (pos < 0L || pos >= this.mLength) {
            return -1;
        }
        if ((long)len < this.mContext.getPathConf(new AlluxioURI(this.mStatus.getPath())).getBytes(PropertyKey.USER_FILE_SEQUENTIAL_PREAD_THRESHOLD)) {
            this.mOptions.setPositionShort(true);
        }
        int lenCopy = len;
        RetryPolicy retry = (RetryPolicy)this.mRetryPolicySupplier.get();
        IOException lastException = null;
        while (len > 0 && retry.attempt() && pos < this.mLength) {
            long blockId = (Long)this.mStatus.getBlockIds().get(Math.toIntExact(pos / this.mBlockSize));
            try {
                BlockInfo blockInfo;
                BlockInfo blockInfo2 = blockInfo = this.isStatusOutdated() || lastException != null ? this.mBlockStore.getInfo(blockId) : this.mStatus.getBlockInfo(blockId);
                if (this.mCachedPositionedReadStream == null) {
                    this.mCachedPositionedReadStream = this.mBlockStore.getInStream(blockInfo, this.mOptions, this.mFailedWorkers);
                } else if (this.mCachedPositionedReadStream.getId() != blockId) {
                    this.closeBlockInStream(this.mCachedPositionedReadStream);
                    this.mCachedPositionedReadStream = this.mBlockStore.getInStream(blockInfo, this.mOptions, this.mFailedWorkers);
                }
                long offset = pos % this.mBlockSize;
                int bytesRead = this.mCachedPositionedReadStream.positionedRead(offset, b, off, (int)Math.min(this.mBlockSize - offset, (long)len));
                Preconditions.checkState((bytesRead > 0 ? 1 : 0) != 0, (Object)"No data is read before EOF");
                pos += (long)bytesRead;
                off += bytesRead;
                len -= bytesRead;
                retry = (RetryPolicy)this.mRetryPolicySupplier.get();
                lastException = null;
                BlockInStream.BlockInStreamSource source = this.mCachedPositionedReadStream.getSource();
                if (source != BlockInStream.BlockInStreamSource.NODE_LOCAL && source != BlockInStream.BlockInStreamSource.PROCESS_LOCAL) {
                    this.triggerAsyncCaching(this.mCachedPositionedReadStream);
                }
                if ((long)bytesRead != this.mBlockSize - offset) continue;
                this.mCachedPositionedReadStream.close();
                this.mCachedPositionedReadStream = null;
            }
            catch (IOException e) {
                lastException = e;
                if (this.mCachedPositionedReadStream == null) continue;
                this.handleRetryableException(this.mCachedPositionedReadStream, e);
                this.mCachedPositionedReadStream = null;
            }
        }
        if (lastException != null) {
            throw lastException;
        }
        return lenCopy - len;
    }

    public long getPos() {
        return this.mPosition;
    }

    public void seek(long pos) throws IOException {
        if (this.mPosition == pos) {
            return;
        }
        Preconditions.checkArgument((pos >= 0L ? 1 : 0) != 0, (String)PreconditionMessage.ERR_SEEK_NEGATIVE.toString(), (long)pos);
        Preconditions.checkArgument((pos <= this.mLength ? 1 : 0) != 0, (String)PreconditionMessage.ERR_SEEK_PAST_END_OF_FILE.toString(), (long)pos);
        if (this.mBlockInStream == null) {
            this.mPosition = pos;
            return;
        }
        long delta = pos - this.mPosition;
        if (delta <= this.mBlockInStream.remaining() && delta >= -this.mBlockInStream.getPos()) {
            this.mBlockInStream.seek(this.mBlockInStream.getPos() + delta);
        } else {
            this.closeBlockInStream(this.mBlockInStream);
        }
        this.mPosition += delta;
    }

    private void updateStream() throws IOException {
        long blockId;
        BlockInfo blockInfo;
        if (this.mBlockInStream != null && this.mBlockInStream.remaining() > 0L) {
            return;
        }
        if (this.mBlockInStream != null && this.mBlockInStream.remaining() == 0L) {
            this.closeBlockInStream(this.mBlockInStream);
        }
        if ((blockInfo = this.mStatus.getBlockInfo(blockId = ((Long)this.mStatus.getBlockIds().get(Math.toIntExact(this.mPosition / this.mBlockSize))).longValue())) == null) {
            throw new IOException("No BlockInfo for block(id=" + blockId + ") of file(id=" + this.mStatus.getFileId() + ", path=" + this.mStatus.getPath() + ")");
        }
        boolean isBlockInfoOutdated = true;
        if (this.mFailedWorkers.isEmpty() || this.mFailedWorkers.size() < blockInfo.getLocations().size()) {
            isBlockInfoOutdated = false;
        } else {
            List locs = blockInfo.getLocations();
            for (BlockLocation location : locs) {
                if (this.mFailedWorkers.containsKey(location.getWorkerAddress())) continue;
                isBlockInfoOutdated = false;
                break;
            }
        }
        this.mBlockInStream = isBlockInfoOutdated || this.isStatusOutdated() ? this.mBlockStore.getInStream(blockId, this.mOptions, this.mFailedWorkers) : this.mBlockStore.getInStream(blockInfo, this.mOptions, this.mFailedWorkers);
        long offset = this.mPosition % this.mBlockSize;
        this.mBlockInStream.seek(offset);
    }

    @VisibleForTesting
    public boolean isStatusOutdated() {
        return System.currentTimeMillis() >= this.mStatusOutdatedTime;
    }

    private void closeBlockInStream(BlockInStream stream) throws IOException {
        if (stream != null) {
            BlockInStream.BlockInStreamSource blockSource = stream.getSource();
            stream.close();
            if (stream == this.mBlockInStream) {
                this.mBlockInStream = null;
            }
            if (stream == this.mCachedPositionedReadStream) {
                this.mCachedPositionedReadStream = null;
            }
            if (blockSource == BlockInStream.BlockInStreamSource.NODE_LOCAL || blockSource == BlockInStream.BlockInStreamSource.PROCESS_LOCAL) {
                return;
            }
            this.triggerAsyncCaching(stream);
        }
    }

    @VisibleForTesting
    boolean triggerAsyncCaching(BlockInStream stream) {
        long blockId = stream.getId();
        BlockInfo blockInfo = this.mStatus.getBlockInfo(blockId);
        if (blockInfo == null) {
            return false;
        }
        try {
            boolean cache = ReadType.fromProto((ReadPType)this.mOptions.getOptions().getReadType()).isCache();
            boolean overReplicated = this.mStatus.getReplicationMax() > 0 && blockInfo.getLocations().size() >= this.mStatus.getReplicationMax();
            cache = cache && !overReplicated;
            WorkerNetAddress dataSource = stream.getAddress();
            if (cache && this.mLastBlockIdCached != blockId) {
                WorkerNetAddress worker;
                long blockLength = this.mOptions.getBlockInfo(blockId).getLength();
                String host = dataSource.getHost();
                if (!dataSource.getContainerHost().equals("")) {
                    LOG.debug("Worker is in a container. Use container host {} instead of physical host {}", (Object)dataSource.getContainerHost(), (Object)host);
                    host = dataSource.getContainerHost();
                }
                CacheRequest request = CacheRequest.newBuilder().setBlockId(blockId).setLength(blockLength).setOpenUfsBlockOptions(this.mOptions.getOpenUfsBlockOptions(blockId)).setSourceHost(host).setSourcePort(dataSource.getDataPort()).setAsync(true).build();
                if (this.mPassiveCachingEnabled && this.mContext.hasProcessLocalWorker()) {
                    this.mContext.getProcessLocalWorker().orElseThrow(NullPointerException::new).cache(request);
                    this.mLastBlockIdCached = blockId;
                    return true;
                }
                if (this.mPassiveCachingEnabled && this.mContext.hasNodeLocalWorker()) {
                    worker = this.mContext.getNodeLocalWorker();
                } else {
                    if (blockInfo.getLocations().stream().anyMatch(it -> Objects.equals(it.getWorkerAddress(), dataSource))) {
                        this.mLastBlockIdCached = blockId;
                        return false;
                    }
                    worker = dataSource;
                }
                try (CloseableResource<BlockWorkerClient> blockWorker = this.mContext.acquireBlockWorkerClient(worker);){
                    ((BlockWorkerClient)blockWorker.get()).cache(request);
                    this.mLastBlockIdCached = blockId;
                }
            }
            return true;
        }
        catch (Exception e) {
            LOG.warn("Failed to complete async cache request (best effort) for block {} of file {}: {}", new Object[]{stream.getId(), this.mStatus.getPath(), e.toString()});
            return false;
        }
    }

    private void handleRetryableException(BlockInStream stream, IOException e) {
        boolean causedByClientOOM;
        WorkerNetAddress workerAddress = stream.getAddress();
        boolean bl = causedByClientOOM = e.getCause() instanceof StatusRuntimeException && e.getCause().getCause() instanceof OutOfDirectMemoryError;
        if (causedByClientOOM) {
            LOG.warn("Failed to read block {} of file {} from worker {}, will retry: {}.", new Object[]{stream.getId(), this.mStatus.getPath(), workerAddress, e.toString()});
        } else {
            LOG.warn("Failed to read block {} of file {} from worker {}. This worker will be skipped for future read operations, will retry: {}.", new Object[]{stream.getId(), this.mStatus.getPath(), workerAddress, e.toString()});
        }
        try {
            stream.close();
        }
        catch (Exception ex) {
            LOG.warn("Failed to close input stream for block {} of file {}: {}", new Object[]{stream.getId(), this.mStatus.getPath(), ex.toString()});
        }
        if (!causedByClientOOM) {
            this.mFailedWorkers.put(workerAddress, System.currentTimeMillis());
        }
    }

    @Override
    public void unbuffer() {
        if (this.mBlockInStream != null) {
            this.mBlockInStream.unbuffer();
        }
        if (this.mCachedPositionedReadStream != null) {
            this.mCachedPositionedReadStream.unbuffer();
        }
    }
}

