/*
 * Decompiled with CFR 0.152.
 */
package io.trino.plugin.hive.orc;

import com.google.common.base.MoreObjects;
import com.google.common.base.Preconditions;
import com.google.common.base.VerifyException;
import com.google.common.collect.ImmutableList;
import io.airlift.log.Logger;
import io.airlift.slice.SizeOf;
import io.trino.orc.OrcDataSink;
import io.trino.orc.OrcDataSource;
import io.trino.orc.OrcWriteValidation;
import io.trino.orc.OrcWriter;
import io.trino.orc.OrcWriterOptions;
import io.trino.orc.OrcWriterStats;
import io.trino.orc.metadata.ColumnMetadata;
import io.trino.orc.metadata.CompressionKind;
import io.trino.orc.metadata.OrcType;
import io.trino.plugin.hive.FileWriter;
import io.trino.plugin.hive.HiveErrorCode;
import io.trino.plugin.hive.WriterKind;
import io.trino.plugin.hive.acid.AcidTransaction;
import io.trino.spi.ErrorCodeSupplier;
import io.trino.spi.Page;
import io.trino.spi.TrinoException;
import io.trino.spi.block.Block;
import io.trino.spi.block.BlockBuilder;
import io.trino.spi.block.LongArrayBlock;
import io.trino.spi.block.RowBlock;
import io.trino.spi.block.RunLengthEncodedBlock;
import io.trino.spi.type.BigintType;
import io.trino.spi.type.IntegerType;
import io.trino.spi.type.Type;
import java.io.Closeable;
import java.io.IOException;
import java.io.UncheckedIOException;
import java.lang.management.ManagementFactory;
import java.lang.management.ThreadMXBean;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.OptionalInt;
import java.util.OptionalLong;
import java.util.function.Supplier;

public class OrcFileWriter
implements FileWriter {
    private static final Logger log = Logger.get(OrcFileWriter.class);
    private static final int INSTANCE_SIZE = SizeOf.instanceSize(OrcFileWriter.class);
    private static final ThreadMXBean THREAD_MX_BEAN = ManagementFactory.getThreadMXBean();
    protected final OrcWriter orcWriter;
    private final WriterKind writerKind;
    private final AcidTransaction transaction;
    private final boolean useAcidSchema;
    private final OptionalInt bucketNumber;
    private final Closeable rollbackAction;
    private final int[] fileInputColumnIndexes;
    private final List<Block> nullBlocks;
    private final Optional<Supplier<OrcDataSource>> validationInputFactory;
    private OptionalLong maxWriteId = OptionalLong.empty();
    private long nextRowId;
    private long validationCpuNanos;

    public OrcFileWriter(OrcDataSink orcDataSink, WriterKind writerKind, AcidTransaction transaction, boolean useAcidSchema, OptionalInt bucketNumber, Closeable rollbackAction, List<String> columnNames, List<Type> fileColumnTypes, ColumnMetadata<OrcType> fileColumnOrcTypes, CompressionKind compression, OrcWriterOptions options, int[] fileInputColumnIndexes, Map<String, String> metadata, Optional<Supplier<OrcDataSource>> validationInputFactory, OrcWriteValidation.OrcWriteValidationMode validationMode, OrcWriterStats stats) {
        Objects.requireNonNull(orcDataSink, "orcDataSink is null");
        this.writerKind = Objects.requireNonNull(writerKind, "writerKind is null");
        this.transaction = Objects.requireNonNull(transaction, "transaction is null");
        this.useAcidSchema = useAcidSchema;
        this.bucketNumber = Objects.requireNonNull(bucketNumber, "bucketNumber is null");
        this.rollbackAction = Objects.requireNonNull(rollbackAction, "rollbackAction is null");
        this.fileInputColumnIndexes = Objects.requireNonNull(fileInputColumnIndexes, "fileInputColumnIndexes is null");
        ImmutableList.Builder nullBlocks = ImmutableList.builder();
        for (Type fileColumnType : fileColumnTypes) {
            BlockBuilder blockBuilder = fileColumnType.createBlockBuilder(null, 1, 0);
            blockBuilder.appendNull();
            nullBlocks.add((Object)blockBuilder.build());
        }
        this.nullBlocks = nullBlocks.build();
        this.validationInputFactory = validationInputFactory;
        this.orcWriter = new OrcWriter(orcDataSink, columnNames, fileColumnTypes, fileColumnOrcTypes, compression, options, metadata, validationInputFactory.isPresent(), validationMode, stats);
        if (transaction.isTransactional()) {
            this.setMaxWriteId(transaction.getWriteId());
        }
    }

    @Override
    public long getWrittenBytes() {
        return this.orcWriter.getWrittenBytes() + this.orcWriter.getBufferedBytes();
    }

    @Override
    public long getMemoryUsage() {
        return (long)INSTANCE_SIZE + this.orcWriter.getRetainedBytes();
    }

    @Override
    public void appendRows(Page dataPage) {
        Block[] blocks = new Block[this.fileInputColumnIndexes.length];
        boolean[] nullBlocksArray = new boolean[this.fileInputColumnIndexes.length];
        boolean hasNullBlocks = false;
        int positionCount = dataPage.getPositionCount();
        for (int i = 0; i < this.fileInputColumnIndexes.length; ++i) {
            int inputColumnIndex = this.fileInputColumnIndexes[i];
            if (inputColumnIndex < 0) {
                hasNullBlocks = true;
                blocks[i] = RunLengthEncodedBlock.create((Block)this.nullBlocks.get(i), (int)positionCount);
            } else {
                blocks[i] = dataPage.getBlock(inputColumnIndex);
            }
            nullBlocksArray[i] = inputColumnIndex < 0;
        }
        if (this.transaction.isInsert() && this.useAcidSchema) {
            Optional<Object> nullBlocks = hasNullBlocks ? Optional.of(nullBlocksArray) : Optional.empty();
            Block rowBlock = RowBlock.fromFieldBlocks((int)positionCount, nullBlocks, (Block[])blocks);
            blocks = this.buildAcidColumns(rowBlock, this.transaction);
        }
        Page page = new Page(dataPage.getPositionCount(), blocks);
        try {
            this.orcWriter.write(page);
        }
        catch (IOException | UncheckedIOException e) {
            throw new TrinoException((ErrorCodeSupplier)HiveErrorCode.HIVE_WRITER_DATA_ERROR, (Throwable)e);
        }
    }

    @Override
    public Closeable commit() {
        try {
            if (this.transaction.isAcidTransactionRunning() && this.useAcidSchema) {
                this.updateUserMetadata();
            }
            this.orcWriter.close();
        }
        catch (IOException | UncheckedIOException e) {
            try {
                this.rollbackAction.close();
            }
            catch (Exception ignored) {
                log.error((Throwable)ignored, "Exception when committing file");
            }
            throw new TrinoException((ErrorCodeSupplier)HiveErrorCode.HIVE_WRITER_CLOSE_ERROR, "Error committing write to Hive", (Throwable)e);
        }
        if (this.validationInputFactory.isPresent()) {
            try (OrcDataSource input = this.validationInputFactory.get().get();){
                long startThreadCpuTime = THREAD_MX_BEAN.getCurrentThreadCpuTime();
                this.orcWriter.validate(input);
                this.validationCpuNanos += THREAD_MX_BEAN.getCurrentThreadCpuTime() - startThreadCpuTime;
            }
            catch (IOException | UncheckedIOException e) {
                throw new TrinoException((ErrorCodeSupplier)HiveErrorCode.HIVE_WRITE_VALIDATION_FAILED, (Throwable)e);
            }
        }
        return this.rollbackAction;
    }

    private void updateUserMetadata() {
        long writeId;
        int bucketValue = OrcFileWriter.computeBucketValue(this.bucketNumber.orElse(0), 0);
        long l = writeId = this.maxWriteId.isPresent() ? this.maxWriteId.getAsLong() : this.transaction.getWriteId();
        if (this.transaction.isAcidTransactionRunning()) {
            int stripeRowCount = this.orcWriter.getStripeRowCount();
            HashMap<String, String> userMetadata = new HashMap<String, String>();
            switch (this.writerKind) {
                case INSERT: {
                    userMetadata.put("hive.acid.stats", String.format("%s,0,0", stripeRowCount));
                    break;
                }
                case DELETE: {
                    userMetadata.put("hive.acid.stats", String.format("0,0,%s", stripeRowCount));
                    break;
                }
                default: {
                    throw new IllegalStateException("In updateUserMetadata, unknown writerKind " + this.writerKind);
                }
            }
            userMetadata.put("hive.acid.key.index", String.format("%s,%s,%s;", writeId, bucketValue, stripeRowCount - 1));
            userMetadata.put("hive.acid.version", "2");
            this.orcWriter.updateUserMetadata(userMetadata);
        }
    }

    @Override
    public void rollback() {
        try (Closeable closeable = this.rollbackAction;){
            this.orcWriter.close();
        }
        catch (Exception e) {
            throw new TrinoException((ErrorCodeSupplier)HiveErrorCode.HIVE_WRITER_CLOSE_ERROR, "Error rolling back write to Hive", (Throwable)e);
        }
    }

    @Override
    public long getValidationCpuNanos() {
        return this.validationCpuNanos;
    }

    public void setMaxWriteId(long maxWriteId) {
        this.maxWriteId = OptionalLong.of(maxWriteId);
    }

    public String toString() {
        return MoreObjects.toStringHelper((Object)this).add("writer", (Object)this.orcWriter).toString();
    }

    private Block[] buildAcidColumns(Block rowBlock, AcidTransaction transaction) {
        int positionCount = rowBlock.getPositionCount();
        int bucketValue = OrcFileWriter.computeBucketValue(this.bucketNumber.orElse(0), 0);
        return new Block[]{RunLengthEncodedBlock.create((Type)IntegerType.INTEGER, (Object)this.getOrcOperation(transaction), (int)positionCount), RunLengthEncodedBlock.create((Type)BigintType.BIGINT, (Object)transaction.getWriteId(), (int)positionCount), RunLengthEncodedBlock.create((Type)IntegerType.INTEGER, (Object)bucketValue, (int)positionCount), this.buildAcidRowIdsColumn(positionCount), RunLengthEncodedBlock.create((Type)BigintType.BIGINT, (Object)transaction.getWriteId(), (int)positionCount), rowBlock};
    }

    private int getOrcOperation(AcidTransaction transaction) {
        switch (transaction.getOperation()) {
            case INSERT: {
                return 0;
            }
        }
        throw new VerifyException("In getOrcOperation, the transaction operation is not allowed, transaction " + transaction);
    }

    private Block buildAcidRowIdsColumn(int positionCount) {
        long[] rowIds = new long[positionCount];
        for (int i = 0; i < positionCount; ++i) {
            ++this.nextRowId;
        }
        return new LongArrayBlock(positionCount, Optional.empty(), rowIds);
    }

    public static int computeBucketValue(int bucketId, int statementId) {
        Preconditions.checkArgument((statementId >= 0 && statementId < 65536 ? 1 : 0) != 0, (String)"statementId should be non-negative and less than 1 << 16, but is %s", (int)statementId);
        Preconditions.checkArgument((bucketId >= 0 && bucketId <= 8192 ? 1 : 0) != 0, (String)"bucketId should be non-negative and less than 1 << 13, but is %s", (int)bucketId);
        return 0x20000000 | bucketId << 16 | statementId;
    }
}

