/*
 * Decompiled with CFR 0.152.
 */
package io.trino.plugin.hive.util;

import com.google.common.base.CharMatcher;
import com.google.common.collect.ImmutableList;
import com.google.common.io.BaseEncoding;
import io.airlift.slice.Slice;
import io.trino.filesystem.Location;
import io.trino.hdfs.FileSystemUtils;
import io.trino.hdfs.HdfsContext;
import io.trino.hdfs.HdfsEnvironment;
import io.trino.hdfs.rubix.CachingTrinoS3FileSystem;
import io.trino.hdfs.s3.TrinoS3FileSystem;
import io.trino.plugin.hive.HiveErrorCode;
import io.trino.plugin.hive.HiveReadOnlyException;
import io.trino.plugin.hive.HiveType;
import io.trino.plugin.hive.TableType;
import io.trino.plugin.hive.metastore.Database;
import io.trino.plugin.hive.metastore.MetastoreUtil;
import io.trino.plugin.hive.metastore.Partition;
import io.trino.plugin.hive.metastore.ProtectMode;
import io.trino.plugin.hive.metastore.SemiTransactionalHiveMetastore;
import io.trino.plugin.hive.metastore.Storage;
import io.trino.plugin.hive.metastore.Table;
import io.trino.plugin.hive.type.ListTypeInfo;
import io.trino.plugin.hive.type.MapTypeInfo;
import io.trino.plugin.hive.type.PrimitiveCategory;
import io.trino.plugin.hive.type.PrimitiveTypeInfo;
import io.trino.plugin.hive.type.StructTypeInfo;
import io.trino.plugin.hive.type.TypeInfo;
import io.trino.plugin.hive.util.HiveUtil;
import io.trino.spi.ErrorCodeSupplier;
import io.trino.spi.Page;
import io.trino.spi.StandardErrorCode;
import io.trino.spi.TrinoException;
import io.trino.spi.block.Block;
import io.trino.spi.connector.SchemaNotFoundException;
import io.trino.spi.connector.SchemaTableName;
import io.trino.spi.type.BigintType;
import io.trino.spi.type.BooleanType;
import io.trino.spi.type.CharType;
import io.trino.spi.type.Chars;
import io.trino.spi.type.DateType;
import io.trino.spi.type.DecimalType;
import io.trino.spi.type.Decimals;
import io.trino.spi.type.DoubleType;
import io.trino.spi.type.IntegerType;
import io.trino.spi.type.RealType;
import io.trino.spi.type.SmallintType;
import io.trino.spi.type.TimestampType;
import io.trino.spi.type.TinyintType;
import io.trino.spi.type.Type;
import io.trino.spi.type.VarcharType;
import java.io.IOException;
import java.nio.charset.StandardCharsets;
import java.time.LocalDate;
import java.time.LocalDateTime;
import java.time.ZoneOffset;
import java.time.format.DateTimeFormatter;
import java.time.format.DateTimeFormatterBuilder;
import java.time.temporal.ChronoField;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.UUID;
import org.apache.hadoop.fs.FileStatus;
import org.apache.hadoop.fs.FileSystem;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.fs.permission.FsPermission;
import org.apache.hadoop.fs.viewfs.ViewFileSystem;
import org.apache.hadoop.hdfs.DistributedFileSystem;

public final class HiveWriteUtils {
    private static final DateTimeFormatter HIVE_DATE_FORMATTER = DateTimeFormatter.ofPattern("yyyy-MM-dd");
    private static final DateTimeFormatter HIVE_TIMESTAMP_FORMATTER = new DateTimeFormatterBuilder().append(DateTimeFormatter.ofPattern("yyyy-MM-dd HH:mm:ss")).optionalStart().appendFraction(ChronoField.NANO_OF_SECOND, 0, 9, true).optionalEnd().toFormatter();

    private HiveWriteUtils() {
    }

    public static List<String> createPartitionValues(List<Type> partitionColumnTypes, Page partitionColumns, int position) {
        ImmutableList.Builder partitionValues = ImmutableList.builder();
        for (int field = 0; field < partitionColumns.getChannelCount(); ++field) {
            String value = HiveWriteUtils.toPartitionValue(partitionColumnTypes.get(field), partitionColumns.getBlock(field), position);
            if (!CharMatcher.inRange((char)' ', (char)'~').matchesAllOf((CharSequence)value)) {
                String encoded = BaseEncoding.base16().withSeparator(" ", 2).encode(value.getBytes(StandardCharsets.UTF_8));
                throw new TrinoException((ErrorCodeSupplier)HiveErrorCode.HIVE_INVALID_PARTITION_VALUE, "Hive partition keys can only contain printable ASCII characters (0x20 - 0x7E). Invalid value: " + encoded);
            }
            partitionValues.add((Object)value);
        }
        return partitionValues.build();
    }

    private static String toPartitionValue(Type type, Block block, int position) {
        if (block.isNull(position)) {
            return "__HIVE_DEFAULT_PARTITION__";
        }
        if (BooleanType.BOOLEAN.equals((Object)type)) {
            return String.valueOf(BooleanType.BOOLEAN.getBoolean(block, position));
        }
        if (BigintType.BIGINT.equals((Object)type)) {
            return String.valueOf(BigintType.BIGINT.getLong(block, position));
        }
        if (IntegerType.INTEGER.equals((Object)type)) {
            return String.valueOf(IntegerType.INTEGER.getInt(block, position));
        }
        if (SmallintType.SMALLINT.equals((Object)type)) {
            return String.valueOf(SmallintType.SMALLINT.getShort(block, position));
        }
        if (TinyintType.TINYINT.equals((Object)type)) {
            return String.valueOf(TinyintType.TINYINT.getByte(block, position));
        }
        if (RealType.REAL.equals((Object)type)) {
            return String.valueOf(RealType.REAL.getFloat(block, position));
        }
        if (DoubleType.DOUBLE.equals((Object)type)) {
            return String.valueOf(DoubleType.DOUBLE.getDouble(block, position));
        }
        if (type instanceof VarcharType) {
            VarcharType varcharType = (VarcharType)type;
            return varcharType.getSlice(block, position).toStringUtf8();
        }
        if (type instanceof CharType) {
            CharType charType = (CharType)type;
            return Chars.padSpaces((Slice)charType.getSlice(block, position), (CharType)charType).toStringUtf8();
        }
        if (DateType.DATE.equals((Object)type)) {
            return LocalDate.ofEpochDay(DateType.DATE.getInt(block, position)).format(HIVE_DATE_FORMATTER);
        }
        if (TimestampType.TIMESTAMP_MILLIS.equals((Object)type)) {
            long epochMicros = type.getLong(block, position);
            long epochSeconds = Math.floorDiv(epochMicros, 1000000);
            int nanosOfSecond = Math.floorMod(epochMicros, 1000000) * 1000;
            return LocalDateTime.ofEpochSecond(epochSeconds, nanosOfSecond, ZoneOffset.UTC).format(HIVE_TIMESTAMP_FORMATTER);
        }
        if (type instanceof DecimalType) {
            DecimalType decimalType = (DecimalType)type;
            return Decimals.readBigDecimal((DecimalType)decimalType, (Block)block, (int)position).stripTrailingZeros().toPlainString();
        }
        throw new TrinoException((ErrorCodeSupplier)StandardErrorCode.NOT_SUPPORTED, "Unsupported type for partition: " + type);
    }

    public static void checkTableIsWritable(Table table, boolean writesToNonManagedTablesEnabled) {
        if (table.getTableType().equals(TableType.MATERIALIZED_VIEW.name())) {
            throw new TrinoException((ErrorCodeSupplier)StandardErrorCode.NOT_SUPPORTED, "Cannot write to Hive materialized view");
        }
        if (!writesToNonManagedTablesEnabled && !table.getTableType().equals(TableType.MANAGED_TABLE.name())) {
            throw new TrinoException((ErrorCodeSupplier)StandardErrorCode.NOT_SUPPORTED, "Cannot write to non-managed Hive table");
        }
        HiveWriteUtils.checkWritable(table.getSchemaTableName(), Optional.empty(), MetastoreUtil.getProtectMode(table), table.getParameters(), table.getStorage());
    }

    public static void checkPartitionIsWritable(String partitionName, Partition partition) {
        HiveWriteUtils.checkWritable(partition.getSchemaTableName(), Optional.of(partitionName), MetastoreUtil.getProtectMode(partition), partition.getParameters(), partition.getStorage());
    }

    private static void checkWritable(SchemaTableName tableName, Optional<String> partitionName, ProtectMode protectMode, Map<String, String> parameters, Storage storage) {
        String tablePartitionDescription = "Table '" + tableName + "'";
        if (partitionName.isPresent()) {
            tablePartitionDescription = tablePartitionDescription + " partition '" + partitionName.get() + "'";
        }
        MetastoreUtil.verifyOnline(tableName, partitionName, protectMode, parameters);
        if (protectMode.readOnly()) {
            throw new HiveReadOnlyException(tableName, partitionName);
        }
        if (storage.isSkewed()) {
            throw new TrinoException((ErrorCodeSupplier)StandardErrorCode.NOT_SUPPORTED, String.format("Inserting into bucketed tables with skew is not supported. %s", tablePartitionDescription));
        }
    }

    public static Location getTableDefaultLocation(HdfsContext context, SemiTransactionalHiveMetastore metastore, HdfsEnvironment hdfsEnvironment, String schemaName, String tableName) {
        Database database = metastore.getDatabase(schemaName).orElseThrow(() -> new SchemaNotFoundException(schemaName));
        return HiveWriteUtils.getTableDefaultLocation(database, context, hdfsEnvironment, schemaName, tableName);
    }

    public static Location getTableDefaultLocation(Database database, HdfsContext context, HdfsEnvironment hdfsEnvironment, String schemaName, String tableName) {
        String location = database.getLocation().orElseThrow(() -> new TrinoException((ErrorCodeSupplier)HiveErrorCode.HIVE_DATABASE_LOCATION_ERROR, String.format("Database '%s' location is not set", schemaName)));
        Path databasePath = new Path(location);
        if (!HiveWriteUtils.isS3FileSystem(context, hdfsEnvironment, databasePath)) {
            if (!HiveWriteUtils.pathExists(context, hdfsEnvironment, databasePath)) {
                throw new TrinoException((ErrorCodeSupplier)HiveErrorCode.HIVE_DATABASE_LOCATION_ERROR, String.format("Database '%s' location does not exist: %s", schemaName, databasePath));
            }
            if (!HiveWriteUtils.isDirectory(context, hdfsEnvironment, databasePath)) {
                throw new TrinoException((ErrorCodeSupplier)HiveErrorCode.HIVE_DATABASE_LOCATION_ERROR, String.format("Database '%s' location is not a directory: %s", schemaName, databasePath));
            }
        }
        Location databaseLocation = Location.of((String)databasePath.toString());
        return databaseLocation.appendPath(HiveUtil.escapeTableName(tableName));
    }

    public static boolean pathExists(HdfsContext context, HdfsEnvironment hdfsEnvironment, Path path) {
        try {
            return hdfsEnvironment.getFileSystem(context, path).exists(path);
        }
        catch (IOException e) {
            throw new TrinoException((ErrorCodeSupplier)HiveErrorCode.HIVE_FILESYSTEM_ERROR, "Failed checking path: " + path, (Throwable)e);
        }
    }

    public static boolean isS3FileSystem(HdfsContext context, HdfsEnvironment hdfsEnvironment, Path path) {
        try {
            FileSystem fileSystem = FileSystemUtils.getRawFileSystem((FileSystem)hdfsEnvironment.getFileSystem(context, path));
            return fileSystem instanceof TrinoS3FileSystem || fileSystem.getClass().getName().equals("com.amazon.ws.emr.hadoop.fs.EmrFileSystem") || fileSystem instanceof CachingTrinoS3FileSystem;
        }
        catch (IOException e) {
            throw new TrinoException((ErrorCodeSupplier)HiveErrorCode.HIVE_FILESYSTEM_ERROR, "Failed checking path: " + path, (Throwable)e);
        }
    }

    public static boolean isViewFileSystem(HdfsContext context, HdfsEnvironment hdfsEnvironment, Path path) {
        try {
            return FileSystemUtils.getRawFileSystem((FileSystem)hdfsEnvironment.getFileSystem(context, path)) instanceof ViewFileSystem;
        }
        catch (IOException e) {
            throw new TrinoException((ErrorCodeSupplier)HiveErrorCode.HIVE_FILESYSTEM_ERROR, "Failed checking path: " + path, (Throwable)e);
        }
    }

    private static boolean isDirectory(HdfsContext context, HdfsEnvironment hdfsEnvironment, Path path) {
        try {
            return hdfsEnvironment.getFileSystem(context, path).isDirectory(path);
        }
        catch (IOException e) {
            throw new TrinoException((ErrorCodeSupplier)HiveErrorCode.HIVE_FILESYSTEM_ERROR, "Failed checking path: " + path, (Throwable)e);
        }
    }

    public static boolean isHdfsEncrypted(HdfsContext context, HdfsEnvironment hdfsEnvironment, Path path) {
        try {
            FileSystem fileSystem = FileSystemUtils.getRawFileSystem((FileSystem)hdfsEnvironment.getFileSystem(context, path));
            if (fileSystem instanceof DistributedFileSystem) {
                return ((DistributedFileSystem)fileSystem).getEZForPath(path) != null;
            }
            return false;
        }
        catch (IOException e) {
            throw new TrinoException((ErrorCodeSupplier)HiveErrorCode.HIVE_FILESYSTEM_ERROR, "Failed checking encryption status for path: " + path, (Throwable)e);
        }
    }

    public static boolean isFileCreatedByQuery(String fileName, String queryId) {
        return fileName.startsWith(queryId) || fileName.endsWith(queryId);
    }

    public static Location createTemporaryPath(HdfsContext context, HdfsEnvironment hdfsEnvironment, Path targetPath, String temporaryStagingDirectoryPath) {
        String temporaryPrefix = temporaryStagingDirectoryPath.replace("${USER}", context.getIdentity().getUser());
        if (HiveWriteUtils.isViewFileSystem(context, hdfsEnvironment, targetPath)) {
            temporaryPrefix = ".hive-staging";
        }
        Path temporaryRoot = new Path(targetPath, temporaryPrefix);
        Path temporaryPath = new Path(temporaryRoot, UUID.randomUUID().toString());
        HiveWriteUtils.createDirectory(context, hdfsEnvironment, temporaryPath);
        if (hdfsEnvironment.isNewFileInheritOwnership()) {
            HiveWriteUtils.setDirectoryOwner(context, hdfsEnvironment, temporaryPath, targetPath);
        }
        return Location.of((String)temporaryPath.toString());
    }

    private static void setDirectoryOwner(HdfsContext context, HdfsEnvironment hdfsEnvironment, Path path, Path targetPath) {
        try {
            FileStatus fileStatus;
            FileSystem fileSystem = hdfsEnvironment.getFileSystem(context, path);
            if (!fileSystem.exists(targetPath)) {
                Path parent = targetPath.getParent();
                if (!fileSystem.exists(parent)) {
                    return;
                }
                fileStatus = fileSystem.getFileStatus(parent);
            } else {
                fileStatus = fileSystem.getFileStatus(targetPath);
            }
            String owner = fileStatus.getOwner();
            String group = fileStatus.getGroup();
            fileSystem.setOwner(path, owner, group);
        }
        catch (IOException e) {
            throw new TrinoException((ErrorCodeSupplier)HiveErrorCode.HIVE_FILESYSTEM_ERROR, String.format("Failed to set owner on %s based on %s", path, targetPath), (Throwable)e);
        }
    }

    public static void createDirectory(HdfsContext context, HdfsEnvironment hdfsEnvironment, Path path) {
        try {
            if (!hdfsEnvironment.getFileSystem(context, path).mkdirs(path, (FsPermission)hdfsEnvironment.getNewDirectoryPermissions().orElse(null))) {
                throw new IOException("mkdirs returned false");
            }
        }
        catch (IOException e) {
            throw new TrinoException((ErrorCodeSupplier)HiveErrorCode.HIVE_FILESYSTEM_ERROR, "Failed to create directory: " + path, (Throwable)e);
        }
        if (hdfsEnvironment.getNewDirectoryPermissions().isPresent()) {
            try {
                hdfsEnvironment.getFileSystem(context, path).setPermission(path, (FsPermission)hdfsEnvironment.getNewDirectoryPermissions().get());
            }
            catch (IOException e) {
                throw new TrinoException((ErrorCodeSupplier)HiveErrorCode.HIVE_FILESYSTEM_ERROR, "Failed to set permission on directory: " + path, (Throwable)e);
            }
        }
    }

    public static boolean isWritableType(HiveType hiveType) {
        return HiveWriteUtils.isWritableType(hiveType.getTypeInfo());
    }

    private static boolean isWritableType(TypeInfo typeInfo) {
        switch (typeInfo.getCategory()) {
            case PRIMITIVE: {
                PrimitiveCategory primitiveCategory = ((PrimitiveTypeInfo)typeInfo).getPrimitiveCategory();
                return HiveWriteUtils.isWritablePrimitiveType(primitiveCategory);
            }
            case MAP: {
                MapTypeInfo mapTypeInfo = (MapTypeInfo)typeInfo;
                return HiveWriteUtils.isWritableType(mapTypeInfo.getMapKeyTypeInfo()) && HiveWriteUtils.isWritableType(mapTypeInfo.getMapValueTypeInfo());
            }
            case LIST: {
                ListTypeInfo listTypeInfo = (ListTypeInfo)typeInfo;
                return HiveWriteUtils.isWritableType(listTypeInfo.getListElementTypeInfo());
            }
            case STRUCT: {
                StructTypeInfo structTypeInfo = (StructTypeInfo)typeInfo;
                return structTypeInfo.getAllStructFieldTypeInfos().stream().allMatch(HiveWriteUtils::isWritableType);
            }
        }
        return false;
    }

    private static boolean isWritablePrimitiveType(PrimitiveCategory primitiveCategory) {
        switch (primitiveCategory) {
            case BOOLEAN: 
            case LONG: 
            case INT: 
            case SHORT: 
            case BYTE: 
            case FLOAT: 
            case DOUBLE: 
            case STRING: 
            case DATE: 
            case TIMESTAMP: 
            case BINARY: 
            case DECIMAL: 
            case VARCHAR: 
            case CHAR: {
                return true;
            }
        }
        return false;
    }
}

