/*
 * Decompiled with CFR 0.152.
 */
package alluxio.client.file.cache;

import alluxio.AlluxioURI;
import alluxio.client.file.CacheContext;
import alluxio.client.file.FileInStream;
import alluxio.client.file.URIStatus;
import alluxio.client.file.cache.CacheManager;
import alluxio.client.file.cache.PageId;
import alluxio.client.file.cache.context.CachePerThreadContext;
import alluxio.conf.AlluxioConfiguration;
import alluxio.conf.PropertyKey;
import alluxio.exception.AlluxioException;
import alluxio.file.ByteArrayTargetBuffer;
import alluxio.file.ByteBufferTargetBuffer;
import alluxio.file.ReadTargetBuffer;
import alluxio.metrics.MetricKey;
import alluxio.metrics.MetricsSystem;
import alluxio.metrics.MultiDimensionalMetricsSystem;
import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import com.google.common.base.Stopwatch;
import com.google.common.base.Ticker;
import com.google.common.io.Closer;
import java.io.Closeable;
import java.io.IOException;
import java.nio.ByteBuffer;
import java.util.Optional;
import java.util.concurrent.TimeUnit;
import javax.annotation.concurrent.NotThreadSafe;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@NotThreadSafe
public class LocalCacheFileInStream
extends FileInStream {
    private static final Logger LOG = LoggerFactory.getLogger(LocalCacheFileInStream.class);
    protected final long mPageSize;
    private final Closer mCloser = Closer.create();
    private final CacheManager mCacheManager;
    private final boolean mIsDora;
    private final boolean mQuotaEnabled;
    private final CacheContext mCacheContext;
    private final URIStatus mStatus;
    private final FileInStreamOpener mExternalFileInStreamOpener;
    private final int mBufferSize;
    private final boolean mFallbackEnabled;
    private byte[] mBuffer = null;
    private long mBufferStartOffset;
    private long mBufferEndOffset;
    private Optional<FileInStream> mExternalFileInStream;
    private long mPosition = 0L;
    private boolean mClosed = false;
    private boolean mEOF = false;

    public static void registerMetrics() {
        Metrics.registerGauges();
    }

    public LocalCacheFileInStream(URIStatus status, FileInStreamOpener fileOpener, CacheManager cacheManager, AlluxioConfiguration conf, Optional<FileInStream> externalFileInStream) {
        this.mPageSize = conf.getBytes(PropertyKey.USER_CLIENT_CACHE_PAGE_SIZE);
        this.mExternalFileInStreamOpener = fileOpener;
        this.mExternalFileInStream = externalFileInStream;
        this.mCacheManager = cacheManager;
        this.mStatus = status;
        this.mIsDora = conf.getBoolean(PropertyKey.DORA_ENABLED);
        this.mQuotaEnabled = conf.getBoolean(PropertyKey.USER_CLIENT_CACHE_QUOTA_ENABLED);
        this.mCacheContext = this.mQuotaEnabled && status.getCacheContext() != null ? status.getCacheContext() : CacheContext.defaults();
        Metrics.registerGauges();
        this.mBufferSize = (int)conf.getBytes(PropertyKey.USER_CLIENT_CACHE_IN_STREAM_BUFFER_SIZE);
        Preconditions.checkArgument((this.mBufferSize >= 0 ? 1 : 0) != 0, (String)"Buffer size cannot be negative. %s", (long)this.mPageSize);
        if (this.mBufferSize > 0) {
            this.mBuffer = new byte[this.mBufferSize];
        }
        this.mFallbackEnabled = conf.getBoolean(PropertyKey.USER_CLIENT_CACHE_FALLBACK_ENABLED);
    }

    @Override
    public int read(byte[] bytesBuffer, int offset, int length) throws IOException {
        return this.readInternal((ReadTargetBuffer)new ByteArrayTargetBuffer(bytesBuffer, offset), offset, length, ReadType.READ_INTO_BYTE_ARRAY, this.mPosition, false);
    }

    @Override
    public int read(ByteBuffer buffer, int offset, int length) throws IOException {
        int totalBytesRead = this.readInternal((ReadTargetBuffer)new ByteBufferTargetBuffer(buffer), offset, length, ReadType.READ_INTO_BYTE_BUFFER, this.mPosition, false);
        if (totalBytesRead == -1) {
            return -1;
        }
        return totalBytesRead;
    }

    private int bufferedRead(ReadTargetBuffer targetBuffer, int length, ReadType readType, long position, Stopwatch stopwatch) throws IOException {
        if (this.mBuffer == null) {
            return this.localCachedRead(targetBuffer, length, readType, position, stopwatch);
        }
        if (position > this.mBufferStartOffset && position < this.mBufferEndOffset) {
            int lengthToReadFromBuffer = (int)Math.min((long)length, this.mBufferEndOffset - position);
            targetBuffer.writeBytes(this.mBuffer, (int)(position - this.mBufferStartOffset), lengthToReadFromBuffer);
            return lengthToReadFromBuffer;
        }
        if (length >= this.mBufferSize) {
            return this.localCachedRead(targetBuffer, length, readType, position, stopwatch);
        }
        int bytesLoadToBuffer = (int)Math.min((long)this.mBufferSize, this.mStatus.getLength() - position);
        int bytesRead = this.localCachedRead((ReadTargetBuffer)new ByteArrayTargetBuffer(this.mBuffer, 0), bytesLoadToBuffer, readType, position, stopwatch);
        this.mBufferStartOffset = position;
        this.mBufferEndOffset = position + (long)bytesRead;
        int dataReadFromBuffer = Math.min(bytesRead, length);
        targetBuffer.writeBytes(this.mBuffer, 0, dataReadFromBuffer);
        MetricsSystem.meter((String)MetricKey.CLIENT_CACHE_BYTES_READ_IN_STREAM_BUFFER.getName()).mark((long)dataReadFromBuffer);
        return dataReadFromBuffer;
    }

    private int localCachedRead(ReadTargetBuffer bytesBuffer, int length, ReadType readType, long position, Stopwatch stopwatch) throws IOException {
        PageId pageId;
        long currentPage = position / this.mPageSize;
        CacheContext cacheContext = this.mStatus.getCacheContext();
        if (cacheContext != null && cacheContext.getCacheIdentifier() != null) {
            pageId = new PageId(cacheContext.getCacheIdentifier(), currentPage);
        } else {
            String fileId = this.mIsDora ? new AlluxioURI(this.mStatus.getUfsPath()).hash() : Long.toString(this.mStatus.getFileId());
            pageId = new PageId(fileId, currentPage);
        }
        int currentPageOffset = (int)(position % this.mPageSize);
        int bytesLeftInPage = (int)(this.mPageSize - (long)currentPageOffset);
        int bytesToReadInPage = Math.min(bytesLeftInPage, length);
        stopwatch.reset().start();
        int bytesRead = this.mCacheManager.get(pageId, currentPageOffset, bytesToReadInPage, bytesBuffer, this.mCacheContext);
        stopwatch.stop();
        if (bytesRead > 0) {
            MetricsSystem.counter((String)MetricKey.CLIENT_CACHE_HIT_REQUESTS.getName()).inc();
            if (cacheContext != null) {
                cacheContext.incrementCounter(MetricKey.CLIENT_CACHE_PAGE_READ_CACHE_TIME_NS.getMetricName(), CacheContext.StatsUnit.NANO, stopwatch.elapsed(TimeUnit.NANOSECONDS));
            }
            return bytesRead;
        }
        stopwatch.reset().start();
        byte[] page = this.readExternalPage(position, readType);
        stopwatch.stop();
        if (page.length > 0) {
            bytesBuffer.writeBytes(page, currentPageOffset, bytesToReadInPage);
            MetricsSystem.meter((String)MetricKey.CLIENT_CACHE_BYTES_REQUESTED_EXTERNAL.getName()).mark((long)bytesToReadInPage);
            MetricsSystem.counter((String)MetricKey.CLIENT_CACHE_EXTERNAL_REQUESTS.getName()).inc();
            if (cacheContext != null) {
                cacheContext.incrementCounter(MetricKey.CLIENT_CACHE_BYTES_REQUESTED_EXTERNAL.getMetricName(), CacheContext.StatsUnit.BYTE, (long)bytesToReadInPage);
                cacheContext.incrementCounter(MetricKey.CLIENT_CACHE_PAGE_READ_EXTERNAL_TIME_NS.getMetricName(), CacheContext.StatsUnit.NANO, stopwatch.elapsed(TimeUnit.NANOSECONDS));
            }
            this.mCacheManager.put(pageId, page, this.mCacheContext);
        }
        return bytesToReadInPage;
    }

    private int readInternal(ReadTargetBuffer targetBuffer, int offset, int length, ReadType readType, long position, boolean isPositionedRead) throws IOException {
        Preconditions.checkArgument((length >= 0 ? 1 : 0) != 0, (Object)"length should be non-negative");
        Preconditions.checkArgument((offset >= 0 ? 1 : 0) != 0, (Object)"offset should be non-negative");
        Preconditions.checkArgument((position >= 0L ? 1 : 0) != 0, (Object)"position should be non-negative");
        if (length == 0) {
            return 0;
        }
        if (position >= this.mStatus.getLength()) {
            return -1;
        }
        int totalBytesRead = 0;
        long currentPosition = position;
        long lengthToRead = Math.min((long)length, this.mStatus.getLength() - position);
        Stopwatch stopwatch = this.createUnstartedStopwatch();
        while ((long)totalBytesRead < lengthToRead) {
            int bytesRead = this.bufferedRead(targetBuffer, (int)(lengthToRead - (long)totalBytesRead), readType, currentPosition, stopwatch);
            totalBytesRead += bytesRead;
            currentPosition += (long)bytesRead;
            if (isPositionedRead) continue;
            this.mPosition = currentPosition;
        }
        if (totalBytesRead > length || totalBytesRead < length && currentPosition < this.mStatus.getLength()) {
            throw new IOException(String.format("Invalid number of bytes read - bytes to read = %d, actual bytes read = %d, bytes remains in file %d", length, totalBytesRead, this.remaining()));
        }
        return totalBytesRead;
    }

    @VisibleForTesting
    protected Stopwatch createUnstartedStopwatch() {
        return Stopwatch.createUnstarted((Ticker)Ticker.systemTicker());
    }

    @Override
    public long skip(long n) {
        this.checkIfClosed();
        if (n <= 0L) {
            return 0L;
        }
        long toSkip = Math.min(this.remaining(), n);
        this.mPosition += toSkip;
        return toSkip;
    }

    @Override
    public void close() throws IOException {
        this.mCloser.close();
    }

    @Override
    public long remaining() {
        return this.mEOF ? 0L : this.mStatus.getLength() - this.mPosition;
    }

    @Override
    public int positionedRead(long pos, byte[] b, int off, int len) throws IOException {
        if (!CachePerThreadContext.get().getCacheEnabled()) {
            MetricsSystem.meter((String)MetricKey.CLIENT_CACHE_BYTES_REQUESTED_EXTERNAL.getName()).mark((long)len);
            MetricsSystem.counter((String)MetricKey.CLIENT_CACHE_EXTERNAL_REQUESTS.getName()).inc();
            len = this.getExternalFileInStream().positionedRead(pos, b, off, len);
            MultiDimensionalMetricsSystem.EXTERNAL_DATA_READ.inc((long)len);
            return len;
        }
        try {
            return this.readInternal((ReadTargetBuffer)new ByteArrayTargetBuffer(b, off), off, len, ReadType.READ_INTO_BYTE_ARRAY, pos, true);
        }
        catch (IOException | RuntimeException e) {
            LOG.warn("Failed to read from Alluxio's page cache.", (Throwable)e);
            if (this.mFallbackEnabled) {
                MetricsSystem.counter((String)MetricKey.CLIENT_CACHE_POSITION_READ_FALLBACK.getName()).inc();
                len = this.getExternalFileInStream().positionedRead(pos, b, off, len);
                MultiDimensionalMetricsSystem.EXTERNAL_DATA_READ.inc((long)len);
                return len;
            }
            throw e;
        }
    }

    public long getPos() {
        return this.mPosition;
    }

    public void seek(long pos) {
        this.checkIfClosed();
        Preconditions.checkArgument((pos >= 0L ? 1 : 0) != 0, (String)"Seek position is negative: %s", (long)pos);
        Preconditions.checkArgument((pos <= this.mStatus.getLength() ? 1 : 0) != 0, (String)"Seek position (%s) exceeds the length of the file (%s)", (long)pos, (long)this.mStatus.getLength());
        if (pos == this.mPosition) {
            return;
        }
        if (pos < this.mPosition) {
            this.mEOF = false;
        }
        this.mPosition = pos;
    }

    @Override
    public void unbuffer() {
        this.mExternalFileInStream.ifPresent(stream -> stream.unbuffer());
    }

    private void checkIfClosed() {
        Preconditions.checkState((!this.mClosed ? 1 : 0) != 0, (Object)"Cannot operate on a closed stream");
    }

    private FileInStream getExternalFileInStream(long position) throws IOException {
        FileInStream externalFileInStream = this.getExternalFileInStream();
        long pageStart = position - position % this.mPageSize;
        if (externalFileInStream.getPos() != pageStart) {
            externalFileInStream.seek(pageStart);
        }
        return externalFileInStream;
    }

    private FileInStream getExternalFileInStream() throws IOException {
        try {
            if (!this.mExternalFileInStream.isPresent()) {
                FileInStream externalFileInStream = this.mExternalFileInStreamOpener.open(this.mStatus);
                this.mExternalFileInStream = Optional.of(externalFileInStream);
                this.mCloser.register((Closeable)externalFileInStream);
                return externalFileInStream;
            }
            return this.mExternalFileInStream.get();
        }
        catch (AlluxioException e) {
            throw new IOException(e);
        }
    }

    private synchronized byte[] readExternalPage(long position, ReadType readType) throws IOException {
        int totalBytesRead;
        int bytesRead;
        long pageStart = position - position % this.mPageSize;
        FileInStream stream = this.getExternalFileInStream(pageStart);
        int pageSize = (int)Math.min(this.mPageSize, this.mStatus.getLength() - pageStart);
        byte[] page = new byte[pageSize];
        ByteBuffer buffer = readType == ReadType.READ_INTO_BYTE_BUFFER ? ByteBuffer.wrap(page) : null;
        for (totalBytesRead = 0; totalBytesRead < pageSize; totalBytesRead += bytesRead) {
            switch (readType) {
                case READ_INTO_BYTE_ARRAY: {
                    bytesRead = stream.read(page, totalBytesRead, pageSize - totalBytesRead);
                    break;
                }
                case READ_INTO_BYTE_BUFFER: {
                    bytesRead = stream.read(buffer);
                    break;
                }
                default: {
                    throw new IOException("unsupported read type = " + (Object)((Object)readType));
                }
            }
            if (bytesRead <= 0) break;
        }
        MultiDimensionalMetricsSystem.EXTERNAL_DATA_READ.inc((long)totalBytesRead);
        MetricsSystem.meter((String)MetricKey.CLIENT_CACHE_BYTES_READ_EXTERNAL.getName()).mark((long)totalBytesRead);
        if (totalBytesRead != pageSize) {
            throw new IOException("Failed to read complete page from external storage. Bytes read: " + totalBytesRead + " Page size: " + pageSize);
        }
        return page;
    }

    static enum ReadType {
        READ_INTO_BYTE_ARRAY,
        READ_INTO_BYTE_BUFFER;

    }

    private static final class Metrics {
        private Metrics() {
        }

        private static void registerGauges() {
            MetricsSystem.registerGaugeIfAbsent((String)MetricsSystem.getMetricName((String)MetricKey.CLIENT_CACHE_HIT_RATE.getName()), () -> {
                long cacheMisses;
                long cacheHits = MetricsSystem.meter((String)MetricKey.CLIENT_CACHE_BYTES_READ_CACHE.getName()).getCount();
                long total = cacheHits + (cacheMisses = MetricsSystem.meter((String)MetricKey.CLIENT_CACHE_BYTES_REQUESTED_EXTERNAL.getName()).getCount());
                if (total > 0L) {
                    return (double)cacheHits / (1.0 * (double)total);
                }
                return 0;
            });
        }
    }

    public static interface FileInStreamOpener {
        public FileInStream open(URIStatus var1) throws IOException, AlluxioException;
    }
}

