/*
 * Decompiled with CFR 0.152.
 */
package io.trino.plugin.hive.fs;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import com.google.common.base.Throwables;
import com.google.common.cache.Cache;
import com.google.common.util.concurrent.UncheckedExecutionException;
import com.google.errorprone.annotations.concurrent.GuardedBy;
import io.airlift.slice.SizeOf;
import io.trino.filesystem.Location;
import io.trino.filesystem.TrinoFileSystem;
import io.trino.plugin.hive.fs.DirectoryLister;
import io.trino.plugin.hive.fs.RemoteIterator;
import io.trino.plugin.hive.fs.SimpleRemoteIterator;
import io.trino.plugin.hive.fs.TransactionDirectoryListingCacheKey;
import io.trino.plugin.hive.fs.TrinoFileStatus;
import io.trino.plugin.hive.metastore.Partition;
import io.trino.plugin.hive.metastore.Storage;
import io.trino.plugin.hive.metastore.Table;
import jakarta.annotation.Nullable;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Iterator;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.atomic.AtomicLong;

public class TransactionScopeCachingDirectoryLister
implements DirectoryLister {
    private final long transactionId;
    private final Cache<TransactionDirectoryListingCacheKey, FetchingValueHolder> cache;
    private final DirectoryLister delegate;

    public TransactionScopeCachingDirectoryLister(DirectoryLister delegate, long transactionId, Cache<TransactionDirectoryListingCacheKey, FetchingValueHolder> cache) {
        this.delegate = Objects.requireNonNull(delegate, "delegate is null");
        this.transactionId = transactionId;
        this.cache = Objects.requireNonNull(cache, "cache is null");
    }

    @Override
    public RemoteIterator<TrinoFileStatus> listFilesRecursively(TrinoFileSystem fs, Table table, Location location) throws IOException {
        return this.listInternal(fs, table, new TransactionDirectoryListingCacheKey(this.transactionId, location));
    }

    private RemoteIterator<TrinoFileStatus> listInternal(TrinoFileSystem fs, Table table, TransactionDirectoryListingCacheKey cacheKey) throws IOException {
        FetchingValueHolder cachedValueHolder;
        try {
            cachedValueHolder = (FetchingValueHolder)this.cache.get((Object)cacheKey, () -> new FetchingValueHolder(this.createListingRemoteIterator(fs, table, cacheKey)));
        }
        catch (UncheckedExecutionException | ExecutionException e) {
            Throwable throwable = e.getCause();
            Throwables.throwIfInstanceOf((Throwable)throwable, IOException.class);
            Throwables.throwIfUnchecked((Throwable)throwable);
            throw new RuntimeException("Failed to list directory: " + String.valueOf(cacheKey.getPath()), throwable);
        }
        if (cachedValueHolder.isFullyCached()) {
            return new SimpleRemoteIterator(cachedValueHolder.getCachedFiles());
        }
        return this.cachingRemoteIterator(cachedValueHolder, cacheKey);
    }

    private RemoteIterator<TrinoFileStatus> createListingRemoteIterator(TrinoFileSystem fs, Table table, TransactionDirectoryListingCacheKey cacheKey) throws IOException {
        return this.delegate.listFilesRecursively(fs, table, cacheKey.getPath());
    }

    @Override
    public void invalidate(Table table) {
        if (TransactionScopeCachingDirectoryLister.isLocationPresent(table.getStorage())) {
            if (table.getPartitionColumns().isEmpty()) {
                this.cache.invalidate((Object)new TransactionDirectoryListingCacheKey(this.transactionId, Location.of((String)table.getStorage().getLocation())));
            } else {
                this.cache.invalidateAll();
            }
        }
        this.delegate.invalidate(table);
    }

    @Override
    public void invalidate(Partition partition) {
        if (TransactionScopeCachingDirectoryLister.isLocationPresent(partition.getStorage())) {
            this.cache.invalidate((Object)new TransactionDirectoryListingCacheKey(this.transactionId, Location.of((String)partition.getStorage().getLocation())));
        }
        this.delegate.invalidate(partition);
    }

    private RemoteIterator<TrinoFileStatus> cachingRemoteIterator(final FetchingValueHolder cachedValueHolder, final TransactionDirectoryListingCacheKey cacheKey) {
        return new RemoteIterator<TrinoFileStatus>(){
            private int fileIndex;

            @Override
            public boolean hasNext() throws IOException {
                try {
                    boolean hasNext = cachedValueHolder.getCachedFile(this.fileIndex).isPresent();
                    TransactionScopeCachingDirectoryLister.this.cache.asMap().replace(cacheKey, cachedValueHolder, cachedValueHolder);
                    return hasNext;
                }
                catch (Exception exception) {
                    TransactionScopeCachingDirectoryLister.this.cache.invalidate((Object)cacheKey);
                    throw exception;
                }
            }

            @Override
            public TrinoFileStatus next() throws IOException {
                Preconditions.checkState((boolean)this.hasNext());
                return cachedValueHolder.getCachedFile(this.fileIndex++).orElseThrow();
            }
        };
    }

    @VisibleForTesting
    boolean isCached(Location location) {
        return this.isCached(new TransactionDirectoryListingCacheKey(this.transactionId, location));
    }

    @VisibleForTesting
    boolean isCached(TransactionDirectoryListingCacheKey cacheKey) {
        FetchingValueHolder cached = (FetchingValueHolder)this.cache.getIfPresent((Object)cacheKey);
        return cached != null && cached.isFullyCached();
    }

    private static boolean isLocationPresent(Storage storage) {
        return storage.getOptionalLocation().isPresent() && !storage.getLocation().isEmpty();
    }

    static class FetchingValueHolder {
        private static final long ATOMIC_LONG_SIZE = SizeOf.instanceSize(AtomicLong.class);
        private static final long INSTANCE_SIZE = SizeOf.instanceSize(FetchingValueHolder.class);
        private final List<TrinoFileStatus> cachedFiles = Collections.synchronizedList(new ArrayList());
        private final AtomicLong cachedFilesSize = new AtomicLong();
        @Nullable
        @GuardedBy(value="this")
        private RemoteIterator<TrinoFileStatus> fileIterator;
        @Nullable
        @GuardedBy(value="this")
        private Exception exception;

        public FetchingValueHolder(RemoteIterator<TrinoFileStatus> fileIterator) {
            this.fileIterator = Objects.requireNonNull(fileIterator, "fileIterator is null");
        }

        public synchronized boolean isFullyCached() {
            return this.fileIterator == null && this.exception == null;
        }

        public long getRetainedSizeInBytes() {
            return INSTANCE_SIZE + ATOMIC_LONG_SIZE + SizeOf.sizeOfObjectArray((int)this.cachedFiles.size()) + this.cachedFilesSize.get();
        }

        public Iterator<TrinoFileStatus> getCachedFiles() {
            Preconditions.checkState((boolean)this.isFullyCached());
            return this.cachedFiles.iterator();
        }

        public Optional<TrinoFileStatus> getCachedFile(int index) throws IOException {
            int filesSize = this.cachedFiles.size();
            Preconditions.checkArgument((index >= 0 && index <= filesSize ? 1 : 0) != 0, (String)"File index (%s) out of bounds [0, %s]", (int)index, (int)filesSize);
            if (index < filesSize) {
                return Optional.of(this.cachedFiles.get(index));
            }
            return this.fetchNextCachedFile(index);
        }

        private synchronized Optional<TrinoFileStatus> fetchNextCachedFile(int index) throws IOException {
            if (this.exception != null) {
                throw new IOException("Exception while listing directory", this.exception);
            }
            if (index < this.cachedFiles.size()) {
                return Optional.of(this.cachedFiles.get(index));
            }
            try {
                if (this.fileIterator == null || !this.fileIterator.hasNext()) {
                    this.fileIterator = null;
                    return Optional.empty();
                }
                TrinoFileStatus fileStatus = this.fileIterator.next();
                this.cachedFiles.add(fileStatus);
                this.cachedFilesSize.addAndGet(fileStatus.getRetainedSizeInBytes());
                return Optional.of(fileStatus);
            }
            catch (Exception exception) {
                this.fileIterator = null;
                this.exception = exception;
                throw exception;
            }
        }
    }
}

