/*
 * Decompiled with CFR 0.152.
 */
package io.trino.plugin.hive.util;

import com.google.common.base.CharMatcher;
import com.google.common.collect.ImmutableList;
import com.google.common.io.BaseEncoding;
import io.airlift.slice.Slice;
import io.trino.filesystem.Location;
import io.trino.filesystem.TrinoFileSystem;
import io.trino.plugin.hive.HiveErrorCode;
import io.trino.plugin.hive.HiveReadOnlyException;
import io.trino.plugin.hive.HiveType;
import io.trino.plugin.hive.TableType;
import io.trino.plugin.hive.metastore.Database;
import io.trino.plugin.hive.metastore.MetastoreUtil;
import io.trino.plugin.hive.metastore.Partition;
import io.trino.plugin.hive.metastore.ProtectMode;
import io.trino.plugin.hive.metastore.SemiTransactionalHiveMetastore;
import io.trino.plugin.hive.metastore.Storage;
import io.trino.plugin.hive.metastore.Table;
import io.trino.plugin.hive.type.ListTypeInfo;
import io.trino.plugin.hive.type.MapTypeInfo;
import io.trino.plugin.hive.type.PrimitiveCategory;
import io.trino.plugin.hive.type.PrimitiveTypeInfo;
import io.trino.plugin.hive.type.StructTypeInfo;
import io.trino.plugin.hive.type.TypeInfo;
import io.trino.plugin.hive.util.HiveUtil;
import io.trino.spi.ErrorCodeSupplier;
import io.trino.spi.Page;
import io.trino.spi.StandardErrorCode;
import io.trino.spi.TrinoException;
import io.trino.spi.block.Block;
import io.trino.spi.connector.SchemaNotFoundException;
import io.trino.spi.connector.SchemaTableName;
import io.trino.spi.security.ConnectorIdentity;
import io.trino.spi.type.BigintType;
import io.trino.spi.type.BooleanType;
import io.trino.spi.type.CharType;
import io.trino.spi.type.Chars;
import io.trino.spi.type.DateType;
import io.trino.spi.type.DecimalType;
import io.trino.spi.type.Decimals;
import io.trino.spi.type.DoubleType;
import io.trino.spi.type.IntegerType;
import io.trino.spi.type.RealType;
import io.trino.spi.type.SmallintType;
import io.trino.spi.type.TimestampType;
import io.trino.spi.type.TinyintType;
import io.trino.spi.type.Type;
import io.trino.spi.type.VarcharType;
import java.io.IOException;
import java.nio.charset.StandardCharsets;
import java.time.LocalDate;
import java.time.LocalDateTime;
import java.time.ZoneOffset;
import java.time.format.DateTimeFormatter;
import java.time.format.DateTimeFormatterBuilder;
import java.time.temporal.ChronoField;
import java.util.List;
import java.util.Map;
import java.util.Optional;

public final class HiveWriteUtils {
    private static final DateTimeFormatter HIVE_DATE_FORMATTER = DateTimeFormatter.ofPattern("yyyy-MM-dd");
    private static final DateTimeFormatter HIVE_TIMESTAMP_FORMATTER = new DateTimeFormatterBuilder().append(DateTimeFormatter.ofPattern("yyyy-MM-dd HH:mm:ss")).optionalStart().appendFraction(ChronoField.NANO_OF_SECOND, 0, 9, true).optionalEnd().toFormatter();

    private HiveWriteUtils() {
    }

    public static List<String> createPartitionValues(List<Type> partitionColumnTypes, Page partitionColumns, int position) {
        ImmutableList.Builder partitionValues = ImmutableList.builder();
        for (int field = 0; field < partitionColumns.getChannelCount(); ++field) {
            String value = HiveWriteUtils.toPartitionValue(partitionColumnTypes.get(field), partitionColumns.getBlock(field), position);
            if (!CharMatcher.inRange((char)' ', (char)'~').matchesAllOf((CharSequence)value)) {
                String encoded = BaseEncoding.base16().withSeparator(" ", 2).encode(value.getBytes(StandardCharsets.UTF_8));
                throw new TrinoException((ErrorCodeSupplier)HiveErrorCode.HIVE_INVALID_PARTITION_VALUE, "Hive partition keys can only contain printable ASCII characters (0x20 - 0x7E). Invalid value: " + encoded);
            }
            partitionValues.add((Object)value);
        }
        return partitionValues.build();
    }

    private static String toPartitionValue(Type type, Block block, int position) {
        if (block.isNull(position)) {
            return "__HIVE_DEFAULT_PARTITION__";
        }
        if (BooleanType.BOOLEAN.equals((Object)type)) {
            return String.valueOf(BooleanType.BOOLEAN.getBoolean(block, position));
        }
        if (BigintType.BIGINT.equals((Object)type)) {
            return String.valueOf(BigintType.BIGINT.getLong(block, position));
        }
        if (IntegerType.INTEGER.equals((Object)type)) {
            return String.valueOf(IntegerType.INTEGER.getInt(block, position));
        }
        if (SmallintType.SMALLINT.equals((Object)type)) {
            return String.valueOf(SmallintType.SMALLINT.getShort(block, position));
        }
        if (TinyintType.TINYINT.equals((Object)type)) {
            return String.valueOf(TinyintType.TINYINT.getByte(block, position));
        }
        if (RealType.REAL.equals((Object)type)) {
            return String.valueOf(RealType.REAL.getFloat(block, position));
        }
        if (DoubleType.DOUBLE.equals((Object)type)) {
            return String.valueOf(DoubleType.DOUBLE.getDouble(block, position));
        }
        if (type instanceof VarcharType) {
            VarcharType varcharType = (VarcharType)type;
            return varcharType.getSlice(block, position).toStringUtf8();
        }
        if (type instanceof CharType) {
            CharType charType = (CharType)type;
            return Chars.padSpaces((Slice)charType.getSlice(block, position), (CharType)charType).toStringUtf8();
        }
        if (DateType.DATE.equals((Object)type)) {
            return LocalDate.ofEpochDay(DateType.DATE.getInt(block, position)).format(HIVE_DATE_FORMATTER);
        }
        if (TimestampType.TIMESTAMP_MILLIS.equals((Object)type)) {
            long epochMicros = type.getLong(block, position);
            long epochSeconds = Math.floorDiv(epochMicros, 1000000);
            int nanosOfSecond = Math.floorMod(epochMicros, 1000000) * 1000;
            return LocalDateTime.ofEpochSecond(epochSeconds, nanosOfSecond, ZoneOffset.UTC).format(HIVE_TIMESTAMP_FORMATTER);
        }
        if (type instanceof DecimalType) {
            DecimalType decimalType = (DecimalType)type;
            return Decimals.readBigDecimal((DecimalType)decimalType, (Block)block, (int)position).stripTrailingZeros().toPlainString();
        }
        throw new TrinoException((ErrorCodeSupplier)StandardErrorCode.NOT_SUPPORTED, "Unsupported type for partition: " + String.valueOf(type));
    }

    public static void checkTableIsWritable(Table table, boolean writesToNonManagedTablesEnabled) {
        if (table.getTableType().equals(TableType.MATERIALIZED_VIEW.name())) {
            throw new TrinoException((ErrorCodeSupplier)StandardErrorCode.NOT_SUPPORTED, "Cannot write to Hive materialized view");
        }
        if (!writesToNonManagedTablesEnabled && !table.getTableType().equals(TableType.MANAGED_TABLE.name())) {
            throw new TrinoException((ErrorCodeSupplier)StandardErrorCode.NOT_SUPPORTED, "Cannot write to non-managed Hive table");
        }
        HiveWriteUtils.checkWritable(table.getSchemaTableName(), Optional.empty(), MetastoreUtil.getProtectMode(table), table.getParameters(), table.getStorage());
    }

    public static void checkPartitionIsWritable(String partitionName, Partition partition) {
        HiveWriteUtils.checkWritable(partition.getSchemaTableName(), Optional.of(partitionName), MetastoreUtil.getProtectMode(partition), partition.getParameters(), partition.getStorage());
    }

    private static void checkWritable(SchemaTableName tableName, Optional<String> partitionName, ProtectMode protectMode, Map<String, String> parameters, Storage storage) {
        String tablePartitionDescription = "Table '" + String.valueOf(tableName) + "'";
        if (partitionName.isPresent()) {
            tablePartitionDescription = tablePartitionDescription + " partition '" + partitionName.get() + "'";
        }
        MetastoreUtil.verifyOnline(tableName, partitionName, protectMode, parameters);
        if (protectMode.readOnly()) {
            throw new HiveReadOnlyException(tableName, partitionName);
        }
        if (storage.isSkewed()) {
            throw new TrinoException((ErrorCodeSupplier)StandardErrorCode.NOT_SUPPORTED, String.format("Inserting into bucketed tables with skew is not supported. %s", tablePartitionDescription));
        }
    }

    public static Location getTableDefaultLocation(SemiTransactionalHiveMetastore metastore, TrinoFileSystem fileSystem, String schemaName, String tableName) {
        Database database = metastore.getDatabase(schemaName).orElseThrow(() -> new SchemaNotFoundException(schemaName));
        return HiveWriteUtils.getTableDefaultLocation(database, fileSystem, schemaName, tableName);
    }

    public static Location getTableDefaultLocation(Database database, TrinoFileSystem fileSystem, String schemaName, String tableName) {
        Location location = database.getLocation().map(Location::of).orElseThrow(() -> new TrinoException((ErrorCodeSupplier)HiveErrorCode.HIVE_DATABASE_LOCATION_ERROR, String.format("Database '%s' location is not set", schemaName)));
        if (!HiveWriteUtils.directoryExists(fileSystem, location).orElse(true).booleanValue()) {
            throw new TrinoException((ErrorCodeSupplier)HiveErrorCode.HIVE_DATABASE_LOCATION_ERROR, String.format("Database '%s' location does not exist: %s", schemaName, location));
        }
        return location.appendPath(HiveUtil.escapeTableName(tableName));
    }

    public static Optional<Boolean> directoryExists(TrinoFileSystem fileSystem, Location path) {
        try {
            return fileSystem.directoryExists(path);
        }
        catch (IOException e) {
            throw new TrinoException((ErrorCodeSupplier)HiveErrorCode.HIVE_FILESYSTEM_ERROR, "Failed checking path: " + String.valueOf(path), (Throwable)e);
        }
    }

    public static boolean isFileCreatedByQuery(String fileName, String queryId) {
        return fileName.startsWith(queryId) || fileName.endsWith(queryId);
    }

    public static Optional<Location> createTemporaryPath(TrinoFileSystem fileSystem, ConnectorIdentity identity, Location targetPath, String temporaryStagingDirectoryPath) {
        String temporaryPrefix = temporaryStagingDirectoryPath.replace("${USER}", identity.getUser());
        try {
            return fileSystem.createTemporaryDirectory(targetPath, temporaryPrefix, ".hive-staging");
        }
        catch (IOException e) {
            throw new TrinoException((ErrorCodeSupplier)HiveErrorCode.HIVE_FILESYSTEM_ERROR, (Throwable)e);
        }
    }

    public static boolean isWritableType(HiveType hiveType) {
        return HiveWriteUtils.isWritableType(hiveType.getTypeInfo());
    }

    private static boolean isWritableType(TypeInfo typeInfo) {
        switch (typeInfo.getCategory()) {
            case PRIMITIVE: {
                PrimitiveCategory primitiveCategory = ((PrimitiveTypeInfo)typeInfo).getPrimitiveCategory();
                return HiveWriteUtils.isWritablePrimitiveType(primitiveCategory);
            }
            case MAP: {
                MapTypeInfo mapTypeInfo = (MapTypeInfo)typeInfo;
                return HiveWriteUtils.isWritableType(mapTypeInfo.getMapKeyTypeInfo()) && HiveWriteUtils.isWritableType(mapTypeInfo.getMapValueTypeInfo());
            }
            case LIST: {
                ListTypeInfo listTypeInfo = (ListTypeInfo)typeInfo;
                return HiveWriteUtils.isWritableType(listTypeInfo.getListElementTypeInfo());
            }
            case STRUCT: {
                StructTypeInfo structTypeInfo = (StructTypeInfo)typeInfo;
                return structTypeInfo.getAllStructFieldTypeInfos().stream().allMatch(HiveWriteUtils::isWritableType);
            }
        }
        return false;
    }

    private static boolean isWritablePrimitiveType(PrimitiveCategory primitiveCategory) {
        switch (primitiveCategory) {
            case BOOLEAN: 
            case LONG: 
            case INT: 
            case SHORT: 
            case BYTE: 
            case FLOAT: 
            case DOUBLE: 
            case STRING: 
            case DATE: 
            case TIMESTAMP: 
            case BINARY: 
            case DECIMAL: 
            case VARCHAR: 
            case CHAR: {
                return true;
            }
        }
        return false;
    }
}

