/*
 * Decompiled with CFR 0.152.
 */
package io.trino.filesystem.alluxio;

import alluxio.client.file.URIStatus;
import alluxio.client.file.cache.CacheManager;
import alluxio.conf.AlluxioConfiguration;
import com.google.common.base.Verify;
import com.google.common.primitives.Ints;
import com.google.common.primitives.Longs;
import io.opentelemetry.api.trace.Span;
import io.opentelemetry.api.trace.Tracer;
import io.trino.filesystem.Location;
import io.trino.filesystem.TrinoInputFile;
import io.trino.filesystem.TrinoInputStream;
import io.trino.filesystem.alluxio.AlluxioCacheStats;
import io.trino.filesystem.alluxio.AlluxioInputHelper;
import io.trino.filesystem.alluxio.AlluxioTracing;
import io.trino.filesystem.tracing.CacheSystemAttributes;
import java.io.EOFException;
import java.io.IOException;
import java.util.Objects;

public class AlluxioInputStream
extends TrinoInputStream {
    private final TrinoInputFile inputFile;
    private final long fileLength;
    private final Location location;
    private final AlluxioCacheStats statistics;
    private final String key;
    private final AlluxioInputHelper helper;
    private final Tracer tracer;
    private TrinoInputStream externalStream;
    private long position;
    private boolean closed;

    public AlluxioInputStream(Tracer tracer, TrinoInputFile inputFile, String key, URIStatus status, CacheManager cacheManager, AlluxioConfiguration configuration, AlluxioCacheStats statistics) {
        this.tracer = Objects.requireNonNull(tracer, "tracer is null");
        this.inputFile = Objects.requireNonNull(inputFile, "inputFile is null");
        this.fileLength = Objects.requireNonNull(status, "status is null").getLength();
        this.location = inputFile.location();
        this.statistics = Objects.requireNonNull(statistics, "statistics is null");
        this.key = Objects.requireNonNull(key, "key is null");
        this.helper = new AlluxioInputHelper(tracer, inputFile.location(), key, status, cacheManager, configuration, statistics);
    }

    public int available() throws IOException {
        this.ensureOpen();
        return Ints.saturatedCast((long)(this.fileLength - this.position));
    }

    public long getPosition() {
        return this.position;
    }

    private void ensureOpen() throws IOException {
        if (this.closed) {
            throw new IOException("Output stream closed: " + String.valueOf(this.location));
        }
    }

    public int read() throws IOException {
        this.ensureOpen();
        byte[] bytes = new byte[1];
        int n = this.read(bytes, 0, 1);
        if (n == 1) {
            return bytes[0] & 0xFF;
        }
        if (n == -1) {
            return -1;
        }
        throw new IOException(String.format("%d bytes read", n));
    }

    public int read(byte[] bytes, int offset, int length) throws IOException {
        this.ensureOpen();
        Objects.checkFromIndexSize(offset, length, bytes.length);
        if (length == 0) {
            return 0;
        }
        if (this.position >= this.fileLength) {
            return -1;
        }
        int bytesRead = this.doRead(bytes, offset, length);
        this.position += (long)bytesRead;
        return bytesRead;
    }

    private int doRead(byte[] bytes, int offset, int length) throws IOException {
        int bytesRead = this.helper.doCacheRead(this.position, bytes, offset, length);
        return Math.addExact(bytesRead, this.doExternalRead0(this.position + (long)bytesRead, bytes, offset + bytesRead, length - bytesRead));
    }

    private int doExternalRead0(long readPosition, byte[] buffer, int offset, int length) throws IOException {
        if (length == 0) {
            return 0;
        }
        Span span = this.tracer.spanBuilder("Alluxio.readExternal").setAttribute(CacheSystemAttributes.CACHE_KEY, (Object)this.key).setAttribute(CacheSystemAttributes.CACHE_FILE_LOCATION, (Object)this.inputFile.location().toString()).setAttribute(CacheSystemAttributes.CACHE_FILE_READ_SIZE, (Object)length).setAttribute(CacheSystemAttributes.CACHE_FILE_READ_POSITION, (Object)readPosition).startSpan();
        return AlluxioTracing.withTracing(span, () -> this.doExternalReadInternal(readPosition, buffer, offset, length));
    }

    private int doExternalReadInternal(long readPosition, byte[] buffer, int offset, int length) throws IOException {
        Verify.verify((length > 0 ? 1 : 0) != 0, (String)"zero-length or negative read", (Object[])new Object[0]);
        AlluxioInputHelper.PageAlignedRead aligned = this.helper.alignRead(readPosition, length);
        if (this.externalStream == null) {
            this.externalStream = this.inputFile.newStream();
        }
        this.externalStream.seek(aligned.pageStart());
        byte[] readBuffer = new byte[aligned.length()];
        int externalBytesRead = this.externalStream.readNBytes(readBuffer, 0, aligned.length());
        if (externalBytesRead < 0) {
            throw new IOException("Unexpected end of stream");
        }
        Verify.verify((aligned.length() == externalBytesRead ? 1 : 0) != 0, (String)"invalid number of external bytes read", (Object[])new Object[0]);
        this.helper.putCache(aligned.pageStart(), aligned.pageEnd(), readBuffer, externalBytesRead);
        int bytesToCopy = Math.min(length, Integer.max(externalBytesRead - aligned.pageOffset(), 0));
        System.arraycopy(readBuffer, aligned.pageOffset(), buffer, offset, bytesToCopy);
        this.statistics.recordExternalRead(externalBytesRead);
        return bytesToCopy;
    }

    public long skip(long n) throws IOException {
        this.ensureOpen();
        n = Longs.constrainToRange((long)n, (long)0L, (long)(this.fileLength - this.position));
        this.position += n;
        return n;
    }

    public void skipNBytes(long n) throws IOException {
        long position;
        this.ensureOpen();
        if (n <= 0L) {
            return;
        }
        try {
            position = Math.addExact(this.position, n);
        }
        catch (ArithmeticException e) {
            throw new EOFException("Unable to skip %s bytes (position=%s, fileSize=%s): %s".formatted(n, this.position, this.fileLength, this.location));
        }
        if (position > this.fileLength) {
            throw new EOFException("Unable to skip %s bytes (position=%s, fileSize=%s): %s".formatted(n, this.position, this.fileLength, this.location));
        }
        this.position = position;
    }

    public void seek(long position) throws IOException {
        this.ensureOpen();
        if (position < 0L) {
            throw new IOException("Negative seek offset");
        }
        if (position > this.fileLength) {
            throw new IOException("Cannot seek to %s. File size is %s: %s".formatted(position, this.fileLength, this.location));
        }
        this.position = position;
    }

    public void close() throws IOException {
        if (!this.closed) {
            this.closed = true;
            if (this.externalStream != null) {
                this.externalStream.close();
                this.externalStream = null;
            }
        }
    }
}

