/*
 * Decompiled with CFR 0.152.
 */
package io.trino.filesystem.azure;

import com.azure.core.http.HttpClient;
import com.azure.core.http.rest.PagedIterable;
import com.azure.core.util.ClientOptions;
import com.azure.core.util.TracingOptions;
import com.azure.storage.blob.BlobClient;
import com.azure.storage.blob.BlobContainerClient;
import com.azure.storage.blob.BlobContainerClientBuilder;
import com.azure.storage.blob.models.BlobItem;
import com.azure.storage.blob.models.ListBlobsOptions;
import com.azure.storage.common.Utility;
import com.azure.storage.file.datalake.DataLakeDirectoryClient;
import com.azure.storage.file.datalake.DataLakeFileClient;
import com.azure.storage.file.datalake.DataLakeFileSystemClient;
import com.azure.storage.file.datalake.DataLakeServiceClient;
import com.azure.storage.file.datalake.DataLakeServiceClientBuilder;
import com.azure.storage.file.datalake.models.DataLakeRequestConditions;
import com.azure.storage.file.datalake.models.DataLakeStorageException;
import com.azure.storage.file.datalake.models.ListPathsOptions;
import com.azure.storage.file.datalake.models.PathItem;
import com.azure.storage.file.datalake.options.DataLakePathDeleteOptions;
import com.google.common.base.Preconditions;
import com.google.common.collect.ImmutableSet;
import io.airlift.units.DataSize;
import io.trino.filesystem.FileIterator;
import io.trino.filesystem.Location;
import io.trino.filesystem.TrinoFileSystem;
import io.trino.filesystem.TrinoInputFile;
import io.trino.filesystem.TrinoOutputFile;
import io.trino.filesystem.azure.AzureAuth;
import io.trino.filesystem.azure.AzureBlobFileIterator;
import io.trino.filesystem.azure.AzureDataLakeFileIterator;
import io.trino.filesystem.azure.AzureInputFile;
import io.trino.filesystem.azure.AzureLocation;
import io.trino.filesystem.azure.AzureOutputFile;
import io.trino.filesystem.azure.AzureUtils;
import java.io.IOException;
import java.util.Objects;
import java.util.Optional;
import java.util.OptionalLong;
import java.util.Set;
import java.util.UUID;
import java.util.function.Predicate;

public class AzureFileSystem
implements TrinoFileSystem {
    private final HttpClient httpClient;
    private final TracingOptions tracingOptions;
    private final AzureAuth azureAuth;
    private final int readBlockSizeBytes;
    private final long writeBlockSizeBytes;
    private final int maxWriteConcurrency;
    private final long maxSingleUploadSizeBytes;

    public AzureFileSystem(HttpClient httpClient, TracingOptions tracingOptions, AzureAuth azureAuth, DataSize readBlockSize, DataSize writeBlockSize, int maxWriteConcurrency, DataSize maxSingleUploadSize) {
        this.httpClient = Objects.requireNonNull(httpClient, "httpClient is null");
        this.tracingOptions = Objects.requireNonNull(tracingOptions, "tracingOptions is null");
        this.azureAuth = Objects.requireNonNull(azureAuth, "azureAuth is null");
        this.readBlockSizeBytes = Math.toIntExact(readBlockSize.toBytes());
        this.writeBlockSizeBytes = writeBlockSize.toBytes();
        Preconditions.checkArgument((maxWriteConcurrency >= 0 ? 1 : 0) != 0, (Object)"maxWriteConcurrency is negative");
        this.maxWriteConcurrency = maxWriteConcurrency;
        this.maxSingleUploadSizeBytes = maxSingleUploadSize.toBytes();
    }

    public TrinoInputFile newInputFile(Location location) {
        AzureLocation azureLocation = new AzureLocation(location);
        BlobClient client = this.createBlobClient(azureLocation);
        return new AzureInputFile(azureLocation, OptionalLong.empty(), client, this.readBlockSizeBytes);
    }

    public TrinoInputFile newInputFile(Location location, long length) {
        AzureLocation azureLocation = new AzureLocation(location);
        BlobClient client = this.createBlobClient(azureLocation);
        return new AzureInputFile(azureLocation, OptionalLong.of(length), client, this.readBlockSizeBytes);
    }

    public TrinoOutputFile newOutputFile(Location location) {
        AzureLocation azureLocation = new AzureLocation(location);
        BlobClient client = this.createBlobClient(azureLocation);
        return new AzureOutputFile(azureLocation, client, this.writeBlockSizeBytes, this.maxWriteConcurrency, this.maxSingleUploadSizeBytes);
    }

    public void deleteFile(Location location) throws IOException {
        location.verifyValidFileLocation();
        AzureLocation azureLocation = new AzureLocation(location);
        BlobClient client = this.createBlobClient(azureLocation);
        try {
            client.delete();
        }
        catch (RuntimeException e) {
            if (AzureUtils.isFileNotFoundException(e)) {
                return;
            }
            throw AzureUtils.handleAzureException(e, "deleting file", azureLocation);
        }
    }

    public void deleteDirectory(Location location) throws IOException {
        AzureLocation azureLocation = new AzureLocation(location);
        try {
            if (this.isHierarchicalNamespaceEnabled(azureLocation)) {
                this.deleteGen2Directory(azureLocation);
            } else {
                this.deleteBlobDirectory(azureLocation);
            }
        }
        catch (RuntimeException e) {
            throw AzureUtils.handleAzureException(e, "deleting directory", azureLocation);
        }
    }

    private void deleteGen2Directory(AzureLocation location) throws IOException {
        DataLakeFileSystemClient fileSystemClient = this.createFileSystemClient(location);
        DataLakePathDeleteOptions deleteRecursiveOptions = new DataLakePathDeleteOptions().setIsRecursive(true);
        if (location.path().isEmpty()) {
            for (PathItem pathItem : fileSystemClient.listPaths()) {
                if (pathItem.isDirectory()) {
                    fileSystemClient.deleteDirectoryIfExistsWithResponse(pathItem.getName(), deleteRecursiveOptions, null, null);
                    continue;
                }
                fileSystemClient.deleteFileIfExists(pathItem.getName());
            }
        } else {
            DataLakeDirectoryClient directoryClient = AzureFileSystem.createDirectoryClient(fileSystemClient, location.path());
            if (directoryClient.exists().booleanValue()) {
                if (!directoryClient.getProperties().isDirectory().booleanValue()) {
                    throw new IOException("Location is not a directory: " + String.valueOf(location));
                }
                directoryClient.deleteIfExistsWithResponse(deleteRecursiveOptions, null, null);
            }
        }
    }

    private void deleteBlobDirectory(AzureLocation location) {
        BlobContainerClient blobContainerClient = this.createBlobContainerClient(location);
        PagedIterable blobItems = blobContainerClient.listBlobs(new ListBlobsOptions().setPrefix(location.directoryPath()), null);
        for (BlobItem item : blobItems) {
            String blobUrl = Utility.urlEncode((String)item.getName());
            blobContainerClient.getBlobClient(blobUrl).deleteIfExists();
        }
    }

    public void renameFile(Location source, Location target) throws IOException {
        source.verifyValidFileLocation();
        target.verifyValidFileLocation();
        AzureLocation sourceLocation = new AzureLocation(source);
        AzureLocation targetLocation = new AzureLocation(target);
        if (!sourceLocation.account().equals(targetLocation.account())) {
            throw new IOException("Cannot rename across storage accounts");
        }
        if (!Objects.equals(sourceLocation.container(), targetLocation.container())) {
            throw new IOException("Cannot rename across storage account containers");
        }
        this.renameGen2File(sourceLocation, targetLocation);
    }

    private void renameGen2File(AzureLocation source, AzureLocation target) throws IOException {
        try {
            DataLakeFileSystemClient fileSystemClient = this.createFileSystemClient(source);
            DataLakeFileClient dataLakeFileClient = AzureFileSystem.createFileClient(fileSystemClient, source.path());
            if (dataLakeFileClient.getProperties().isDirectory().booleanValue()) {
                throw new IOException("Rename file from %s to %s, source is a directory".formatted(source, target));
            }
            AzureFileSystem.createDirectoryIfNotExists(fileSystemClient, target.location().parentDirectory().path());
            dataLakeFileClient.renameWithResponse(null, Utility.urlEncode((String)target.path()), null, new DataLakeRequestConditions().setIfNoneMatch("*"), null, null);
        }
        catch (RuntimeException e) {
            throw new IOException("Rename file from %s to %s failed".formatted(source, target), e);
        }
    }

    public FileIterator listFiles(Location location) throws IOException {
        AzureLocation azureLocation = new AzureLocation(location);
        try {
            return this.isHierarchicalNamespaceEnabled(azureLocation) ? this.listGen2Files(azureLocation) : this.listBlobFiles(azureLocation);
        }
        catch (RuntimeException e) {
            throw AzureUtils.handleAzureException(e, "listing files", azureLocation);
        }
    }

    private FileIterator listGen2Files(AzureLocation location) throws IOException {
        PagedIterable pathItems;
        DataLakeFileSystemClient fileSystemClient = this.createFileSystemClient(location);
        if (location.path().isEmpty()) {
            pathItems = fileSystemClient.listPaths(new ListPathsOptions().setRecursive(true), null);
        } else {
            DataLakeDirectoryClient directoryClient = AzureFileSystem.createDirectoryClient(fileSystemClient, location.path());
            if (!directoryClient.exists().booleanValue()) {
                return FileIterator.empty();
            }
            if (!directoryClient.getProperties().isDirectory().booleanValue()) {
                throw new IOException("Location is not a directory: " + String.valueOf(location));
            }
            pathItems = directoryClient.listPaths(true, false, null, null);
        }
        return new AzureDataLakeFileIterator(location, pathItems.stream().filter(Predicate.not(PathItem::isDirectory)).iterator());
    }

    private FileIterator listBlobFiles(AzureLocation location) {
        PagedIterable blobItems = this.createBlobContainerClient(location).listBlobs(new ListBlobsOptions().setPrefix(location.directoryPath()), null);
        return new AzureBlobFileIterator(location, blobItems.iterator());
    }

    public Optional<Boolean> directoryExists(Location location) throws IOException {
        AzureLocation azureLocation = new AzureLocation(location);
        if (location.path().isEmpty()) {
            return Optional.of(true);
        }
        if (!this.isHierarchicalNamespaceEnabled(azureLocation)) {
            if (this.listFiles(location).hasNext()) {
                return Optional.of(true);
            }
            return Optional.empty();
        }
        try {
            DataLakeFileSystemClient fileSystemClient = this.createFileSystemClient(azureLocation);
            DataLakeFileClient fileClient = AzureFileSystem.createFileClient(fileSystemClient, azureLocation.path());
            return Optional.of(fileClient.getProperties().isDirectory());
        }
        catch (DataLakeStorageException e) {
            if (e.getStatusCode() == 404) {
                return Optional.of(false);
            }
            throw AzureUtils.handleAzureException((RuntimeException)((Object)e), "checking directory existence", azureLocation);
        }
        catch (RuntimeException e) {
            throw AzureUtils.handleAzureException(e, "checking directory existence", azureLocation);
        }
    }

    public void createDirectory(Location location) throws IOException {
        AzureLocation azureLocation = new AzureLocation(location);
        if (!this.isHierarchicalNamespaceEnabled(azureLocation)) {
            return;
        }
        try {
            DataLakeFileSystemClient fileSystemClient = this.createFileSystemClient(azureLocation);
            DataLakeDirectoryClient directoryClient = AzureFileSystem.createDirectoryIfNotExists(fileSystemClient, azureLocation.path());
            if (!directoryClient.getProperties().isDirectory().booleanValue()) {
                throw new IOException("Location is not a directory: " + String.valueOf(azureLocation));
            }
        }
        catch (RuntimeException e) {
            throw AzureUtils.handleAzureException(e, "creating directory", azureLocation);
        }
    }

    public void renameDirectory(Location source, Location target) throws IOException {
        AzureLocation sourceLocation = new AzureLocation(source);
        AzureLocation targetLocation = new AzureLocation(target);
        if (!sourceLocation.account().equals(targetLocation.account())) {
            throw new IOException("Cannot rename across storage accounts");
        }
        if (!Objects.equals(sourceLocation.container(), targetLocation.container())) {
            throw new IOException("Cannot rename across storage account containers");
        }
        if (!this.isHierarchicalNamespaceEnabled(sourceLocation)) {
            throw new IOException("Azure non-hierarchical does not support directory renames");
        }
        if (sourceLocation.path().isEmpty() || targetLocation.path().isEmpty()) {
            throw new IOException("Cannot rename %s to %s".formatted(source, target));
        }
        try {
            DataLakeFileSystemClient fileSystemClient = this.createFileSystemClient(sourceLocation);
            DataLakeDirectoryClient directoryClient = AzureFileSystem.createDirectoryClient(fileSystemClient, sourceLocation.path());
            if (!directoryClient.exists().booleanValue()) {
                throw new IOException("Source directory does not exist: " + String.valueOf(source));
            }
            if (!directoryClient.getProperties().isDirectory().booleanValue()) {
                throw new IOException("Source is not a directory: " + String.valueOf(source));
            }
            directoryClient.rename(null, Utility.urlEncode((String)targetLocation.path()));
        }
        catch (RuntimeException e) {
            throw new IOException("Rename directory from %s to %s failed".formatted(source, target), e);
        }
    }

    public Set<Location> listDirectories(Location location) throws IOException {
        AzureLocation azureLocation = new AzureLocation(location);
        try {
            return this.isHierarchicalNamespaceEnabled(azureLocation) ? this.listGen2Directories(azureLocation) : this.listBlobDirectories(azureLocation);
        }
        catch (RuntimeException e) {
            throw AzureUtils.handleAzureException(e, "listing files", azureLocation);
        }
    }

    public Optional<Location> createTemporaryDirectory(Location targetPath, String temporaryPrefix, String relativePrefix) throws IOException {
        Location temporary;
        AzureLocation azureLocation = new AzureLocation(targetPath);
        if (!this.isHierarchicalNamespaceEnabled(azureLocation)) {
            return Optional.empty();
        }
        if (temporaryPrefix.startsWith("/")) {
            String prefix = temporaryPrefix;
            while (prefix.startsWith("/")) {
                prefix = prefix.substring(1);
            }
            temporary = azureLocation.baseLocation().appendPath(prefix);
        } else {
            temporary = targetPath.appendPath(temporaryPrefix);
        }
        temporary = temporary.appendPath(UUID.randomUUID().toString());
        this.createDirectory(temporary);
        return Optional.of(temporary);
    }

    private Set<Location> listGen2Directories(AzureLocation location) throws IOException {
        PagedIterable pathItems;
        DataLakeFileSystemClient fileSystemClient = this.createFileSystemClient(location);
        if (location.path().isEmpty()) {
            pathItems = fileSystemClient.listPaths();
        } else {
            DataLakeDirectoryClient directoryClient = AzureFileSystem.createDirectoryClient(fileSystemClient, location.path());
            if (!directoryClient.exists().booleanValue()) {
                return ImmutableSet.of();
            }
            if (!directoryClient.getProperties().isDirectory().booleanValue()) {
                throw new IOException("Location is not a directory: " + String.valueOf(location));
            }
            pathItems = directoryClient.listPaths(false, false, null, null);
        }
        Location baseLocation = location.baseLocation();
        return (Set)pathItems.stream().filter(PathItem::isDirectory).map(item -> baseLocation.appendPath(item.getName() + "/")).collect(ImmutableSet.toImmutableSet());
    }

    private Set<Location> listBlobDirectories(AzureLocation location) {
        Location baseLocation = location.baseLocation();
        return (Set)this.createBlobContainerClient(location).listBlobsByHierarchy(location.directoryPath()).stream().filter(BlobItem::isPrefix).map(item -> baseLocation.appendPath(item.getName())).collect(ImmutableSet.toImmutableSet());
    }

    private boolean isHierarchicalNamespaceEnabled(AzureLocation location) throws IOException {
        try {
            DataLakeFileSystemClient fileSystemClient = this.createFileSystemClient(location);
            return fileSystemClient.getDirectoryClient("/").exists();
        }
        catch (RuntimeException e) {
            throw new IOException("Checking whether hierarchical namespace is enabled for the location %s failed".formatted(location), e);
        }
    }

    private BlobClient createBlobClient(AzureLocation location) {
        return this.createBlobContainerClient(location).getBlobClient(Utility.urlEncode((String)location.path()));
    }

    private BlobContainerClient createBlobContainerClient(AzureLocation location) {
        Objects.requireNonNull(location, "location is null");
        BlobContainerClientBuilder builder = new BlobContainerClientBuilder().httpClient(this.httpClient).clientOptions(new ClientOptions().setTracingOptions(this.tracingOptions)).endpoint(String.format("https://%s.blob.core.windows.net", location.account()));
        this.azureAuth.setAuth(location.account(), builder);
        location.container().ifPresent(arg_0 -> ((BlobContainerClientBuilder)builder).containerName(arg_0));
        return builder.buildClient();
    }

    private DataLakeFileSystemClient createFileSystemClient(AzureLocation location) {
        Objects.requireNonNull(location, "location is null");
        DataLakeServiceClientBuilder builder = new DataLakeServiceClientBuilder().httpClient(this.httpClient).clientOptions(new ClientOptions().setTracingOptions(this.tracingOptions)).endpoint(String.format("https://%s.dfs.core.windows.net", location.account()));
        this.azureAuth.setAuth(location.account(), builder);
        DataLakeServiceClient client = builder.buildClient();
        DataLakeFileSystemClient fileSystemClient = client.getFileSystemClient(location.container().orElseThrow());
        if (!fileSystemClient.exists()) {
            throw new IllegalArgumentException();
        }
        return fileSystemClient;
    }

    private static DataLakeDirectoryClient createDirectoryClient(DataLakeFileSystemClient fileSystemClient, String directoryName) {
        return fileSystemClient.getDirectoryClient(Utility.urlEncode((String)directoryName));
    }

    private static DataLakeFileClient createFileClient(DataLakeFileSystemClient fileSystemClient, String fileName) {
        return fileSystemClient.getFileClient(Utility.urlEncode((String)fileName));
    }

    private static DataLakeDirectoryClient createDirectoryIfNotExists(DataLakeFileSystemClient fileSystemClient, String name) {
        return fileSystemClient.createDirectoryIfNotExists(Utility.urlEncode((String)name));
    }
}

