/*
 * Decompiled with CFR 0.152.
 */
package io.trino.plugin.hive.metastore.cache;

import com.google.common.base.Preconditions;
import com.google.common.base.Throwables;
import com.google.common.cache.Cache;
import com.google.common.cache.CacheBuilder;
import com.google.common.cache.CacheLoader;
import com.google.common.cache.CacheStats;
import com.google.common.cache.LoadingCache;
import com.google.common.math.LongMath;
import com.google.common.util.concurrent.UncheckedExecutionException;
import io.airlift.concurrent.Threads;
import io.airlift.units.Duration;
import io.trino.collect.cache.SafeCaches;
import io.trino.plugin.base.CatalogName;
import io.trino.plugin.hive.metastore.HiveMetastore;
import io.trino.plugin.hive.metastore.HiveMetastoreFactory;
import io.trino.plugin.hive.metastore.cache.CachingHiveMetastore;
import io.trino.plugin.hive.metastore.cache.CachingHiveMetastoreConfig;
import io.trino.plugin.hive.metastore.cache.ImpersonationCachingConfig;
import io.trino.plugin.hive.metastore.cache.ReentrantBoundedExecutor;
import io.trino.spi.NodeManager;
import io.trino.spi.TrinoException;
import io.trino.spi.security.ConnectorIdentity;
import java.util.Objects;
import java.util.Optional;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.TimeUnit;
import java.util.function.Function;
import javax.annotation.PostConstruct;
import javax.annotation.PreDestroy;
import javax.inject.Inject;
import org.weakref.jmx.Flatten;
import org.weakref.jmx.Managed;
import org.weakref.jmx.Nested;

public class SharedHiveMetastoreCache {
    private final boolean enabled;
    private final CatalogName catalogName;
    private final Duration metastoreCacheTtl;
    private final Optional<Duration> metastoreRefreshInterval;
    private final long metastoreCacheMaximumSize;
    private final int maxMetastoreRefreshThreads;
    private final Duration userMetastoreCacheTtl;
    private final long userMetastoreCacheMaximumSize;
    private final boolean metastorePartitionCacheEnabled;
    private ExecutorService executorService;

    @Inject
    public SharedHiveMetastoreCache(CatalogName catalogName, NodeManager nodeManager, CachingHiveMetastoreConfig config, ImpersonationCachingConfig impersonationCachingConfig) {
        Objects.requireNonNull(nodeManager, "nodeManager is null");
        Objects.requireNonNull(config, "config is null");
        Objects.requireNonNull(catalogName, "catalogName is null");
        this.catalogName = catalogName;
        this.maxMetastoreRefreshThreads = config.getMaxMetastoreRefreshThreads();
        this.metastoreCacheTtl = config.getMetastoreCacheTtl();
        this.metastoreRefreshInterval = config.getMetastoreRefreshInterval();
        this.metastoreCacheMaximumSize = config.getMetastoreCacheMaximumSize();
        this.metastorePartitionCacheEnabled = config.isPartitionCacheEnabled();
        this.userMetastoreCacheTtl = impersonationCachingConfig.getUserMetastoreCacheTtl();
        this.userMetastoreCacheMaximumSize = impersonationCachingConfig.getUserMetastoreCacheMaximumSize();
        this.enabled = nodeManager.getCurrentNode().isCoordinator() && this.metastoreCacheTtl.toMillis() > 0L && this.metastoreCacheMaximumSize > 0L;
    }

    @PostConstruct
    public void start() {
        if (this.enabled) {
            this.executorService = Executors.newCachedThreadPool(Threads.daemonThreadsNamed((String)("hive-metastore-" + this.catalogName + "-%s")));
        }
    }

    @PreDestroy
    public void stop() {
        if (this.executorService != null) {
            this.executorService.shutdownNow();
            this.executorService = null;
        }
    }

    public boolean isEnabled() {
        return this.enabled;
    }

    public HiveMetastoreFactory createCachingHiveMetastoreFactory(HiveMetastoreFactory metastoreFactory) {
        if (!this.enabled) {
            return metastoreFactory;
        }
        if (metastoreFactory.isImpersonationEnabled()) {
            if (this.userMetastoreCacheMaximumSize == 0L || this.userMetastoreCacheTtl.toMillis() == 0L) {
                return metastoreFactory;
            }
            return new ImpersonationCachingHiveMetastoreFactory(metastoreFactory);
        }
        CachingHiveMetastore cachingHiveMetastore = CachingHiveMetastore.cachingHiveMetastore(metastoreFactory.createMetastore(Optional.empty()), new ReentrantBoundedExecutor(this.executorService, this.maxMetastoreRefreshThreads), this.metastoreCacheTtl, this.metastoreRefreshInterval, this.metastoreCacheMaximumSize, this.metastorePartitionCacheEnabled);
        return new CachingHiveMetastoreFactory(cachingHiveMetastore);
    }

    public class ImpersonationCachingHiveMetastoreFactory
    implements HiveMetastoreFactory {
        private final HiveMetastoreFactory metastoreFactory;
        private final LoadingCache<String, CachingHiveMetastore> cache;

        public ImpersonationCachingHiveMetastoreFactory(HiveMetastoreFactory metastoreFactory) {
            this.metastoreFactory = metastoreFactory;
            this.cache = SafeCaches.buildNonEvictableCache((CacheBuilder)CacheBuilder.newBuilder().expireAfterWrite(SharedHiveMetastoreCache.this.userMetastoreCacheTtl.toMillis(), TimeUnit.MILLISECONDS).maximumSize(SharedHiveMetastoreCache.this.userMetastoreCacheMaximumSize), (CacheLoader)CacheLoader.from(this::createUserCachingMetastore));
        }

        @Override
        public boolean isImpersonationEnabled() {
            return true;
        }

        @Override
        public HiveMetastore createMetastore(Optional<ConnectorIdentity> identity) {
            Preconditions.checkArgument((boolean)identity.isPresent(), (Object)"Identity must be present for impersonation cache");
            try {
                return (HiveMetastore)this.cache.getUnchecked((Object)identity.get().getUser());
            }
            catch (UncheckedExecutionException e) {
                Throwables.throwIfInstanceOf((Throwable)e.getCause(), TrinoException.class);
                throw e;
            }
        }

        private CachingHiveMetastore createUserCachingMetastore(String user) {
            ConnectorIdentity identity = ConnectorIdentity.ofUser((String)user);
            return CachingHiveMetastore.cachingHiveMetastore(this.metastoreFactory.createMetastore(Optional.of(identity)), new ReentrantBoundedExecutor(SharedHiveMetastoreCache.this.executorService, SharedHiveMetastoreCache.this.maxMetastoreRefreshThreads), SharedHiveMetastoreCache.this.metastoreCacheTtl, SharedHiveMetastoreCache.this.metastoreRefreshInterval, SharedHiveMetastoreCache.this.metastoreCacheMaximumSize, SharedHiveMetastoreCache.this.metastorePartitionCacheEnabled);
        }

        @Managed
        public void flushCache() {
            this.cache.asMap().values().forEach(CachingHiveMetastore::flushCache);
        }

        @Managed
        @Nested
        public AggregateCacheStatsMBean getDatabaseStats() {
            return new AggregateCacheStatsMBean(CachingHiveMetastore::getDatabaseCache);
        }

        @Managed
        @Nested
        public AggregateCacheStatsMBean getDatabaseNamesStats() {
            return new AggregateCacheStatsMBean(CachingHiveMetastore::getDatabaseNamesCache);
        }

        @Managed
        @Nested
        public AggregateCacheStatsMBean getTableStats() {
            return new AggregateCacheStatsMBean(CachingHiveMetastore::getTableCache);
        }

        @Managed
        @Nested
        public AggregateCacheStatsMBean getTableNamesStats() {
            return new AggregateCacheStatsMBean(CachingHiveMetastore::getTableNamesCache);
        }

        @Managed
        @Nested
        public AggregateCacheStatsMBean getTableWithParameterStats() {
            return new AggregateCacheStatsMBean(CachingHiveMetastore::getTablesWithParameterCache);
        }

        @Managed
        @Nested
        public AggregateCacheStatsMBean getTableStatisticsStats() {
            return new AggregateCacheStatsMBean(CachingHiveMetastore::getTableStatisticsCache);
        }

        @Managed
        @Nested
        public AggregateCacheStatsMBean getPartitionStatisticsStats() {
            return new AggregateCacheStatsMBean(CachingHiveMetastore::getPartitionStatisticsCache);
        }

        @Managed
        @Nested
        public AggregateCacheStatsMBean getViewNamesStats() {
            return new AggregateCacheStatsMBean(CachingHiveMetastore::getViewNamesCache);
        }

        @Managed
        @Nested
        public AggregateCacheStatsMBean getPartitionStats() {
            return new AggregateCacheStatsMBean(CachingHiveMetastore::getPartitionCache);
        }

        @Managed
        @Nested
        public AggregateCacheStatsMBean getPartitionFilterStats() {
            return new AggregateCacheStatsMBean(CachingHiveMetastore::getPartitionFilterCache);
        }

        @Managed
        @Nested
        public AggregateCacheStatsMBean getTablePrivilegesStats() {
            return new AggregateCacheStatsMBean(CachingHiveMetastore::getTablePrivilegesCache);
        }

        @Managed
        @Nested
        public AggregateCacheStatsMBean getRolesStats() {
            return new AggregateCacheStatsMBean(CachingHiveMetastore::getRolesCache);
        }

        @Managed
        @Nested
        public AggregateCacheStatsMBean getRoleGrantsStats() {
            return new AggregateCacheStatsMBean(CachingHiveMetastore::getRoleGrantsCache);
        }

        @Managed
        @Nested
        public AggregateCacheStatsMBean getGrantedPrincipalsStats() {
            return new AggregateCacheStatsMBean(CachingHiveMetastore::getGrantedPrincipalsCache);
        }

        @Managed
        @Nested
        public AggregateCacheStatsMBean getConfigValuesStats() {
            return new AggregateCacheStatsMBean(CachingHiveMetastore::getConfigValuesCache);
        }

        public class AggregateCacheStatsMBean {
            private final Function<CachingHiveMetastore, Cache<?, ?>> cacheExtractor;

            public AggregateCacheStatsMBean(Function<CachingHiveMetastore, Cache<?, ?>> cacheExtractor) {
                this.cacheExtractor = Objects.requireNonNull(cacheExtractor, "cacheExtractor is null");
            }

            @Managed
            public long size() {
                return ImpersonationCachingHiveMetastoreFactory.this.cache.asMap().values().stream().map(this.cacheExtractor).mapToLong(Cache::size).reduce(0L, LongMath::saturatedAdd);
            }

            @Managed
            public Double getHitRate() {
                return this.aggregateStats().getHitRate();
            }

            @Managed
            public Double getMissRate() {
                return this.aggregateStats().getMissRate();
            }

            @Managed
            public long getRequestCount() {
                return this.aggregateStats().getRequestCount();
            }

            private CacheStatsAggregator aggregateStats() {
                CacheStatsAggregator aggregator = new CacheStatsAggregator();
                for (CachingHiveMetastore metastore : ImpersonationCachingHiveMetastoreFactory.this.cache.asMap().values()) {
                    aggregator.add(this.cacheExtractor.apply(metastore).stats());
                }
                return aggregator;
            }
        }
    }

    public static class CachingHiveMetastoreFactory
    implements HiveMetastoreFactory {
        private final CachingHiveMetastore metastore;

        private CachingHiveMetastoreFactory(CachingHiveMetastore metastore) {
            this.metastore = Objects.requireNonNull(metastore, "metastore is null");
        }

        @Override
        public boolean isImpersonationEnabled() {
            return false;
        }

        @Override
        public HiveMetastore createMetastore(Optional<ConnectorIdentity> identity) {
            return this.metastore;
        }

        @Nested
        @Flatten
        public CachingHiveMetastore getMetastore() {
            return this.metastore;
        }
    }

    private static final class CacheStatsAggregator {
        private long requestCount;
        private long hitCount;
        private long missCount;

        private CacheStatsAggregator() {
        }

        void add(CacheStats stats) {
            this.requestCount += stats.requestCount();
            this.hitCount += stats.hitCount();
            this.missCount += stats.missCount();
        }

        public long getRequestCount() {
            return this.requestCount;
        }

        public long getHitCount() {
            return this.hitCount;
        }

        public long getMissCount() {
            return this.missCount;
        }

        public double getHitRate() {
            return this.requestCount == 0L ? 1.0 : (double)this.hitCount / (double)this.requestCount;
        }

        public double getMissRate() {
            return this.requestCount == 0L ? 0.0 : (double)this.missCount / (double)this.requestCount;
        }
    }
}

