/*
 * Decompiled with CFR 0.152.
 */
package io.trino.plugin.iceberg;

import com.google.common.base.Preconditions;
import com.google.common.base.Splitter;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.ImmutableMap;
import com.google.common.collect.ImmutableSet;
import io.airlift.units.DataSize;
import io.trino.orc.OrcDataSink;
import io.trino.orc.OrcDataSource;
import io.trino.orc.OrcDataSourceId;
import io.trino.orc.OrcReaderOptions;
import io.trino.orc.OrcWriterOptions;
import io.trino.orc.OrcWriterStats;
import io.trino.orc.OutputStreamOrcDataSink;
import io.trino.parquet.writer.ParquetWriterOptions;
import io.trino.plugin.hive.FileFormatDataSourceStats;
import io.trino.plugin.hive.HdfsEnvironment;
import io.trino.plugin.hive.NodeVersion;
import io.trino.plugin.hive.orc.HdfsOrcDataSource;
import io.trino.plugin.hive.orc.OrcWriterConfig;
import io.trino.plugin.iceberg.FileIoProvider;
import io.trino.plugin.iceberg.IcebergAvroFileWriter;
import io.trino.plugin.iceberg.IcebergErrorCode;
import io.trino.plugin.iceberg.IcebergFileFormat;
import io.trino.plugin.iceberg.IcebergFileWriter;
import io.trino.plugin.iceberg.IcebergOrcFileWriter;
import io.trino.plugin.iceberg.IcebergParquetFileWriter;
import io.trino.plugin.iceberg.IcebergSessionProperties;
import io.trino.plugin.iceberg.TypeConverter;
import io.trino.plugin.iceberg.util.PrimitiveTypeMapBuilder;
import io.trino.spi.ErrorCodeSupplier;
import io.trino.spi.StandardErrorCode;
import io.trino.spi.TrinoException;
import io.trino.spi.connector.ConnectorSession;
import io.trino.spi.type.TypeManager;
import java.io.IOException;
import java.io.OutputStream;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.Callable;
import java.util.function.Supplier;
import java.util.stream.IntStream;
import javax.inject.Inject;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.FSDataInputStream;
import org.apache.hadoop.fs.FileSystem;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.mapred.JobConf;
import org.apache.iceberg.MetricsConfig;
import org.apache.iceberg.Schema;
import org.apache.iceberg.io.DeleteSchemaUtil;
import org.apache.iceberg.io.FileIO;
import org.apache.iceberg.parquet.ParquetSchemaUtil;
import org.apache.iceberg.types.Types;
import org.weakref.jmx.Managed;

public class IcebergFileWriterFactory {
    private static final Schema POSITION_DELETE_SCHEMA = DeleteSchemaUtil.pathPosSchema();
    private static final MetricsConfig FULL_METRICS_CONFIG = MetricsConfig.fromProperties((Map)ImmutableMap.of((Object)"write.metadata.metrics.default", (Object)"full"));
    private static final Splitter COLUMN_NAMES_SPLITTER = Splitter.on((char)',').trimResults().omitEmptyStrings();
    private final HdfsEnvironment hdfsEnvironment;
    private final TypeManager typeManager;
    private final NodeVersion nodeVersion;
    private final FileFormatDataSourceStats readStats;
    private final OrcWriterStats orcWriterStats = new OrcWriterStats();
    private final OrcWriterOptions orcWriterOptions;

    @Inject
    public IcebergFileWriterFactory(HdfsEnvironment hdfsEnvironment, TypeManager typeManager, NodeVersion nodeVersion, FileFormatDataSourceStats readStats, OrcWriterConfig orcWriterConfig) {
        Preconditions.checkArgument((!Objects.requireNonNull(orcWriterConfig, "orcWriterConfig is null").isUseLegacyVersion() ? 1 : 0) != 0, (Object)"the ORC writer shouldn't be configured to use a legacy version");
        this.hdfsEnvironment = Objects.requireNonNull(hdfsEnvironment, "hdfsEnvironment is null");
        this.typeManager = Objects.requireNonNull(typeManager, "typeManager is null");
        this.nodeVersion = Objects.requireNonNull(nodeVersion, "nodeVersion is null");
        this.readStats = Objects.requireNonNull(readStats, "readStats is null");
        this.orcWriterOptions = orcWriterConfig.toOrcWriterOptions();
    }

    @Managed
    public OrcWriterStats getOrcWriterStats() {
        return this.orcWriterStats;
    }

    public IcebergFileWriter createDataFileWriter(Path outputPath, Schema icebergSchema, JobConf jobConf, ConnectorSession session, HdfsEnvironment.HdfsContext hdfsContext, FileIoProvider fileIoProvider, IcebergFileFormat fileFormat, MetricsConfig metricsConfig, Map<String, String> storageProperties) {
        switch (fileFormat) {
            case PARQUET: {
                return this.createParquetWriter(MetricsConfig.getDefault(), outputPath, icebergSchema, jobConf, session, hdfsContext);
            }
            case ORC: {
                return this.createOrcWriter(metricsConfig, outputPath, icebergSchema, jobConf, session, storageProperties, IcebergSessionProperties.getOrcStringStatisticsLimit(session));
            }
            case AVRO: {
                return this.createAvroWriter(fileIoProvider.createFileIo(hdfsContext, session.getQueryId()), outputPath, icebergSchema, jobConf, session);
            }
        }
        throw new TrinoException((ErrorCodeSupplier)StandardErrorCode.NOT_SUPPORTED, "File format not supported: " + fileFormat);
    }

    public IcebergFileWriter createPositionDeleteWriter(Path outputPath, JobConf jobConf, ConnectorSession session, HdfsEnvironment.HdfsContext hdfsContext, FileIoProvider fileIoProvider, IcebergFileFormat fileFormat, Map<String, String> storageProperties) {
        switch (fileFormat) {
            case PARQUET: {
                return this.createParquetWriter(FULL_METRICS_CONFIG, outputPath, POSITION_DELETE_SCHEMA, jobConf, session, hdfsContext);
            }
            case ORC: {
                return this.createOrcWriter(FULL_METRICS_CONFIG, outputPath, POSITION_DELETE_SCHEMA, jobConf, session, storageProperties, DataSize.ofBytes((long)Integer.MAX_VALUE));
            }
            case AVRO: {
                return this.createAvroWriter(fileIoProvider.createFileIo(hdfsContext, session.getQueryId()), outputPath, POSITION_DELETE_SCHEMA, jobConf, session);
            }
        }
        throw new TrinoException((ErrorCodeSupplier)StandardErrorCode.NOT_SUPPORTED, "File format not supported: " + fileFormat);
    }

    private IcebergFileWriter createParquetWriter(MetricsConfig metricsConfig, Path outputPath, Schema icebergSchema, JobConf jobConf, ConnectorSession session, HdfsEnvironment.HdfsContext hdfsContext) {
        List fileColumnNames = (List)icebergSchema.columns().stream().map(Types.NestedField::name).collect(ImmutableList.toImmutableList());
        List fileColumnTypes = (List)icebergSchema.columns().stream().map(column -> TypeConverter.toTrinoType(column.type(), this.typeManager)).collect(ImmutableList.toImmutableList());
        try {
            FileSystem fileSystem = this.hdfsEnvironment.getFileSystem(session.getIdentity(), outputPath, (Configuration)jobConf);
            Callable<Void> rollbackAction = () -> {
                fileSystem.delete(outputPath, false);
                return null;
            };
            ParquetWriterOptions parquetWriterOptions = ParquetWriterOptions.builder().setMaxPageSize(IcebergSessionProperties.getParquetWriterPageSize(session)).setMaxBlockSize(IcebergSessionProperties.getParquetWriterBlockSize(session)).setBatchSize(IcebergSessionProperties.getParquetWriterBatchSize(session)).build();
            return new IcebergParquetFileWriter(metricsConfig, (OutputStream)this.hdfsEnvironment.doAs(session.getIdentity(), () -> fileSystem.create(outputPath)), rollbackAction, fileColumnTypes, ParquetSchemaUtil.convert((Schema)icebergSchema, (String)"table"), PrimitiveTypeMapBuilder.makeTypeMap(fileColumnTypes, fileColumnNames), parquetWriterOptions, IntStream.range(0, fileColumnNames.size()).toArray(), IcebergSessionProperties.getCompressionCodec(session).getParquetCompressionCodec(), this.nodeVersion.toString(), outputPath, this.hdfsEnvironment, hdfsContext);
        }
        catch (IOException e) {
            throw new TrinoException((ErrorCodeSupplier)IcebergErrorCode.ICEBERG_WRITER_OPEN_ERROR, "Error creating Parquet file", (Throwable)e);
        }
    }

    private IcebergFileWriter createOrcWriter(MetricsConfig metricsConfig, Path outputPath, Schema icebergSchema, JobConf jobConf, ConnectorSession session, Map<String, String> storageProperties, DataSize stringStatisticsLimit) {
        try {
            FileSystem fileSystem = this.hdfsEnvironment.getFileSystem(session.getIdentity(), outputPath, (Configuration)jobConf);
            OrcDataSink orcDataSink = (OrcDataSink)this.hdfsEnvironment.doAs(session.getIdentity(), () -> new OutputStreamOrcDataSink((OutputStream)fileSystem.create(outputPath)));
            Callable<Void> rollbackAction = () -> {
                this.hdfsEnvironment.doAs(session.getIdentity(), () -> fileSystem.delete(outputPath, false));
                return null;
            };
            List columnFields = icebergSchema.columns();
            List fileColumnNames = (List)columnFields.stream().map(Types.NestedField::name).collect(ImmutableList.toImmutableList());
            List fileColumnTypes = (List)columnFields.stream().map(Types.NestedField::type).map(type -> TypeConverter.toTrinoType(type, this.typeManager)).collect(ImmutableList.toImmutableList());
            Optional<Supplier<OrcDataSource>> validationInputFactory = Optional.empty();
            if (IcebergSessionProperties.isOrcWriterValidate(session)) {
                validationInputFactory = Optional.of(() -> {
                    try {
                        return new HdfsOrcDataSource(new OrcDataSourceId(outputPath.toString()), ((Long)this.hdfsEnvironment.doAs(session.getIdentity(), () -> fileSystem.getFileStatus(outputPath).getLen())).longValue(), new OrcReaderOptions(), (FSDataInputStream)this.hdfsEnvironment.doAs(session.getIdentity(), () -> fileSystem.open(outputPath)), this.readStats);
                    }
                    catch (IOException e) {
                        throw new TrinoException((ErrorCodeSupplier)IcebergErrorCode.ICEBERG_WRITE_VALIDATION_FAILED, (Throwable)e);
                    }
                });
            }
            return new IcebergOrcFileWriter(metricsConfig, icebergSchema, orcDataSink, rollbackAction, fileColumnNames, fileColumnTypes, TypeConverter.toOrcType(icebergSchema), IcebergSessionProperties.getCompressionCodec(session).getOrcCompressionKind(), IcebergFileWriterFactory.withBloomFilterOptions(this.orcWriterOptions, storageProperties).withStripeMinSize(IcebergSessionProperties.getOrcWriterMinStripeSize(session)).withStripeMaxSize(IcebergSessionProperties.getOrcWriterMaxStripeSize(session)).withStripeMaxRowCount(IcebergSessionProperties.getOrcWriterMaxStripeRows(session)).withDictionaryMaxMemory(IcebergSessionProperties.getOrcWriterMaxDictionaryMemory(session)).withMaxStringStatisticsLimit(stringStatisticsLimit), IntStream.range(0, fileColumnNames.size()).toArray(), (Map<String, String>)ImmutableMap.builder().put((Object)"presto_version", (Object)this.nodeVersion.toString()).put((Object)"presto_query_id", (Object)session.getQueryId()).buildOrThrow(), validationInputFactory, IcebergSessionProperties.getOrcWriterValidateMode(session), this.orcWriterStats);
        }
        catch (IOException e) {
            throw new TrinoException((ErrorCodeSupplier)IcebergErrorCode.ICEBERG_WRITER_OPEN_ERROR, "Error creating ORC file", (Throwable)e);
        }
    }

    public static OrcWriterOptions withBloomFilterOptions(OrcWriterOptions orcWriterOptions, Map<String, String> storageProperties) {
        if (storageProperties.containsKey("orc.bloom.filter.columns")) {
            if (!storageProperties.containsKey("orc.bloom.filter.fpp")) {
                throw new TrinoException((ErrorCodeSupplier)IcebergErrorCode.ICEBERG_INVALID_METADATA, "FPP for Bloom filter is missing");
            }
            try {
                double fpp = Double.parseDouble(storageProperties.get("orc.bloom.filter.fpp"));
                return OrcWriterOptions.builderFrom((OrcWriterOptions)orcWriterOptions).setBloomFilterColumns((Set)ImmutableSet.copyOf((Collection)COLUMN_NAMES_SPLITTER.splitToList((CharSequence)storageProperties.get("orc.bloom.filter.columns")))).setBloomFilterFpp(fpp).build();
            }
            catch (NumberFormatException e) {
                throw new TrinoException((ErrorCodeSupplier)IcebergErrorCode.ICEBERG_INVALID_METADATA, String.format("Invalid value for %s property: %s", "orc_bloom_filter_fpp", storageProperties.get("orc.bloom.filter.fpp")));
            }
        }
        return orcWriterOptions;
    }

    private IcebergFileWriter createAvroWriter(FileIO fileIo, Path outputPath, Schema icebergSchema, JobConf jobConf, ConnectorSession session) {
        try {
            FileSystem fileSystem = this.hdfsEnvironment.getFileSystem(session.getIdentity(), outputPath, (Configuration)jobConf);
            Callable<Void> rollbackAction = () -> {
                fileSystem.delete(outputPath, false);
                return null;
            };
            List columnTypes = (List)icebergSchema.columns().stream().map(column -> TypeConverter.toTrinoType(column.type(), this.typeManager)).collect(ImmutableList.toImmutableList());
            return new IcebergAvroFileWriter(fileIo, outputPath, rollbackAction, icebergSchema, columnTypes, IcebergSessionProperties.getCompressionCodec(session));
        }
        catch (IOException e) {
            throw new TrinoException((ErrorCodeSupplier)IcebergErrorCode.ICEBERG_WRITER_OPEN_ERROR, "Error creating AVRO file", (Throwable)e);
        }
    }
}

