/*
 * Decompiled with CFR 0.152.
 */
package org.apache.iceberg.hadoop;

import java.io.Closeable;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.UncheckedIOException;
import java.nio.file.AccessDeniedException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import org.apache.hadoop.conf.Configurable;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.FileStatus;
import org.apache.hadoop.fs.FileSystem;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.fs.PathFilter;
import org.apache.hadoop.fs.RemoteIterator;
import org.apache.iceberg.BaseMetastoreCatalog;
import org.apache.iceberg.CatalogUtil;
import org.apache.iceberg.LockManager;
import org.apache.iceberg.Schema;
import org.apache.iceberg.TableMetadata;
import org.apache.iceberg.TableOperations;
import org.apache.iceberg.catalog.Catalog;
import org.apache.iceberg.catalog.Namespace;
import org.apache.iceberg.catalog.SupportsNamespaces;
import org.apache.iceberg.catalog.TableIdentifier;
import org.apache.iceberg.exceptions.AlreadyExistsException;
import org.apache.iceberg.exceptions.NamespaceNotEmptyException;
import org.apache.iceberg.exceptions.NoSuchNamespaceException;
import org.apache.iceberg.exceptions.NoSuchTableException;
import org.apache.iceberg.exceptions.RuntimeIOException;
import org.apache.iceberg.hadoop.HadoopFileIO;
import org.apache.iceberg.hadoop.HadoopTableOperations;
import org.apache.iceberg.hadoop.Util;
import org.apache.iceberg.io.CloseableGroup;
import org.apache.iceberg.io.FileIO;
import org.apache.iceberg.relocated.com.google.common.base.Joiner;
import org.apache.iceberg.relocated.com.google.common.base.MoreObjects;
import org.apache.iceberg.relocated.com.google.common.base.Preconditions;
import org.apache.iceberg.relocated.com.google.common.collect.ImmutableMap;
import org.apache.iceberg.relocated.com.google.common.collect.Lists;
import org.apache.iceberg.relocated.com.google.common.collect.Sets;
import org.apache.iceberg.util.LocationUtil;
import org.apache.iceberg.util.LockManagers;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class HadoopCatalog
extends BaseMetastoreCatalog
implements Closeable,
SupportsNamespaces,
Configurable {
    private static final Logger LOG = LoggerFactory.getLogger(HadoopCatalog.class);
    private static final String TABLE_METADATA_FILE_EXTENSION = ".metadata.json";
    private static final Joiner SLASH = Joiner.on((String)"/");
    private static final PathFilter TABLE_FILTER = path -> path.getName().endsWith(TABLE_METADATA_FILE_EXTENSION);
    private static final String HADOOP_SUPPRESS_PERMISSION_ERROR = "suppress-permission-error";
    private String catalogName;
    private Configuration conf;
    private CloseableGroup closeableGroup;
    private String warehouseLocation;
    private FileSystem fs;
    private FileIO fileIO;
    private LockManager lockManager;
    private boolean suppressPermissionError = false;
    private Map<String, String> catalogProperties;

    public HadoopCatalog() {
    }

    public void initialize(String name, Map<String, String> properties) {
        this.catalogProperties = ImmutableMap.copyOf(properties);
        String inputWarehouseLocation = properties.get("warehouse");
        Preconditions.checkArgument((inputWarehouseLocation != null && inputWarehouseLocation.length() > 0 ? 1 : 0) != 0, (Object)"Cannot initialize HadoopCatalog because warehousePath must not be null or empty");
        this.catalogName = name;
        this.warehouseLocation = LocationUtil.stripTrailingSlash(inputWarehouseLocation);
        this.fs = Util.getFs(new Path(this.warehouseLocation), this.conf);
        String fileIOImpl = properties.get("io-impl");
        this.fileIO = fileIOImpl == null ? new HadoopFileIO(this.conf) : CatalogUtil.loadFileIO(fileIOImpl, properties, this.conf);
        this.lockManager = LockManagers.from(properties);
        this.closeableGroup = new CloseableGroup();
        this.closeableGroup.addCloseable((AutoCloseable)this.lockManager);
        this.closeableGroup.setSuppressCloseFailure(true);
        this.suppressPermissionError = Boolean.parseBoolean(properties.get(HADOOP_SUPPRESS_PERMISSION_ERROR));
    }

    public HadoopCatalog(Configuration conf, String warehouseLocation) {
        this.setConf(conf);
        this.initialize("hadoop", (Map<String, String>)ImmutableMap.of((Object)"warehouse", (Object)warehouseLocation));
    }

    public String name() {
        return this.catalogName;
    }

    private boolean shouldSuppressPermissionError(IOException ioException) {
        if (this.suppressPermissionError) {
            return ioException instanceof AccessDeniedException || ioException.getMessage() != null && ioException.getMessage().contains("AuthorizationPermissionMismatch");
        }
        return false;
    }

    private boolean isTableDir(Path path) {
        Path metadataPath = new Path(path, "metadata");
        try {
            return this.fs.listStatus(metadataPath, TABLE_FILTER).length >= 1;
        }
        catch (FileNotFoundException e) {
            return false;
        }
        catch (IOException e) {
            if (this.shouldSuppressPermissionError(e)) {
                LOG.warn("Unable to list metadata directory {}: {}", (Object)metadataPath, (Object)e);
                return false;
            }
            throw new UncheckedIOException(e);
        }
    }

    private boolean isDirectory(Path path) {
        try {
            return this.fs.getFileStatus(path).isDirectory();
        }
        catch (FileNotFoundException e) {
            return false;
        }
        catch (IOException e) {
            if (this.shouldSuppressPermissionError(e)) {
                LOG.warn("Unable to list directory {}: {}", (Object)path, (Object)e);
                return false;
            }
            throw new UncheckedIOException(e);
        }
    }

    public List<TableIdentifier> listTables(Namespace namespace) {
        Preconditions.checkArgument((namespace.levels().length >= 1 ? 1 : 0) != 0, (String)"Missing database in table identifier: %s", (Object)namespace);
        Path nsPath = new Path(this.warehouseLocation, SLASH.join((Object[])namespace.levels()));
        HashSet tblIdents = Sets.newHashSet();
        try {
            if (!this.isDirectory(nsPath)) {
                throw new NoSuchNamespaceException("Namespace does not exist: %s", new Object[]{namespace});
            }
            RemoteIterator it = this.fs.listStatusIterator(nsPath);
            while (it.hasNext()) {
                Path path;
                FileStatus status = (FileStatus)it.next();
                if (!status.isDirectory() || !this.isTableDir(path = status.getPath())) continue;
                TableIdentifier tblIdent = TableIdentifier.of((Namespace)namespace, (String)path.getName());
                tblIdents.add(tblIdent);
            }
        }
        catch (IOException ioe) {
            throw new RuntimeIOException(ioe, "Failed to list tables under: %s", new Object[]{namespace});
        }
        return Lists.newArrayList((Iterable)tblIdents);
    }

    @Override
    protected boolean isValidIdentifier(TableIdentifier identifier) {
        return true;
    }

    @Override
    protected TableOperations newTableOps(TableIdentifier identifier) {
        return new HadoopTableOperations(new Path(this.defaultWarehouseLocation(identifier)), this.fileIO, this.conf, this.lockManager);
    }

    @Override
    protected String defaultWarehouseLocation(TableIdentifier tableIdentifier) {
        String tableName = tableIdentifier.name();
        StringBuilder sb = new StringBuilder();
        sb.append(this.warehouseLocation).append('/');
        for (String level : tableIdentifier.namespace().levels()) {
            sb.append(level).append('/');
        }
        sb.append(tableName);
        return sb.toString();
    }

    public boolean dropTable(TableIdentifier identifier, boolean purge) {
        if (!this.isValidIdentifier(identifier)) {
            throw new NoSuchTableException("Invalid identifier: %s", new Object[]{identifier});
        }
        Path tablePath = new Path(this.defaultWarehouseLocation(identifier));
        TableOperations ops = this.newTableOps(identifier);
        TableMetadata lastMetadata = ops.current();
        try {
            if (lastMetadata == null) {
                LOG.debug("Not an iceberg table: {}", (Object)identifier);
                return false;
            }
            if (purge) {
                CatalogUtil.dropTableData(ops.io(), lastMetadata);
            }
            return this.fs.delete(tablePath, true);
        }
        catch (IOException e) {
            throw new RuntimeIOException(e, "Failed to delete file: %s", new Object[]{tablePath});
        }
    }

    public void renameTable(TableIdentifier from, TableIdentifier to) {
        throw new UnsupportedOperationException("Cannot rename Hadoop tables");
    }

    public void createNamespace(Namespace namespace, Map<String, String> meta) {
        Preconditions.checkArgument((!namespace.isEmpty() ? 1 : 0) != 0, (String)"Cannot create namespace with invalid name: %s", (Object)namespace);
        if (!meta.isEmpty()) {
            throw new UnsupportedOperationException("Cannot create namespace " + namespace + ": metadata is not supported");
        }
        Path nsPath = new Path(this.warehouseLocation, SLASH.join((Object[])namespace.levels()));
        if (this.isNamespace(nsPath)) {
            throw new AlreadyExistsException("Namespace already exists: %s", new Object[]{namespace});
        }
        try {
            this.fs.mkdirs(nsPath);
        }
        catch (IOException e) {
            throw new RuntimeIOException(e, "Create namespace failed: %s", new Object[]{namespace});
        }
    }

    public List<Namespace> listNamespaces(Namespace namespace) {
        Path nsPath;
        Path path = nsPath = namespace.isEmpty() ? new Path(this.warehouseLocation) : new Path(this.warehouseLocation, SLASH.join((Object[])namespace.levels()));
        if (!this.isNamespace(nsPath)) {
            throw new NoSuchNamespaceException("Namespace does not exist: %s", new Object[]{namespace});
        }
        try {
            ArrayList namespaces = Lists.newArrayList();
            RemoteIterator it = this.fs.listStatusIterator(nsPath);
            while (it.hasNext()) {
                Path path2 = ((FileStatus)it.next()).getPath();
                if (!this.isNamespace(path2)) continue;
                namespaces.add(this.append(namespace, path2.getName()));
            }
            return namespaces;
        }
        catch (IOException ioe) {
            throw new RuntimeIOException(ioe, "Failed to list namespace under: %s", new Object[]{namespace});
        }
    }

    private Namespace append(Namespace ns, String name) {
        String[] levels = Arrays.copyOfRange(ns.levels(), 0, ns.levels().length + 1);
        levels[ns.levels().length] = name;
        return Namespace.of((String[])levels);
    }

    public boolean dropNamespace(Namespace namespace) {
        Path nsPath = new Path(this.warehouseLocation, SLASH.join((Object[])namespace.levels()));
        if (!this.isNamespace(nsPath) || namespace.isEmpty()) {
            return false;
        }
        try {
            if (this.fs.listStatusIterator(nsPath).hasNext()) {
                throw new NamespaceNotEmptyException("Namespace %s is not empty.", new Object[]{namespace});
            }
            return this.fs.delete(nsPath, false);
        }
        catch (IOException e) {
            throw new RuntimeIOException(e, "Namespace delete failed: %s", new Object[]{namespace});
        }
    }

    public boolean setProperties(Namespace namespace, Map<String, String> properties) {
        throw new UnsupportedOperationException("Cannot set namespace properties " + namespace + " : setProperties is not supported");
    }

    public boolean removeProperties(Namespace namespace, Set<String> properties) {
        throw new UnsupportedOperationException("Cannot remove properties " + namespace + " : removeProperties is not supported");
    }

    public Map<String, String> loadNamespaceMetadata(Namespace namespace) {
        Path nsPath = new Path(this.warehouseLocation, SLASH.join((Object[])namespace.levels()));
        if (!this.isNamespace(nsPath) || namespace.isEmpty()) {
            throw new NoSuchNamespaceException("Namespace does not exist: %s", new Object[]{namespace});
        }
        return ImmutableMap.of((Object)"location", (Object)nsPath.toString());
    }

    private boolean isNamespace(Path path) {
        return this.isDirectory(path) && !this.isTableDir(path);
    }

    @Override
    public void close() throws IOException {
        this.closeableGroup.close();
    }

    @Override
    public String toString() {
        return MoreObjects.toStringHelper((Object)this).add("name", (Object)this.catalogName).add("location", (Object)this.warehouseLocation).toString();
    }

    @Override
    public Catalog.TableBuilder buildTable(TableIdentifier identifier, Schema schema) {
        return new HadoopCatalogTableBuilder(identifier, schema);
    }

    public void setConf(Configuration conf) {
        this.conf = conf;
    }

    public Configuration getConf() {
        return this.conf;
    }

    @Override
    protected Map<String, String> properties() {
        return this.catalogProperties == null ? ImmutableMap.of() : this.catalogProperties;
    }

    private class HadoopCatalogTableBuilder
    extends BaseMetastoreCatalog.BaseMetastoreCatalogTableBuilder {
        private final String defaultLocation;

        private HadoopCatalogTableBuilder(TableIdentifier identifier, Schema schema) {
            super(identifier, schema);
            this.defaultLocation = HadoopCatalog.this.defaultWarehouseLocation(identifier);
        }

        @Override
        public Catalog.TableBuilder withLocation(String location) {
            Preconditions.checkArgument((location == null || location.equals(this.defaultLocation) ? 1 : 0) != 0, (Object)("Cannot set a custom location for a path-based table. Expected " + this.defaultLocation + " but got " + location));
            return this;
        }
    }
}

