/*
 * Decompiled with CFR 0.152.
 */
package org.apache.iceberg.rest.auth;

import com.fasterxml.jackson.core.JsonGenerator;
import com.fasterxml.jackson.databind.JsonNode;
import java.io.IOException;
import java.io.StringWriter;
import java.io.Writer;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicReference;
import java.util.regex.Pattern;
import org.apache.iceberg.exceptions.RuntimeIOException;
import org.apache.iceberg.relocated.com.google.common.base.Joiner;
import org.apache.iceberg.relocated.com.google.common.base.Preconditions;
import org.apache.iceberg.relocated.com.google.common.base.Splitter;
import org.apache.iceberg.relocated.com.google.common.collect.ImmutableList;
import org.apache.iceberg.relocated.com.google.common.collect.ImmutableMap;
import org.apache.iceberg.relocated.com.google.common.collect.Sets;
import org.apache.iceberg.rest.ErrorHandlers;
import org.apache.iceberg.rest.RESTClient;
import org.apache.iceberg.rest.RESTUtil;
import org.apache.iceberg.rest.ResourcePaths;
import org.apache.iceberg.rest.responses.OAuthTokenResponse;
import org.apache.iceberg.util.JsonUtil;
import org.apache.iceberg.util.Pair;
import org.apache.iceberg.util.Tasks;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class OAuth2Util {
    private static final Logger LOG = LoggerFactory.getLogger(OAuth2Util.class);
    private static final Pattern VALID_SCOPE_TOKEN = Pattern.compile("^[!-~&&[^\"\\\\]]+$");
    private static final Splitter SCOPE_DELIMITER = Splitter.on((String)" ");
    private static final Joiner SCOPE_JOINER = Joiner.on((String)" ");
    private static final String AUTHORIZATION_HEADER = "Authorization";
    private static final String BEARER_PREFIX = "Bearer ";
    private static final Splitter CREDENTIAL_SPLITTER = Splitter.on((String)":").limit(2).trimResults();
    private static final String GRANT_TYPE = "grant_type";
    private static final String CLIENT_CREDENTIALS = "client_credentials";
    private static final String TOKEN_EXCHANGE = "urn:ietf:params:oauth:grant-type:token-exchange";
    private static final String SCOPE = "scope";
    private static final String CATALOG = "catalog";
    private static final String CLIENT_ID = "client_id";
    private static final String CLIENT_SECRET = "client_secret";
    private static final String SUBJECT_TOKEN = "subject_token";
    private static final String SUBJECT_TOKEN_TYPE = "subject_token_type";
    private static final String ACTOR_TOKEN = "actor_token";
    private static final String ACTOR_TOKEN_TYPE = "actor_token_type";
    private static final Set<String> VALID_TOKEN_TYPES = Sets.newHashSet((Object[])new String[]{"urn:ietf:params:oauth:token-type:access_token", "urn:ietf:params:oauth:token-type:refresh_token", "urn:ietf:params:oauth:token-type:id_token", "urn:ietf:params:oauth:token-type:saml1", "urn:ietf:params:oauth:token-type:saml2", "urn:ietf:params:oauth:token-type:jwt"});
    private static final String ACCESS_TOKEN = "access_token";
    private static final String TOKEN_TYPE = "token_type";
    private static final String EXPIRES_IN = "expires_in";
    private static final String ISSUED_TOKEN_TYPE = "issued_token_type";
    private static final String REFRESH_TOKEN = "refresh_token";

    private OAuth2Util() {
    }

    public static Map<String, String> authHeaders(String token) {
        if (token != null) {
            return ImmutableMap.of((Object)AUTHORIZATION_HEADER, (Object)(BEARER_PREFIX + token));
        }
        return ImmutableMap.of();
    }

    public static boolean isValidScopeToken(String scopeToken) {
        return VALID_SCOPE_TOKEN.matcher(scopeToken).matches();
    }

    public static List<String> parseScope(String scope) {
        return SCOPE_DELIMITER.splitToList((CharSequence)scope);
    }

    public static String toScope(Iterable<String> scopes) {
        return SCOPE_JOINER.join(scopes);
    }

    private static OAuthTokenResponse refreshToken(RESTClient client, Map<String, String> headers, String subjectToken, String subjectTokenType, String scope) {
        Map<String, String> request = OAuth2Util.tokenExchangeRequest(subjectToken, subjectTokenType, (List<String>)(scope != null ? ImmutableList.of((Object)scope) : ImmutableList.of()));
        OAuthTokenResponse response = client.postForm(ResourcePaths.tokens(), request, OAuthTokenResponse.class, headers, ErrorHandlers.defaultErrorHandler());
        response.validate();
        return response;
    }

    public static OAuthTokenResponse exchangeToken(RESTClient client, Map<String, String> headers, String subjectToken, String subjectTokenType, String actorToken, String actorTokenType, String scope) {
        Map<String, String> request = OAuth2Util.tokenExchangeRequest(subjectToken, subjectTokenType, actorToken, actorTokenType, (List<String>)(scope != null ? ImmutableList.of((Object)scope) : ImmutableList.of()));
        OAuthTokenResponse response = client.postForm(ResourcePaths.tokens(), request, OAuthTokenResponse.class, headers, ErrorHandlers.defaultErrorHandler());
        response.validate();
        return response;
    }

    public static OAuthTokenResponse fetchToken(RESTClient client, Map<String, String> headers, String credential, String scope) {
        Map<String, String> request = OAuth2Util.clientCredentialsRequest(credential, (List<String>)(scope != null ? ImmutableList.of((Object)scope) : ImmutableList.of()));
        OAuthTokenResponse response = client.postForm(ResourcePaths.tokens(), request, OAuthTokenResponse.class, headers, ErrorHandlers.defaultErrorHandler());
        response.validate();
        return response;
    }

    private static Map<String, String> tokenExchangeRequest(String subjectToken, String subjectTokenType, List<String> scopes) {
        return OAuth2Util.tokenExchangeRequest(subjectToken, subjectTokenType, null, null, scopes);
    }

    private static Map<String, String> tokenExchangeRequest(String subjectToken, String subjectTokenType, String actorToken, String actorTokenType, List<String> scopes) {
        Preconditions.checkArgument((boolean)VALID_TOKEN_TYPES.contains(subjectTokenType), (String)"Invalid token type: %s", (Object)subjectTokenType);
        Preconditions.checkArgument((actorToken == null || VALID_TOKEN_TYPES.contains(actorTokenType) ? 1 : 0) != 0, (String)"Invalid token type: %s", (Object)actorTokenType);
        ImmutableMap.Builder formData = ImmutableMap.builder();
        formData.put((Object)GRANT_TYPE, (Object)TOKEN_EXCHANGE);
        formData.put((Object)SCOPE, (Object)OAuth2Util.toScope(scopes));
        formData.put((Object)SUBJECT_TOKEN, (Object)subjectToken);
        formData.put((Object)SUBJECT_TOKEN_TYPE, (Object)subjectTokenType);
        if (actorToken != null) {
            formData.put((Object)ACTOR_TOKEN, (Object)actorToken);
            formData.put((Object)ACTOR_TOKEN_TYPE, (Object)actorTokenType);
        }
        return formData.build();
    }

    private static Pair<String, String> parseCredential(String credential) {
        Preconditions.checkNotNull((Object)credential, (Object)"Invalid credential: null");
        List parts = CREDENTIAL_SPLITTER.splitToList((CharSequence)credential);
        switch (parts.size()) {
            case 2: {
                return Pair.of((String)parts.get(0), (String)parts.get(1));
            }
            case 1: {
                return Pair.of(null, (String)parts.get(0));
            }
        }
        throw new IllegalArgumentException("Invalid credential: " + credential);
    }

    private static Map<String, String> clientCredentialsRequest(String credential, List<String> scopes) {
        Pair<String, String> credentialPair = OAuth2Util.parseCredential(credential);
        return OAuth2Util.clientCredentialsRequest(credentialPair.first(), credentialPair.second(), scopes);
    }

    private static Map<String, String> clientCredentialsRequest(String clientId, String clientSecret, List<String> scopes) {
        ImmutableMap.Builder formData = ImmutableMap.builder();
        formData.put((Object)GRANT_TYPE, (Object)CLIENT_CREDENTIALS);
        if (clientId != null) {
            formData.put((Object)CLIENT_ID, (Object)clientId);
        }
        formData.put((Object)CLIENT_SECRET, (Object)clientSecret);
        formData.put((Object)SCOPE, (Object)OAuth2Util.toScope(scopes));
        return formData.build();
    }

    public static String tokenResponseToJson(OAuthTokenResponse response) {
        try {
            StringWriter writer = new StringWriter();
            JsonGenerator generator = JsonUtil.factory().createGenerator((Writer)writer);
            OAuth2Util.tokenResponseToJson(response, generator);
            generator.flush();
            return writer.toString();
        }
        catch (IOException e) {
            throw new RuntimeIOException(e);
        }
    }

    public static void tokenResponseToJson(OAuthTokenResponse response, JsonGenerator gen) throws IOException {
        response.validate();
        gen.writeStartObject();
        gen.writeStringField(ACCESS_TOKEN, response.token());
        gen.writeStringField(TOKEN_TYPE, response.tokenType());
        if (response.issuedTokenType() != null) {
            gen.writeStringField(ISSUED_TOKEN_TYPE, response.issuedTokenType());
        }
        if (response.expiresInSeconds() != null) {
            gen.writeNumberField(EXPIRES_IN, response.expiresInSeconds().intValue());
        }
        if (response.scopes() != null && !response.scopes().isEmpty()) {
            gen.writeStringField(SCOPE, OAuth2Util.toScope(response.scopes()));
        }
        gen.writeEndObject();
    }

    public static OAuthTokenResponse tokenResponseFromJson(String json) {
        try {
            return OAuth2Util.tokenResponseFromJson((JsonNode)JsonUtil.mapper().readValue(json, JsonNode.class));
        }
        catch (IOException e) {
            throw new RuntimeIOException(e);
        }
    }

    public static OAuthTokenResponse tokenResponseFromJson(JsonNode json) {
        Preconditions.checkArgument((boolean)json.isObject(), (String)"Cannot parse token response from non-object: %s", (Object)json);
        OAuthTokenResponse.Builder builder = OAuthTokenResponse.builder().withToken(JsonUtil.getString(ACCESS_TOKEN, json)).withTokenType(JsonUtil.getString(TOKEN_TYPE, json)).withIssuedTokenType(JsonUtil.getStringOrNull(ISSUED_TOKEN_TYPE, json));
        if (json.has(EXPIRES_IN)) {
            builder.setExpirationInSeconds(JsonUtil.getInt(EXPIRES_IN, json));
        }
        if (json.has(SCOPE)) {
            builder.addScopes(OAuth2Util.parseScope(JsonUtil.getString(SCOPE, json)));
        }
        return builder.build();
    }

    public static class AuthSession {
        private Map<String, String> headers;
        private String token;
        private String tokenType;
        private volatile boolean keepRefreshed = true;

        public AuthSession(Map<String, String> baseHeaders, String token, String tokenType) {
            this.headers = RESTUtil.merge(baseHeaders, OAuth2Util.authHeaders(token));
            this.token = token;
            this.tokenType = tokenType;
        }

        public Map<String, String> headers() {
            return this.headers;
        }

        public String token() {
            return this.token;
        }

        public String tokenType() {
            return this.tokenType;
        }

        public void stopRefreshing() {
            this.keepRefreshed = false;
        }

        public Pair<Integer, TimeUnit> refresh(RESTClient client) {
            if (this.token != null && this.keepRefreshed) {
                AtomicReference<Object> ref = new AtomicReference<Object>(null);
                boolean isSuccessful = Tasks.foreach(ref).suppressFailureWhenFinished().retry(5).onFailure((task, err) -> LOG.warn("Failed to refresh token", (Throwable)err)).exponentialBackoff(100L, 60000L, 1800000L, 2.0).run(holder -> holder.set(OAuth2Util.refreshToken(client, this.headers(), this.token, this.tokenType, OAuth2Util.CATALOG)));
                if (!isSuccessful || ref.get() == null) {
                    return null;
                }
                OAuthTokenResponse response = ref.get();
                this.token = response.token();
                this.tokenType = response.issuedTokenType();
                this.headers = RESTUtil.merge(this.headers, OAuth2Util.authHeaders(this.token));
                if (response.expiresInSeconds() != null) {
                    return Pair.of(response.expiresInSeconds(), TimeUnit.SECONDS);
                }
            }
            return null;
        }
    }
}

