/*
 * Decompiled with CFR 0.152.
 */
package io.trino.plugin.iceberg;

import com.google.common.base.Preconditions;
import com.google.common.collect.ImmutableList;
import io.airlift.slice.Slice;
import io.airlift.slice.Slices;
import io.trino.plugin.iceberg.util.Timestamps;
import io.trino.spi.ErrorCodeSupplier;
import io.trino.spi.Page;
import io.trino.spi.StandardErrorCode;
import io.trino.spi.TrinoException;
import io.trino.spi.block.ArrayBlockBuilder;
import io.trino.spi.block.Block;
import io.trino.spi.block.BlockBuilder;
import io.trino.spi.block.MapBlockBuilder;
import io.trino.spi.block.RowBlockBuilder;
import io.trino.spi.type.ArrayType;
import io.trino.spi.type.BigintType;
import io.trino.spi.type.BooleanType;
import io.trino.spi.type.DateType;
import io.trino.spi.type.DecimalType;
import io.trino.spi.type.Decimals;
import io.trino.spi.type.DoubleType;
import io.trino.spi.type.Int128;
import io.trino.spi.type.IntegerType;
import io.trino.spi.type.MapType;
import io.trino.spi.type.RealType;
import io.trino.spi.type.RowType;
import io.trino.spi.type.TimeType;
import io.trino.spi.type.TimestampType;
import io.trino.spi.type.TimestampWithTimeZoneType;
import io.trino.spi.type.UuidType;
import io.trino.spi.type.VarbinaryType;
import io.trino.spi.type.VarcharType;
import jakarta.annotation.Nullable;
import java.math.BigDecimal;
import java.math.BigInteger;
import java.nio.ByteBuffer;
import java.time.LocalDate;
import java.time.LocalDateTime;
import java.time.LocalTime;
import java.time.OffsetDateTime;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.UUID;
import java.util.stream.IntStream;
import org.apache.iceberg.Schema;
import org.apache.iceberg.data.GenericRecord;
import org.apache.iceberg.data.Record;
import org.apache.iceberg.types.Type;
import org.apache.iceberg.types.Types;
import org.apache.iceberg.util.DateTimeUtil;

public final class IcebergAvroDataConversion {
    private IcebergAvroDataConversion() {
    }

    public static Iterable<Record> toIcebergRecords(Page page, List<io.trino.spi.type.Type> types, Schema icebergSchema) {
        return () -> new RecordIterator(page, types, icebergSchema);
    }

    @Nullable
    public static Object toIcebergAvroObject(io.trino.spi.type.Type type, Type icebergType, Block block, int position) {
        if (block.isNull(position)) {
            return null;
        }
        if (type.equals(BooleanType.BOOLEAN)) {
            return BooleanType.BOOLEAN.getBoolean(block, position);
        }
        if (type.equals(IntegerType.INTEGER)) {
            return IntegerType.INTEGER.getInt(block, position);
        }
        if (type.equals(BigintType.BIGINT)) {
            return BigintType.BIGINT.getLong(block, position);
        }
        if (type.equals(RealType.REAL)) {
            return Float.valueOf(RealType.REAL.getFloat(block, position));
        }
        if (type.equals(DoubleType.DOUBLE)) {
            return DoubleType.DOUBLE.getDouble(block, position);
        }
        if (type instanceof DecimalType) {
            DecimalType decimalType = (DecimalType)type;
            return Decimals.readBigDecimal((DecimalType)decimalType, (Block)block, (int)position);
        }
        if (type instanceof VarcharType) {
            VarcharType varcharType = (VarcharType)type;
            return varcharType.getSlice(block, position).toStringUtf8();
        }
        if (type instanceof VarbinaryType) {
            VarbinaryType varbinaryType = (VarbinaryType)type;
            if (icebergType.typeId().equals((Object)Type.TypeID.FIXED)) {
                return varbinaryType.getSlice(block, position).getBytes();
            }
            return ByteBuffer.wrap(varbinaryType.getSlice(block, position).getBytes());
        }
        if (type.equals(DateType.DATE)) {
            int epochDays = DateType.DATE.getInt(block, position);
            return LocalDate.ofEpochDay(epochDays);
        }
        if (type.equals(TimeType.TIME_MICROS)) {
            long microsOfDay = TimeType.TIME_MICROS.getLong(block, position) / 1000000L;
            return DateTimeUtil.timeFromMicros((long)microsOfDay);
        }
        if (type.equals(TimestampType.TIMESTAMP_MICROS)) {
            long epochMicros = TimestampType.TIMESTAMP_MICROS.getLong(block, position);
            return DateTimeUtil.timestampFromMicros((long)epochMicros);
        }
        if (type.equals(TimestampWithTimeZoneType.TIMESTAMP_TZ_MICROS)) {
            long epochUtcMicros = Timestamps.timestampTzToMicros(Timestamps.getTimestampTz(block, position));
            return DateTimeUtil.timestamptzFromMicros((long)epochUtcMicros);
        }
        if (type.equals(UuidType.UUID)) {
            return UuidType.trinoUuidToJavaUuid((Slice)UuidType.UUID.getSlice(block, position));
        }
        if (type instanceof ArrayType) {
            ArrayType arrayType = (ArrayType)type;
            io.trino.spi.type.Type elementType = arrayType.getElementType();
            Type elementIcebergType = icebergType.asListType().elementType();
            Block arrayBlock = (Block)block.getObject(position, Block.class);
            ArrayList<Object> list = new ArrayList<Object>(arrayBlock.getPositionCount());
            for (int i = 0; i < arrayBlock.getPositionCount(); ++i) {
                Object element = IcebergAvroDataConversion.toIcebergAvroObject(elementType, elementIcebergType, arrayBlock, i);
                list.add(element);
            }
            return Collections.unmodifiableList(list);
        }
        if (type instanceof MapType) {
            MapType mapType = (MapType)type;
            io.trino.spi.type.Type keyType = mapType.getKeyType();
            io.trino.spi.type.Type valueType = mapType.getValueType();
            Type keyIcebergType = icebergType.asMapType().keyType();
            Type valueIcebergType = icebergType.asMapType().valueType();
            Block mapBlock = (Block)block.getObject(position, Block.class);
            HashMap<Object, Object> map = new HashMap<Object, Object>();
            for (int i = 0; i < mapBlock.getPositionCount(); i += 2) {
                Object key = IcebergAvroDataConversion.toIcebergAvroObject(keyType, keyIcebergType, mapBlock, i);
                Object value = IcebergAvroDataConversion.toIcebergAvroObject(valueType, valueIcebergType, mapBlock, i + 1);
                map.put(key, value);
            }
            return Collections.unmodifiableMap(map);
        }
        if (type instanceof RowType) {
            RowType rowType = (RowType)type;
            Block rowBlock = (Block)block.getObject(position, Block.class);
            List fieldTypes = rowType.getTypeParameters();
            Preconditions.checkArgument((fieldTypes.size() == rowBlock.getPositionCount() ? 1 : 0) != 0, (Object)"Expected row value field count does not match type field count");
            List icebergFields = icebergType.asStructType().fields();
            GenericRecord record = GenericRecord.create((Types.StructType)icebergType.asStructType());
            for (int i = 0; i < rowBlock.getPositionCount(); ++i) {
                Object element = IcebergAvroDataConversion.toIcebergAvroObject((io.trino.spi.type.Type)fieldTypes.get(i), ((Types.NestedField)icebergFields.get(i)).type(), rowBlock, i);
                record.set(i, element);
            }
            return record;
        }
        throw new TrinoException((ErrorCodeSupplier)StandardErrorCode.NOT_SUPPORTED, "unsupported type: " + type);
    }

    public static void serializeToTrinoBlock(io.trino.spi.type.Type type, Type icebergType, BlockBuilder builder, Object object) {
        if (object == null) {
            builder.appendNull();
            return;
        }
        if (type.equals(BooleanType.BOOLEAN)) {
            BooleanType.BOOLEAN.writeBoolean(builder, ((Boolean)object).booleanValue());
            return;
        }
        if (type.equals(IntegerType.INTEGER)) {
            IntegerType.INTEGER.writeLong(builder, (long)((Integer)object).intValue());
            return;
        }
        if (type.equals(BigintType.BIGINT)) {
            BigintType.BIGINT.writeLong(builder, ((Long)object).longValue());
            return;
        }
        if (type.equals(RealType.REAL)) {
            RealType.REAL.writeLong(builder, (long)Float.floatToRawIntBits(((Float)object).floatValue()));
            return;
        }
        if (type.equals(DoubleType.DOUBLE)) {
            DoubleType.DOUBLE.writeDouble(builder, ((Double)object).doubleValue());
            return;
        }
        if (type instanceof DecimalType) {
            DecimalType decimalType = (DecimalType)type;
            BigDecimal decimal = (BigDecimal)object;
            BigInteger unscaledValue = decimal.unscaledValue();
            if (decimalType.isShort()) {
                decimalType.writeLong(builder, unscaledValue.longValue());
            } else {
                decimalType.writeObject(builder, (Object)Int128.valueOf((BigInteger)unscaledValue));
            }
            return;
        }
        if (type instanceof VarcharType) {
            type.writeSlice(builder, Slices.utf8Slice((String)((String)object)));
            return;
        }
        if (type instanceof VarbinaryType) {
            if (icebergType.typeId().equals((Object)Type.TypeID.FIXED)) {
                VarbinaryType.VARBINARY.writeSlice(builder, Slices.wrappedBuffer((byte[])((byte[])object)));
            }
            VarbinaryType.VARBINARY.writeSlice(builder, Slices.wrappedHeapBuffer((ByteBuffer)((ByteBuffer)object)));
            return;
        }
        if (type.equals(DateType.DATE)) {
            DateType.DATE.writeLong(builder, ((LocalDate)object).toEpochDay());
            return;
        }
        if (type.equals(TimeType.TIME_MICROS)) {
            type.writeLong(builder, ((LocalTime)object).toNanoOfDay() * 1000L);
            return;
        }
        if (type.equals(TimestampType.TIMESTAMP_MICROS)) {
            long epochMicros = DateTimeUtil.microsFromTimestamp((LocalDateTime)((LocalDateTime)object));
            type.writeLong(builder, epochMicros);
            return;
        }
        if (type.equals(TimestampWithTimeZoneType.TIMESTAMP_TZ_MICROS)) {
            long epochUtcMicros = DateTimeUtil.microsFromTimestamptz((OffsetDateTime)((OffsetDateTime)object));
            type.writeObject(builder, (Object)Timestamps.timestampTzFromMicros(epochUtcMicros));
            return;
        }
        if (type.equals(UuidType.UUID)) {
            type.writeSlice(builder, UuidType.javaUuidToTrinoUuid((UUID)((UUID)object)));
            return;
        }
        if (type instanceof ArrayType) {
            Collection array = (Collection)object;
            io.trino.spi.type.Type elementType = ((ArrayType)type).getElementType();
            Type elementIcebergType = icebergType.asListType().elementType();
            ((ArrayBlockBuilder)builder).buildEntry(elementBuilder -> {
                for (Object element : array) {
                    IcebergAvroDataConversion.serializeToTrinoBlock(elementType, elementIcebergType, elementBuilder, element);
                }
            });
            return;
        }
        if (type instanceof MapType) {
            Map map = (Map)object;
            io.trino.spi.type.Type keyType = ((MapType)type).getKeyType();
            io.trino.spi.type.Type valueType = ((MapType)type).getValueType();
            Type keyIcebergType = icebergType.asMapType().keyType();
            Type valueIcebergType = icebergType.asMapType().valueType();
            ((MapBlockBuilder)builder).buildEntry((keyBuilder, valueBuilder) -> {
                for (Map.Entry entry : map.entrySet()) {
                    IcebergAvroDataConversion.serializeToTrinoBlock(keyType, keyIcebergType, keyBuilder, entry.getKey());
                    IcebergAvroDataConversion.serializeToTrinoBlock(valueType, valueIcebergType, valueBuilder, entry.getValue());
                }
            });
            return;
        }
        if (type instanceof RowType) {
            Record record = (Record)object;
            List typeParameters = type.getTypeParameters();
            List icebergFields = icebergType.asStructType().fields();
            ((RowBlockBuilder)builder).buildEntry(fieldBuilders -> {
                for (int i = 0; i < typeParameters.size(); ++i) {
                    IcebergAvroDataConversion.serializeToTrinoBlock((io.trino.spi.type.Type)typeParameters.get(i), ((Types.NestedField)icebergFields.get(i)).type(), (BlockBuilder)fieldBuilders.get(i), record.get(i));
                }
            });
            return;
        }
        throw new TrinoException((ErrorCodeSupplier)StandardErrorCode.NOT_SUPPORTED, "unsupported type: " + type);
    }

    private static class RecordIterator
    implements Iterator<Record> {
        private final List<Block> columnBlocks;
        private final List<io.trino.spi.type.Type> types;
        private final List<Type> icebergTypes;
        private final Schema icebergSchema;
        private final int positionCount;
        private int position;

        public RecordIterator(Page page, List<io.trino.spi.type.Type> types, Schema icebergSchema) {
            Objects.requireNonNull(page, "page is null");
            this.types = ImmutableList.copyOf((Collection)Objects.requireNonNull(types, "types is null"));
            this.icebergSchema = Objects.requireNonNull(icebergSchema, "icebergSchema is null");
            Preconditions.checkArgument((page.getChannelCount() == types.size() ? 1 : 0) != 0, (Object)"the page's channel count must be equal to the size of types");
            Preconditions.checkArgument((types.size() == icebergSchema.columns().size() ? 1 : 0) != 0, (Object)"the size of types must be equal to the number of columns in icebergSchema");
            this.icebergTypes = (List)icebergSchema.columns().stream().map(Types.NestedField::type).collect(ImmutableList.toImmutableList());
            this.columnBlocks = (List)IntStream.range(0, types.size()).mapToObj(arg_0 -> ((Page)page).getBlock(arg_0)).collect(ImmutableList.toImmutableList());
            this.positionCount = page.getPositionCount();
        }

        @Override
        public boolean hasNext() {
            return this.position < this.positionCount;
        }

        @Override
        public Record next() {
            GenericRecord record = GenericRecord.create((Schema)this.icebergSchema);
            for (int channel = 0; channel < this.types.size(); ++channel) {
                Object element = IcebergAvroDataConversion.toIcebergAvroObject(this.types.get(channel), this.icebergTypes.get(channel), this.columnBlocks.get(channel), this.position);
                record.set(channel, element);
            }
            ++this.position;
            return record;
        }
    }
}

