/*
 * Decompiled with CFR 0.152.
 */
package io.trino.plugin.hive.coercions;

import com.google.common.base.Preconditions;
import io.airlift.slice.Slice;
import io.airlift.slice.Slices;
import io.trino.plugin.hive.HiveErrorCode;
import io.trino.plugin.hive.coercions.TypeCoercer;
import io.trino.spi.ErrorCodeSupplier;
import io.trino.spi.TrinoException;
import io.trino.spi.block.Block;
import io.trino.spi.block.BlockBuilder;
import io.trino.spi.type.LongTimestamp;
import io.trino.spi.type.TimestampType;
import io.trino.spi.type.Timestamps;
import io.trino.spi.type.VarcharType;
import io.trino.spi.type.Varchars;
import java.time.LocalDate;
import java.time.LocalDateTime;
import java.time.ZoneOffset;
import java.time.chrono.IsoChronology;
import java.time.format.DateTimeFormatter;
import java.time.format.DateTimeFormatterBuilder;
import java.time.format.DateTimeParseException;
import java.time.format.ResolverStyle;

public final class TimestampCoercer {
    private static final DateTimeFormatter LOCAL_DATE_TIME = new DateTimeFormatterBuilder().parseCaseInsensitive().append(DateTimeFormatter.ISO_LOCAL_DATE).appendLiteral(' ').append(DateTimeFormatter.ISO_LOCAL_TIME).toFormatter().withResolverStyle(ResolverStyle.STRICT).withChronology(IsoChronology.INSTANCE);
    private static final long START_OF_MODERN_ERA_SECONDS = LocalDate.of(1900, 1, 1).toEpochDay() * 86400L;

    private TimestampCoercer() {
    }

    public static class VarcharToLongTimestampCoercer
    extends TypeCoercer<VarcharType, TimestampType> {
        public VarcharToLongTimestampCoercer(VarcharType fromType, TimestampType toType) {
            super(fromType, toType);
            Preconditions.checkArgument((!toType.isShort() ? 1 : 0) != 0, (Object)String.format("Precision must be in the range [%s, %s]", 7, 12));
        }

        @Override
        protected void applyCoercedValue(BlockBuilder blockBuilder, Block block, int position) {
            try {
                Slice value = ((VarcharType)this.fromType).getSlice(block, position);
                LocalDateTime dateTime = LOCAL_DATE_TIME.parse((CharSequence)value.toStringUtf8(), LocalDateTime::from);
                long epochSecond = dateTime.toEpochSecond(ZoneOffset.UTC);
                if (epochSecond < START_OF_MODERN_ERA_SECONDS) {
                    throw new TrinoException((ErrorCodeSupplier)HiveErrorCode.HIVE_INVALID_TIMESTAMP_COERCION, "Coercion on historical dates is not supported");
                }
                long epochMicros = epochSecond * 1000000L + (long)(dateTime.getNano() / 1000);
                int picosOfMicro = dateTime.getNano() % 1000 * 1000;
                ((TimestampType)this.toType).writeObject(blockBuilder, (Object)new LongTimestamp(epochMicros, picosOfMicro));
            }
            catch (DateTimeParseException ignored) {
                blockBuilder.appendNull();
            }
        }
    }

    public static class VarcharToShortTimestampCoercer
    extends TypeCoercer<VarcharType, TimestampType> {
        public VarcharToShortTimestampCoercer(VarcharType fromType, TimestampType toType) {
            super(fromType, toType);
            Preconditions.checkArgument((boolean)toType.isShort(), (Object)String.format("TIMESTAMP precision must be in range [0, %s]: %s", 12, toType.getPrecision()));
        }

        @Override
        protected void applyCoercedValue(BlockBuilder blockBuilder, Block block, int position) {
            try {
                Slice value = ((VarcharType)this.fromType).getSlice(block, position);
                LocalDateTime dateTime = LOCAL_DATE_TIME.parse((CharSequence)value.toStringUtf8(), LocalDateTime::from);
                long epochSecond = dateTime.toEpochSecond(ZoneOffset.UTC);
                if (epochSecond < START_OF_MODERN_ERA_SECONDS) {
                    throw new TrinoException((ErrorCodeSupplier)HiveErrorCode.HIVE_INVALID_TIMESTAMP_COERCION, "Coercion on historical dates is not supported");
                }
                long roundedNanos = Timestamps.round((long)dateTime.getNano(), (int)(9 - ((TimestampType)this.toType).getPrecision()));
                long epochMicros = epochSecond * 1000000L + Timestamps.roundDiv((long)roundedNanos, (long)1000L);
                ((TimestampType)this.toType).writeLong(blockBuilder, epochMicros);
            }
            catch (DateTimeParseException ignored) {
                blockBuilder.appendNull();
            }
        }
    }

    public static class LongTimestampToVarcharCoercer
    extends TypeCoercer<TimestampType, VarcharType> {
        public LongTimestampToVarcharCoercer(TimestampType fromType, VarcharType toType) {
            super(fromType, toType);
        }

        @Override
        protected void applyCoercedValue(BlockBuilder blockBuilder, Block block, int position) {
            LongTimestamp timestamp = (LongTimestamp)((TimestampType)this.fromType).getObject(block, position);
            long epochSecond = Math.floorDiv(timestamp.getEpochMicros(), 1000000);
            long microsFraction = Math.floorMod(timestamp.getEpochMicros(), 1000000);
            long nanosFraction = microsFraction * 1000L + (long)(timestamp.getPicosOfMicro() / 1000);
            if (epochSecond < START_OF_MODERN_ERA_SECONDS) {
                throw new TrinoException((ErrorCodeSupplier)HiveErrorCode.HIVE_INVALID_TIMESTAMP_COERCION, "Coercion on historical dates is not supported");
            }
            ((VarcharType)this.toType).writeSlice(blockBuilder, Varchars.truncateToLength((Slice)Slices.utf8Slice((String)LOCAL_DATE_TIME.format(LocalDateTime.ofEpochSecond(epochSecond, Math.toIntExact(nanosFraction), ZoneOffset.UTC))), (VarcharType)((VarcharType)this.toType)));
        }
    }
}

