/*
 * Decompiled with CFR 0.152.
 */
package io.trino.plugin.iceberg;

import com.google.common.base.Preconditions;
import com.google.common.base.Verify;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.Streams;
import io.airlift.json.JsonCodec;
import io.airlift.slice.Slice;
import io.airlift.slice.Slices;
import io.airlift.units.DataSize;
import io.trino.filesystem.Location;
import io.trino.filesystem.TrinoFileSystem;
import io.trino.plugin.iceberg.CommitTaskData;
import io.trino.plugin.iceberg.IcebergColumnHandle;
import io.trino.plugin.iceberg.IcebergErrorCode;
import io.trino.plugin.iceberg.IcebergFileFormat;
import io.trino.plugin.iceberg.IcebergFileWriter;
import io.trino.plugin.iceberg.IcebergFileWriterFactory;
import io.trino.plugin.iceberg.IcebergSessionProperties;
import io.trino.plugin.iceberg.IcebergSortingFileWriter;
import io.trino.plugin.iceberg.IcebergUtil;
import io.trino.plugin.iceberg.MetricsWrapper;
import io.trino.plugin.iceberg.PartitionData;
import io.trino.plugin.iceberg.PartitionTransforms;
import io.trino.plugin.iceberg.TrinoSortField;
import io.trino.plugin.iceberg.util.Timestamps;
import io.trino.spi.ErrorCodeSupplier;
import io.trino.spi.Page;
import io.trino.spi.PageIndexer;
import io.trino.spi.PageIndexerFactory;
import io.trino.spi.PageSorter;
import io.trino.spi.TrinoException;
import io.trino.spi.block.Block;
import io.trino.spi.connector.ConnectorPageSink;
import io.trino.spi.connector.ConnectorSession;
import io.trino.spi.connector.SortOrder;
import io.trino.spi.type.BigintType;
import io.trino.spi.type.BooleanType;
import io.trino.spi.type.DateType;
import io.trino.spi.type.DecimalType;
import io.trino.spi.type.Decimals;
import io.trino.spi.type.DoubleType;
import io.trino.spi.type.IntegerType;
import io.trino.spi.type.RealType;
import io.trino.spi.type.SmallintType;
import io.trino.spi.type.TimeType;
import io.trino.spi.type.TimestampType;
import io.trino.spi.type.TimestampWithTimeZoneType;
import io.trino.spi.type.TinyintType;
import io.trino.spi.type.TypeManager;
import io.trino.spi.type.UuidType;
import io.trino.spi.type.VarbinaryType;
import io.trino.spi.type.VarcharType;
import java.io.Closeable;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.UUID;
import java.util.concurrent.CompletableFuture;
import java.util.function.Function;
import java.util.stream.Stream;
import org.apache.iceberg.FileContent;
import org.apache.iceberg.MetricsConfig;
import org.apache.iceberg.PartitionField;
import org.apache.iceberg.PartitionSpec;
import org.apache.iceberg.PartitionSpecParser;
import org.apache.iceberg.Schema;
import org.apache.iceberg.StructLike;
import org.apache.iceberg.io.LocationProvider;
import org.apache.iceberg.transforms.Transform;
import org.apache.iceberg.types.Type;
import org.apache.iceberg.types.Types;

public class IcebergPageSink
implements ConnectorPageSink {
    private static final int MAX_PAGE_POSITIONS = 4096;
    private final int maxOpenWriters;
    private final Schema outputSchema;
    private final PartitionSpec partitionSpec;
    private final LocationProvider locationProvider;
    private final IcebergFileWriterFactory fileWriterFactory;
    private final TrinoFileSystem fileSystem;
    private final JsonCodec<CommitTaskData> jsonCodec;
    private final ConnectorSession session;
    private final IcebergFileFormat fileFormat;
    private final MetricsConfig metricsConfig;
    private final PagePartitioner pagePartitioner;
    private final long targetMaxFileSize;
    private final Map<String, String> storageProperties;
    private final List<TrinoSortField> sortOrder;
    private final boolean sortedWritingEnabled;
    private final DataSize sortingFileWriterBufferSize;
    private final Integer sortingFileWriterMaxOpenFiles;
    private final Location tempDirectory;
    private final TypeManager typeManager;
    private final PageSorter pageSorter;
    private final List<io.trino.spi.type.Type> columnTypes;
    private final List<Integer> sortColumnIndexes;
    private final List<SortOrder> sortOrders;
    private final List<WriteContext> writers = new ArrayList<WriteContext>();
    private final List<Closeable> closedWriterRollbackActions = new ArrayList<Closeable>();
    private final Collection<Slice> commitTasks = new ArrayList<Slice>();
    private long writtenBytes;
    private long memoryUsage;
    private long validationCpuNanos;

    public IcebergPageSink(Schema outputSchema, PartitionSpec partitionSpec, LocationProvider locationProvider, IcebergFileWriterFactory fileWriterFactory, PageIndexerFactory pageIndexerFactory, TrinoFileSystem fileSystem, List<IcebergColumnHandle> inputColumns, JsonCodec<CommitTaskData> jsonCodec, ConnectorSession session, IcebergFileFormat fileFormat, Map<String, String> storageProperties, int maxOpenWriters, List<TrinoSortField> sortOrder, DataSize sortingFileWriterBufferSize, int sortingFileWriterMaxOpenFiles, TypeManager typeManager, PageSorter pageSorter) {
        Objects.requireNonNull(inputColumns, "inputColumns is null");
        this.outputSchema = Objects.requireNonNull(outputSchema, "outputSchema is null");
        this.partitionSpec = Objects.requireNonNull(partitionSpec, "partitionSpec is null");
        this.locationProvider = Objects.requireNonNull(locationProvider, "locationProvider is null");
        this.fileWriterFactory = Objects.requireNonNull(fileWriterFactory, "fileWriterFactory is null");
        this.fileSystem = Objects.requireNonNull(fileSystem, "fileSystem is null");
        this.jsonCodec = Objects.requireNonNull(jsonCodec, "jsonCodec is null");
        this.session = Objects.requireNonNull(session, "session is null");
        this.fileFormat = Objects.requireNonNull(fileFormat, "fileFormat is null");
        this.metricsConfig = MetricsConfig.fromProperties(Objects.requireNonNull(storageProperties, "storageProperties is null"));
        this.maxOpenWriters = maxOpenWriters;
        this.pagePartitioner = new PagePartitioner(pageIndexerFactory, IcebergPageSink.toPartitionColumns(inputColumns, partitionSpec));
        this.targetMaxFileSize = IcebergSessionProperties.getTargetMaxFileSize(session);
        this.storageProperties = Objects.requireNonNull(storageProperties, "storageProperties is null");
        this.sortOrder = Objects.requireNonNull(sortOrder, "sortOrder is null");
        this.sortedWritingEnabled = IcebergSessionProperties.isSortedWritingEnabled(session);
        this.sortingFileWriterBufferSize = Objects.requireNonNull(sortingFileWriterBufferSize, "sortingFileWriterBufferSize is null");
        this.sortingFileWriterMaxOpenFiles = sortingFileWriterMaxOpenFiles;
        this.tempDirectory = Location.of((String)locationProvider.newDataLocation("trino-tmp-files"));
        this.typeManager = Objects.requireNonNull(typeManager, "typeManager is null");
        this.pageSorter = Objects.requireNonNull(pageSorter, "pageSorter is null");
        this.columnTypes = (List)IcebergUtil.getColumns(outputSchema, typeManager).stream().map(IcebergColumnHandle::getType).collect(ImmutableList.toImmutableList());
        if (this.sortedWritingEnabled) {
            ImmutableList.Builder sortColumnIndexes = ImmutableList.builder();
            ImmutableList.Builder sortOrders = ImmutableList.builder();
            for (TrinoSortField sortField : sortOrder) {
                Types.NestedField column = outputSchema.findField(sortField.getSourceColumnId());
                if (column == null) {
                    throw new TrinoException((ErrorCodeSupplier)IcebergErrorCode.ICEBERG_INVALID_METADATA, "Unable to find sort field source column in the table schema: " + sortField);
                }
                sortColumnIndexes.add((Object)outputSchema.columns().indexOf(column));
                sortOrders.add((Object)sortField.getSortOrder());
            }
            this.sortColumnIndexes = sortColumnIndexes.build();
            this.sortOrders = sortOrders.build();
        } else {
            this.sortColumnIndexes = ImmutableList.of();
            this.sortOrders = ImmutableList.of();
        }
    }

    public long getCompletedBytes() {
        return this.writtenBytes;
    }

    public long getMemoryUsage() {
        return this.memoryUsage;
    }

    public long getValidationCpuNanos() {
        return this.validationCpuNanos;
    }

    public CompletableFuture<?> appendPage(Page page) {
        this.doAppend(page);
        return NOT_BLOCKED;
    }

    public CompletableFuture<Collection<Slice>> finish() {
        for (int writerIndex = 0; writerIndex < this.writers.size(); ++writerIndex) {
            this.closeWriter(writerIndex);
        }
        this.writers.clear();
        return CompletableFuture.completedFuture(this.commitTasks);
    }

    public void abort() {
        List rollbackActions = (List)Streams.concat((Stream[])new Stream[]{this.writers.stream().filter(Objects::nonNull).map(writer -> writer::rollback), this.closedWriterRollbackActions.stream()}).collect(ImmutableList.toImmutableList());
        Throwable error = null;
        for (Closeable rollbackAction : rollbackActions) {
            try {
                rollbackAction.close();
            }
            catch (Throwable t) {
                if (error == null) {
                    error = new RuntimeException("Exception during rollback");
                }
                error.addSuppressed(t);
            }
        }
        if (error != null) {
            throw error;
        }
    }

    private void doAppend(Page page) {
        Page chunk;
        for (int writeOffset = 0; writeOffset < page.getPositionCount(); writeOffset += chunk.getPositionCount()) {
            chunk = page.getRegion(writeOffset, Math.min(page.getPositionCount() - writeOffset, 4096));
            this.writePage(chunk);
        }
    }

    private void writePage(Page page) {
        int index;
        int[] writerIndexes = this.getWriterIndexes(page);
        int[] sizes = new int[this.writers.size()];
        int[] nArray = writerIndexes;
        int n = nArray.length;
        for (int i = 0; i < n; ++i) {
            int n2 = index = nArray[i];
            sizes[n2] = sizes[n2] + 1;
        }
        int[][] writerPositions = new int[this.writers.size()][];
        int[] counts = new int[this.writers.size()];
        int position = 0;
        while (position < page.getPositionCount()) {
            index = writerIndexes[position];
            int count = counts[index];
            if (count == 0) {
                writerPositions[index] = new int[sizes[index]];
            }
            writerPositions[index][count] = position++;
            int n3 = index;
            counts[n3] = counts[n3] + 1;
        }
        for (int index2 = 0; index2 < writerPositions.length; ++index2) {
            int[] positions = writerPositions[index2];
            if (positions == null) continue;
            Page pageForWriter = page;
            if (positions.length != page.getPositionCount()) {
                Verify.verify((positions.length == counts[index2] ? 1 : 0) != 0);
                pageForWriter = pageForWriter.getPositions(positions, 0, positions.length);
            }
            IcebergFileWriter writer = this.writers.get(index2).getWriter();
            long currentWritten = writer.getWrittenBytes();
            long currentMemory = writer.getMemoryUsage();
            writer.appendRows(pageForWriter);
            this.writtenBytes += writer.getWrittenBytes() - currentWritten;
            this.memoryUsage += writer.getMemoryUsage() - currentMemory;
        }
    }

    private int[] getWriterIndexes(Page page) {
        int[] writerIndexes = this.pagePartitioner.partitionPage(page);
        if (this.pagePartitioner.getMaxIndex() >= this.maxOpenWriters) {
            throw new TrinoException((ErrorCodeSupplier)IcebergErrorCode.ICEBERG_TOO_MANY_OPEN_PARTITIONS, String.format("Exceeded limit of %s open writers for partitions", this.maxOpenWriters));
        }
        while (this.writers.size() <= this.pagePartitioner.getMaxIndex()) {
            this.writers.add(null);
        }
        for (int position = 0; position < page.getPositionCount(); ++position) {
            int writerIndex = writerIndexes[position];
            WriteContext writer = this.writers.get(writerIndex);
            if (writer != null) {
                if (writer.getWrittenBytes() <= this.targetMaxFileSize) continue;
                this.closeWriter(writerIndex);
            }
            Optional<PartitionData> partitionData = this.getPartitionData(this.pagePartitioner.getColumns(), page, position);
            String fileName = this.fileFormat.toIceberg().addExtension(this.session.getQueryId() + "-" + UUID.randomUUID());
            String outputPath = partitionData.map(partition -> this.locationProvider.newDataLocation(this.partitionSpec, (StructLike)partition, fileName)).orElseGet(() -> this.locationProvider.newDataLocation(fileName));
            if (!this.sortOrder.isEmpty() && this.sortedWritingEnabled) {
                String tempName = "sorting-file-writer-%s-%s".formatted(this.session.getQueryId(), UUID.randomUUID());
                Location tempFilePrefix = this.tempDirectory.appendPath(tempName);
                WriteContext writerContext = this.createWriter(outputPath, partitionData);
                IcebergSortingFileWriter sortedFileWriter = new IcebergSortingFileWriter(this.fileSystem, tempFilePrefix, writerContext.getWriter(), this.sortingFileWriterBufferSize, this.sortingFileWriterMaxOpenFiles, this.columnTypes, this.sortColumnIndexes, this.sortOrders, this.pageSorter, this.typeManager.getTypeOperators());
                writer = new WriteContext(sortedFileWriter, outputPath, partitionData);
            } else {
                writer = this.createWriter(outputPath, partitionData);
            }
            this.writers.set(writerIndex, writer);
            this.memoryUsage += writer.getWriter().getMemoryUsage();
        }
        Verify.verify((this.writers.size() == this.pagePartitioner.getMaxIndex() + 1 ? 1 : 0) != 0);
        Verify.verify((!this.writers.contains(null) ? 1 : 0) != 0);
        return writerIndexes;
    }

    private void closeWriter(int writerIndex) {
        WriteContext writeContext = this.writers.get(writerIndex);
        IcebergFileWriter writer = writeContext.getWriter();
        long currentWritten = writer.getWrittenBytes();
        long currentMemory = writer.getMemoryUsage();
        this.closedWriterRollbackActions.add(writer.commit());
        this.validationCpuNanos += writer.getValidationCpuNanos();
        this.writtenBytes += writer.getWrittenBytes() - currentWritten;
        this.memoryUsage -= currentMemory;
        this.writers.set(writerIndex, null);
        CommitTaskData task = new CommitTaskData(writeContext.getPath(), this.fileFormat, writer.getWrittenBytes(), new MetricsWrapper(writer.getMetrics()), PartitionSpecParser.toJson((PartitionSpec)this.partitionSpec), writeContext.getPartitionData().map(PartitionData::toJson), FileContent.DATA, Optional.empty());
        this.commitTasks.add(Slices.wrappedBuffer((byte[])this.jsonCodec.toJsonBytes((Object)task)));
    }

    private WriteContext createWriter(String outputPath, Optional<PartitionData> partitionData) {
        IcebergFileWriter writer = this.fileWriterFactory.createDataFileWriter(this.fileSystem, Location.of((String)outputPath), this.outputSchema, this.session, this.fileFormat, this.metricsConfig, this.storageProperties);
        return new WriteContext(writer, outputPath, partitionData);
    }

    private Optional<PartitionData> getPartitionData(List<PartitionColumn> columns, Page page, int position) {
        if (columns.isEmpty()) {
            return Optional.empty();
        }
        Object[] values = new Object[columns.size()];
        for (int i = 0; i < columns.size(); ++i) {
            PartitionColumn column = columns.get(i);
            Block block = page.getBlock(column.getSourceChannel());
            io.trino.spi.type.Type type = column.getSourceType();
            Type icebergType = this.outputSchema.findType(column.getField().sourceId());
            Object value = IcebergPageSink.getIcebergValue(block, position, type);
            values[i] = IcebergPageSink.applyTransform(column.getField().transform(), icebergType, value);
        }
        return Optional.of(new PartitionData(values));
    }

    private static Object applyTransform(Transform<?, ?> transform, Type icebergType, Object value) {
        return transform.bind(icebergType).apply(value);
    }

    public static Object getIcebergValue(Block block, int position, io.trino.spi.type.Type type) {
        if (block.isNull(position)) {
            return null;
        }
        if (type.equals(BigintType.BIGINT)) {
            return BigintType.BIGINT.getLong(block, position);
        }
        if (type.equals(TinyintType.TINYINT)) {
            return (int)TinyintType.TINYINT.getByte(block, position);
        }
        if (type.equals(SmallintType.SMALLINT)) {
            return (int)SmallintType.SMALLINT.getShort(block, position);
        }
        if (type.equals(IntegerType.INTEGER)) {
            return IntegerType.INTEGER.getInt(block, position);
        }
        if (type.equals(DateType.DATE)) {
            return DateType.DATE.getInt(block, position);
        }
        if (type.equals(BooleanType.BOOLEAN)) {
            return BooleanType.BOOLEAN.getBoolean(block, position);
        }
        if (type instanceof DecimalType) {
            DecimalType decimalType = (DecimalType)type;
            return Decimals.readBigDecimal((DecimalType)decimalType, (Block)block, (int)position);
        }
        if (type.equals(RealType.REAL)) {
            return Float.valueOf(RealType.REAL.getFloat(block, position));
        }
        if (type.equals(DoubleType.DOUBLE)) {
            return DoubleType.DOUBLE.getDouble(block, position);
        }
        if (type.equals(TimeType.TIME_MICROS)) {
            return TimeType.TIME_MICROS.getLong(block, position) / 1000000L;
        }
        if (type.equals(TimestampType.TIMESTAMP_MICROS)) {
            return TimestampType.TIMESTAMP_MICROS.getLong(block, position);
        }
        if (type.equals(TimestampWithTimeZoneType.TIMESTAMP_TZ_MICROS)) {
            return Timestamps.timestampTzToMicros(Timestamps.getTimestampTz(block, position));
        }
        if (type instanceof VarbinaryType) {
            VarbinaryType varbinaryType = (VarbinaryType)type;
            return varbinaryType.getSlice(block, position).getBytes();
        }
        if (type instanceof VarcharType) {
            VarcharType varcharType = (VarcharType)type;
            return varcharType.getSlice(block, position).toStringUtf8();
        }
        if (type.equals(UuidType.UUID)) {
            return UuidType.trinoUuidToJavaUuid((Slice)UuidType.UUID.getSlice(block, position));
        }
        throw new UnsupportedOperationException("Type not supported as partition column: " + type.getDisplayName());
    }

    private static List<PartitionColumn> toPartitionColumns(List<IcebergColumnHandle> handles, PartitionSpec partitionSpec) {
        HashMap<Integer, Integer> idChannels = new HashMap<Integer, Integer>();
        for (int i = 0; i < handles.size(); ++i) {
            idChannels.put(handles.get(i).getId(), i);
        }
        return (List)partitionSpec.fields().stream().map(field -> {
            Integer channel = (Integer)idChannels.get(field.sourceId());
            Preconditions.checkArgument((channel != null ? 1 : 0) != 0, (String)"partition field not found: %s", (Object)field);
            io.trino.spi.type.Type inputType = ((IcebergColumnHandle)handles.get(channel)).getType();
            PartitionTransforms.ColumnTransform transform = PartitionTransforms.getColumnTransform(field, inputType);
            return new PartitionColumn((PartitionField)field, channel, inputType, transform.getType(), transform.getBlockTransform());
        }).collect(ImmutableList.toImmutableList());
    }

    private static class PagePartitioner {
        private final PageIndexer pageIndexer;
        private final List<PartitionColumn> columns;

        public PagePartitioner(PageIndexerFactory pageIndexerFactory, List<PartitionColumn> columns) {
            this.pageIndexer = pageIndexerFactory.createPageIndexer((List)columns.stream().map(PartitionColumn::getResultType).collect(ImmutableList.toImmutableList()));
            this.columns = ImmutableList.copyOf(columns);
        }

        public int[] partitionPage(Page page) {
            Block[] blocks = new Block[this.columns.size()];
            for (int i = 0; i < this.columns.size(); ++i) {
                PartitionColumn column = this.columns.get(i);
                Block block = page.getBlock(column.getSourceChannel());
                blocks[i] = column.getBlockTransform().apply(block);
            }
            Page transformed = new Page(page.getPositionCount(), blocks);
            return this.pageIndexer.indexPage(transformed);
        }

        public int getMaxIndex() {
            return this.pageIndexer.getMaxIndex();
        }

        public List<PartitionColumn> getColumns() {
            return this.columns;
        }
    }

    private static class WriteContext {
        private final IcebergFileWriter writer;
        private final String path;
        private final Optional<PartitionData> partitionData;

        public WriteContext(IcebergFileWriter writer, String path, Optional<PartitionData> partitionData) {
            this.writer = Objects.requireNonNull(writer, "writer is null");
            this.path = Objects.requireNonNull(path, "path is null");
            this.partitionData = Objects.requireNonNull(partitionData, "partitionData is null");
        }

        public IcebergFileWriter getWriter() {
            return this.writer;
        }

        public String getPath() {
            return this.path;
        }

        public Optional<PartitionData> getPartitionData() {
            return this.partitionData;
        }

        public long getWrittenBytes() {
            return this.writer.getWrittenBytes();
        }

        public void rollback() {
            this.writer.rollback();
        }
    }

    private static class PartitionColumn {
        private final PartitionField field;
        private final int sourceChannel;
        private final io.trino.spi.type.Type sourceType;
        private final io.trino.spi.type.Type resultType;
        private final Function<Block, Block> blockTransform;

        public PartitionColumn(PartitionField field, int sourceChannel, io.trino.spi.type.Type sourceType, io.trino.spi.type.Type resultType, Function<Block, Block> blockTransform) {
            this.field = Objects.requireNonNull(field, "field is null");
            this.sourceChannel = sourceChannel;
            this.sourceType = Objects.requireNonNull(sourceType, "sourceType is null");
            this.resultType = Objects.requireNonNull(resultType, "resultType is null");
            this.blockTransform = Objects.requireNonNull(blockTransform, "blockTransform is null");
        }

        public PartitionField getField() {
            return this.field;
        }

        public int getSourceChannel() {
            return this.sourceChannel;
        }

        public io.trino.spi.type.Type getSourceType() {
            return this.sourceType;
        }

        public io.trino.spi.type.Type getResultType() {
            return this.resultType;
        }

        public Function<Block, Block> getBlockTransform() {
            return this.blockTransform;
        }
    }
}

